package jp.co.sra.jun.goodies.track;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Point;
import java.awt.Rectangle;

import jp.co.sra.smalltalk.DependentEvent;
import jp.co.sra.smalltalk.StController;
import jp.co.sra.smalltalk.StModel;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StSymbol;

import jp.co.sra.jun.system.framework.JunAbstractViewCanvas;

/**
 * JunTrackSliderViewAwt class
 * 
 *  @author    MATSUDA Ryouichi
 *  @created   1998/11/19 (by MATSUDA Ryouichi)
 *  @updated   1999/12/10 (by MATSUDA Ryouichi)
 *  @updated   2002/10/28 (by nisinaka)
 *  @updated   2003/03/24 (by nisinaka)
 *  @updated   2004/09/21 (by nisinaka)
 *  @updated   2005/03/03 (by nisinaka)
 *  @updated   2006/04/20 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on Jun519 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunTrackSliderViewAwt.java,v 8.10 2008/02/20 06:32:04 nisinaka Exp $
 */
public class JunTrackSliderViewAwt extends JunAbstractViewCanvas implements JunTrackSliderView {
	protected StRectangle sliderRectangle;
	protected StRectangle savedSliderRectangle;
	protected StRectangle firstMarkerRectangle;
	protected StRectangle lastMarkerRectangle;
	protected boolean notDisplaySlider;

	protected Image _offScreenImage;
	protected Dimension _offScreenSize;
	protected Graphics _offScreenGraphics;

	/**
	 * Create a new instance of <code>JunTrackSliderViewAwt</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunTrackSliderViewAwt() {
		super();
	}

	/**
	 * Create a new instance of <code>JunTrackSliderViewAwt</code> and initialize it.
	 * 
	 * @param aTrackSliderModel jp.co.sra.jun.goodies.track.JunTrackSliderModel
	 * @category Instance creation
	 */
	public JunTrackSliderViewAwt(JunTrackSliderModel aTrackSliderModel) {
		super(aTrackSliderModel);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StViewJPanel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		this.flushSliderRectangles();
		this.flushMarkerRectangles();
		_offScreenImage = null;
		_offScreenSize = null;
		_offScreenGraphics = null;
	}

	/**
	 * Build this component.
	 * 
	 * @see jp.co.sra.smalltalk.StViewCanvas#buildComponent()
	 * @category initialize-release
	 */
	protected void buildComponent() {
		this.setSize(new Dimension(150, 20));
	}

	/**
	 * Answer the current another interval rectangle.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @see jp.co.sra.jun.goodies.track.JunTrackSliderView#anotherIntervalRectangle()
	 * @category accessing
	 */
	public StRectangle anotherIntervalRectangle() {
		return this.intervalRectangle().translatedBy_(0, this.margin().bottom() + 1);
	}

	/**
	 * Answer the current first marker rectangle.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @see jp.co.sra.jun.goodies.track.JunTrackSliderView#firstMarkerRectangle()
	 * @category accessing
	 */
	public StRectangle firstMarkerRectangle() {
		if (firstMarkerRectangle == null) {
			StRectangle box = this.sliderArea();
			box = box.insetBy_(new StRectangle(0, 0, 1, 1));
			int x = (int) Math.round(box.width() * this.getTrackSliderModel().interval()[0]) + box.originX();
			StRectangle marker = new StRectangle(x, box.top(), x, box.bottom());
			marker = marker.expandedBy_(this.margin());
			marker = StRectangle.Vertex_vertex_(new Point(marker.left() + 2, marker.bottom() - 1), marker.center());
			firstMarkerRectangle = marker;
		}
		return firstMarkerRectangle;
	}

	/**
	 * Answer the current interval rectangle.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category accessing
	 */
	protected StRectangle intervalRectangle() {
		StRectangle box1 = this.firstMarkerRectangle();
		StRectangle box2 = this.lastMarkerRectangle();
		StRectangle box = new StRectangle(box1.right() + 1, box1.top(), box2.left(), box2.bottom());
		box = box.intersect_(this.sliderArea()).insetBy_(new StRectangle(0, 0, 0, 2));
		return box;
	}

	/**
	 * Answer the current last marker rectangle.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @see jp.co.sra.jun.goodies.track.JunTrackSliderView#lastMarkerRectangle()
	 * @category accessing
	 */
	public StRectangle lastMarkerRectangle() {
		if (lastMarkerRectangle == null) {
			StRectangle box = this.sliderArea();
			box = box.insetBy_(new StRectangle(0, 0, 1, 1));
			int x = (int) Math.round(box.width() * this.getTrackSliderModel().interval()[1]) + box.originX();
			StRectangle marker = new StRectangle(x, box.top(), x, box.bottom());
			marker = marker.expandedBy_(this.margin());
			marker = StRectangle.Vertex_vertex_(new Point(marker.right() - 2, marker.bottom() - 1), marker.center());
			lastMarkerRectangle = marker;
		}
		return lastMarkerRectangle;
	}

	/**
	 * Answer the Rectangle of margin.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category accessing
	 */
	protected StRectangle margin() {
		return new StRectangle(6, 4, 6, 4);
	}

	/**
	 * Set the new first marker to the model via view.
	 * 
	 * @param x int
	 * @param y int
	 * @see jp.co.sra.jun.goodies.track.JunTrackSliderView#setModelLastMarker_(int, int)
	 * @category accessing
	 */
	public void setModelFirstMarker_(int x, int y) {
		StRectangle box = this.sliderArea();
		box = box.insetBy_(new StRectangle(0, 0, 1, 1));
		double value = (double) (x - box.originX()) / box.width();
		this.getTrackSliderModel().firstMarker_(Math.max(0, Math.min(value, 1)));
	}

	/**
	 * Set the new interval to the model via view.
	 * 
	 * @param x int
	 * @param y int
	 * @see jp.co.sra.jun.goodies.track.JunTrackSliderView#setModelInterval_(int, int)
	 * @category accessing
	 */
	public void setModelInterval_(int x, int y) {
		StRectangle box = this.sliderArea();
		box = box.insetBy_(new StRectangle(0, 0, 1, 1));
		double value = (double) (x - box.originX()) / box.width();
		double difference = this.getTrackSliderModel().lastMarker() - this.getTrackSliderModel().firstMarker();
		double half = difference / 2;
		double first = value - half;
		double last = first + difference;
		if (first < 0) {
			first = 0;
			last = 0 + difference;
		}
		if (last > 1) {
			first = 1 - difference;
			last = 1;
		}
		this.getTrackSliderModel().interval_(new double[] { first, last });
	}

	/**
	 * Set the new last marker to the model via view.
	 * 
	 * @param x int
	 * @param y int
	 * @see jp.co.sra.jun.goodies.track.JunTrackSliderView#setModelLastMarker_(int, int)
	 * @category accessing
	 */
	public void setModelLastMarker_(int x, int y) {
		StRectangle box = this.sliderArea();
		box = box.insetBy_(new StRectangle(0, 0, 1, 1));
		double value = (double) (x - box.originX()) / box.width();
		this.getTrackSliderModel().lastMarker_(Math.max(0, Math.min(value, 1)));
	}

	/**
	 * Set the new model value via view.
	 * 
	 * @param x int
	 * @param y int
	 * @see jp.co.sra.jun.goodies.track.JunTrackSliderView#setModelValue_(int, int)
	 * @category accessing
	 */
	public void setModelValue_(int x, int y) {
		StRectangle box = this.sliderArea();
		box = box.insetBy_(new StRectangle(0, 0, 1, 1));
		double value = (double) (x - box.originX()) / box.width();
		this.getTrackSliderModel().value_(Math.max(0, Math.min(value, 1)));
	}

	/**
	 * Answer my slider area.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @see jp.co.sra.jun.goodies.track.JunTrackSliderView#sliderArea()
	 * @category accessing
	 */
	public StRectangle sliderArea() {
		return (new StRectangle(this.getSize())).insetBy_(this.margin());
	}

	/**
	 * Answer a slider rectangle.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @see jp.co.sra.jun.goodies.track.JunTrackSliderView#sliderRectangle()
	 * @category accessing
	 */
	public synchronized StRectangle sliderRectangle() {
		if (sliderRectangle == null) {
			sliderRectangle = this.sliderRectangleForValue_(this.getTrackSliderModel().doubleValue());
		}
		return (StRectangle) sliderRectangle.copy();
	}

	/**
	 * Answer a slider rectangle for a value.
	 * 
	 * @param aValue double
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category accessing
	 */
	protected StRectangle sliderRectangleForValue_(double aValue) {
		StRectangle box = this.sliderArea();
		box = box.insetBy_(new StRectangle(0, 0, 1, 1));
		int x = (int) Math.round(box.width() * aValue) + box.originX();
		StRectangle slider = new StRectangle(x, box.top(), x, box.bottom());
		slider = slider.expandedBy_(this.margin().insetBy_(new StRectangle(0, 0, 0, this.margin().bottom())));
		return slider;
	}

	/**
	 * Set the bounds of the view.
	 * Also needs to flush the slider rectangles which need to redraw.
	 *
	 * @param x The new <i>x</i>-coordinate of this component.
	 * @param y The new <i>y</i>-coordinate of this component.
	 * @param width The new <code>width</code> of this component.
	 * @param height The new <code>height</code> of this component.
	 * @see java.awt.Component#setBounds(int, int, int, int)
	 * @category bounds accessing
	 */
	public void setBounds(int x, int y, int width, int height) {
		super.setBounds(x, y, width, height);
		this.flushSliderRectangles();
		this.flushMarkerRectangles();
	}

	/**
	 * Answer my default controller.
	 * 
	 * @return jp.co.sra.smalltalk.StController
	 * @see jp.co.sra.smalltalk.StViewJPanel#defaultController()
	 * @category defaults
	 */
	protected StController defaultController() {
		return new JunTrackSliderController();
	}

	/**
	 * Answer my default model.
	 * 
	 * @return jp.co.sra.smalltalk.StModel
	 * @see jp.co.sra.smalltalk.StViewJPanel#defaultModel()
	 * @category defaults
	 */
	protected StModel defaultModel() {
		return new JunTrackSliderModel();
	}

	/**
	 * Display the background on the graphics.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @category displaying
	 */
	protected void displayBackgroundOn_(Graphics graphicsContext) {
		if (this.getTrackSliderModel().parentTracker() == null) {
			StRectangle box = new StRectangle(this.getSize());
			graphicsContext.clipRect(box.left(), box.top(), box.right(), box.bottom());
			graphicsContext.setColor(Gray70);
			graphicsContext.fillRect(box.left(), box.top(), box.right(), box.bottom());
			graphicsContext.drawRect(box.left(), box.top(), box.right(), box.bottom());
			graphicsContext.setColor(Gray80);
			graphicsContext.drawLine(box.left(), box.top(), box.right() - 1, box.top());
			graphicsContext.drawLine(box.left(), box.top(), box.left(), box.bottom() - 1);
			graphicsContext.setColor(Gray20);
			graphicsContext.drawLine(box.left(), box.bottom() - 1, box.right() - 1, box.bottom() - 1);
			graphicsContext.drawLine(box.right() - 1, box.top(), box.right() - 1, box.bottom() - 1);
			box = this.sliderArea();
			graphicsContext.setColor(Gray20);
			graphicsContext.drawLine(box.left(), box.top(), box.right() - 1, box.top());
			graphicsContext.drawLine(box.left(), box.top(), box.left(), box.bottom() - 1);
			graphicsContext.setColor(Gray80);
			graphicsContext.drawLine(box.left() + 1, box.top() + 1, box.right() - 1, box.top() + 1);
			graphicsContext.drawLine(box.left() + 1, box.top() + 1, box.left() + 1, box.bottom() - 1);
			graphicsContext.setColor(Gray20);
			graphicsContext.drawLine(box.left() + 1, box.bottom() - 2, box.right() - 1, box.bottom() - 2);
			graphicsContext.drawLine(box.right() - 2, box.top() + 1, box.right() - 2, box.bottom() - 1);
			graphicsContext.setColor(Gray80);
			graphicsContext.drawLine(box.left(), box.bottom() - 1, box.right() - 1, box.bottom() - 1);
			graphicsContext.drawLine(box.right() - 1, box.top(), box.right() - 1, box.bottom() - 1);
		} else {
			StRectangle box = new StRectangle(this.getSize());
			graphicsContext.clipRect(box.left(), box.top(), box.right(), box.bottom());
			graphicsContext.setColor(Gray70);
			graphicsContext.fillRect(box.left(), box.top(), box.right(), box.bottom());
			graphicsContext.drawRect(box.left(), box.top(), box.right(), box.bottom());
			graphicsContext.setColor(Gray80);
			graphicsContext.drawLine(box.left(), box.top(), box.right() - 1, box.top());
			graphicsContext.drawLine(box.left(), box.top(), box.left(), box.bottom() - 1);
			graphicsContext.setColor(Gray20);
			graphicsContext.drawLine(box.left(), box.bottom() - 1, box.right() - 1, box.bottom() - 1);
			graphicsContext.drawLine(box.right() - 1, box.top(), box.right() - 1, box.bottom() - 1);
			box = this.sliderArea();
			graphicsContext.setColor(Gray20);
			graphicsContext.drawLine(box.left(), box.top(), box.right() - 1, box.top());
			graphicsContext.setColor(Gray80);
			graphicsContext.drawLine(box.left(), box.top() + 1, box.right() - 1, box.top() + 1);
			graphicsContext.setColor(Gray20);
			graphicsContext.drawLine(box.left(), box.bottom() - 2, box.right() - 1, box.bottom() - 2);
			graphicsContext.setColor(Gray80);
			graphicsContext.drawLine(box.left(), box.bottom() - 1, box.right() - 1, box.bottom() - 1);
		}
	}

	/**
	 * Display a box with a color on a graphics context.
	 * 
	 * @param area jp.co.sra.smalltalk.StRectangle
	 * @param color java.awt.Color
	 * @param graphicsContext java.awt.Graphics
	 * @category displaying
	 */
	protected void displayBox_color_on_(StRectangle area, Color color, Graphics graphicsContext) {
		graphicsContext.setColor(color);
		graphicsContext.drawLine(area.left(), area.top(), area.right() - 1, area.top());
		graphicsContext.drawLine(area.left(), area.top(), area.left(), area.bottom() - 1);
		graphicsContext.drawLine(area.left(), area.bottom() - 1, area.right() - 1, area.bottom() - 1);
		graphicsContext.drawLine(area.right() - 1, area.top(), area.right() - 1, area.bottom() - 1);
	}

	/**
	 * Display an interval on a graphics context.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @category displaying
	 */
	protected void displayIntervalOn_(Graphics graphicsContext) {
		if (this.getTrackSliderModel().areMarkersActive() == false) {
			return;
		}

		StRectangle box1 = this.firstMarkerRectangle();
		StRectangle box2 = this.lastMarkerRectangle();
		if (this.getTrackSliderModel().parentTracker() == null) {
			graphicsContext.setColor(Color.blue);
			if (this.getTrackSliderModel().fixFirstMarker()) {
				graphicsContext.drawPolyline(new int[] { box1.right(), box1.right() }, new int[] { box1.top(), box1.bottom() }, 2);
			} else {
				graphicsContext.drawPolyline(new int[] { box1.right(), box1.right(), box1.left(), box1.right() }, new int[] { box1.top(), box1.bottom(), box1.bottom(), box1.top() }, 4);
			}
			graphicsContext.setColor(Color.red);
			if (this.getTrackSliderModel().fixLastMarker()) {
				graphicsContext.drawPolyline(new int[] { box2.left(), box2.left() }, new int[] { box2.top(), box2.bottom() }, 2);
			} else {
				graphicsContext.drawPolyline(new int[] { box2.left(), box2.left(), box2.right(), box2.left() }, new int[] { box2.top(), box2.bottom(), box2.bottom(), box2.top() }, 4);
			}
		} else {
			graphicsContext.setColor(Color.blue);
			graphicsContext.drawPolyline(new int[] { box1.right(), box1.right() }, new int[] { box1.top(), box1.bottom() }, 2);
			graphicsContext.setColor(Color.red);
			graphicsContext.drawPolyline(new int[] { box2.left(), box2.left() }, new int[] { box2.top(), box2.bottom() }, 2);
		}
		Rectangle box = this.intervalRectangle().toRectangle();
		graphicsContext.setColor(Color.green);
		graphicsContext.fillRect(box.x, box.y, box.width, box.height);
	}

	/**
	 * Display on a graphics context.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_(java.awt.Graphics)
	 * @category displaying
	 */
	public void displayOn_(Graphics graphicsContext) {
		if (this.isShowing() == false) {
			return;
		}

		Dimension size = this.getSize();
		if (!size.equals(_offScreenSize)) {
			_offScreenImage = this.createImage(size.width, size.height);
			_offScreenGraphics = _offScreenImage.getGraphics();
			_offScreenSize = size;
		}
		this.displayBackgroundOn_(_offScreenGraphics);
		this.displayIntervalOn_(_offScreenGraphics);
		this.displaySliderOn_withColor_(_offScreenGraphics, Gray90);
		graphicsContext.drawImage(_offScreenImage, 0, 0, this);
	}

	/**
	 * Display the slider for the value on the graphics context with the color.
	 * 
	 * @param value double
	 * @param aGraphics java.awt.Graphics
	 * @param aColor java.awt.Color
	 * @see jp.co.sra.jun.goodies.track.JunTrackSliderView#displaySliderForValue_on_withColor_(double, java.awt.Graphics, java.awt.Color)
	 * @category displaying
	 */
	public void displaySliderForValue_on_withColor_(double value, Graphics aGraphics, Color aColor) {
		StRectangle box = this.sliderRectangleForValue_(value);
		StRectangle area = box.insetBy_(new StRectangle(1, 0, 1, 0));
		this.displayBox_color_on_(area, Gray20, aGraphics);
		this.displayBox_color_on_(area.insetBy_(1), aColor, aGraphics);
		this.displayBox_color_on_(area.insetBy_(2), Gray20, aGraphics);
		savedSliderRectangle = box;
	}

	/**
	 * Display the slider on the graphics context with the color.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @param color java.awt.Color
	 * @see jp.co.sra.jun.goodies.track.JunTrackSliderView#displaySliderOn_withColor_(java.awt.Graphics, java.awt.Color)
	 * @category displaying
	 */
	public void displaySliderOn_withColor_(Graphics graphicsContext, Color color) {
		StRectangle box = this.sliderRectangle();
		if (this.notDisplaySlider() == false) {
			StRectangle area = box.insetBy_(new StRectangle(1, 0, 1, 0));
			this.displayBox_color_on_(area, Gray20, graphicsContext);
			this.displayBox_color_on_(area.insetBy_(1), color, graphicsContext);
			this.displayBox_color_on_(area.insetBy_(2), Gray20, graphicsContext);
		}
		savedSliderRectangle = box;
	}

	/**
	 * Flush the marker rectangles.
	 * 
	 * @category flushing
	 */
	protected void flushMarkerRectangles() {
		firstMarkerRectangle = null;
		lastMarkerRectangle = null;
	}

	/**
	 * Flush the slider rectangles.
	 * 
	 * @category flushing
	 */
	protected synchronized void flushSliderRectangles() {
		sliderRectangle = null;
		savedSliderRectangle = null;
	}

	/**
	 * Answer the current JunTrackSliderModel.
	 * 
	 * @return jp.co.sra.jun.goodies.track.JunTrackSliderModel
	 * @deprecated since Jun379
	 * @category model accessing
	 */
	public JunTrackSliderModel getModel() {
		return (JunTrackSliderModel) this.model();
	}

	/**
	 * Answer the current model as a JunTrackSliderModel.
	 * 
	 * @return jp.co.sra.jun.goodies.track.JunTrackSliderModel
	 * @see jp.co.sra.jun.goodies.track.JunTrackSliderView#getTrackSliderModel()
	 * @category model accessing
	 */
	public JunTrackSliderModel getTrackSliderModel() {
		return (JunTrackSliderModel) this.model();
	}

	/**
	 * Set the new JunTrackSliderModel.
	 * 
	 * @param aModel jp.co.sra.jun.goodies.track.JunTrackSliderModel
	 * @deprecated since Jun379
	 * @category model accessing
	 */
	public void setModel(JunTrackSliderModel aModel) {
		this.model_(aModel);
	}

	/**
	 * Show the popup menu at the specified point on the view.
	 *
	 * @param x int
	 * @param y int
	 * @see jp.co.sra.smalltalk.StView#_showPopupMenu(int, int)
	 * @category popup menu
	 */
	public void _showPopupMenu(int x, int y) {
		this.popupMenu_(this.getTrackSliderModel()._popupMenu());
		super._showPopupMenu(x, y);
	}

	/**
	 * Action for the update notification.
	 * 
	 * @param evt jp.co.sra.smalltalk.DependentEvent
	 * @see jp.co.sra.smalltalk.DependentListener#update_(jp.co.sra.smalltalk.DependentEvent)
	 * @category updating
	 */
	public void update_(DependentEvent evt) {
		if (this.isShowing() == false) {
			return;
		}

		StSymbol aSymbol = evt.getAspect();
		if (aSymbol == $("value")) {
			synchronized (this) {
				sliderRectangle = null;
				StRectangle clipBox = this.sliderRectangle();
				if (clipBox.equals(savedSliderRectangle) == false) {
					if (savedSliderRectangle != null) {
						clipBox = clipBox.merge_(savedSliderRectangle);
					}
					Graphics graphics = this.getGraphics();
					if (graphics != null) {
						try {
							graphics.clipRect(clipBox.originX(), clipBox.originY(), clipBox.width(), clipBox.height());
							this.displayOn_(graphics);
						} finally {
							graphics.dispose();
						}
					}
				}
			}
		} else if ((aSymbol == $("interval")) || (aSymbol == $("useMarkers"))) {
			this.flushMarkerRectangles();
			Graphics graphics = this.getGraphics();
			if (graphics != null) {
				try {
					this.displayOn_(graphics);
				} finally {
					graphics.dispose();
				}
			}
		} else {
			super.update_(evt);
		}
	}

	/**
	 * Answer true if the receiver need not to display slider, otherwise false.
	 * 
	 * @return boolean
	 * @category private
	 */
	protected boolean notDisplaySlider() {
		return notDisplaySlider;
	}

	/**
	 * Set whether the slider is needed to be displayed or not.
	 * 
	 * @param aBoolean
	 * @see jp.co.sra.jun.goodies.track.JunTrackSliderView#notDisplaySlider_(boolean)
	 * @category private
	 */
	public void notDisplaySlider_(boolean aBoolean) {
		notDisplaySlider = aBoolean;

		if (this.getTrackSliderModel().childTracker() == null) {
			return;
		}

		JunTrackSliderView aView = null;
		Object[] dependents = this.getTrackSliderModel().childTracker().dependents();
		for (int i = 0; i < dependents.length; i++) {
			if (dependents[i] instanceof JunTrackSliderView) {
				aView = (JunTrackSliderView) dependents[i];
				break;
			}
		}
		if (aView == null) {
			return;
		}

		aView.notDisplaySlider_(aBoolean);
	}
}
