package jp.co.sra.jun.goodies.track;

import java.awt.Color;
import java.awt.Image;
import java.awt.event.WindowEvent;
import java.awt.image.BufferedImage;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StValueHolder;
import jp.co.sra.smalltalk.StView;

import jp.co.sra.jun.goodies.button.JunButtonModel;
import jp.co.sra.jun.goodies.button.JunButtonWithMenuModel;
import jp.co.sra.jun.goodies.cursors.JunCursors;

/**
 * JunTrackerModel class
 * 
 *  @author    MATSUDA Ryouichi
 *  @created   1998/11/24 (by MATSUDA Ryouichi)
 *  @updated   1999/12/10 (by MATSUDA Ryouichi)
 *  @updated   2002/10/28 (by nisinaka)
 *  @updated   2003/03/24 (by nisinaka)
 *  @updated   2006/02/23 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun651 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunTrackerModel.java,v 8.12 2008/02/20 06:32:04 nisinaka Exp $
 */
public class JunTrackerModel extends JunTrackSliderModel {
	protected Thread trackingProcess;
	protected StValueHolder stepHolder;
	protected StSymbol loopCondition;
	protected StSymbol loopDirection;
	protected JunButtonModel playButton;
	protected JunButtonModel loopButton;
	protected JunButtonModel previousButton;
	protected JunButtonModel nextButton;
	protected JunButtonModel firstButton;
	protected JunButtonModel lastButton;
	protected JunButtonModel firstMarkerButton;
	protected JunButtonModel lastMarkerButton;
	protected JunButtonWithMenuModel speakerButton;
	protected JunTrackSliderModel childTracker;

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.goodies.track.JunTrackSliderModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		trackingProcess = null;
		stepHolder = null;
		loopCondition = null;
		loopDirection = $("forth");
		playButton = null;
		loopButton = null;
		previousButton = null;
		nextButton = null;
		firstButton = null;
		lastButton = null;
		firstMarkerButton = null;
		lastMarkerButton = null;
		speakerButton = null;
		childTracker = null;
		this.stepHolder();
		this.loopCondition();
	}

	/**
	 * Answer the step value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double step() {
		return ((Number) stepHolder.value()).doubleValue();
	}

	/**
	 * Set number to step.
	 * 
	 * @param aNumber double
	 * @category accessing
	 */
	public void step_(double aNumber) {
		if (aNumber <= 0) {
			return;
		}

		stepHolder.value_(Math.max(0, Math.min(aNumber, 1)));
		this.value_(this.doubleValue());
	}

	/**
	 * Answer the stepHolder.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category accessing
	 */
	protected StValueHolder stepHolder() {
		if (stepHolder == null) {
			stepHolder = new StValueHolder(0.01d);
		}
		return stepHolder;
	}

	/**
	 * Answer the symbol as loopCondition.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category accessing
	 */
	public StSymbol loopCondition() {
		if (loopCondition == null) {
			loopCondition = $("oneWay");
		}
		return loopCondition;
	}

	/**
	 * Set a symbol to loopCondition.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @category accessing
	 */
	public void loopCondition_(StSymbol aSymbol) {
		if ((aSymbol == $("oneWay")) || (aSymbol == $("loop")) || (aSymbol == $("backAndForth"))) {
			loopCondition = aSymbol;
			this.loopButton().visual_(this.loopImage());
		}
	}

	/**
	 * Set #oneWay to the loopCondition.
	 * 
	 * @category accessing
	 */
	public void oneWay() {
		this.loopCondition_($("oneWay"));
	}

	/**
	 * Set #loop to the loopCondition.
	 * 
	 * @category accessing
	 */
	public void loop() {
		this.loopCondition_($("loop"));
	}

	/**
	 * Set #backAndForth to the loopCondition.
	 * 
	 * @category accessing
	 */
	public void backAndForth() {
		this.loopCondition_($("backAndForth"));
	}

	/**
	 * Set number to this value.
	 * 
	 * @param aNumber double
	 * @see jp.co.sra.jun.goodies.track.JunTrackSliderModel#value_(double)
	 * @category accessing
	 */
	public void value_(double aNumber) {
		if (aNumber <= 0) {
			super.value_(0);
		} else if (1 <= aNumber) {
			super.value_(1);
		} else {
			super.value_(Math.round(aNumber / this.step()) * this.step());
		}
	}

	/**
	 * Move to first with a Tracker.
	 * 
	 * @category accessing
	 */
	public void first() {
		if (this.areMarkersActive()) {
			this.value_(this.firstMarker());
		} else {
			this.value_(0);
		}
	}

	/**
	 * Move to last with a Tracker.
	 * 
	 * @category accessing
	 */
	public void last() {
		if (this.areMarkersActive()) {
			this.value_(this.lastMarker());
		} else {
			this.value_(1);
		}
	}

	/**
	 * Set the first marker.
	 * 
	 * @param normalizedValue double
	 * @see jp.co.sra.jun.goodies.track.JunTrackSliderModel#firstMarker_(double)
	 * @category accessing
	 */
	public void firstMarker_(double normalizedValue) {
		if (this.fixFirstMarker() == false) {
			double first = Math.max(normalizedValue, this.intervalBounds()[0]);
			double[] interval = new double[] { first, Math.max(first, this.lastMarker()) };
			this.interval_(interval);
		}
	}

	/**
	 * Set the last marker.
	 * 
	 * @param normalizedValue double
	 * @see jp.co.sra.jun.goodies.track.JunTrackSliderModel#lastMarker_(double)
	 * @category accessing
	 */
	public void lastMarker_(double normalizedValue) {
		if (this.fixLastMarker() == false) {
			double last = Math.min(normalizedValue, this.intervalBounds()[1]);
			double[] interval = new double[] { Math.min(this.firstMarker(), last), last };
			this.interval_(interval);
		}
	}

	/**
	 * Adjust the interval.
	 * 
	 * @param anArray double[]
	 * @return double[]
	 * @see jp.co.sra.jun.goodies.track.JunTrackSliderModel#adjustInterval_(double[])
	 * @category accessing
	 */
	protected double[] adjustInterval_(double[] anArray) {
		double first = Math.min(anArray[0], anArray[1]);
		double last = Math.max(anArray[0], anArray[1]);
		first = Math.round(Math.max(0, Math.min(first, 1)) / this.step()) * this.step();
		last = Math.round(Math.max(first, Math.min(last, 1)) / this.step()) * this.step();
		return new double[] { first, last };
	}

	/**
	 * Enable markers.
	 * 
	 * @see jp.co.sra.jun.goodies.track.JunTrackSliderModel#enableMarkers()
	 * @category accessing
	 */
	public void enableMarkers() {
		super.enableMarkers();

		this.firstButton().enable();
		this.lastButton().enable();
		this.firstMarkerButton().enable();
		this.lastMarkerButton().enable();
	}

	/**
	 * Disable markers.
	 * 
	 * @see jp.co.sra.jun.goodies.track.JunTrackSliderModel#disableMarkers()
	 * @category accessing
	 */
	public void disableMarkers() {
		super.disableMarkers();

		this.firstButton().disable();
		this.lastButton().disable();
		this.firstMarkerButton().disable();
		this.lastMarkerButton().disable();
	}

	/**
	 * Answer my child tracker.
	 *
	 * @return jp.co.sra.jun.goodies.track.JunTrackSliderModel
	 * @see jp.co.sra.jun.goodies.track.JunTrackSliderModel#childTracker()
	 * @category accessing
	 */
	public JunTrackSliderModel childTracker() {
		return childTracker;
	}

	/**
	 * Set my child tracker.
	 * 
	 * @param aTracker jp.co.sra.jun.goodies.track.JunTrackSliderModel
	 * @category accessing
	 */
	public void childTracker_(JunTrackSliderModel aTracker) {
		childTracker = aTracker;
	}

	/**
	 * Answer the playButton model.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel playButton() {
		if (playButton == null) {
			JunButtonModel button = new JunButtonModel(false, this.playImage(), new StBlockClosure() {
				public Object value_(Object o) {
					playAction_((JunButtonModel) o);
					return null;
				}
			});
			playButton = button;
		}
		return playButton;
	}

	/**
	 * Answer an image of the current play button.
	 * 
	 * @return java.awt.Image
	 * @category button images
	 */
	public Image playImage() {
		if (playButton != null) {
			if (this.playButton().value()) {
				return JunCursors.PauseCursorImage();
			} else {
				return JunCursors.PlayCursorImage();
			}
		}
		return JunCursors.PlayCursorImage();
	}

	/**
	 * Action to do when the next button is pressed.
	 * 
	 * @param aModel jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category button actions
	 */
	protected void playAction_(JunButtonModel aModel) {
		if (aModel.value()) {
			this.end();
		} else {
			if (aModel._isPressedWithShiftDown()) {
				this.first();
			}
			this.start();
		}
	}

	/**
	 * Set the new value of the play button.
	 * Also update the image at the same time.
	 *
	 * @param aBoolean boolean
	 * @category button actions
	 */
	public void playButtonVisual_(boolean aBoolean) {
		this.playButton().value_(aBoolean);
		this.playButton().visual_(this.playImage());
	}

	/**
	 * Answer the loopButton model.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel loopButton() {
		if (loopButton == null) {
			JunButtonModel button = new JunButtonModel(false, this.loopImage(), new StBlockClosure() {
				public Object value_(Object o) {
					loopAction_((JunButtonModel) o);
					return null;
				}
			});
			loopButton = button;
		}
		return loopButton;
	}

	/**
	 * Answer an Image of current loopCondition.
	 * 
	 * @return java.awt.Image
	 * @category button images
	 */
	public Image loopImage() {
		if (this.loopCondition() == $("oneWay")) {
			return JunCursors.ArrowCursorImage();
		} else if (this.loopCondition() == $("loop")) {
			return JunCursors.RepeatCursorImage();
		} else if (this.loopCondition() == $("backAndForth")) {
			return JunCursors.RoundtripCursorImage();
		}
		return JunCursors.ArrowCursorImage();
	}

	/**
	 * Action to do when the loop button is pressed.
	 * 
	 * @param aModel jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category button actions
	 */
	protected void loopAction_(JunButtonModel aModel) {
		if (this.loopCondition() == $("oneWay")) {
			this.loop();
		} else if (this.loopCondition() == $("loop")) {
			this.backAndForth();
		} else if (this.loopCondition() == $("backAndForth")) {
			this.oneWay();
		} else {
			this.oneWay();
		}
	}

	/**
	 * Answer the previousButton model.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel previousButton() {
		if (previousButton == null) {
			JunButtonModel button = new JunButtonModel(false, this.previousButtonImage(), new StBlockClosure() {
				public Object value_(Object o) {
					previousAction_((JunButtonModel) o);
					return null;
				}
			});
			button.repeatAction_(true);
			previousButton = button;
		}
		return previousButton;
	}

	/**
	 * Answer a previous button image. 
	 * 
	 * @return java.awt.Image
	 * @category button images
	 */
	public Image previousButtonImage() {
		return JunCursors.PreviousCursorImage();
	}

	/**
	 * Action to do when the previous button is pressed.
	 * 
	 * @param aModel jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category button actions
	 */
	protected void previousAction_(JunButtonModel aModel) {
		if (aModel._isPressedWithShiftDown()) {
			this.first();
		} else if (this.doubleValue() > 0) {
			this.previous();
		}
	}

	/**
	 * Answer the nextButton model.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel nextButton() {
		if (nextButton == null) {
			JunButtonModel button = new JunButtonModel(false, this.nextButtonImage(), new StBlockClosure() {
				public Object value_(Object o) {
					nextAction_((JunButtonModel) o);
					return null;
				}
			});
			button.repeatAction_(true);
			nextButton = button;
		}
		return nextButton;
	}

	/**
	 * Answer a next button image. 
	 * 
	 * @return java.awt.Image
	 * @category button images
	 */
	public Image nextButtonImage() {
		return JunCursors.NextCursorImage();
	}

	/**
	 * Action to do when the next button is pressed.
	 * 
	 * @param aModel jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category button actions
	 */
	protected void nextAction_(JunButtonModel aModel) {
		if (aModel._isPressedWithShiftDown()) {
			this.last();
		} else if (this.doubleValue() < 1) {
			this.next();
		}
	}

	/**
	 * Answer the first button model.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel firstButton() {
		if (firstButton == null) {
			JunButtonModel button = new JunButtonModel(false, this.firstButtonImage(), new StBlockClosure() {
				public Object value_(Object o) {
					firstAction_((JunButtonModel) o);
					return null;
				}
			});
			button.repeatAction_(true);
			firstButton = button;
		}
		return firstButton;
	}

	/**
	 * Answer the first button image.
	 *
	 * @return java.awt.Image
	 * @category button images
	 */
	protected Image firstButtonImage() {
		BufferedImage image = ((StImage) (new StImage(JunCursors.PrologueCursorImage())).clone()).image();
		int blue = Color.blue.getRGB();
		for (int y = 2; y <= 13; y++) {
			image.setRGB(3, y, blue);
		}
		return image;
	}

	/**
	 * Action to do when the first button is pressed.
	 * 
	 * @param aModel jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category button actions
	 */
	protected void firstAction_(JunButtonModel aModel) {
		if (this.areMarkersActive() == false) {
			return;
		}

		JunCursors cursor = new JunCursors(JunCursors.PrologueCursor());
		try {
			cursor._show();

			double position = this.firstMarker();
			double step = 0.02;
			if (aModel._isPressedWithShiftDown()) {
				double currentPosition = this.doubleValue();
				if (position - currentPosition > 0) {
					while ((currentPosition = currentPosition + step) < position) {
						this.value_(currentPosition);
					}
				} else {
					while ((currentPosition = currentPosition - step) > position) {
						this.value_(currentPosition);
					}
				}
			}
			this.value_(position);
		} finally {
			cursor._restore();
		}
	}

	/**
	 * Answer the last button model.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel lastButton() {
		if (lastButton == null) {
			JunButtonModel button = new JunButtonModel(false, this.lastButtonImage(), new StBlockClosure() {
				public Object value_(Object o) {
					lastAction_((JunButtonModel) o);
					return null;
				}
			});
			button.repeatAction_(true);
			lastButton = button;
		}
		return lastButton;
	}

	/**
	 * Answer the last button image.
	 *
	 * @return java.awt.Image
	 * @category button images
	 */
	protected Image lastButtonImage() {
		BufferedImage image = ((StImage) (new StImage(JunCursors.EpilogueCursorImage())).clone()).image();
		int red = Color.red.getRGB();
		for (int y = 2; y <= 13; y++) {
			image.setRGB(12, y, red);
		}
		return image;
	}

	/**
	 * Action to do when the last button is pressed.
	 * 
	 * @param aModel jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category button actions
	 */
	protected void lastAction_(JunButtonModel aModel) {
		if (this.areMarkersActive() == false) {
			return;
		}

		JunCursors cursor = new JunCursors(JunCursors.EpilogueCursor());
		try {
			cursor._show();

			double position = this.lastMarker();
			double step = 0.02;
			if (aModel._isPressedWithShiftDown()) {
				double currentPosition = this.doubleValue();
				if (position - currentPosition > 0) {
					while ((currentPosition = currentPosition + step) < position) {
						this.value_(currentPosition);
					}
				} else {
					while ((currentPosition = currentPosition - step) > position) {
						this.value_(currentPosition);
					}
				}
			}
			this.value_(position);
		} finally {
			cursor._restore();
		}
	}

	/**
	 * Answer the first marker button model.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel firstMarkerButton() {
		if (firstMarkerButton == null) {
			JunButtonModel button = new JunButtonModel(false, this.firstMarkerImage(), new StBlockClosure() {
				public Object value_(Object o) {
					firstMarkerAction_((JunButtonModel) o);
					return null;
				}
			});
			firstMarkerButton = button;
		}
		return firstMarkerButton;
	}

	/**
	 * Answer the first marker image.
	 *
	 * @return java.awt.Image
	 * @category button images
	 */
	protected Image firstMarkerImage() {
		BufferedImage anImage = new BufferedImage(16, 16, BufferedImage.TYPE_INT_ARGB);

		int white = Color.white.getRGB();
		anImage.setRGB(9, 3, white);
		anImage.setRGB(8, 4, white);
		anImage.setRGB(9, 4, white);
		anImage.setRGB(10, 4, white);
		anImage.setRGB(7, 5, white);
		anImage.setRGB(8, 5, white);
		anImage.setRGB(9, 5, white);
		anImage.setRGB(10, 5, white);
		anImage.setRGB(7, 6, white);
		anImage.setRGB(8, 6, white);
		anImage.setRGB(9, 6, white);
		anImage.setRGB(10, 6, white);
		anImage.setRGB(6, 7, white);
		anImage.setRGB(7, 7, white);
		anImage.setRGB(8, 7, white);
		anImage.setRGB(9, 7, white);
		anImage.setRGB(10, 7, white);
		anImage.setRGB(6, 8, white);
		anImage.setRGB(7, 8, white);
		anImage.setRGB(8, 8, white);
		anImage.setRGB(9, 8, white);
		anImage.setRGB(10, 8, white);
		anImage.setRGB(5, 9, white);
		anImage.setRGB(6, 9, white);
		anImage.setRGB(7, 9, white);
		anImage.setRGB(8, 9, white);
		anImage.setRGB(9, 9, white);
		anImage.setRGB(10, 9, white);
		anImage.setRGB(5, 10, white);
		anImage.setRGB(6, 10, white);
		anImage.setRGB(7, 10, white);
		anImage.setRGB(8, 10, white);
		anImage.setRGB(9, 10, white);
		anImage.setRGB(10, 10, white);
		anImage.setRGB(4, 11, white);
		anImage.setRGB(5, 11, white);
		anImage.setRGB(6, 11, white);
		anImage.setRGB(7, 11, white);
		anImage.setRGB(8, 11, white);
		anImage.setRGB(9, 11, white);
		anImage.setRGB(10, 11, white);
		anImage.setRGB(4, 12, white);
		anImage.setRGB(5, 12, white);
		anImage.setRGB(6, 12, white);
		anImage.setRGB(7, 12, white);
		anImage.setRGB(8, 12, white);
		anImage.setRGB(9, 12, white);
		anImage.setRGB(10, 12, white);
		anImage.setRGB(5, 13, white);
		anImage.setRGB(6, 13, white);
		anImage.setRGB(7, 13, white);
		anImage.setRGB(8, 13, white);
		anImage.setRGB(9, 13, white);

		int blue = Color.blue.getRGB();
		anImage.setRGB(9, 4, blue);
		anImage.setRGB(8, 5, blue);
		anImage.setRGB(9, 5, blue);
		anImage.setRGB(8, 6, blue);
		anImage.setRGB(9, 6, blue);
		anImage.setRGB(7, 7, blue);
		anImage.setRGB(9, 7, blue);
		anImage.setRGB(7, 8, blue);
		anImage.setRGB(9, 8, blue);
		anImage.setRGB(6, 9, blue);
		anImage.setRGB(9, 9, blue);
		anImage.setRGB(6, 10, blue);
		anImage.setRGB(9, 10, blue);
		anImage.setRGB(5, 11, blue);
		anImage.setRGB(9, 11, blue);
		anImage.setRGB(5, 12, blue);
		anImage.setRGB(6, 12, blue);
		anImage.setRGB(7, 12, blue);
		anImage.setRGB(8, 12, blue);
		anImage.setRGB(9, 12, blue);

		return anImage;
	}

	/**
	 * Action to do when the first marker button is pressed.
	 * 
	 * @param aModel jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category button actions
	 */
	protected void firstMarkerAction_(JunButtonModel aModel) {
		if (this.areMarkersActive() == false) {
			return;
		}

		if (this.fixFirstMarker()) {
			this.value_(this.firstMarker());
		} else {
			this.firstMarker_(this.doubleValue());
		}
	}

	/**
	 * Answer the last marker button model.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel lastMarkerButton() {
		if (lastMarkerButton == null) {
			JunButtonModel button = new JunButtonModel(false, this.lastMarkerImage(), new StBlockClosure() {
				public Object value_(Object o) {
					lastMarkerAction_((JunButtonModel) o);
					return null;
				}
			});
			lastMarkerButton = button;
		}
		return lastMarkerButton;
	}

	/**
	 * Answer the last marker image.
	 *
	 * @return java.awt.Image
	 * @category button images
	 */
	protected Image lastMarkerImage() {
		BufferedImage anImage = new BufferedImage(16, 16, BufferedImage.TYPE_INT_ARGB);

		int white = Color.white.getRGB();
		anImage.setRGB(6, 3, white);
		anImage.setRGB(5, 4, white);
		anImage.setRGB(6, 4, white);
		anImage.setRGB(7, 4, white);
		anImage.setRGB(5, 5, white);
		anImage.setRGB(6, 5, white);
		anImage.setRGB(7, 5, white);
		anImage.setRGB(8, 5, white);
		anImage.setRGB(5, 6, white);
		anImage.setRGB(6, 6, white);
		anImage.setRGB(7, 6, white);
		anImage.setRGB(8, 6, white);
		anImage.setRGB(5, 7, white);
		anImage.setRGB(6, 7, white);
		anImage.setRGB(7, 7, white);
		anImage.setRGB(8, 7, white);
		anImage.setRGB(9, 7, white);
		anImage.setRGB(5, 8, white);
		anImage.setRGB(6, 8, white);
		anImage.setRGB(7, 8, white);
		anImage.setRGB(8, 8, white);
		anImage.setRGB(9, 8, white);
		anImage.setRGB(5, 9, white);
		anImage.setRGB(6, 9, white);
		anImage.setRGB(7, 9, white);
		anImage.setRGB(8, 9, white);
		anImage.setRGB(9, 9, white);
		anImage.setRGB(10, 9, white);
		anImage.setRGB(5, 10, white);
		anImage.setRGB(6, 10, white);
		anImage.setRGB(7, 10, white);
		anImage.setRGB(8, 10, white);
		anImage.setRGB(9, 10, white);
		anImage.setRGB(10, 10, white);
		anImage.setRGB(5, 11, white);
		anImage.setRGB(6, 11, white);
		anImage.setRGB(7, 11, white);
		anImage.setRGB(8, 11, white);
		anImage.setRGB(9, 11, white);
		anImage.setRGB(10, 11, white);
		anImage.setRGB(11, 11, white);
		anImage.setRGB(5, 12, white);
		anImage.setRGB(6, 12, white);
		anImage.setRGB(7, 12, white);
		anImage.setRGB(8, 12, white);
		anImage.setRGB(9, 12, white);
		anImage.setRGB(10, 12, white);
		anImage.setRGB(11, 12, white);
		anImage.setRGB(6, 13, white);
		anImage.setRGB(7, 13, white);
		anImage.setRGB(8, 13, white);
		anImage.setRGB(9, 13, white);
		anImage.setRGB(10, 13, white);

		int red = Color.red.getRGB();
		anImage.setRGB(6, 4, red);
		anImage.setRGB(6, 5, red);
		anImage.setRGB(7, 5, red);
		anImage.setRGB(6, 6, red);
		anImage.setRGB(7, 6, red);
		anImage.setRGB(6, 7, red);
		anImage.setRGB(8, 7, red);
		anImage.setRGB(6, 8, red);
		anImage.setRGB(8, 8, red);
		anImage.setRGB(6, 9, red);
		anImage.setRGB(9, 9, red);
		anImage.setRGB(6, 10, red);
		anImage.setRGB(9, 10, red);
		anImage.setRGB(6, 11, red);
		anImage.setRGB(10, 11, red);
		anImage.setRGB(6, 12, red);
		anImage.setRGB(7, 12, red);
		anImage.setRGB(8, 12, red);
		anImage.setRGB(9, 12, red);
		anImage.setRGB(10, 12, red);

		return anImage;
	}

	/**
	 * Action to do when the last marker button is pressed.
	 * 
	 * @param aModel jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category button actions
	 */
	protected void lastMarkerAction_(JunButtonModel aModel) {
		if (this.areMarkersActive() == false) {
			return;
		}

		if (this.fixLastMarker()) {
			this.value_(this.lastMarker());
		} else {
			this.lastMarker_(this.doubleValue());
		}
	}

	/**
	 * Answer the speaker button model.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonWithMenuModel speakerButton() {
		if (speakerButton == null) {
			int size = 11;
			String[] labels = new String[size];
			Float[] values = new Float[size];
			for (int i = 0; i < size; i++) {
				int volume = 100 - i * 10;
				labels[i] = String.valueOf(volume);
				values[i] = new Float((float) volume / 100);
			}
			Image[] labelImages = new Image[] {
					JunCursors.Speaker10CursorImage(),
					JunCursors.Speaker09CursorImage(),
					JunCursors.Speaker08CursorImage(),
					JunCursors.Speaker07CursorImage(),
					JunCursors.Speaker06CursorImage(),
					JunCursors.Speaker05CursorImage(),
					JunCursors.Speaker04CursorImage(),
					JunCursors.Speaker03CursorImage(),
					JunCursors.Speaker02CursorImage(),
					JunCursors.Speaker01CursorImage(),
					JunCursors.Speaker00CursorImage() };

			JunButtonWithMenuModel button = new JunButtonWithMenuModel(labels, labelImages, values);
			button.action_(new StBlockClosure());
			button.repeatTick_(10);
			button.markerDisplaying_(false);

			speakerButton = button;
		}

		return speakerButton;
	}

	/**
	 * Answer true if the tracker state is at the beginning, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean atStart() {
		double nextValue = this.doubleValue() - this.step();
		if (nextValue < 0) {
			return true;
		}
		return (this.isIntervalNotEmpty() && (this.doubleValue() >= this.firstMarker()) && nextValue < this.firstMarker());
	}

	/**
	 * Answer true if the tracker state is at the end, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean atEnd() {
		double nextValue = this.doubleValue() + this.step();
		if (nextValue > 1) {
			return true;
		}
		return (this.isIntervalNotEmpty() && (this.doubleValue() <= this.lastMarker()) && nextValue > this.lastMarker());
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category interface opening
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunTrackerViewAwt(this);
		} else {
			return new JunTrackerViewSwing(this);
		}
	}

	/**
	 * Answer a window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("Tracker");
	}

	/**
	 * Invoked when a window is in the process of being closed.
	 * 
	 * @param e java.awt.event.WindowEvent
	 * @see jp.co.sra.smalltalk.StApplicationModel#noticeOfWindowClose()
	 * @category interface closing
	 */
	public void noticeOfWindowClose(WindowEvent e) {
		super.noticeOfWindowClose(e);
		this.end();
	}

	/**
	 * Start the trackingProcess.
	 * 
	 * @category scheduling
	 */
	public void start() {
		trackingProcess = new Thread() {
			public void run() {
				try {
					loopDirection = $("forth");
					playButtonVisual_(true);
					if (atEnd()) {
						first();
					}
					while (playButton().value()) {
						next();
						trackingProcess.yield();
					}
				} finally {
					playButtonVisual_(false);
					trackingProcess = null;
				}
			}
		};
		trackingProcess.setPriority(Thread.NORM_PRIORITY);
		trackingProcess.start();
	}

	/**
	 * End the trackingProcess.
	 * 
	 * @category scheduling
	 */
	public void end() {
		this.playButton().value_(false);
		this.playButton().visual_(this.playImage());
	}

	/**
	 * Move to next position.
	 * 
	 * @category scheduling
	 */
	public void next() {
		if (this.loopCondition() == $("oneWay")) {
			this.nextOneWay();
		} else if (this.loopCondition() == $("loop")) {
			this.nextLoop();
		} else if (this.loopCondition() == $("backAndForth")) {
			this.nextBackAndForth();
		}
	}

	/**
	 * Action for the next when the loop condition is in OneWay.
	 * 
	 * @category scheduling
	 */
	protected void nextOneWay() {
		if (this.atEnd()) {
			this.end();
		} else {
			this.nextStep();
		}
	}

	/**
	 * Action for the next when the loop condition is in Loop.
	 * 
	 * @category scheduling
	 */
	protected void nextLoop() {
		if (this.atEnd()) {
			this.first();
		} else {
			this.nextStep();
		}
	}

	/**
	 * Action for the next when the loop condition is in BackAndForth.
	 * 
	 * @category scheduling
	 */
	protected void nextBackAndForth() {
		if (loopDirection == $("forth")) {
			this.nextForth();
		} else {
			this.nextBack();
		}
	}

	/**
	 * Action for the next to the back direction.
	 * 
	 * @category scheduling
	 */
	protected void nextBack() {
		if (this.atStart()) {
			loopDirection = $("forth");
		} else {
			this.previousStep();
		}
	}

	/**
	 * Action for the next to the forth direction.
	 * 
	 * @category scheduling
	 */
	protected void nextForth() {
		if (this.atEnd()) {
			loopDirection = $("back");
		} else {
			this.nextStep();
		}
	}

	/**
	 * Action for the basic next.
	 * 
	 * @category scheduling
	 */
	protected void nextStep() {
		this.value_(this.doubleValue() + this.step());
	}

	/**
	 * Move to previous position.
	 * 
	 * @category scheduling
	 */
	public void previous() {
		if (this.loopCondition() == $("oneWay")) {
			this.previousOneWay();
		} else if (this.loopCondition() == $("loop")) {
			this.previousLoop();
		} else if (this.loopCondition() == $("backAndForth")) {
			this.previousBackAndForth();
		}
	}

	/**
	 * Action for the previous when the loop condition is in OneWay.
	 * 
	 * @category scheduling
	 */
	protected void previousOneWay() {
		if (this.atStart()) {
			this.first();
		} else {
			this.previousStep();
		}
	}

	/**
	 * Action for the previous when the loop condition is in Loop.
	 * 
	 * @category scheduling
	 */
	protected void previousLoop() {
		if (this.atStart()) {
			this.last();
		} else {
			this.previousStep();
		}
	}

	/**
	 * Action for the previous when the loop condition is in BackAndForth.
	 * 
	 * @category scheduling
	 */
	protected void previousBackAndForth() {
		double nextValue = (loopDirection == $("forth")) ? (this.doubleValue() - this.step()) : this.doubleValue() + this.step();
		if (this.isIntervalNotEmpty() && (nextValue < this.firstMarker() || nextValue > this.lastMarker()) && (this.firstMarker() <= this.doubleValue()) && (this.doubleValue() <= this.lastMarker())) {
			if (loopDirection == $("forth")) {
				loopDirection = $("back");
			} else {
				loopDirection = $("forth");
			}
			if (loopDirection == $("forth")) {
				nextValue = Math.max(this.firstMarker(), Math.min(this.doubleValue() - this.step(), this.lastMarker()));
			} else {
				nextValue = Math.max(this.firstMarker(), Math.min(this.doubleValue() + this.step(), this.lastMarker()));
			}
		} else if ((nextValue < 0) || (nextValue > 1)) {
			if (loopDirection == $("forth")) {
				loopDirection = $("back");
			} else {
				loopDirection = $("forth");
			}
			if (loopDirection == $("forth")) {
				nextValue = Math.max(0, Math.min(this.doubleValue() - this.step(), 1));
			} else {
				nextValue = Math.max(0, Math.min(this.doubleValue() + this.step(), 1));
			}
		}
		this.value_(nextValue);
	}

	/**
	 * Action for the basic previous
	 * 
	 * @category scheduling
	 */
	protected void previousStep() {
		this.value_(this.doubleValue() - this.step());
	}
}
