package jp.co.sra.jun.goodies.utilities;

import jp.co.sra.smalltalk.StBlockClosure;

import jp.co.sra.jun.system.framework.JunAbstractObject;

/**
 * JunControlUtility class
 * 
 *  @author    nisinaka
 *  @created   2006/09/29 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun697 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunControlUtility.java,v 8.18 2008/02/20 06:32:14 nisinaka Exp $
 */
public class JunControlUtility extends JunAbstractObject {

	/**
	 * Wait 1 millisecond.
	 * 
	 * @category Clock inquiries 
	 */
	public static long NextMillisecondClockValue() {
		return NextMillisecondClockValue_(1);
	}

	/**
	 * Wait the specified tick time.
	 * 
	 * @param tickTime int
	 * @category Clock inquiries
	 */
	public static long NextMillisecondClockValue_(int tickTime) {
		long nextTime = System.currentTimeMillis() + tickTime;
		nextTime = nextTime / tickTime * tickTime;
		WaitUntilTimeMillis_(nextTime);
		return nextTime;
	}

	/**
	 * Evaluate the block closure and wait at least for the specified time.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param millisecondTime int
	 * @category Evaluating
	 */
	public static Object Do_forMilliseconds_(StBlockClosure aBlock, int millisecondTime) {
		if (millisecondTime <= 0) {
			return aBlock.value();
		}

		long timeLimit = System.currentTimeMillis() + millisecondTime;
		Object returnValue = null;
		try {
			returnValue = aBlock.value();
		} finally {
			WaitUntilTimeMillis_(timeLimit);
		}
		return returnValue;
	}

	/**
	 * Evaluate the block closure to handle the specified number of frames per second.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param aNumber int
	 * @category Evaluating
	 */
	public static Object Do_framesPerSecond_(StBlockClosure aBlock, int aNumber) {
		if (aNumber <= 0) {
			return aBlock.value();
		}

		return Do_forMilliseconds_(aBlock, (int) Math.round(1000 / Math.max(0.001, Math.min(aNumber, 1000))));
	}

	/**
	 * Hold for the specified time.
	 * 
	 * @param millisecondTime int
	 * @return boolean
	 * @category Holding
	 */
	public static boolean HoldForMilliseconds_(int millisecondTime) {
		return HoldForMilliseconds_untilDo_(millisecondTime, new StBlockClosure() {
			public Object value() {
				return Boolean.FALSE;
			}
		});
	}

	/**
	 * Evaluate the block and hold for the specified time.
	 * 
	 * @param millisecondTime int
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category Holding
	 */
	public static Object HoldForMilliseconds_do_(int millisecondTime, StBlockClosure aBlock) {
		Thread currentThread = Thread.currentThread();
		int priority = currentThread.getPriority();
		try {
			currentThread.setPriority(Math.max(priority - 1, Thread.MIN_PRIORITY));

			long timeLimit = System.currentTimeMillis() + millisecondTime;
			Object resultValue = aBlock.value();
			while (System.currentTimeMillis() < timeLimit) {
				currentThread.yield();
			}
			return resultValue;

		} finally {
			currentThread.setPriority(priority);
		}
	}

	/**
	 * Evaluate the block and hold for the specified time.
	 * 
	 * @param millisecondTime int
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return boolean
	 * @category Holding
	 */
	public static boolean HoldForMilliseconds_untilDo_(int millisecondTime, StBlockClosure aBlock) {
		Thread currentThread = Thread.currentThread();
		int priority = currentThread.getPriority();
		try {
			currentThread.setPriority(Math.max(priority - 1, Thread.MIN_PRIORITY));

			long timeLimit = System.currentTimeMillis() + millisecondTime;
			while (System.currentTimeMillis() < timeLimit) {
				if (Boolean.TRUE.equals(aBlock.value())) {
					return true;
				}
				currentThread.yield();
			}
			return false;

		} finally {
			currentThread.setPriority(priority);
		}
	}

	/**
	 * Evaluate the block and hold for the specified time.
	 * 
	 * @param millisecondTime int
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return boolean
	 * @category Holding
	 */
	public static boolean HoldForMilliseconds_whileDo_(int millisecondTime, final StBlockClosure aBlock) {
		return HoldForMilliseconds_untilDo_(millisecondTime, new StBlockClosure() {
			public Object value() {
				return Boolean.valueOf(Boolean.FALSE.equals(aBlock.value()));
			}
		});
	}

	/**
	 * Wait for the specified time.
	 * 
	 * @param millisecondTime long
	 * @category Waiting
	 */
	public static void WaitForMilliseconds_(long millisecondTime) {
		WaitUntilTimeMillis_(System.currentTimeMillis() + millisecondTime);
	}

	/**
	 * Wait until the system's millisecond clock reaches the specified time.
	 * 
	 * @param time long
	 * @category Waiting
	 */
	public static void WaitUntilTimeMillis_(long time) {
		long waitTime;
		while ((waitTime = time - System.currentTimeMillis()) > 0) {
			try {
				Thread.sleep(waitTime);
			} catch (InterruptedException e) {
				e.printStackTrace();
			}
		}

		/*
		long waitTime = time - System.currentTimeMillis();
		if (waitTime > 0) {
			try {
				Thread.sleep(waitTime);
			} catch (InterruptedException e) {
			}

			while (time > System.currentTimeMillis()) {
				Thread.yield();
			}
		}
		*/
	}

}
