package jp.co.sra.jun.graphics.item;

import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StComposedText;
import jp.co.sra.smalltalk.StDisplayable;
import jp.co.sra.smalltalk.StObject;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StSymbol;

import jp.co.sra.jun.graphics.abstracts.JunAbstractItem;

/**
 * JunPrimitiveItem class
 * 
 *  @author    Ryouichi Matsuda
 *  @created   2003/12/17 (by Ryouichi Matsuda)
 *  @updated   2006/11/22 (by m-asada)
 *  @version   699 (with StPL8.9) based on Jun637 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunPrimitiveItem.java,v 8.10 2008/02/20 06:32:15 nisinaka Exp $
 */
public class JunPrimitiveItem extends JunAbstractItem {
	protected StDisplayable visualComponent;
	protected StSymbol visualAlignment;

	/**
	 * Create a new instance of <code>JunPrimitiveItem</code> and initialize it.
	 * 
	 * @param origin java.awt.Point
	 * @param extent java.awt.Dimension
	 * @param visualComponent jp.co.sra.smalltalk.StDisplayable
	 * @category Instance creation
	 */
	public JunPrimitiveItem(Point origin, Dimension extent, StDisplayable visualComponent) {
		super(origin, new Point(extent.width, extent.height));
		this.visualComponent_(visualComponent);
	}

	/**
	 * Create a new instance of <code>JunPrimitiveItem</code> and initialize it.
	 * 
	 * @param origin java.awt.Point
	 * @param extent java.awt.Point
	 * @param visualComponent jp.co.sra.smalltalk.StDisplayable
	 * @category Instance creation
	 */
	public JunPrimitiveItem(Point origin, Point extent, StDisplayable visualComponent) {
		super(origin, extent);
		this.visualComponent_(visualComponent);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.graphics.abstracts.JunAbstractVisual#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		visualComponent = null;
		visualAlignment = null;
	}

	/**
	 * Answer the visual alignment.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category accessing
	 */
	public StSymbol visualAlignment() {
		if (visualAlignment == null) {
			visualAlignment = $("topLeft");
		}
		return visualAlignment;
	}

	/**
	 * Set the visual alignment.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @category accessing
	 */
	public void visualAlignment_(StSymbol aSymbol) {
		visualAlignment = aSymbol;
	}

	/**
	 * Answer the visual component.
	 * 
	 * @return visualComponent jp.co.sra.smalltalk.StDisplayable
	 * @category accessing
	 */
	public StDisplayable visualComponent() {
		return visualComponent;
	}

	/**
	 * Set the visual component.
	 * 
	 * @param aVisualComponent jp.co.sra.smalltalk.StDisplayable
	 * @category accessing
	 */
	public void visualComponent_(StDisplayable aVisualComponent) {
		visualComponent = aVisualComponent;
	}

	/**
	 * Display background on graphics context in specified box.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @param insetBox java.awt.Rectangle
	 * @category displaying
	 */
	public void displayBackgroundOn_insetBox_(Graphics graphicsContext, Rectangle insetBox) {
		Graphics gc = graphicsContext.create();
		try {
			if (this.isEmphasis()) {
				gc.setColor(this.selectionBackgroundColor());
			} else {
				gc.setColor(this.backgroundColor());
			}
			gc.fillRect(insetBox.x, insetBox.y, insetBox.width, insetBox.height);
		} finally {
			if (gc != null) {
				gc.dispose();
				gc = null;
			}
		}
	}

	/**
	 * Display border on graphics context in specified box.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @param insetBox java.awt.Rectangle
	 * @category displaying
	 */
	public void displayBorderOn_insetBox_(Graphics graphicsContext, Rectangle insetBox) {
		Graphics gc = graphicsContext.create();
		try {
			gc.setColor(this.borderColor());
			StRectangle[] areas = new StRectangle(this.bounds()).areasOutside_(new StRectangle(insetBox));
			for (int i = 0; i < areas.length; i++) {
				Rectangle box = areas[i].toRectangle();
				gc.fillRect(box.x, box.y, box.width, box.height);
			}
		} finally {
			if (gc != null) {
				gc.dispose();
				gc = null;
			}
		}
	}

	/**
	 * Display on graphics context.
	 * 
	 * @param aGraphicsContext java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_(java.awt.Graphics)
	 * @category displaying
	 */
	public void displayOn_(Graphics aGraphicsContext) {
		Rectangle clipBounds = aGraphicsContext.getClipBounds();
		if (clipBounds == null || clipBounds.intersects(this.bounds())) {
			Rectangle insetBox = this.insideBounds();
			this.displayBorderOn_insetBox_(aGraphicsContext, insetBox);
			this.displayBackgroundOn_insetBox_(aGraphicsContext, insetBox);
			this.displayVisualOn_insetBox_(aGraphicsContext, insetBox);
		}
	}

	/**
	 * Display visual on graphics context in specified box.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @param insetBox java.awt.Rectangle
	 * @category displaying
	 */
	public void displayVisualOn_insetBox_(Graphics graphicsContext, Rectangle insetBox) {
		Graphics gc = graphicsContext.create();
		try {
			StDisplayable aVisual = this.visualComponent();
			if (aVisual == null) {
				return;
			}
			Rectangle clipBounds = gc.getClipBounds();
			if (clipBounds != null) {
				Rectangle clipBox = clipBounds.intersection(insetBox);
				gc.setClip(clipBox.x, clipBox.y, clipBox.width, clipBox.height);
			}
			if (this.isEmphasis()) {
				gc.setColor(this.selectionForegroundColor());
			} else {
				gc.setColor(this.foregroundColor());
			}
			Rectangle aBox = aVisual.bounds();
			if (aVisual instanceof StComposedText && aBox.width > insetBox.width) {
				StComposedText _aVisual = (StComposedText) aVisual;
				Font _textStyle = _aVisual.font();
				FontMetrics _fm = gc.getFontMetrics(_textStyle);
				String aString = ContractString_compositionWidth_textStyle_(_aVisual.string(), insetBox.width, _fm);
				aVisual = (StDisplayable) StObject._New(aVisual.getClass(), new Object[] { aString, _textStyle });
				aBox = aVisual.bounds();
			}

			try {
				Point _align = (Point) new StRectangle(aBox).perform_(this.visualAlignment().toString());
				Point _with = (Point) new StRectangle(insetBox).perform_(this.visualAlignment().toString());
				aBox = new StRectangle(aBox).align_with_(_align, _with).toRectangle();
				aVisual.displayOn_at_(gc, aBox.getLocation());
			} catch (Exception e) {
				e.printStackTrace();
			}
		} finally {
			if (gc != null) {
				gc.dispose();
				gc = null;
			}
		}
	}

	/**
	 * Enumerate all items and evaluate the block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @see jp.co.sra.jun.graphics.abstracts.JunAbstractVisual#itemsDo_(jp.co.sra.smalltalk.StBlockClosure)
	 * @category enumerating
	 */
	public Object itemsDo_(StBlockClosure aBlock) {
		return aBlock.value_(this);
	}

	/**
	 * Answer true if the receiver is primitive object, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.graphics.abstracts.JunAbstractVisual#isPrimitive()
	 * @category testing
	 */
	public boolean isPrimitive() {
		return true;
	}
}
