package jp.co.sra.jun.graphics.navigator;

import java.util.LinkedList;

import jp.co.sra.smalltalk.StBlockClosure;

import jp.co.sra.jun.graphics.list.JunFileList;
import jp.co.sra.jun.system.framework.JunAbstractObject;

/**
 * JunFileListDispenser class
 * 
 *  @author    nisinaka
 *  @created   2004/01/13 (by nisinaka)
 *  @updated   2006/11/29 (by m-asada)
 *  @version   699 (with StPL8.9) based on Jun697 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunFileListDispenser.java,v 8.10 2008/02/20 06:32:16 nisinaka Exp $
 */
public class JunFileListDispenser extends JunAbstractObject {
	protected int cacheSize;
	protected LinkedList cachedFileLists;

	public static JunFileListDispenser SoleFileListDispenser;

	static {
		FlushSole();
	}

	/**
	 * Flush the sole instance of the file list dispenser.
	 * 
	 * @category Class initialization
	 */
	public static void FlushSole() {
		SoleFileListDispenser = null;
	}

	/**
	 * Answer the default instance of JunFileListDispenser.
	 * 
	 * @return jp.co.sra.jun.graphics.navigator.JunFileListDispenser
	 * @category Public access
	 */
	public static JunFileListDispenser Default() {
		if (SoleFileListDispenser == null) {
			SoleFileListDispenser = new JunFileListDispenser();
		}
		return SoleFileListDispenser;
	}

	/**
	 * Answer the default cache size.
	 * 
	 * @return int
	 * @category Defaults
	 */
	protected static int DefaultCacheSize() {
		return 32;
	}

	/**
	 * Create a new instance of JunFileListDispenser and initialize it with a default cache size.
	 * 
	 * @category Instance creation 
	 */
	private JunFileListDispenser() {
		this(DefaultCacheSize());
	}

	/**
	 * Create a new instance of JunFileListDispenser and initialize it with the specified cache size.
	 * 
	 * @param sizeInteger
	 * @category Instance creation
	 */
	private JunFileListDispenser(int sizeInteger) {
		super();
		this.cacheSize_(sizeInteger);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunAbstractObject#initialize()
	 * @category initialize-release
	 */
	protected synchronized void initialize() {
		super.initialize();
		cachedFileLists = null;
		cacheSize = 0;
	}

	/**
	 * Answer the JunFileList at the specified place.
	 * 
	 * @param aFileList jp.co.sra.jun.graphics.graphics.list.JunFileList
	 * @return jp.co.sra.jun.graphics.graphics.list.JunFileList
	 * @category accessing
	 */
	public synchronized JunFileList at_(JunFileList aFileList) {
		return this.at_ifAbsent_(aFileList, new StBlockClosure() {
			public Object value() {
				return null;
			}
		});
	}

	/**
	 * Answer the JunFileList at the specified place.
	 * If absent, evaluate the block and return it as an JunFileList.
	 * 
	 * @param aFileList jp.co.sra.jun.graphics.graphics.list.JunFileList
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.jun.graphics.graphics.list.JunFileList
	 * @category accessing
	 */
	public synchronized JunFileList at_ifAbsent_(JunFileList aFileList, StBlockClosure aBlock) {
		int index = this.cachedFileLists().indexOf(aFileList);
		if (index > 0) {
			return (JunFileList) this.cachedFileLists().get(index);
		} else {
			return (JunFileList) aBlock.value();
		}
	}

	/**
	 * Answer true if the receiver contains the specified file list, otherwise false.
	 *  
	 * @param aFileList jp.co.sra.jun.graphics.graphics.list.JunFileList
	 * @return boolean
	 * @category testing
	 */
	public synchronized boolean includes_(JunFileList aFileList) {
		return this.cachedFileLists().contains(aFileList);
	}

	/**
	 * Add the copy of the file list to the cache.
	 * 
	 * @param aFileList jp.co.sra.jun.graphics.graphics.list.JunFileList
	 * @category adding
	 */
	public synchronized void add_(JunFileList aFileList) {
		if (aFileList.isDirectory() && !aFileList.isRoot() && !this.includes_(aFileList)) {
			JunFileList theFileList = (JunFileList) aFileList.copy();
			if (this.cachedFileLists().size() >= this.cacheSize()) {
				this.cachedFileLists().removeLast();
			}
			this.cachedFileLists().addFirst(theFileList);
		}
	}

	/**
	 * Remove the copy of the file list from the cache.
	 * 
	 * @param aFileList jp.co.sra.jun.graphics.graphics.list.JunFileList
	 * @category removing
	 */
	public synchronized void remove_(JunFileList aFileList) {
		if (aFileList.isDirectory()) {
			JunFileList theFileList = this.at_(aFileList);
			if (theFileList != null) {
				this.cachedFileLists().remove(theFileList);
			}
		}
	}

	/**
	 * Answre my current cache size.
	 * 
	 * @return int
	 * @category caching
	 */
	public int cacheSize() {
		if (cacheSize <= 0) {
			cacheSize = DefaultCacheSize();
		}
		return cacheSize;
	}

	/**
	 * Set my new cache size.
	 * 
	 * @param anInteger int
	 * @category caching
	 */
	public void cacheSize_(int anInteger) {
		cacheSize = Math.max(1, anInteger);
	}

	/**
	 * Answer my cached file lists.
	 * 
	 * @return java.util.LinkedList
	 * @category caching
	 */
	public synchronized LinkedList cachedFileLists() {
		if (cachedFileLists == null) {
			cachedFileLists = new LinkedList();
		}
		return cachedFileLists;
	}

	/**
	 * Flush my cached file lists.
	 * 
	 * @category caching
	 */
	public synchronized void flushCachedFileLists() {
		cachedFileLists = null;
	}
}
