package jp.co.sra.jun.octree.basic;

import java.awt.Color;
import java.util.Vector;
import jp.co.sra.jun.geometry.basic.*;
import jp.co.sra.jun.geometry.boundaries.*;
import jp.co.sra.jun.octree.nodes.JunOctreeNode;
import jp.co.sra.jun.opengl.objects.*;
import jp.co.sra.smalltalk.*;

// import jp.co.sra.jun.octree.editor.JunOctreeEditModel;

/**
 * JunOctree class
 * 
 *  @author    nisinaka
 *  @created   1999/12/21 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOctree.java,v 8.10 2008/02/20 06:32:17 nisinaka Exp $
 */
public class JunOctree extends StObject {
	/** The node of the octree. */
	protected JunOctreeNode node = null;

	/** The bounding box of the octree. */
	protected Jun3dBoundingBox bounds = null;

	/**
	 * Create an instance of JunOctree and initialize it.
	 * 
	 * @param aJun3dBoundingBox jp.co.sra.jun.geometry.basic.Jun3dBoundingBox
	 * 
	 * @return jp.co.sra.jun.octree.basic.JunOctree
	 */
	public static JunOctree Bounds_(Jun3dBoundingBox aJun3dBoundingBox) {
		JunOctree octree = new JunOctree();
		octree.setBounds_(aJun3dBoundingBox);

		return octree;
	}

	/**
	 * Create an instance of JunOctree and initialize it.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param anInteger int
	 * @param aJun3dBoundingBox jp.co.sra.jun.geometry.basic.Jun3dBoundingBox
	 * 
	 * @return jp.co.sra.jun.octree.basic.JunOctree
	 */
	public static JunOctree FromBlock_depth_bounds_(StBlockClosure aBlock, int anInteger, Jun3dBoundingBox aJun3dBoundingBox) {
		JunOctree octree = JunOctree.Bounds_(aJun3dBoundingBox);
		octree.block_depth_(aBlock, anInteger);

		return octree;
	}

	/**
	 * Create a unit JunOctree.
	 * 
	 * @return jp.co.sra.jun.octree.basic.JunOctree
	 */
	public static JunOctree Unity() {
		return JunOctree.Bounds_(Jun3dBoundingBox.Origin_corner_(Jun3dPoint.Zero(), Jun3dPoint.Unity()));
	}

	/**
	 * Convert to JunOpenGL3dObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		final Vector components = new Vector();
		this.originExtentTipsDo_(new StBlockClosure() {
			public Object value_value_value_(Object o1, Object o2, Object o3) {
				Jun3dPoint origin = (Jun3dPoint) o1;
				Jun3dPoint extent = (Jun3dPoint) o2;
				JunOctreeNode tip = (JunOctreeNode) o3;
				JunOpenGL3dObject component = tip.asJunOpenGL3dObject();

				if (component != null) {
					components.addElement(component.scaledBy_(extent).translatedBy_(origin));
				}

				return null;
			}
		});

		return new JunOpenGL3dCompoundObject(components);
	}

	/**
	 * Make the octree to be compact.
	 */
	public void beCompact() {
		node.beCompact();
	}

	/**
	 * Set the block and the accuracy.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param aNumber double
	 */
	public void block_accuracy_(StBlockClosure aBlock, double aNumber) {
		node.origin_extent_accuracy_block_(this.origin(), this.extent(), aNumber, aBlock);
	}

	/**
	 * Set the block and the depth.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param anInteger int
	 */
	public void block_depth_(StBlockClosure aBlock, int anInteger) {
		node.origin_extent_depth_block_(this.origin(), this.extent(), anInteger, aBlock);
	}

	/**
	 * Answer the bounds.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dBoundingBox
	 */
	public Jun3dBoundingBox bounds() {
		return bounds;
	}

	/**
	 * Set the block and the depth.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param aNumber double
	 */
	public void brepsBlock_accuracy_(StBlockClosure aBlock, double aNumber) {
		node.brepsOrigin_extent_accuracy_block_(this.origin(), this.extent(), aNumber, aBlock);
	}

	/**
	 * Set the block and the depth.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param anInteger int
	 */
	public void brepsBlock_depth_(StBlockClosure aBlock, int anInteger) {
		node.brepsOrigin_extent_depth_block_(this.origin(), this.extent(), anInteger, aBlock);
	}

	/**
	 * Answer the center point of the octree.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 */
	public Jun3dPoint center() {
		return this.bounds().center();
	}

	/**
	 * Answer true if the octree contains the Jun3dPoint.
	 * 
	 * @param aJun3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * 
	 * @return boolean
	 */
	public boolean containsPoint_(Jun3dPoint aJun3dPoint) {
		return this.bounds().containsPoint_(aJun3dPoint) && this.tipForX_y_z_(aJun3dPoint.x(), aJun3dPoint.y(), aJun3dPoint.z()).containsPoint_(aJun3dPoint);
	}

	/**
	 * Answer the corner point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 */
	public Jun3dPoint corner() {
		return (Jun3dPoint) this.bounds().corner();
	}

	/**
	 * Answer the depth of the octree.
	 * 
	 * @return double
	 */
	public double depth() {
		return this.bounds().depth();
	}

	/**
	 * Divide the node at the specified point.
	 * 
	 * @param aJun3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 */
	public void divideTipForPoint_(Jun3dPoint aJun3dPoint) {
		this.tipForX_y_z_(aJun3dPoint.x(), aJun3dPoint.y(), aJun3dPoint.z()).beDivided();
	}

	/**
	 * Edit the JunOctree.
	 */
	public void edit() {
		// JunOctreeEditModel.Edit_(this);
	}

	/**
	 * Empty the node at the specified point.
	 * 
	 * @param aJun3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 */
	public void emptyTipForPoint_(Jun3dPoint aJun3dPoint) {
		this.tipForX_y_z_(aJun3dPoint.x(), aJun3dPoint.y(), aJun3dPoint.z()).beEmpty();
	}

	/**
	 * Answer the extent.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 */
	public Jun3dPoint extent() {
		return (Jun3dPoint) this.bounds().extent();
	}

	/**
	 * Fill the node at the specified point.
	 * 
	 * @param aJun3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 */
	public void fillTipForPoint_(Jun3dPoint aJun3dPoint) {
		this.tipForX_y_z_(aJun3dPoint.x(), aJun3dPoint.y(), aJun3dPoint.z()).beFilled();
	}

	/**
	 * Answer the height of the octree.
	 * 
	 * @return double
	 */
	public double height() {
		return this.bounds().height();
	}

	/**
	 * Answer the origin point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 */
	public Jun3dPoint origin() {
		return (Jun3dPoint) this.bounds().origin();
	}

	/**
	 * Enumerate the nodes and evaluate the block closure.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * 
	 * @return java.lang.Object
	 */
	public Object originExtentNodesDo_(StBlockClosure aBlock) {
		return node.origin_extent_nodesDo_(this.origin(), this.extent(), aBlock);
	}

	/**
	 * Emumerate all nodes except aJunOctreeNode and evaluate the block.
	 * 
	 * @param aJunOctreeNode jp.co.sra.jun.octree.nodes.JunOctreeNode
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * 
	 * @return java.lang.Object
	 */
	public Object originExtentNodesWithout_do_(JunOctreeNode aJunOctreeNode, StBlockClosure aBlock) {
		return node.origin_extent_nodesWithout_do_(this.origin(), this.extent(), aJunOctreeNode, aBlock);
	}

	/**
	 * Emumerate all nodes except aCollectionOfJunOctreeNode and evaluate the
	 * block.
	 * 
	 * @param aCollectionOfJunOctreeNode java.util.Vector
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * 
	 * @return java.lang.Object
	 */
	public Object originExtentNodesWithoutAll_do_(Vector aCollectionOfJunOctreeNode, StBlockClosure aBlock) {
		return node.origin_extent_nodesWithoutAll_do_(this.origin(), this.extent(), aCollectionOfJunOctreeNode, aBlock);
	}

	/**
	 * Enumerate the tips and evaluate the block closure.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * 
	 * @return java.lang.Object
	 */
	public Object originExtentTipsDo_(StBlockClosure aBlock) {
		return node.origin_extent_tipsDo_(this.origin(), this.extent(), aBlock);
	}

	/**
	 * Set the node.
	 * 
	 * @param aJunOctreeNode jp.co.sra.jun.octree.nodes.JunOctreeNode
	 */
	public void setNode_(JunOctreeNode aJunOctreeNode) {
		node = aJunOctreeNode;
	}

	/**
	 * Show the JunOctree as a JunOpenGL3dObject.
	 */
	public void show() {
		JunOpenGL3dObject anObject = this.asJunOpenGL3dObject();
		anObject.paint_(Color.white);
		anObject.show();
	}

	/**
	 * Answer the width of the octree.
	 * 
	 * @return double
	 */
	public double width() {
		return this.bounds().width();
	}

	/**
	 * Create a new node.
	 * 
	 * @return jp.co.sra.jun.octree.nodes.JunOctreeNode
	 */
	protected JunOctreeNode _newNode() {
		return JunOctreeNode.Filled();
	}

	/**
	 * Set the bounds.
	 * 
	 * @param aJun3dBoundingBox jp.co.sra.jun.geometry.basic.Jun3dBoundingBox
	 */
	protected void setBounds_(Jun3dBoundingBox aJun3dBoundingBox) {
		node = this._newNode();
		bounds = aJun3dBoundingBox;
	}

	/**
	 * Answer the node at the specified point.
	 * 
	 * @param x double
	 * @param y double
	 * @param z double
	 * 
	 * @return jp.co.sra.jun.octree.nodes.JunOctreeNode
	 */
	protected JunOctreeNode tipForX_y_z_(double x, double y, double z) {
		JunOctreeNode n = node;

		while (n.isTip() == false) {
			int childIndex = 0;

			if (x >= 0.5d) {
				childIndex++;
				x -= 0.5d;
			}

			x *= 2.0d;

			if (y >= 0.5d) {
				childIndex += 2;
				y -= 0.5d;
			}

			y *= 2.0d;

			if (z >= 0.5d) {
				childIndex += 4;
				z -= 0.5d;
			}

			z *= 2.0d;
			n = n.childAt_(childIndex);
		}

		return n;
	}
}
