package jp.co.sra.jun.opengl.flux;

import java.util.ArrayList;
import java.util.Collection;

import jp.co.sra.jun.geometry.abstracts.JunGeometry;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.curves.Jun3dLine;
import jp.co.sra.jun.geometry.surfaces.JunPlane;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.system.framework.JunAbstractObject;

/**
 * JunOpenGLFluxMaker class
 * 
 *  @author    MATSUDA Ryouichi
 *  @created   1998/12/02 (by MATSUDA Ryouichi)
 *  @updated   2006/10/12 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun697 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGLFluxMaker.java,v 8.12 2008/02/20 06:32:34 nisinaka Exp $
 */
public class JunOpenGLFluxMaker extends JunAbstractObject {

	protected ArrayList lineSegments; // (Jun3dPoint[2])[]
	protected JunCoordinateAngles originalCoordinateAngles;
	protected JunCoordinateAngles[] coordinateAnglesCollection;
	protected Jun3dTransformation[] rotationTransformations;
	protected Jun3dTransformation[] revisionTransformations;
	protected Jun3dTransformation[] scalingTransformations;
	protected Jun3dTransformation[] translationTransformations;
	protected Jun3dTransformation[] segmentTransformations;

	/**
	 * Create a new instance of <code>JunOpenGLFluxMaker</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunOpenGLFluxMaker() {
		super();
	}

	/**
	 * Create a new instance of <code>JunOpenGLFluxMaker</code> and initialize it.
	 *
	 * @param lineSegments jp.co.sra.jun.geometry.basic.Jun3dPoint[][]
	 * @category Instance creation
	 */
	public JunOpenGLFluxMaker(Jun3dPoint[][] lineSegments) {
		this();
		this.lineSegments_(lineSegments);
	}

	/**
	 * Create a new instance of JunOpenGLFluxMaker and initialize it.
	 *
	 * @param aCollectionOfLineSegments java.util.Collection
	 * @category Instance creation
	 */
	public JunOpenGLFluxMaker(Collection aCollectionOfLineSegments) {
		this((Jun3dPoint[][]) aCollectionOfLineSegments.toArray(new Jun3dPoint[aCollectionOfLineSegments.size()][]));
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunAbstractObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		lineSegments = null;
		this.flushTransformations();
	}

	/**
	 * Answer my current line segments.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[][]
	 * @category accessing
	 */
	public Jun3dPoint[][] lineSegments() {
		return (Jun3dPoint[][]) this._lineSegments().toArray(new Jun3dPoint[this._lineSegments().size()][]);
	}

	/**
	 * Set my new line segments.
	 * 
	 * @param lineSegments jp.co.sra.jun.geometry.basic.Jun3dPoint[][]
	 * @category accessing
	 */
	public void lineSegments_(Jun3dPoint[][] lineSegments) {
		if (lineSegments == null) {
			return;
		}

		for (int i = 0; i < lineSegments.length; i++) {
			this.add_(lineSegments[i]);
		}
	}

	/**
	 * Answer the array list of the line segments.
	 * 
	 * @return java.util.ArrayList
	 * @category accessing
	 */
	protected ArrayList _lineSegments() {
		if (lineSegments == null) {
			lineSegments = new ArrayList();
		}
		return lineSegments;
	}

	/**
	 * Answer the direction vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint directionVector() {
		return new Jun3dPoint(1, 0, 0);
	}

	/**
	 * Answer the unit vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint unitVector() {
		return new Jun3dPoint(0, 0, 1);
	}

	/**
	 * Answer my segment transformations.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation[]
	 * @category accessing
	 */
	public Jun3dTransformation[] segmentTransformations() {
		if (segmentTransformations == null) {
			this.computeTransformations();
		}
		return segmentTransformations;
	}

	/**
	 * Add a line segment.
	 * 
	 * @param aSegment jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category adding
	 */
	public void add_(Jun3dPoint[] aSegment) {
		this._lineSegments().add(aSegment);
		this.flushTransformations();
	}

	/**
	 * Answer the original coordinate angles.
	 * 
	 * @return jp.co.sra.jun.opengl.flux.JunCoordinateAngles
	 * @category transformations
	 */
	public JunCoordinateAngles originalCoordinateAngles() {
		if (originalCoordinateAngles == null) {
			this.computeTransformations();
		}
		return originalCoordinateAngles;
	}

	/**
	 * Answer the collection of coordinate angles.
	 * 
	 * @return jp.co.sra.jun.opengl.flux.JunCoordinateAngles[]
	 * @category transformations
	 */
	public JunCoordinateAngles[] coordinateAnglesCollection() {
		if (coordinateAnglesCollection == null) {
			this.computeTransformations();
		}
		return coordinateAnglesCollection;
	}

	/**
	 * Answer the rotation transformations.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation[]
	 * @category transformations
	 */
	public Jun3dTransformation[] rotationTransformations() {
		if (rotationTransformations == null) {
			this.computeTransformations();
		}
		return rotationTransformations;
	}

	/**
	 * Answer the revision transformations.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation[]
	 * @category transformations
	 */
	public Jun3dTransformation[] revisionTransformations() {
		if (revisionTransformations == null) {
			this.computeTransformations();
		}
		return revisionTransformations;
	}

	/**
	 * Answer the scaling transformations.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation[]
	 * @category transformations
	 */
	public Jun3dTransformation[] scalingTransformations() {
		if (scalingTransformations == null) {
			this.computeTransformations();
		}
		return scalingTransformations;
	}

	/**
	 * Answer the translation transformations.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation[]
	 * @category transformations
	 */
	public Jun3dTransformation[] translationTransformations() {
		if (translationTransformations == null) {
			this.computeTransformations();
		}
		return translationTransformations;
	}

	/**
	 * Flush transformations.
	 * 
	 * @category private
	 */
	protected void flushTransformations() {
		originalCoordinateAngles = null;
		coordinateAnglesCollection = null;
		rotationTransformations = null;
		revisionTransformations = null;
		scalingTransformations = null;
		translationTransformations = null;
		segmentTransformations = null;
	}

	/**
	 * Compute transformations.
	 * 
	 * @category private
	 */
	protected void computeTransformations() {
		originalCoordinateAngles = JunCoordinateAngles.V1_v2_(this.unitVector(), this.directionVector());
		coordinateAnglesCollection = new JunCoordinateAngles[this._lineSegments().size()];
		rotationTransformations = new Jun3dTransformation[this._lineSegments().size()];
		revisionTransformations = new Jun3dTransformation[this._lineSegments().size()];
		scalingTransformations = new Jun3dTransformation[this._lineSegments().size()];
		translationTransformations = new Jun3dTransformation[this._lineSegments().size()];
		segmentTransformations = new Jun3dTransformation[this._lineSegments().size()];

		for (int index = 0; index < this._lineSegments().size(); index++) {
			Jun3dPoint[] lineSegment = (Jun3dPoint[]) this._lineSegments().get(index);
			Jun3dPoint aPoint0 = lineSegment[0];
			Jun3dPoint aPoint1 = lineSegment[1];
			double aLength = aPoint0.distance_(aPoint1);
			JunCoordinateAngles coordinateAngles = JunCoordinateAngles.V1_v2_((Jun3dPoint) aPoint1.minus_(aPoint0), this.directionVector());
			coordinateAnglesCollection[index] = coordinateAngles;

			Jun3dTransformation rotationT = Jun3dTransformation.Unity();
			rotationT = rotationT.product_(originalCoordinateAngles.asInverseTransformation());
			rotationT = rotationT.product_(coordinateAngles.asTransformation());
			rotationTransformations[index] = rotationT;

			Jun3dPoint aVector = this.directionVector().transform_(rotationT);
			JunPlane aPlane = new JunPlane(new Jun3dPoint(0, 0, 0), this.unitVector(), this.directionVector());
			Jun3dLine rotationLine = new Jun3dLine(new Jun3dPoint(0, 0, 0), (Jun3dPoint) this.unitVector().transform_(rotationT));
			Jun3dTransformation revisionT = this.transformationToBeParallel_and_byRotationAround_(aPlane, aVector, rotationLine);
			lineSegment = (Jun3dPoint[]) this._lineSegments().get(index);
			aPoint0 = lineSegment[0];
			aPoint1 = lineSegment[1];
			if (aPoint0.z() < aPoint1.z()) {
				Jun3dTransformation verticalT;
				verticalT = Jun3dTransformation.Rotate_around_(JunAngle.FromDeg_(180), rotationLine);
				revisionT = (Jun3dTransformation) revisionT.product_(verticalT);
			}

			revisionTransformations[index] = revisionT;

			Jun3dTransformation scalingT = Jun3dTransformation.Scale_(new Jun3dPoint(aLength, aLength, aLength));
			scalingTransformations[index] = scalingT;

			Jun3dTransformation translationT = Jun3dTransformation.Translate_(aPoint0);
			translationTransformations[index] = translationT;

			Jun3dTransformation aTransformation = Jun3dTransformation.Unity();
			aTransformation = aTransformation.product_(scalingT);
			aTransformation = aTransformation.product_(rotationT);
			aTransformation = aTransformation.product_(revisionT);
			aTransformation = aTransformation.product_(translationT);
			segmentTransformations[index] = aTransformation;
		}
	}

	/**
	 * Transformation to be parallel JunPlane and vector by rotation around rotationLine.
	 * 
	 * @param aPlane jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @param aVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param rotationLine jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category private
	 */
	protected Jun3dTransformation transformationToBeParallel_and_byRotationAround_(JunPlane aPlane, Jun3dPoint aVector, Jun3dLine rotationLine) {
		Jun3dTransformation revisionT = Jun3dTransformation.Unity();
		JunAngle anAngle = JunAngle.FromDeg_(0);
		Jun3dPoint aPoint = aVector;
		Jun3dLine aLine = new Jun3dLine(new Jun3dPoint(0, 0, 0), aPoint);

		final double[] _steps = { 90.0d, 45.0d, 10.0d, 5.0d, 1.0d, 0.5d };
		for (int i = 0; i < _steps.length; i++) {
			double step = _steps[i];
			while (Math.abs(aPlane.angleWithLine_(aLine).deg()) > step) {
				anAngle = JunAngle.FromDeg_(anAngle.deg() + step);
				revisionT = Jun3dTransformation.Rotate_around_(anAngle, rotationLine);
				aPoint = aVector.transform_(revisionT);
				aLine = new Jun3dLine(new Jun3dPoint(0, 0, 0), aPoint);
			}
		}

		int accuracyPlace = (int) Math.round(Math.abs(Math.log(JunGeometry.Accuracy()) / Math.log(10))) - 1;
		for (int it = 1; it <= accuracyPlace; it++) {
			double denominator = Math.pow(10.0d, it);
			final double[] _each = { 1.0d, 0.5d };
			for (int i = 0; i < _each.length; i++) {
				double step = _each[i] / denominator;
				while (Math.abs(aPlane.angleWithLine_(aLine).deg()) > step) {
					anAngle = JunAngle.FromDeg_(anAngle.deg() + step);
					revisionT = Jun3dTransformation.Rotate_around_(anAngle, rotationLine);
					aPoint = aVector.transform_(revisionT);
					aLine = new Jun3dLine(new Jun3dPoint(0, 0, 0), aPoint);
				}
			}
		}

		if (aPoint.x() < 0) {
			anAngle = JunAngle.FromDeg_(180).minus_(anAngle).negated();
			revisionT = Jun3dTransformation.Rotate_around_(anAngle, rotationLine);
		}

		return revisionT;
	}

}
