package jp.co.sra.jun.opengl.grapher;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.text.DateFormat;
import java.util.Date;
import java.util.StringTokenizer;
import java.util.Vector;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.menu.MenuPerformer;
import jp.co.sra.smalltalk.menu.StCheckBoxMenuItem;
import jp.co.sra.smalltalk.menu.StMenu;
import jp.co.sra.smalltalk.menu.StMenuBar;
import jp.co.sra.smalltalk.menu.StMenuItem;

import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.goodies.files.JunFileModel;
import jp.co.sra.jun.graphics.navigator.JunFileRequesterDialog;
import jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel;
import jp.co.sra.jun.opengl.display.JunOpenGLShowModel;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.system.framework.JunDialog;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunOpenGL3dGrapher class
 * 
 *  @author    Hirotsugu Kondo
 *  @created   1998/12/07 (by Hirotsugu Kondo)
 *  @updated   1999/06/25 (by nisinaka)
 *  @updated   2003/05/15 (by nisinaka)
 *  @updated   2005/03/02 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun475 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGL3dGrapher.java,v 8.11 2008/02/20 06:32:34 nisinaka Exp $
 */
public class JunOpenGL3dGrapher extends JunOpenGLDisplayModel {

	/**
	 * Create a new instance of <code>JunOpenGL3dGrapher</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunOpenGL3dGrapher() {
		super();
	}

	/**
	 * Add the JunOpenGL3dObject as one of the selected objects.
	 * 
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#addSelectedObject_(jp.co.sra.jun.opengl.objects.JunOpenGL3dObject)
	 * @category selecting
	 */
	public void addSelectedObject_(JunOpenGL3dObject a3dObject) {
		JunOpenGL3dObject displayObject = this.displayObject();

		if (displayObject == null) {
			return;
		}

		if (displayObject.isPrimitive()) {
			return;
		}

		JunOpenGL3dNode[] nodes = ((JunOpenGL3dGraph) displayObject).nodes();
		int size = nodes.length;
		JunOpenGL3dNode node = null;

		for (int index = 0; index < size; index++) {
			JunOpenGL3dNode each = nodes[index];

			if (each.displayObject() == a3dObject) {
				node = each;
			}
		}

		if (node == null) {
			return;
		}

		super.addSelectedObject_(a3dObject);
	}

	/**
	 * Arrange the receiver's graph.
	 * 
	 * @category menu messages
	 */
	public void arrangeGraph() {
		JunOpenGL3dGraph graph = (JunOpenGL3dGraph) this.displayObject();

		if (graph == null) {
			return;
		}

		this.clearSelectedObjects();
		graph.arrange();
		this.resetView();
	}

	/**
	 * Arrange the receiver's graph.
	 * 
	 * @category menu messages
	 */
	public void arrangeGraph2() {
		JunOpenGL3dGraph graph = (JunOpenGL3dGraph) this.displayObject();

		if (graph == null) {
			return;
		}

		this.clearSelectedObjects();
		graph.arrange2();
		this.resetView();
	}

	/**
	 * Arrange the receiver's balance graph.
	 * 
	 * @category menu messages
	 */
	public void arrangeGraphBalance() {
		JunOpenGL3dGraph graph = (JunOpenGL3dGraph) this.displayObject();

		if (graph == null) {
			return;
		}

		graph.arrangeBalance_(!graph.arrangeBalance());
		graph.arrange();
		this.clearSelectedObjects();
		this.updateGraphMenuIndication();
		this.changed_($("object"));
	}

	/**
	 * Arrange the receiver's crank arcs graph.
	 * 
	 * @category menu messages
	 */
	public void arrangeGraphCrankArcs() {
		JunOpenGL3dGraph graph = (JunOpenGL3dGraph) this.displayObject();

		if (graph == null) {
			return;
		}

		boolean bool = !this.isCrankArcs();
		JunOpenGL3dArc[] arcs = graph.arcs();
		int arcSize = arcs.length;

		for (int index = 0; index < arcSize; index++) {
			arcs[index].crank_(bool);
		}

		graph.flushDisplayObject();
		this.updateGraphMenuIndication();
		this.changed_($("object"));
	}

	/**
	 * Arrange the receiver's interval graph.
	 * 
	 * @category menu messages
	 */
	public void arrangeGraphInterval() {
		JunOpenGL3dGraph graph = (JunOpenGL3dGraph) this.displayObject();

		if (graph == null) {
			return;
		}

		Jun3dPoint interval = graph.arrangeInterval();
		String string = new Double(interval.x()).toString() + "," + new Double(interval.y()).toString() + "," + new Double(interval.z()).toString();
		string = JunDialog.Request_(JunSystem.$String("Arrange interval?") + " (x , y , z)", string);

		if (string == null) {
			return;
		}

		if (string.length() == 0) {
			return;
		}

		float x;
		float y;
		float z;

		try {
			StringTokenizer st = new StringTokenizer(string, "(), ");
			x = Float.valueOf(st.nextToken()).floatValue();
			y = Float.valueOf(st.nextToken()).floatValue();
			z = Float.valueOf(st.nextToken()).floatValue();
		} catch (Exception e) {
			return;
		}

		Jun3dPoint point = new Jun3dPoint(x, y, z);
		graph.arrangeInterval_(point);
		graph.arrange();
		this.clearSelectedObjects();
		this.changed_($("object"));
	}

	/**
	 * Arrange the receiver's graph with interim.
	 * 
	 * @category menu messages
	 */
	public void arrangeGraphWithInterim() {
		final JunOpenGL3dGraph graph = (JunOpenGL3dGraph) this.displayObject();

		if (graph == null) {
			return;
		}

		this.clearSelectedObjects();

		final JunOpenGL3dGrapher this_ = this;
		graph.arrange_(new StBlockClosure() {
			public Object value_(Object arc) {
				graph.flushDisplayObject();
				this_.resetView();

				return null;
			}
		});
		this.resetView();
	}

	/**
	 * Arrange the receiver's properties.
	 * 
	 * @category menu messages
	 */
	public void arrangeProperties() {
		JunOpenGL3dGraph graph = (JunOpenGL3dGraph) this.displayObject();

		if (graph == null) {
			return;
		}

		System.out.print(graph.arrangeFormat());
	}

	/**
	 * Clear the receiver's display object.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#clearObject()
	 * @category menu messages
	 */
	public void clearObject() {
		System.out.print("do nothing");
	}

	/**
	 * Copy the receiver's object.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#copyObject()
	 * @category menu messages
	 */
	public void copyObject() {
		System.out.print("do nothing");
	}

	/**
	 * Cut the selected objects.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#cutObject()
	 * @category menu messages
	 */
	public void cutObject() {
		System.out.print("do nothing");
	}

	/**
	 * Answer true if the receiver's graph is arrange balance, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isArrangeBalance() {
		JunOpenGL3dGraph graph = (JunOpenGL3dGraph) this.displayObject();

		if (graph == null) {
			return new JunOpenGL3dGraph().arrangeBalance();
		}

		return graph.arrangeBalance();
	}

	/**
	 * Answer true if the receiver's graph is crank arcs, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isCrankArcs() {
		JunOpenGL3dGraph graph = (JunOpenGL3dGraph) this.displayObject();

		if (graph == null) {
			return new JunOpenGL3dArc().crank();
		}

		JunOpenGL3dArc[] arcs = graph.arcs();
		int size = arcs.length;

		for (int index = 0; index < size; index++) {
			boolean crank = arcs[index].crank();

			if (crank) {
				return crank;
			}
		}

		return false;
	}

	/**
	 * Answer whether the display object is empty or not.
	 *
	 * @return boolean
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#isEmpty()
	 * @category testing
	 */
	public boolean isEmpty() {
		if (this.displayObject() == null) {
			return true;
		}

		if (this.displayObject().isCompound() == false) {
			return false;
		}

		return ((JunOpenGL3dGraph) this.displayObject()).components().length == 0;
	}

	/**
	 * Load a JunOpenGL3dGraph from the reader.
	 * 
	 * @param aReader java.io.BufferedReader
	 * @return jp.co.sra.jun.opengl.grapher.JunOpenGL3dGraph
	 * @exception java.io.IOException
	 * @category reading
	 */
	public JunOpenGL3dGraph loadFromLGT10_(BufferedReader aReader) throws IOException {
		StringWriter aWriter = new StringWriter();
		JunOpenGL3dGraph anObject = null;

		try {
			int ch;

			while ((ch = aReader.read()) > 0) {
				aWriter.write(ch);
			}

			aWriter.flush();
			anObject = (JunOpenGL3dGraph) JunOpenGL3dGraph.LoadFrom_(aWriter.toString());
		} finally {
			aWriter.close();
		}

		return anObject;
	}

	/**
	 * Open LGT file.
	 * 
	 * @category menu messages
	 */
	public void openLGT() {
		this.openLGT10();
	}

	/**
	 * Open LGT 1.0 file.
	 * 
	 * @category menu messages
	 */
	public void openLGT10() {
		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType(JunSystem.$String("<1p> files", null, "LGT"), new String[] { "*.lgt", "*.LGT" }) };
		File file = JunFileRequesterDialog.RequestFile(JunSystem.$String("Select an <1p> file.", null, "LGT"), fileTypes, fileTypes[0]);
		if (file == null) {
			return;
		}

		try {
			JunOpenGL3dGraph object = this.readFromLGT10_(file);
			if (object == null) {
				return;
			}

			this.displayObject_(object);
			this.resetView();
			JunOpenGLShowModel model = this.showModel();
			if (model != null) {
				model.resetView();
			}
		} catch (Exception e) {
		}
	}

	/**
	 * Paste the object from the clipboard.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#pasteObject()
	 * @category menu messages
	 */
	public void pasteObject() {
		System.out.print("do nothing");
	}

	/**
	 * Read a LGT1.0 file and create a graph object.
	 * 
	 * @param aFile java.io.File
	 * @return jp.co.sra.jun.opengl.flux.JunOpenGLFluxObject
	 * @exception java.io.IOException
	 * @category reading
	 */
	public JunOpenGL3dGraph readFromLGT10_(File aFile) throws IOException {
		FileReader aFileReader;

		try {
			aFileReader = new FileReader(aFile);
		} catch (FileNotFoundException e) {
			return null;
		}

		BufferedReader aReader = new BufferedReader(aFileReader);
		JunOpenGL3dGraph anObject = null;

		try {
			anObject = this.loadFromLGT10_(aReader);
		} finally {
			aReader.close();
		}

		return anObject;
	}

	/**
	 * Remove the JunOpenGL3dObject from the selected objects.
	 * 
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#removeSelectedObject_(jp.co.sra.jun.opengl.objects.JunOpenGL3dObject)
	 * @category selecting
	 */
	public void removeSelectedObject_(JunOpenGL3dObject a3dObject) {
		JunOpenGL3dObject displayObject = this.displayObject();

		if (displayObject == null) {
			return;
		}

		if (displayObject.isPrimitive()) {
			return;
		}

		JunOpenGL3dNode[] nodes = ((JunOpenGL3dGraph) displayObject).nodes();
		int size = nodes.length;
		JunOpenGL3dNode node = null;

		for (int index = 0; index < size; index++) {
			JunOpenGL3dNode each = nodes[index];

			if (each.displayObject() == a3dObject) {
				node = each;
			}
		}

		if (node == null) {
			return;
		}

		if (this.selectedObjects().contains(a3dObject)) {
			this.selectedObjects().removeElement(a3dObject);
		}
	}

	/**
	 * Save the graph object to a file.
	 * 
	 * @category menu messages
	 */
	public void saveLGT() {
		this.saveLGT10();
	}

	/**
	 * Save the graph object to a LGT1.0 file.
	 * 
	 * @throws SmalltalkException DOCUMENT ME!
	 * @category menu messages
	 */
	public void saveLGT10() {
		JunOpenGL3dGraphAbstract graph = (JunOpenGL3dGraphAbstract) this.displayObject();
		if (graph == null) {
			return;
		}

		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType(JunSystem.$String("<1p> files", null, "LGT"), new String[] { "*.lgt", "*.LGT" }) };
		File file = JunFileRequesterDialog.RequestNewFile(JunSystem.$String("Input an <1p> file.", null, "LGT"), fileTypes, fileTypes[0]);
		if (file == null) {
			return;
		}

		try {
			this.writeToLGT10_object_(file, graph);
		} catch (IOException e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Write the JunOpenGL3dObject to the writer.
	 * 
	 * @param aWriter java.io.BufferedWriter
	 * @param aGraph jp.co.sra.jun.opengl.grapher.JunOpenGL3dGraphAbstract
	 * @exception java.io.IOException
	 * @category writing
	 */
	public void saveToLGT10_object_(BufferedWriter aWriter, JunOpenGL3dGraphAbstract aGraph) throws IOException {
		aWriter.write(this.defaultStampForLGT10());
		aGraph.saveOn_(aWriter);
	}

	/**
	 * Save the display object to the VRML1.0 file.
	 * 
	 * @throws SmalltalkException DOCUMENT ME!
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#saveWRL10()
	 * @category menu messages
	 */
	public void saveWRL10() {
		JunOpenGL3dGraphAbstract graph = (JunOpenGL3dGraphAbstract) this.displayObject();
		if (graph == null) {
			return;
		}

		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType(JunSystem.$String("<1p> files", null, "VRML"), new String[] { "*.wrl", "*.WRL" }) };
		File file = JunFileRequesterDialog.RequestNewFile(JunSystem.$String("Input a <1p> file.", null, "WRL"), new File(graph.name() + ".wrl"), fileTypes, fileTypes[0]);
		if (file == null) {
			return;
		}

		try {
			this.writeToWRL20_object_(file, graph.displayObject());
		} catch (IOException e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Save the display object to the VRML2.0 file.
	 * 
	 * @throws SmalltalkException DOCUMENT ME!
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#saveWRL20()
	 * @category menu messages
	 */
	public void saveWRL20() {
		JunOpenGL3dGraphAbstract graph = (JunOpenGL3dGraphAbstract) this.displayObject();

		if (graph == null) {
			return;
		}

		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType(JunSystem.$String("<1p> files", null, "VRML"), new String[] { "*.wrl", "*.WRL" }) };
		File file = JunFileRequesterDialog.RequestNewFile(JunSystem.$String("Input a <1p> file.", null, "WRL"), new File(graph.name() + ".wrl"), fileTypes, fileTypes[0]);
		if (file == null) {
			return;
		}

		Jun3dBoundingBox box = graph.boundingBox();
		Jun3dPoint center = box.center();
		Jun3dTransformation transformation = Jun3dTransformation.Translate_(new Jun3dPoint(0 - center.x(), 0 - center.y(), 0 - center.z()));
		JunOpenGL3dObject object = graph.displayObject().transform_(transformation);

		try {
			this.writeToWRL20_object_(file, object);
		} catch (IOException e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Select all objects.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#selectAll()
	 * @category selecting
	 */
	public void selectAll() {
		JunOpenGL3dObject displayObject = this.displayObject();

		if (displayObject == null) {
			return;
		}

		if (displayObject.isPrimitive()) {
			return;
		}

		Vector selectedObjects = this.selectedObjects();
		this.clearSelectedObjects();

		JunOpenGL3dNode[] nodes = ((JunOpenGL3dGraph) displayObject).nodes();
		int size = nodes.length;

		for (int index = 0; index < size; index++) {
			selectedObjects.addElement(nodes[index]);
		}

		this.changed_($("selection"));
	}

	/**
	 * Answer the selected nodes.
	 * 
	 * @return java.util.Vector
	 * @category selecting
	 */
	public Vector selectedNodes() {
		Vector selectedNodes = new Vector();
		Vector selectedObjects = this.selectedObjects();

		for (int index = 0; index < selectedObjects.size(); index++) {
			JunOpenGL3dGraph each = (JunOpenGL3dGraph) selectedObjects.elementAt(index);
			JunOpenGL3dNode[] nodes = ((JunOpenGL3dGraph) each.displayObject()).nodes();
			int size = nodes.length;

			for (int i = 0; i < size; i++) {
				JunOpenGL3dNode node = nodes[index];

				if (node.displayObject() == each) {
					selectedNodes.addElement(node);
				}
			}
		}

		return selectedNodes;
	}

	/**
	 * Answer the spawning object.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#spawningObject()
	 * @category private
	 */
	public JunOpenGL3dObject spawningObject() {
		JunOpenGL3dGraph spawningObject;
		Vector selectedObjects = this.selectedObjects();

		if (selectedObjects.isEmpty()) {
			spawningObject = (JunOpenGL3dGraph) this.displayObject().copy();
		} else {
			spawningObject = new JunOpenGL3dGraph();

			Vector selectedNodes = this.selectedNodes();

			for (int index = 0; index < selectedNodes.size(); index++) {
				spawningObject.addNode_((JunOpenGL3dNode) selectedNodes.elementAt(index));
			}
		}

		return spawningObject;
	}

	/**
	 * Write the JunOpenGL3dObject to the LGT1.0 file.
	 * 
	 * @param aFile java.io.File
	 * @param aGraph jp.co.sra.jun.opengl.grapher.JunOpenGL3dGraphAbstract
	 * @exception java.io.IOException
	 * @category writing
	 */
	public void writeToLGT10_object_(File aFile, JunOpenGL3dGraphAbstract aGraph) throws IOException {
		if (aFile == null) {
			return;
		}

		BufferedWriter aWriter = new BufferedWriter(new FileWriter(aFile));

		try {
			this.saveToLGT10_object_(aWriter, aGraph);
		} finally {
			aWriter.flush();
			aWriter.close();
		}
	}

	/**
	 * Answer the default stamp string for LGT1.0.
	 * 
	 * @return java.lang.String
	 * @category defaults
	 */
	private String defaultStampForLGT10() {
		StringWriter sw = new StringWriter();
		PrintWriter pw = new PrintWriter(sw);
		pw.println("%LGT V1.0 List Grapher Transmission (Lisp S Expression)");
		pw.println("% This file was created by " + JunSystem.System() + JunSystem.Version());
		pw.println("% " + DateFormat.getInstance().format(new Date()));
		pw.println();
		pw.flush();

		return sw.toString();
	}

	/**
	 * Answer the window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return JunSystem.$String("Grapher");
	}

	/**
	 * Answer my menu bar.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenuBar
	 * @see jp.co.sra.smalltalk.StApplicationModel#_menuBar()
	 * @category resources
	 */
	public StMenuBar _menuBar() {
		if (_menuBar == null) {
			_menuBar = new StMenuBar();
			_menuBar.add(this._createFileMenu());
			_menuBar.add(this._createEditMenu());
			_menuBar.add(this._createViewMenu());
			_menuBar.add(this._createLightMenu());
			_menuBar.add(this._createGraphMenu());
			_menuBar.add(this._createMiscMenu());
		}
		return _menuBar;
	}

	/**
	 * Create a "File" menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#_createFileMenu()
	 * @category resources
	 */
	protected StMenu _createFileMenu() {
		StMenu fileMenu = new StMenu(JunSystem.$String("File"), $("fileMenu"));

		// New
		fileMenu.add(new StMenuItem(JunSystem.$String("New"), new MenuPerformer(this, "newModel")));

		// Open...
		fileMenu.add(new StMenuItem(JunSystem.$String("Open") + "...", new MenuPerformer(this, "openLST")));

		//
		fileMenu.addSeparator();

		// Save...
		fileMenu.add(new StMenuItem(JunSystem.$String("Save") + "...", $("saveMenu"), new MenuPerformer(this, "saveLST")));

		// Save as...
		StMenu saveAsMenu = new StMenu(JunSystem.$String("Save as..."), $("saveAsMenu"));
		saveAsMenu.add(new StMenuItem("VRML1.0...", new MenuPerformer(this, "saveWRL10")));
		saveAsMenu.add(new StMenuItem("VRML97...", new MenuPerformer(this, "saveWRL97")));
		fileMenu.add(saveAsMenu);

		// Save as image...
		fileMenu.add(new StMenuItem(JunSystem.$String("Save as image..."), $("saveAsImageMenu"), new MenuPerformer(this, "saveAsImage")));

		//
		fileMenu.addSeparator();

		// Quit
		fileMenu.add(new StMenuItem(JunSystem.$String("Quit"), new MenuPerformer(this, "quitDoing")));

		return fileMenu;
	}

	/**
	 * Create a "Graph" menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @category resources
	 */
	protected StMenu _createGraphMenu() {
		StMenu graphMenu = new StMenu(JunSystem.$String("Graph"), $("graphMenu"));
		graphMenu.add(new StMenuItem(JunSystem.$String("Arrange"), $("arrangeMenu"), new MenuPerformer(this, "arrangeGraph")));
		graphMenu.add(new StMenuItem(JunSystem.$String("Arrange2"), $("arrange2Menu"), new MenuPerformer(this, "arrangeGraph2")));
		graphMenu.addSeparator();
		graphMenu.add(new StMenuItem(JunSystem.$String("Interval"), $("intervalMenu"), new MenuPerformer(this, "arrangeGraphInterval")));
		graphMenu.add(new StCheckBoxMenuItem(JunSystem.$String("Balance"), $("balanceMenu"), new MenuPerformer(this, "arrangeGraphBalance")));
		graphMenu.addSeparator();
		graphMenu.add(new StCheckBoxMenuItem(JunSystem.$String("Crank arcs"), $("crankArcsMenu"), new MenuPerformer(this, "arrangeGraphCrankArcs")));
		return graphMenu;
	}

	/**
	 * Create a "Misc" menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#_createMiscMenu()
	 * @category resources
	 */
	protected StMenu _createMiscMenu() {
		StMenu miscMenu = new StMenu(JunSystem.$String("Misc"), $("miscMenu"));
		miscMenu.add(new StMenuItem(JunSystem.$String("Spawn"), $("spawnMenu"), new MenuPerformer(this, "spawnObject")));
		miscMenu.add(new StMenuItem(JunSystem.$String("Bounds"), $("boundsMenu"), new MenuPerformer(this, "showBounds")));
		return miscMenu;
	}

	/**
	 * Update the menu indications.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#updateMenuIndication()
	 * @category menu messages
	 */
	public void updateMenuIndication() {
		super.updateMenuIndication();
		this.updateGraphMenuIndication();
	}

	/**
	 * Update the graph menu indication.
	 * 
	 * @category menu messages
	 */
	public void updateGraphMenuIndication() {
		StMenu graphMenu = (StMenu) this._menuBar().atNameKey_($("graphMenu"));
		if (graphMenu == null) {
			return;
		}

		StMenuItem menuItem;
		boolean displayObjectIsNotEmpty = !this.isEmpty();

		// Arrange
		menuItem = graphMenu.atNameKey_($("arrangeMenu"));
		if (menuItem != null) {
			menuItem.beEnabled(displayObjectIsNotEmpty);
		}

		// Arrange2
		menuItem = graphMenu.atNameKey_($("arrange2Menu"));
		if (menuItem != null) {
			menuItem.beEnabled(displayObjectIsNotEmpty);
		}

		// Interval
		menuItem = graphMenu.atNameKey_($("intervalMenu"));
		if (menuItem != null) {
			menuItem.beEnabled(displayObjectIsNotEmpty);
		}

		// Balance
		menuItem = graphMenu.atNameKey_($("balanceMenu"));
		if (menuItem != null) {
			menuItem.beEnabled(displayObjectIsNotEmpty);
			if (displayObjectIsNotEmpty) {
				((StCheckBoxMenuItem) menuItem).beSelected(this.isArrangeBalance());
			}
		}

		// Crank arcs
		menuItem = graphMenu.atNameKey_($("crankArcsMenu"));
		if (menuItem != null) {
			menuItem.beEnabled(displayObjectIsNotEmpty);
			if (displayObjectIsNotEmpty) {
				((StCheckBoxMenuItem) menuItem).beSelected(this.isCrankArcs());
			}
		}
	}

}
