package jp.co.sra.jun.opengl.objects;

import java.io.PrintWriter;
import java.io.Writer;
import java.util.ArrayList;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StObject;
import jp.co.sra.smalltalk.StSymbol;

import jp.co.sra.jun.geometry.abstracts.JunGeometry;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;

/**
 * JunOpenGL3dSphere class
 * 
 *  @author    nisinaka
 *  @created   2004/05/12 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGL3dSphere.java,v 8.12 2008/02/20 06:32:35 nisinaka Exp $
 */
public class JunOpenGL3dSphere extends JunOpenGL3dCompoundObject {

	protected static final Jun3dPoint DefaultCenter = new Jun3dPoint(0, 0, 0);
	protected static final double DefaultRadius = 1.0;
	protected static final int DefaultSlices = 20;
	protected static final int DefaultStacks = 10;

	protected Jun3dPoint center;
	protected double radius;
	protected int slices;
	protected int stacks;

	/**
	 * Create a new instance of JunOpenGL3dShepere and initialize it.
	 * 
	 * @param radius double
	 * @category Instance creation
	 */
	public JunOpenGL3dSphere(double radius) {
		this(DefaultCenter, radius, DefaultSlices, DefaultStacks);
	}

	/**
	 * Create a new instance of JunOpenGL3dShepere and initialize it.
	 *
	 * @param radius double
	 * @param slices int
	 * @param stacks int
	 * @category Instance creation
	 */
	public JunOpenGL3dSphere(double radius, int slices, int stacks) {
		this(DefaultCenter, radius, slices, stacks);
	}

	/**
	 * Create a new instance of JunOpenGL3dShepere and initialize it.
	 * 
	 * @param center jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param radius double
	 * @category Instance creation
	 */
	public JunOpenGL3dSphere(Jun3dPoint center, double radius) {
		this(center, radius, DefaultSlices, DefaultStacks);
	}

	/**
	 * Create a new instance of JunOpenGL3dShepere and initialize it.
	 *
	 * @param center jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param radius double
	 * @param slices int
	 * @param stacks int
	 * @category Instance creation
	 */
	public JunOpenGL3dSphere(Jun3dPoint center, double radius, int slices, int stacks) {
		super();
		this.center_(center);
		this.radius_(radius);
		this.slices_(slices);
		this.stacks_(stacks);
	}

	/**
	 * Create a new instance of JunOpenGL3dSphere and initialize it with the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunOpenGL3dSphere(JunLispList aList) {
		super();
		this.fromLispList(aList);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		center = null;
		radius = Double.NaN;
		slices = DefaultSlices;
		stacks = DefaultStacks;
	}

	/**
	 * Answer my current center point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint center() {
		if (center == null) {
			center = DefaultCenter;
		}
		return center;
	}

	/**
	 * Set my new center point.
	 * 
	 * @param newCenter jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public void center_(Jun3dPoint newCenter) {
		center = newCenter;
		this.flushComponents();
	}

	/**
	 * Answer my current radius.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double radius() {
		if (Double.isNaN(radius)) {
			radius = DefaultRadius;
		}
		return radius;
	}

	/**
	 * Set my new radius.
	 * 
	 * @param newRadius double
	 * @category accessing
	 */
	public void radius_(double newRadius) {
		radius = Math.max(newRadius, 0.001);
		this.flushComponents();
	}

	/**
	 * Answer my current number of slices.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int slices() {
		return slices;
	}

	/**
	 * Set my new number of slices.
	 * 
	 * @param newSlices int
	 * @category accessing
	 */
	public void slices_(int newSlices) {
		slices = Math.max(newSlices, 4);
		this.flushComponents();
	}

	/**
	 * Answer my current number of stacks.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int stacks() {
		return stacks;
	}

	/**
	 * Set my new number of stacks.
	 * 
	 * @param newStacks int
	 * @category accessing
	 */
	public void stacks_(int newStacks) {
		stacks = Math.max(newStacks, 2);
		this.flushComponents();
	}

	/**
	 * Answer my current components as ArrayList.
	 * If not exist, create one.
	 * 
	 * @return java.util.ArrayList
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject#_components()
	 * @category accessing
	 */
	protected ArrayList _components() {
		if (components == null) {
			components = new ArrayList();
			this.flushBounds();

			double dTheta = Math.PI / this.stacks();
			double dPhi = Math.PI * 2 / this.slices();
			for (int stack = 0; stack < this.stacks(); stack++) {
				double theta = dTheta * stack - Math.PI / 2;
				for (int slice = 0; slice < this.slices(); slice++) {
					double phi = dPhi * slice;
					Jun3dPoint[] points = new Jun3dPoint[] { this.pointAt(theta, phi), this.pointAt(theta, phi + dPhi), this.pointAt(theta + dTheta, phi + dPhi), this.pointAt(theta + dTheta, phi) };
					JunOpenGL3dPolygon aPolygon = new JunOpenGL3dPolygon(points);
					aPolygon.paint_(null);
					components.add(aPolygon);
				}
			}
		}
		return components;
	}

	/**
	 * Create a new JunOpenGL3dSphere transformed with the transformation.
	 * 
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#transform_(jp.co.sra.jun.geometry.transformations.Jun3dTransformation)
	 * @category transforming 
	 */
	public JunOpenGL3dObject transform_(Jun3dTransformation aTransformation) {
		JunOpenGL3dSphere aSphere = (JunOpenGL3dSphere) this.copy();

		Jun3dPoint oldSurfacePoint = new Jun3dPoint(this.center().x() + this.radius(), this.center().y(), this.center().z());
		Jun3dPoint newSurfacePoint = oldSurfacePoint.transform_(aTransformation);
		Jun3dPoint newCenterPoint = this.center().transform_(aTransformation);
		double newRadius = newCenterPoint.distance_(newSurfacePoint);

		aSphere.center_(newCenterPoint);
		aSphere.radius_(newRadius);
		return aSphere;
	}

	/**
	 * Do an extra copy of the receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StObject
	 * @see jp.co.sra.smalltalk.StObject#postCopy()
	 * @category copying
	 */
	public StObject postCopy() {
		super.postCopy();

		if (center != null) {
			center = new Jun3dPoint(center.x(), center.y(), center.z());
		}

		return this;
	}

	/**
	 * Answer the 3D point on the surface of the sphere.
	 * 
	 * @param theta double
	 * @param phi double
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category private
	 */
	protected Jun3dPoint pointAt(double theta, double phi) {
		double x = this.radius() * Math.cos(theta) * Math.cos(phi);
		if (Math.abs(x) < JunGeometry.ACCURACY) {
			x = 0.0d;
		}
		double y = this.radius() * Math.cos(theta) * Math.sin(phi);
		if (Math.abs(y) < JunGeometry.ACCURACY) {
			y = 0.0d;
		}
		double z = this.radius() * Math.sin(theta);
		if (Math.abs(z) < JunGeometry.ACCURACY) {
			z = 0.0d;
		}
		return this.center().plus_(new Jun3dPoint(x, y, z));
	}

	/**
	 * Answer the StSymbol which represents the kind of the receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#kindName()
	 * @category lisp support
	 */
	public StSymbol kindName() {
		return $("Sphere");
	}

	/**
	 * Convert the receiver as JunLispCons.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#toLispList()
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons list = this.lispCons();
		list.head_(this.kindName());

		if (this.hasName()) {
			list.add_(this.nameToLispList());
		}
		if (this.hasColor()) {
			list.add_(this.colorToLispList());
		}
		if (this.hasTexture()) {
			list.add_(this.textureToLispList());
		}
		list.add_(this.centerToLispList());
		list.add_(this.radiusToLispList());
		list.add_(this.slicesToLispList());
		list.add_(this.stacksToLispList());

		return list;
	}

	/**
	 * Convert the receiver's center point as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList centerToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("center"));
		list.tail_(this.center());
		return list;
	}

	/**
	 * Convert the receiver's radius as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList radiusToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("radius"));
		list.tail_(new Double(this.radius()));
		return list;
	}

	/**
	 * Convert the receiver's number of slices as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList slicesToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("slices"));
		list.tail_(new Integer(this.slices()));
		return list;
	}

	/**
	 * Convert the receiver's number of stacks as a LispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList stacksToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("stacks"));
		list.tail_(new Integer(this.stacks()));
		return list;
	}

	/**
	 * Get my attributes from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#fromLispList(jp.co.sra.jun.goodies.lisp.JunLispList)
	 * @category lisp support
	 */
	public void fromLispList(JunLispList aList) {
		super.fromLispList(aList);
		this.centerFromLispList(aList);
		this.radiusFromLispList(aList);
		this.slicesFromLispList(aList);
		this.stacksFromLispList(aList);
	}

	/**
	 * Get my center from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void centerFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("center")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.center_((Jun3dPoint) list.tail());
	}

	/**
	 * Get my radius from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void radiusFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("radius")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.radius_(((Number) list.tail()).doubleValue());
	}

	/**
	 * Get my slices from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void slicesFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("slices")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.slices_(((Number) list.tail()).intValue());
	}

	/**
	 * Get my stacks from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void stacksFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("stacks")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.stacks_(((Number) list.tail()).intValue());
	}

	/**
	 * Write the VRML1.0 string of the receiver on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws SmalltalkException 
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#vrml10On_(java.io.Writer)
	 * @category vrml support
	 */
	public void vrml10On_(Writer aWriter) {
		super.vrml10On_(aWriter);
	}

	/**
	 * Write the VRML2.0 string of the receiver on the writer.
	 * 
	 * @param pw java.io.PrintWriter
	 * @param leader java.lang.String
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#vrml20On_(java.io.Writer)
	 * @category vrml support
	 */
	public void vrml20On_(PrintWriter pw, String leader) {
		pw.println(leader + "Transform {");
		pw.println(leader + INDENT + "translation " + this.center().x() + " " + this.center().y() + " " + this.center().z());
		pw.println(leader + INDENT + "children [");
		this.vrml20ShapeOn_(pw, leader + INDENT + INDENT);
		pw.println(leader + INDENT + "] # children");
		pw.println(leader + "} #Transform");
		pw.flush();
	}

	/**
	 * Write my geometry as VRML2.0 to the writer.
	 * 
	 * @param pw java.io.PrintWriter
	 * @param leader java.lang.String
	 * @category vrml support
	 */
	protected void vrml20GeometryOn_(PrintWriter pw, String leader) {
		pw.println(leader + "geometry Sphere { radius " + this.radius() + " }");
	}

}
