package jp.co.sra.jun.opengl.support;

import java.awt.Canvas;
import java.awt.Color;
import java.awt.Component;
import java.awt.Graphics;
import java.awt.Rectangle;

import jp.co.sra.smalltalk.DependentEvent;

import jp.co.sra.gl4jun.GLjRenderingMode;

/**
 * JunGLCanvas class
 * 
 *  @author    nisinaka
 *  @created   2007/08/20 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunGLCanvas.java,v 8.6 2008/02/20 06:32:49 nisinaka Exp $
 */
public class JunGLCanvas extends Canvas implements JunOpenGLDrawable {

	protected JunOpenGLRenderer renderer;
	protected int handle;
	protected boolean isMoved;
	protected boolean isResized;

	/**
	 * Create a new instance of JunGLCanvas and initialize it.
	 *
	 * @category Instance creation
	 */
	public JunGLCanvas() {
		this(null);
	}

	/**
	 * Create a new instance of JunGLCanvas and initialize it.
	 *
	 * @param anOpenGLRenderer jp.co.sra.jun.opengl.support.JunOpenGLRenderer
	 * @category Instance creation
	 */
	public JunGLCanvas(JunOpenGLRenderer anOpenGLRenderer) {
		this.initialize();
		this.setRenderer(anOpenGLRenderer);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @category initialize-release
	 */
	protected void initialize() {
		renderer = null;
		handle = -1;
		isMoved = false;
		isResized = false;
	}

	/**
	 * Release the receiver.
	 * 
	 * @category initialize-release
	 */
	public void release() {
		if (handle >= 0) {
			JunGLInterface.Current.gljDeleteContext(handle, false);
			handle = -1;
		}
	}

	/**
	 * Delete the context.
	 * 
	 * @see java.awt.Component#removeNotify()
	 * @category initialize-release
	 */
	public void removeNotify() {
		this.release();
		super.removeNotify();
	}

	/**
	 * Answer my current OpenGL interface.
	 * 
	 * @return jp.co.sra.jun.opengl.support.JunOpenGLInterface
	 * @see jp.co.sra.jun.opengl.support.JunOpenGLDrawable#getOpenGLInterface()
	 * @category accessing 
	 */
	public JunOpenGLInterface getOpenGLInterface() {
		return JunGLInterface.Current;
	}

	/**
	 * Answer my current background color.
	 * 
	 * @return java.awt.Color
	 * @see jp.co.sra.jun.opengl.support.JunOpenGLDrawable#getBackgroundColor()
	 * @category accessing
	 */
	public Color getBackgroundColor() {
		return this.getBackground();
	}

	/**
	 * Set my new renderer.
	 * 
	 * @param anOpenGLRenderer jp.co.sra.jun.opengl.support.JunOpenGLRenderer
	 * @see jp.co.sra.jun.opengl.support.JunOpenGLDrawable#setRenderer(jp.co.sra.jun.opengl.support.JunOpenGLRenderer)
	 * @category accessing
	 */
	public void setRenderer(JunOpenGLRenderer anOpenGLRenderer) {
		if (renderer == anOpenGLRenderer) {
			return;
		}

		renderer = anOpenGLRenderer;
		this.redisplay();
	}

	/**
	 * Answer my current handle.
	 * 
	 * @return int
	 * @category accessing
	 */
	protected int handle() {
		if (handle <= 0) {
			handle = JunGLInterface.Current.gljCreateContext(this, this.getWidth(), this.getHeight(), GLjRenderingMode.DOUBLE_BUFFER);
		}
		return handle;
	}

	/**
	 * Set the bounds.
	 * 
	 * @param x int
	 * @param y int
	 * @param width int
	 * @param height int
	 * @see java.awt.Component#setBounds(int, int, int, int)
	 * @category bounds accessing
	 */
	public void setBounds(int x, int y, int width, int height) {
		Rectangle bounds = this.getBounds();
		if (bounds.x != x || bounds.y != y) {
			isMoved = true;
		}
		if (bounds.width != width || bounds.height != height) {
			isResized = true;
		}

		super.setBounds(x, y, width, height);
	}

	/**
	 * Convert to a component.
	 * 
	 * @return java.awt.Component
	 * @see jp.co.sra.jun.opengl.support.JunOpenGLDrawable#toComponent()
	 * @category converting
	 */
	public Component toComponent() {
		return this;
	}

	/**
	 * Render the receiver on the rendering context.
	 * 
	 * @param aRenderingContext jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * @see jp.co.sra.jun.opengl.support.JunOpenGLDrawable#renderOn_(jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext)
	 * @category rendering
	 */
	public void renderOn_(JunOpenGLRenderingContext aRenderingContext) {
		if (renderer != null) {
			renderer.renderOn_(aRenderingContext);
		}
	}

	/**
	 * Paint the receiver on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @see java.awt.Canvas#paint(java.awt.Graphics)
	 * @category painting
	 */
	public void paint(Graphics aGraphics) {
		synchronized (JunGLInterface.Current) {
			try {
				if (isResized) {
					JunGLInterface.Current.gljSetSize(this.handle(), this.getWidth(), this.getHeight());
					isResized = false;
				}
				if (isMoved) {
					JunGLInterface.Current.gljSetLocation(this.handle(), this.getX(), this.getY());
					isMoved = false;
				}

				JunGLInterface.Current.gljMakeCurrent(this.handle());

				JunOpenGLRenderingContext renderingContext = new JunOpenGLRenderingContext(this);
				this.renderOn_(renderingContext);

			} finally {
				JunGLInterface.Current.gljFlushBuffer(this.handle());
			}
		}

		if (renderer != null) {
			renderer.superimposeOn_(aGraphics, this);
		}
	}

	/**
	 * Updates the canvas.
	 * Override to suppress clearing.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @category painting
	 */
	public void update(Graphics aGraphics) {
		this.paint(aGraphics);
	}

	/**
	 * Display the entire contents again.
	 * 
	 * @see jp.co.sra.jun.opengl.support.JunOpenGLDrawable#redisplay()
	 * @category displaying
	 */
	public void redisplay() {
		if (this.isShowing()) {
			this.repaint();
		}
	}

	/**
	 * Display the entire contents again immediately.
	 * 
	 * @see jp.co.sra.jun.opengl.support.JunOpenGLDrawable#redisplayImmediately()
	 * @category displaying
	 */
	public void redisplayImmediately() {
		if (this.isShowing() == false) {
			return;
		}

		Graphics aGraphics = null;
		try {
			aGraphics = this.getGraphics();
			this.paint(aGraphics);
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
			}
		}
	}

	/**
	 * Update for the change notice.
	 * 
	 * @param e jp.co.sra.smalltalk.DependentEvent
	 * @see jp.co.sra.smalltalk.DependentListener#update_(jp.co.sra.smalltalk.DependentEvent)
	 * @category updating
	 */
	public void update_(DependentEvent e) {
		this.redisplay();
	}

}
