package jp.co.sra.jun.opengl.support;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StApplicationModel;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StUserMessage;
import jp.co.sra.smalltalk.menu.MenuEvent;
import jp.co.sra.smalltalk.menu.MenuListener;
import jp.co.sra.smalltalk.menu.StMenu;
import jp.co.sra.smalltalk.menu.StRadioButtonGroup;
import jp.co.sra.smalltalk.menu.StRadioButtonMenuItem;

import jp.co.sra.jun.system.framework.JunAbstractObject;

/**
 * JunOpenGLResource class
 * 
 *  @author    nisinaka
 *  @created   2007/08/17 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGLResource.java,v 8.5 2008/02/20 06:32:49 nisinaka Exp $
 */
public abstract class JunOpenGLResource extends JunAbstractObject {

	public static final StMenu DefaultOpenGLViewMenu;
	protected static Class DefaultOpenGLDrawableClass = null;

	static {
		DefaultOpenGLViewMenu = CreateDefaultOpenGLViewMenu();
		InitializeDefaultOpenGLDrawableClass();
	}

	/**
	 * Initialize the default view class.
	 * 
	 * @category Class initialization
	 */
	protected static void InitializeDefaultOpenGLDrawableClass() {
		Class aClass = null;

		if (JunOpenGLResource.UseJOGL()) {
			if (JunOpenGLResource.IsJOGLAvailable()) {
				if (StApplicationModel.GetDefaultViewMode() != StApplicationModel.VIEW_SWING_ONLY) {
					aClass = GetJOGLCanvasClass();
				} else {
					aClass = GetJOGLJPanelClass();
				}
			} else {
				System.err.println("JOGL is specified to be used, but not available.");
			}
		}
		if (aClass == null) {
			if (StApplicationModel.GetDefaultViewMode() != StApplicationModel.VIEW_SWING_ONLY) {
				aClass = JunGLCanvas.class;
			} else {
				aClass = JunGLJPanel.class;
			}
		}

		SetDefaultOpenGLDrawableClass(aClass);
	}

	/**
	 * Answer the default OpenGL drawable class.
	 * 
	 * @return java.lang.Class
	 * @category Accessing
	 */
	public static Class GetDefaultOpenGLDrawableClass() {
		return DefaultOpenGLDrawableClass;
	}

	/**
	 * Set the new default view class.
	 * 
	 * @param aClass java.lang.Class
	 * @category Accessing
	 */
	public static void SetDefaultOpenGLDrawableClass(Class aClass) {
		if (JunOpenGLDrawable.class.isAssignableFrom(aClass) == false) {
			throw SmalltalkException.Error("The specified class is not a kind of JunOpenGLDrawable : " + aClass);
		}

		DefaultOpenGLDrawableClass = aClass;
		System.err.println("The default OpenGL drawable class is set to " + GetDefaultOpenGLDrawableClassName());
		UpdateDefaultOpenGLViewMenu();
	}

	/**
	 * Answer the default view class name.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category Accessing
	 */
	public static StSymbol GetDefaultOpenGLDrawableClassName() {
		String className = DefaultOpenGLDrawableClass.getName();
		return $(className.substring(className.lastIndexOf('.') + 1));
	}

	/**
	 * Answer the JOGLCanvas class.
	 * 
	 * @return java.lang.Class
	 * @category Accessing
	 */
	public static Class GetJOGLCanvasClass() {
		Class aClass = null;
		try {
			aClass = Class.forName("jp.co.sra.jun.opengl.support.JOGLCanvas");
		} catch (ClassNotFoundException e) {
			e.printStackTrace();
		}
		return aClass;
	}

	/**
	 * Answer the JOGLJPanel class.
	 * 
	 * @return java.lang.Class
	 * @category Accessing
	 */
	public static Class GetJOGLJPanelClass() {
		Class aClass = null;
		try {
			aClass = Class.forName("jp.co.sra.jun.opengl.support.JOGLJPanel");
		} catch (ClassNotFoundException e) {
			e.printStackTrace();
		}
		return aClass;
	}

	/**
	 * Create a "Default OpenGL View" menu.
	 * 
	 * @return jp.co.sra.smalltalk.StMenu
	 * @category Resources
	 */
	protected static StMenu CreateDefaultOpenGLViewMenu() {
		StRadioButtonGroup aGroup = new StRadioButtonGroup();
		StMenu aMenu = new StMenu(new StUserMessage("Default OpenGL View"));
		aMenu.add(new StRadioButtonMenuItem(new StUserMessage("JunGL Canvas"), $("JunGLCanvas"), aGroup, new MenuListener() {
			public void menuPerformed(MenuEvent aMenuEvent) {
				SetDefaultOpenGLDrawableClass(JunGLCanvas.class);
			}
		}));
		aMenu.add(new StRadioButtonMenuItem(new StUserMessage("JunGL JPanel"), $("JunGLJPanel"), aGroup, new MenuListener() {
			public void menuPerformed(MenuEvent aMenuEvent) {
				SetDefaultOpenGLDrawableClass(JunGLJPanel.class);
			}
		}));
		aMenu.add(new StRadioButtonMenuItem(new StUserMessage("JOGL Canvas"), $("JOGLCanvas"), aGroup, new MenuListener() {
			public void menuPerformed(MenuEvent aMenuEvent) {
				SetDefaultOpenGLDrawableClass(GetJOGLCanvasClass());
			}
		}));
		aMenu.add(new StRadioButtonMenuItem(new StUserMessage("JOGL JPanel"), $("JOGLJPanel"), aGroup, new MenuListener() {
			public void menuPerformed(MenuEvent aMenuEvent) {
				SetDefaultOpenGLDrawableClass(GetJOGLJPanelClass());
			}
		}));
		return aMenu;
	}

	/**
	 * Update the menu indications of the "Default OpenGL View" menu.
	 * 
	 * @category Resources
	 */
	protected static void UpdateDefaultOpenGLViewMenu() {
		DefaultOpenGLViewMenu.atNameKey_($("JOGLCanvas")).beEnabled(IsJOGLAvailable());
		DefaultOpenGLViewMenu.atNameKey_($("JOGLJPanel")).beEnabled(IsJOGLAvailable());

		StRadioButtonMenuItem aMenuItem = (StRadioButtonMenuItem) DefaultOpenGLViewMenu.atNameKey_(GetDefaultOpenGLDrawableClassName());
		if (aMenuItem != null && aMenuItem.isSelected() == false) {
			aMenuItem.beOn();
		}
	}

	/**
	 * Answer true if the JOGL is available, otherwise false.
	 * 
	 * @return boolean
	 * @category Testing
	 */
	public static boolean IsJOGLAvailable() {
		try {
			Class.forName("javax.media.opengl.GL");
			return true;
		} catch (ClassNotFoundException e) {
			return false;
		}
	}

	/**
	 * Answer true if the JOGL is specified to be used, otherwise false.
	 * 
	 * @return boolean
	 * @category Testing
	 */
	public static boolean UseJOGL() {
		return Boolean.getBoolean("jun.opengl.jogl");
	}

	/**
	 * Create a new instance of <code>JunOpenGLResource</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunOpenGLResource() {
		super();
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunAbstractObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
	}

}
