package jp.co.sra.jun.system.framework;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dialog;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Window;
import java.awt.event.WindowEvent;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import javax.imageio.ImageIO;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StApplicationModel;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StComposedText;
import jp.co.sra.smalltalk.StController;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StInputState;
import jp.co.sra.smalltalk.StObject;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.SystemInterface;

import jp.co.sra.jun.geometry.basic.JunPoint;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.goodies.files.JunFileModel;
import jp.co.sra.jun.goodies.image.streams.JunImageStream;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispNil;
import jp.co.sra.jun.goodies.utilities.JunControlUtility;
import jp.co.sra.jun.goodies.utilities.JunImageUtility;
import jp.co.sra.jun.goodies.utilities.JunSensorUtility;
import jp.co.sra.jun.graphics.navigator.JunFileRequesterDialog;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunApplicationModel class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2002/11/07 (by Mitsuhiro Asada)
 *  @updated   2003/03/03 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on Jun697 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunApplicationModel.java,v 8.26 2008/02/20 06:32:50 nisinaka Exp $
 */
public abstract class JunApplicationModel extends StApplicationModel {

	protected StBlockClosure closeBlock;

	/**
	 * Answer the string of default base name.
	 *
	 * @return java.lang.String
	 * @category Defaults
	 */
	public static String DefaultBaseName() {
		SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMddHHmmss");
		return dateFormat.format(new Date());
	}

	/**
	 * Evaluate the block closure and wait at least for the specified time.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param millisecondTime int
	 * @category Evaluating
	 */
	public static Object Do_forMilliseconds_(StBlockClosure aBlock, int millisecondTime) {
		return JunControlUtility.Do_forMilliseconds_(aBlock, millisecondTime);
	}

	/**
	 * Evaluate the block closure to handle the specified number of frames per second.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param aNumber int
	 * @category Evaluating
	 */
	public static Object Do_framesPerSecond_(StBlockClosure aBlock, int aNumber) {
		return JunControlUtility.Do_framesPerSecond_(aBlock, aNumber);
	}

	/**
	 * An utility method to make nine boxes of the specified rectangle.
	 * 
	 * @param aRectangle java.awt.Rectangle
	 * @return java.util.Map
	 * @category Utilities
	 */
	public static Map NineBoxesOf_(Rectangle aRectangle) {
		HashMap aMap = new HashMap(9);

		StRectangle aBox = new StRectangle(aRectangle);
		StRectangle area = new StRectangle(aRectangle.getLocation(), new Dimension(aRectangle.width / 3, aRectangle.height / 3));
		StSymbol[] symbols = new StSymbol[] { $("topLeft"), $("topCenter"), $("topRight"), $("leftCenter"), $("center"), $("rightCenter"), $("bottomLeft"), $("bottomCenter"), $("bottomRight") };
		for (int i = 0; i < symbols.length; i++) {
			StRectangle box = area.align_with_(area._pointAt(symbols[i]), aBox._pointAt(symbols[i]));
			aMap.put(symbols[i], box);
		}

		return aMap;
	}

	/**
	 * An utility method to make a rectangle visible on the screen.
	 * This method was defined in Screen class.
	 *
	 * @return java.awt.Rectangle
	 * @param aRectangle java.awt.Rectangle
	 * @category Utilities
	 */
	public static Rectangle _MakeRectangleVisible(Rectangle aRectangle) {
		int dx = 0;
		int dy = 0;

		Rectangle screenBounds = _GetScreenBoundsContainsPoint(SystemInterface._MousePoint());
		if (aRectangle.x < screenBounds.x) {
			dx = aRectangle.x;
		} else if (aRectangle.x + aRectangle.width >= screenBounds.x + screenBounds.width) {
			dx = (screenBounds.x + screenBounds.width) - (aRectangle.x + aRectangle.width);
		}
		if (aRectangle.y < screenBounds.y) {
			dy = aRectangle.y;
		} else if (aRectangle.y + aRectangle.height >= screenBounds.y + screenBounds.height) {
			dy = (screenBounds.y + screenBounds.height) - (aRectangle.y + aRectangle.height);
		}

		aRectangle.translate(dx, dy);
		return aRectangle;
	}

	/**
	 * Initialize the ApplicationModel when created.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		closeBlock = null;
	}

	/**
	 * Convert the receiver to an image as StImage.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category converting
	 */
	public StImage asImage() {
		return this.asWindowImage();
	}

	/**
	 * Convert the receiver's window to an image as StImage.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category converting
	 */
	public StImage asWindowImage() {
		Window aWindow = this.getWindow();
		if (aWindow == null) {
			return null;
		}
		if (aWindow.isShowing()) {
			aWindow.setVisible(true);
		}
		StImage anImage = null;
		if (JunApplicationModel.GetDefaultViewMode() == JunApplicationModel.VIEW_AWT) {
			throw SmalltalkException.ShouldNotImplement();
		} else {
			Component rootComponent = aWindow.getComponent(0);
			anImage = new StImage(rootComponent.getWidth(), rootComponent.getHeight());
			Graphics aGraphics = anImage.image().getGraphics();
			try {
				rootComponent.paint(aGraphics);
			} finally {
				if (aGraphics != null) {
					aGraphics.dispose();
				}
			}
		}
		return anImage;
	}

	/**
	 * Convert the receiver to an image as Image.
	 * 
	 * @return java.awt.Image
	 * @category converting
	 */
	public Image toImage() {
		StImage anImage = this.asImage();
		return (anImage == null) ? null : anImage.image();
	}

	/**
	 * Evaluate the specified debug block closure.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category debug
	 */
	public void debug_(StBlockClosure aBlock) {
		// aBlock.value()
	}

	/**
	 * Answer the string of default base name.
	 *
	 * @return java.lang.String
	 * @category defaults
	 */
	public String defaultBaseName() {
		return DefaultBaseName();
	}

	/**
	 * Answer the default image quality value.
	 * 
	 * @return float
	 * @category defaults
	 */
	public float defaultImageQuality() {
		return JunImageUtility.DefaultImageQuality();
	}

	/**
	 * Answer the point of default offset.
	 * 
	 * @return java.awt.Point
	 * @category defaults
	 */
	public Point defaultOffsetPoint() {
		return new Point(25, 25);
	}

	/**
	 * Changed the receiver's view with the specified box.
	 * 
	 * @param newBounds java.awt.Rectangle
	 * @return jp.co.sra.jun.system.framework.JunApplicationModel
	 * @category displaying
	 */
	public JunApplicationModel changedViewBox_(Rectangle newBounds) {
		return this;
	}

	/**
	 * Display the composed text on the graphics with the specified attributes.
	 * 
	 * @param aComposedText jp.co.sra.smalltalk.StComposedText
	 * @param aGraphics java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @category displaying
	 */
	public void displayComposedText_on_at_(StComposedText aComposedText, Graphics aGraphics, Point aPoint) {
		this.displayComposedText_on_at_textColor_vergeColor_alignmentSymbol_(aComposedText, aGraphics, aPoint, Color.black, Color.white, $("origin"));
	}

	/**
	 * Display the composed text on the graphics with the specified attributes.
	 * 
	 * @param aComposedText jp.co.sra.smalltalk.StComposedText
	 * @param aGraphics java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @param alignmentSymbol jp.co.sra.smalltalk.StSymbol
	 * @category displaying
	 */
	public void displayComposedText_on_at_alignmentSymbol_(StComposedText aComposedText, Graphics aGraphics, Point aPoint, StSymbol alignmentSymbol) {
		this.displayComposedText_on_at_textColor_vergeColor_alignmentSymbol_(aComposedText, aGraphics, aPoint, Color.black, Color.white, alignmentSymbol);
	}

	/**
	 * Display the composed text on the graphics with the specified attributes.
	 * 
	 * @param aComposedText jp.co.sra.smalltalk.StComposedText
	 * @param aGraphics java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @param textColor java.awt.Color
	 * @param vergeColor java.awt.Color
	 * @category displaying
	 */
	public void displayComposedText_on_at_textColor_vergeColor_(StComposedText aComposedText, Graphics aGraphics, Point aPoint, Color textColor, Color vergeColor) {
		this.displayComposedText_on_at_textColor_vergeColor_alignmentSymbol_(aComposedText, aGraphics, aPoint, textColor, vergeColor, $("origin"));
	}

	/**
	 * Display the composed text on the graphics with the specified attributes.
	 * 
	 * @param aComposedText jp.co.sra.smalltalk.StComposedText
	 * @param aGraphics java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @param textColor java.awt.Color
	 * @param vergeColor java.awt.Color
	 * @param alignmentSymbol jp.co.sra.smalltalk.StSymbol
	 * @category displaying
	 */
	public void displayComposedText_on_at_textColor_vergeColor_alignmentSymbol_(StComposedText aComposedText, Graphics aGraphics, Point aPoint, Color textColor, Color vergeColor, StSymbol alignmentSymbol) {
		StRectangle composedBox = new StRectangle(aComposedText.bounds());
		Point alignPoint = null;
		try {
			alignPoint = (Point) composedBox.perform_(alignmentSymbol.toString());
		} catch (Exception e) {
			throw new SmalltalkException(e);
		}
		composedBox = composedBox.align_with_(alignPoint, aPoint);
		Point displayPoint = composedBox.origin();

		aGraphics.setColor(vergeColor);
		Point[] points = this.neighborhoodPoints_(displayPoint);
		for (int i = 0; i < points.length; i++) {
			aComposedText.displayOn_at_(aGraphics, points[i]);
		}

		aGraphics.setColor(textColor);
		aComposedText.displayOn_at_(aGraphics, displayPoint);
	}

	/**
	 * Display the image on the graphics with the specified attributes.
	 * 
	 * @param anImage java.awt.Image
	 * @param aGraphics java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @category displaying
	 */
	public void displayImage_on_at_(Image anImage, Graphics aGraphics, Point aPoint) {
		this.displayImage_on_at_alignmentSymbol_(anImage, aGraphics, aPoint, $("origin"));
	}

	/**
	 * Display the image on the graphics with the specified attributes.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @param aGraphics java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @category displaying
	 */
	public void displayImage_on_at_(StImage anImage, Graphics aGraphics, Point aPoint) {
		this.displayImage_on_at_alignmentSymbol_(anImage.image(), aGraphics, aPoint, $("origin"));
	}

	/**
	 * Display the image on the graphics with the specified attributes.
	 * 
	 * @param anImage java.awt.Image
	 * @param aGraphics java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @param alignmentSymbol jp.co.sra.smalltalk.StSymbol
	 * @category displaying
	 */
	public void displayImage_on_at_alignmentSymbol_(Image anImage, Graphics aGraphics, Point aPoint, StSymbol alignmentSymbol) {
		int width, height;
		if (anImage instanceof BufferedImage) {
			BufferedImage theImage = (BufferedImage) anImage;
			width = theImage.getWidth();
			height = theImage.getHeight();
		} else {
			width = anImage.getWidth(null);
			height = anImage.getHeight(null);
		}
		StRectangle composedBox = new StRectangle(0, 0, width, height);
		Point alignPoint = null;
		try {
			alignPoint = (Point) composedBox.perform_(alignmentSymbol.toString());
		} catch (Exception e) {
			throw new SmalltalkException(e);
		}
		composedBox = composedBox.align_with_(alignPoint, aPoint);
		Point displayPoint = composedBox.origin();
		aGraphics.drawImage(anImage, displayPoint.x, displayPoint.y, null);
	}

	/**
	 * Display the image on the graphics with the specified attributes.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @param aGraphics java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @param alignmentSymbol jp.co.sra.smalltalk.StSymbol
	 * @category displaying
	 */
	public void displayImage_on_at_alignmentSymbol_(StImage anImage, Graphics aGraphics, Point aPoint, StSymbol alignmentSymbol) {
		this.displayImage_on_at_alignmentSymbol_(anImage.image(), aGraphics, aPoint, alignmentSymbol);
	}

	/**
	 * Evaluate the block closure and wait at least for the specified time.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param millisecondTime int
	 * @category evaluating
	 */
	public Object do_forMilliseconds_(StBlockClosure aBlock, int millisecondTime) {
		return Do_forMilliseconds_(aBlock, millisecondTime);
	}

	/**
	 * Evaluate the block closure to handle the specified number of frames per second.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param aNumber int
	 * @category evaluating
	 */
	public Object do_framesPerSecond_(StBlockClosure aBlock, int aNumber) {
		return Do_framesPerSecond_(aBlock, aNumber);
	}

	/**
	 * Called by the garbage collector on an object when garbage collection determines
	 * that there are no more references to the object.
	 * 
	 * @throws java.lang.Throwable
	 * @see java.lang.Object#finalize()
	 * @category finalization
	 */
	protected void finalize() throws Throwable {
		this.debug_(new StBlockClosure() {
			public Object value() {
				System.out.println();
				System.out.println(this.printString() + " passwd away.");
				return null;
			}
		});
		super.finalize();
	}

	/**
	 * Clean up this menus.
	 * 
	 * @category flushing
	 */
	protected void flushMenus() {
	}

	/**
	 * Clean up this visuals.
	 * 
	 * @category flushing
	 */
	protected void flushVisuals() {
	}

	/**
	 * Set the receiver's closeBlock.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category interface closing
	 */
	public void closeBlock_(StBlockClosure aBlock) {
		closeBlock = aBlock;
	}

	/**
	 * Set the colleagues to be closed together.
	 * 
	 * @param applicationModels jp.co.sra.smalltalk.StApplicationModel[]
	 * @category interface closing
	 */
	public void closeTogether_(StApplicationModel[] applicationModels) {
		this.closeTogether_(applicationModels, false, false);
	}

	/**
	 * Set the colleagues to be closed together.
	 * 
	 * @param aCollection java.util.Collection
	 * @category interface closing
	 */
	public void closeTogether_(final Collection aCollection) {
		this.closeTogether_((StApplicationModel[]) aCollection.toArray(new StApplicationModel[aCollection.size()]));
	}

	/**
	 * Set the colleagues to be closed together when ALT key is pressed.
	 * 
	 * @param applicationModels jp.co.sra.smalltalk.StApplicationModel[]
	 * @category interface closing
	 */
	public void closeTogetherWhenAltDown_(StApplicationModel[] applicationModels) {
		this.closeTogether_(applicationModels, false, true);
	}

	/**
	 * Set the colleagues to be closed together when SHIFT key is pressed.
	 * 
	 * @param applicationModels jp.co.sra.smalltalk.StApplicationModel[]
	 * @category interface closing
	 */
	public void closeTogetherWhenShiftDown_(StApplicationModel[] applicationModels) {
		this.closeTogether_(applicationModels, true, false);
	}

	/**
	 * Set the colleagues to be closed together when SHIFT or ALT key is pressed.
	 * 
	 * @param applicationModels jp.co.sra.smalltalk.StApplicationModel[]
	 * @category interface closing
	 */
	public void closeTogetherWhenShiftDownOrAltDown_(StApplicationModel[] applicationModels) {
		this.closeTogether_(applicationModels, true, true);
	}

	/**
	 * Set the colleagues to be closed together.
	 * 
	 * @param applicationModels jp.co.sra.smalltalk.StApplicationModel[]
	 * @category interface closing
	 */
	protected void closeTogether_(final StApplicationModel[] applicationModels, final boolean checkShiftDown, final boolean checkAltDown) {
		this.closeBlock_(new StBlockClosure() {
			public Object value_(Object anObject) {
				if (checkShiftDown && JunSensorUtility.ShiftDown() == false) {
					return null;
				}
				if (checkAltDown && JunSensorUtility.AltDown() == false) {
					return null;
				}

				for (int i = 0; i < applicationModels.length; i++) {
					if (applicationModels[i] != anObject) {
						applicationModels[i].closeRequest();
					}
				}
				return null;
			}
		});
	}

	/**
	 * Invoked when a window is in the process of being closed.
	 * 
	 * @param evt java.awt.event.WindowEvent
	 * @see jp.co.sra.smalltalk.StApplicationModel#noticeOfWindowClose(java.awt.event.WindowEvent)
	 * @category interface closing
	 */
	public void noticeOfWindowClose(WindowEvent evt) {
		super.noticeOfWindowClose(evt);

		if (closeBlock != null) {
			switch (closeBlock.numArgs()) {
				case 0:
					closeBlock.value();
					break;
				case 1:
					closeBlock.value_(this);
					break;
				case 2:
					closeBlock.value_value_(this, this.getWindow());
					break;
				case 3:
					closeBlock.value_value_value_(this, this.getWindow(), this.getView());
					break;
				case 4:
					closeBlock.valueWithArguments_(new Object[] { this, this.getWindow(), this.getView(), this.getViews() });
					break;
			}
		}
	}

	/**
	 * Create a Frame for a default view and open it at the specified point.
	 *
	 * @return java.awt.Frame
	 * @param aPoint java.awt.Point
	 * @category interface opening
	 */
	public Frame openAt_(Point aPoint) {
		return this.openViewAt_(this.defaultView(), aPoint);
	}

	/**
	 * Create a Frame for a default view and open it in the specified area.
	 *
	 * @return java.awt.Frame
	 * @param aRectangle java.awt.Rectangle
	 * @category interface opening
	 */
	public Frame openIn_(Rectangle aRectangle) {
		return this.openViewIn_(this.defaultView(), aRectangle);
	}

	/**
	 * Create a Frame for the view and open it at the specified point.
	 *
	 * @return java.awt.Frame
	 * @param aView jp.co.sra.smalltalk.StView
	 * @param aPoint java.awt.Point
	 * @category interface opening
	 */
	public Frame openViewAt_(StView aView, Point aPoint) {
		Frame aFrame = this.allButOpenView_(aView);
		aFrame.setLocation(aPoint);
		aFrame.setVisible(true);
		return aFrame;
	}

	/**
	 * Create a Frame for the view and open it in the specified area.
	 *
	 * @return java.awt.Frame
	 * @param aView jp.co.sra.smalltalk.StView
	 * @param aRectangle java.awt.Rectangle
	 * @category interface opening
	 */
	public Frame openViewIn_(StView aView, Rectangle aRectangle) {
		Frame aFrame = this.allButOpenView_(aView);
		aFrame.setBounds(_MakeRectangleVisible(aRectangle));
		aFrame.setVisible(true);
		return aFrame;
	}

	/**
	 * This method will be sent by the view when its top component is opend.
	 * 
	 * @param aView jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StUIBulider#windowOpened(WindowEvent)
	 * @see jp.co.sra.smalltalk.StApplicationModel#postOpenWith_(jp.co.sra.smalltalk.StView)
	 * @category interface opening
	 */
	public void postOpenWith_(StView aView) {
		super.postOpenWith_(aView);

		this.updateMenuIndication();

		/*
		 * The following statement is defined in Smalltalk version, but
		 * not in Java version.  This is basically done when creating
		 * StApplicationWindow.  With this statement, the window title
		 * specified right after open() is called might be overridden.
		 */
		// this.setWindowLabel();
	}

	/**
	 * Create an application window with the secret view.
	 * 
	 * @return java.awt.Frame
	 * @category interface opening
	 */
	public Frame secretOpen() {
		Frame aFrame = this.allButOpenView_(this.defaultView());
		Dimension screenSize = aFrame.getToolkit().getScreenSize();
		Rectangle aRectangle = new Rectangle(screenSize.width - 25, screenSize.height - 50, aFrame.getBounds().width, aFrame.getBounds().height);
		aFrame.setBounds(aRectangle);
		aFrame.setVisible(false);
		this.postOpenWith_(this.getView());
		return aFrame;
	}

	/**
	 * Set window label.
	 * 
	 * @category interface opening
	 * @deprecated since Jun633, use updateWindowTitle()
	 */
	public void setWindowLabel() {
		Window aWindow = this.getWindow();
		if (aWindow == null) {
			return;
		}

		if (aWindow instanceof Frame) {
			((Frame) aWindow).setTitle(this.windowTitle());
		} else if (aWindow instanceof Dialog) {
			((Dialog) aWindow).setTitle(this.windowTitle());
		}
	}

	/**
	 * Answer a window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("Application");
	}

	/**
	 * Answer the kind name of the element.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category lisp support
	 */
	protected StSymbol kindName() {
		return this._className();
	}

	/**
	 * Answer a new lisp cons cell.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons lispCons() {
		return JunLispCons.Cell();
	}

	/**
	 * Answer a new lisp nil.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispNil lispNil() {
		return JunLispNil.NullList();
	}

	/**
	 * Convert the element map to a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		throw SmalltalkException.SubclassResponsibility();
	}

	/**
	 * Update the menu indication.
	 * 
	 * @category menu accessing
	 */
	public void updateMenuIndication() {
		return;
	}

	/**
	 * Create new model and open it.
	 * 
	 * @return jp.co.sra.jun.system.framework.JunApplicationModel
	 * @category menu messages
	 */
	public JunApplicationModel newModel() {
		try {
			JunApplicationModel aModel = (JunApplicationModel) this.getClass().newInstance();
			aModel.open();
			return aModel;
		} catch (InstantiationException e) {
		} catch (IllegalAccessException e) {
		}
		return null;
	}

	/**
	 * Read an image from the specified file, and answer it.
	 * 
	 * @param aFile java.io.File
	 * @return jp.co.sra.smalltalk.StImage
	 * @category menu messages
	 */
	public StImage readImageFrom_(File aFile) {
		Class aClass = JunImageStream.ImageStreamClassForFileName_(aFile);
		if (aClass == null) {
			return null;
		}

		StImage anImage = null;
		JunImageStream aStream = null;
		JunCursors cursor = new JunCursors(JunCursors.ReadCursor());
		try {
			cursor._show();
			aStream = (JunImageStream) StObject._PerformWith(aClass, "On_", new FileInputStream(aFile));
			anImage = aStream.nextImage();
		} catch (IOException e) {
			System.err.println(e.getMessage());
			e.printStackTrace();
		} catch (NoSuchMethodException e) {
			System.err.println(e.getMessage());
			e.printStackTrace();
		} finally {
			if (aStream != null) {
				try {
					aStream.close();
					aStream = null;
				} catch (IOException e) {
					System.err.println(e.getMessage());
					e.printStackTrace();
				}
			}
			cursor._restore();
		}
		return anImage;
	}

	/**
	 * Request new image filename and answer it.
	 * 
	 * @return java.io.File
	 * @category menu messages
	 */
	public File requestNewImageFilename() {
		String[] writeImageExtensions = null;
		if (Arrays.binarySearch(ImageIO.getWriterFormatNames(), "gif") >= 0) {
			writeImageExtensions = JunSystem.DefaultImageExtensionPatterns();
		} else {
			String[] imageExtensions = JunSystem.DefaultImageExtensionPatterns();
			writeImageExtensions = new String[imageExtensions.length - 2];
			int index = 0;
			for (int i = 0; i < imageExtensions.length; i++) {
				if (imageExtensions[i].toLowerCase().equals("*.gif") == false) {
					writeImageExtensions[index] = imageExtensions[i];
					index++;
				}
			}
		}

		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType($String("<1p> files", null, $String("Image")), writeImageExtensions) };
		String fileName = null;
		if (StInputState.Default().shiftDown() || StInputState.Default().altDown()) {
			Window window = this.getWindow();
			if (window == null || window instanceof Frame == false) {
				fileName = this.defaultBaseName();
			} else {
				if (StInputState.Default().altDown()) {
					fileName = ((Frame) window).getTitle() + this.defaultBaseName();
				} else {
					fileName = ((Frame) window).getTitle();
				}
			}
		} else {
			fileName = this.defaultBaseName();
		}

		File file = JunFileRequesterDialog.RequestNewFile($String("Input an <1p> file.", null, $String("Image")), new File(fileName + ".jpg"), fileTypes, fileTypes[0]);
		if (file == null) {
			return null;
		}
		return file;
	}

	/**
	 * Save the receiver's display object as an image file.
	 * 
	 * @category menu messages
	 */
	public void saveAsImage() {
		StImage anImage = this.asImage();
		if (anImage == null) {
			return;
		}
		File aFile = this.requestNewImageFilename();
		if (aFile == null) {
			return;
		}
		this.writeImage_to_(anImage, aFile);
	}

	/**
	 * Save the receiver's display object as the specified image file.
	 * 
	 * @param aFile java.io.File
	 * @category menu messages
	 */
	public void saveAsImageTo_(File aFile) {
		StImage anImage = this.asImage();
		if (anImage == null) {
			return;
		}
		this.writeImage_to_(anImage, aFile);
	}

	/**
	 * Open dialog for under construction.
	 * 
	 * @category menu messages
	 */
	public void underConstruction() {
		JunDialog.Warn_($String("Sorry, it is under construction."));
	}

	/**
	 * Write the specified image to a file.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @param aFile java.io.File
	 * @category menu messages
	 */
	public void writeImage_to_(StImage anImage, File aFile) {
		try {
			JunImageUtility.WriteImage_to_(anImage, aFile);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Answer the receiver's current controller.
	 * 
	 * @return jp.co.sra.smalltalk.StController
	 * @category private
	 */
	public StController getController() {
		StView aView = this.getView();
		if (aView == null) {
			return null;
		}
		return aView.controller();
	}

	/**
	 * Answer the receiver's controllers.
	 * 
	 * @return jp.co.sra.smalltalk.StController[]
	 * @category private
	 */
	public StController[] getControllers() {
		StView[] views = this.getViews();
		StController[] controllers = new StController[views.length];
		for (int i = 0; i < views.length; i++) {
			controllers[i] = views[i].controller();
		}
		return controllers;
	}

	/**
	 * Answer the receiver's current view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @category private
	 */
	public StView getView() {
		Object[] dependents = this.dependents();
		for (int i = 0; i < dependents.length; i++) {
			Object each = dependents[i];
			if (each instanceof StView && ((StView) each).model() == this) {
				return (StView) each;
			}
		}

		return null;
	}

	/**
	 * Answer the receiver's views.
	 * 
	 * @return jp.co.sra.smalltalk.StView[]
	 * @category private
	 */
	public StView[] getViews() {
		Collection viewCollection = new ArrayList();
		Object[] dependents = this.dependents();
		for (int i = 0; i < dependents.length; i++) {
			Object each = dependents[i];
			if (each instanceof StView && ((StView) each).model() == this) {
				viewCollection.add(each);
			}
		}
		return (StView[]) viewCollection.toArray(new StView[viewCollection.size()]);
	}

	/**
	 * Answer my window.
	 * 
	 * @return java.awt.Window
	 * @category private
	 */
	public Window getWindow() {
		StView aView = this.getView();
		if (aView == null) {
			return null;
		}
		return aView.topComponent();
	}

	/**
	 * Answer a margin rectangle for visible screen area.
	 * 
	 * @return java.awt.Rectangle
	 * @category private
	 */
	protected Rectangle marginRectangleForVisibleScreenArea() {
		return new Rectangle(10, 30, 0, -20);
	}

	/**
	 * Answer the neighborhood points of the specified point.
	 * 
	 * @param aPoint java.awt.Point
	 * @return java.awt.Point[]
	 * @category private
	 */
	protected Point[] neighborhoodPoints_(Point aPoint) {
		return JunPoint.NeighborhoodPoints_(aPoint);
	}

	/**
	 * Set the size for the view.
	 * 
	 * @param extent java.awt.Dimension
	 * @category private
	 */
	protected void setSize_(Dimension extent) {
		StView aView = this.getView();
		if (aView == null) {
			return;
		}

		Window aWindow = aView.topComponent();
		if (aWindow == null) {
			return;
		}

		Dimension oldExtent = ((Component) aView).getSize();
		Dimension newExtent = extent;
		int width = aWindow.getWidth() - oldExtent.width + newExtent.width;
		int height = aWindow.getHeight() - oldExtent.height + newExtent.height;
		aWindow.setSize(width, height);
		aWindow.validate();
	}

	/**
	 * Update this window.
	 * 
	 * @category private
	 */
	public void updateWindow() {
		this.updateMenuBar();
		this.updateWindowTitle();
		this.flushVisuals();
		Window aWindow = this.getWindow();
		if (aWindow != null) {
			aWindow.repaint();
		}
	}

	/**
	 * Update the menu bar of my windows.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#updateMenuBar()
	 * @category updating
	 */
	public void updateMenuBar() {
		this.flushMenus();
		super.updateMenuBar();
	}

}
