package jp.co.sra.jun.terrain.editor;

import java.awt.Color;
import java.awt.Point;
import java.awt.Rectangle;
import java.util.ArrayList;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StInputState;
import jp.co.sra.smalltalk.StInterval;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.menu.MenuPerformer;
import jp.co.sra.smalltalk.menu.StMenu;
import jp.co.sra.smalltalk.menu.StMenuBar;
import jp.co.sra.smalltalk.menu.StMenuItem;

import jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayProcessor;
import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox;
import jp.co.sra.jun.geometry.boundaries.JunBorderGenerator;
import jp.co.sra.jun.geometry.curves.Jun2dLine;
import jp.co.sra.jun.goodies.button.JunButtonModel;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.goodies.progress.JunCursorAnimator;
import jp.co.sra.jun.goodies.wheels.JunThumbWheel;
import jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolyline;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolylineLoop;
import jp.co.sra.jun.opengl.projection.JunOpenGLProjection;
import jp.co.sra.jun.opengl.projection.JunOpenGLProjector;
import jp.co.sra.jun.system.framework.JunDialog;
import jp.co.sra.jun.terrain.display.JunTerrainDisplayModel;
import jp.co.sra.jun.terrain.support.JunTerrainField;
import jp.co.sra.jun.terrain.support.JunTerrainTriangle;

/**
 * JunTerrainEditModel class
 * 
 *  @author    nisinaka
 *  @created   2001/11/19 (by nisinaka)
 *  @updated   2005/03/03 (by nisinaka)
 *  @updated   2007/08/28 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun680 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunTerrainEditModel.java,v 8.15 2008/02/20 06:33:00 nisinaka Exp $
 */
public class JunTerrainEditModel extends JunOpenGLDisplayModel {

	protected ArrayList peaks;
	protected Jun3dPoint[][] contours;
	protected double contourStep;
	protected double contourGrid;
	protected JunTerrainField terrainField;
	protected Jun2dBoundingBox mapBounds;
	protected Jun2dPoint sight;
	protected double zoomHeight;

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		peaks = new ArrayList();
		contours = null;
		contourStep = Double.NaN;
		contourGrid = Double.NaN;
		terrainField = null;
		mapBounds = null;
		sight = null;
		zoomHeight = Double.NaN;
	}

	/**
	 * Answer the array of peaks.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category accessing
	 */
	public Jun3dPoint[] peaks() {
		return (Jun3dPoint[]) peaks.toArray(new Jun3dPoint[peaks.size()]);
	}

	/**
	 * Answer the array of contours.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[][]
	 * @category accessing
	 */
	public Jun3dPoint[][] contours() {
		if (contours == null) {
			contours = this.computeContours();
		}
		return contours;
	}

	/**
	 * Answer my current contour step.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double contourStep() {
		if (Double.isNaN(contourStep)) {
			contourStep = this.defaultContourStep();
		}
		return contourStep;
	}

	/**
	 * Set my new contour step.
	 * 
	 * @param aNumber double
	 * @category accessing
	 */
	public void contourStep_(double aNumber) {
		contourStep = aNumber;
		this.flushContours();
		this.changed();
	}

	/**
	 * Answer my current contour grid.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double contourGrid() {
		if (Double.isNaN(contourGrid)) {
			contourGrid = this.defaultContourGrid();
		}

		return contourGrid;
	}

	/**
	 * Set my new contour grid.
	 * 
	 * @param aNumber double
	 * @category accessing
	 */
	public void contourGrid_(double aNumber) {
		contourGrid = aNumber;
		this.flushContours();
		this.changed();
	}

	/**
	 * Answer my terrain field.
	 * 
	 * @return jp.co.sra.jun.terrain.support.JunTerrainField
	 * @category accessing
	 */
	public JunTerrainField terrainField() {
		if (terrainField == null) {
			terrainField = this.computeTerrainField();
		}
		return terrainField;
	}

	/**
	 * Answer my current display object.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#displayObject()
	 * @category accessing
	 */
	public JunOpenGL3dObject displayObject() {
		if (openGL3dObject == null) {
			this.computeDisplayObject();
		}
		return openGL3dObject;
	}

	/**
	 * Answer my current projection.
	 * 
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLProjection
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#displayProjection()
	 * @category accessing
	 */
	public JunOpenGLProjection displayProjection() {
		return this.displayProjector().projection();
	}

	/**
	 * Answer my current projector.
	 * 
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLProjector
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#displayProjector()
	 * @category accessing
	 */
	public JunOpenGLProjector displayProjector() {
		if (openGLProjector == null) {
			openGLProjector = new JunOpenGLProjector();
			openGLProjector.parallelProjection();
			openGLProjector.sightPoint_(this.sightPoint());
			openGLProjector.eyePoint_(this.eyePoint());
			openGLProjector.upVector_(this.defaultUpVector());
			openGLProjector.viewFactor_(this.defaultViewFactor());
			openGLProjector.zoomHeight_(this.defaultZoomHeight());
		}

		return openGLProjector;
	}

	/**
	 * Answer my current map bounds.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dBoundingBox
	 * @category bounds accessing
	 */
	public Jun2dBoundingBox mapBounds() {
		if (mapBounds == null) {
			mapBounds = this.defaultMapBounds();
		}
		return mapBounds;
	}

	/**
	 * Add a 3D point as a peak.
	 * 
	 * @param aJun3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category adding
	 */
	public Jun3dPoint addPeak_(Jun3dPoint aJun3dPoint) {
		peaks.add(aJun3dPoint);
		this.flushContours();
		this.updateProjection();
		this.updateDisplayObject();
		return aJun3dPoint;
	}

	/**
	 * Answer my peak button.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel peakButton() {
		if (this.pushButtons().containsKey($("peak")) == false) {
			JunButtonModel button = new JunButtonModel();
			button.value_(false);
			button.visual_(JunCursors.PeakCursorImage());
			button.action_(this.peakButtonAction());
			this.pushButtons().put($("peak"), button);
		}
		return (JunButtonModel) this.pushButtons().get($("peak"));
	}

	/**
	 * Answer the button action for the peak button.
	 * 
	 * @return jp.co.sra.smalltalk.StBlockClosure
	 * @category buttons
	 */
	protected StBlockClosure peakButtonAction() {
		return new StBlockClosure() {
			public Object value_(Object o) {
				JunButtonModel model = (JunButtonModel) o;
				JunTerrainEditModel.this.setButtonState($("peak"), !model.value());
				return null;
			}
		};
	}

	/**
	 * Answer my show button.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel showButton() {
		if (this.pushButtons().containsKey($("show")) == false) {
			JunButtonModel button = new JunButtonModel();
			button.value_(false);
			button.visual_(JunCursors.ReadCursorImage());
			button.action_(this.showButtonAction());
			this.pushButtons().put($("show"), button);
		}
		return (JunButtonModel) this.pushButtons().get($("show"));
	}

	/**
	 * Answer the button action for the show button.
	 * 
	 * @return jp.co.sra.smalltalk.StBlockClosure
	 * @category buttons
	 */
	protected StBlockClosure showButtonAction() {
		return new StBlockClosure() {
			public Object value_(Object o) {
				JunButtonModel model = (JunButtonModel) o;
				JunTerrainEditModel.this.setButtonState($("show"), !model.value());
				return null;
			}
		};
	}

	/**
	 * Answer my current button state.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category buttons
	 */
	public StSymbol buttonState() {
		if (this.pushButtons().size() <= 3) {
			this.dragButton();
			this.peakButton();
			this.showButton();
		}
		return super.buttonState();
	}

	/**
	 * Set my new button status.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @param aBoolean boolean
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#setButtonState(jp.co.sra.smalltalk.StSymbol, boolean)
	 * @category buttons
	 */
	protected synchronized void setButtonState(StSymbol aSymbol, boolean aBoolean) {
		super.setButtonState(aSymbol, aBoolean);
	}

	/**
	 * Answer a zThumbWheel.
	 * 
	 * @return jp.co.sra.jun.goodies.wheels.JunThumbWheel
	 * @category wheels
	 */
	public JunThumbWheel zThumbWheel() {
		JunThumbWheel thumbWheel = new JunThumbWheel(true);
		thumbWheel.compute_(new StBlockClosure() {
			public Object value_(Object value) {
				double factor = 1 + (((Number) value).doubleValue() / 100);
				zoom_(factor);
				return null;
			}
		});
		return thumbWheel;
	}

	/**
	 * Answer the default contour grid.
	 * 
	 * @return double
	 * @category defaults
	 */
	public double defaultContourGrid() {
		return Math.min(this.mapBounds().width(), this.mapBounds().height()) / 20.0d;
	}

	/**
	 * Answer the default contour step.
	 * 
	 * @return double
	 * @category defaults
	 */
	public double defaultContourStep() {
		return 10.0d;
	}

	/**
	 * Answer the default value of 'mapBounds'.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dBoundingBox
	 * @category defaults
	 */
	public Jun2dBoundingBox defaultMapBounds() {
		return Jun2dBoundingBox.Origin_corner_(new Jun2dPoint(-1000, -1000), new Jun2dPoint(1000, 1000));
	}

	/**
	 * Answer the default sight point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category defaults
	 */
	public Jun2dPoint defaultSight() {
		return this.mapBounds().center();
	}

	/**
	 * Answer my default up vector.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.Jun3dPoint
	 * @category defaults
	 */
	public Jun3dPoint defaultUpVector() {
		return new Jun3dPoint(0, 1, 0);
	}

	/**
	 * Answer my default view factor.
	 * 
	 * @return double
	 * @category defaults
	 */
	public double defaultViewFactor() {
		return 10;
	}

	/**
	 * Answer the default zoom height.
	 * 
	 * @return double
	 * @category defaults
	 */
	public double defaultZoomHeight() {
		return this.mapBounds().height() * 1.1d;
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @category interface opening
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunTerrainEditViewAwt(this);
		} else {
			return new JunTerrainEditViewSwing(this);
		}
	}

	/**
	 * Answer the title string.
	 * 
	 * @return java.lang.String
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("Terrain Editor");
	}

	/**
	 * Called when a mouse is dragged.
	 * 
	 * @param from2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param to2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category manipulating
	 */
	protected void drag_xy_(Jun2dPoint from2dPoint, Jun2dPoint to2dPoint) {
		if (this.displayObject() == null) {
			return;
		}

		JunOpenGLProjection aProjection = this.displayProjection();
		Jun3dPoint fromPoint = aProjection.translateTo3dPointFromPoint_(from2dPoint);
		Jun3dPoint toPoint = aProjection.translateTo3dPointFromPoint_(to2dPoint);
		Jun3dPoint vector = fromPoint.minus_(toPoint);
		this.sight_(new Jun2dPoint(this.sight().x() + vector.x(), this.sight().y() + vector.y()));
	}

	/**
	 * Plot a peak with the information of the controller.
	 * 
	 * @param aController jp.co.sra.jun.terrain.editor.JunTerrainEditController
	 * @category manipulating
	 */
	protected void plotPeak_(JunTerrainEditController aController) {
		String numberString = JunDialog.Request_($String("Input a peek height."), "100.0");
		if (numberString == null || numberString.length() == 0) {
			return;
		}

		try {
			double height = Double.parseDouble(numberString);
			Jun3dPoint point = this.mousePointFrom_(aController);
			this.addPeak_(new Jun3dPoint(point.x(), point.y(), height));
		} catch (NumberFormatException e) {
			JunDialog.Warn_(numberString + $String(" is invalid value."));
		}
	}

	/**
	 * Show the terrain.
	 * 
	 * @param aController jp.co.sra.jun.terrain.editor.JunTerrainEditController
	 * @return jp.co.sra.jun.terrain.display.JunTerrainDisplayModel
	 * @category manipulating
	 */
	protected JunTerrainDisplayModel showTerrain_(JunTerrainEditController aController) {
		JunAngle zoomAngle = JunAngle.FromDeg_(45);
		Jun2dLine sightLineSegment = aController.mouse2dLineSegment();
		if (sightLineSegment == null) {
			return null;
		}

		Jun3dPoint eyePoint = this.displayProjection().translateTo3dPointFromPoint_(sightLineSegment.from());
		Jun3dPoint sightPoint = this.displayProjection().translateTo3dPointFromPoint_(sightLineSegment.to());
		if (eyePoint.distance_(sightPoint) < (1.0d - 12)) {
			return null;
		}

		JunTerrainDisplayModel displayModel = new JunTerrainDisplayModel(this.terrainField());
		displayModel.defaultZoomHeight_(eyePoint.minus_(sightPoint).length() * zoomAngle.tan());
		displayModel.defaultUpVector_(new Jun3dPoint(0, 0, 1));
		displayModel.moveTo_sight_(new Jun2dPoint(eyePoint.x(), eyePoint.y()), new Jun2dPoint(sightPoint.x(), sightPoint.y()));

		JunOpenGLProjection projection = displayModel.displayProjection();
		displayModel.defaultSightPoint_(projection.sightPoint());
		displayModel.defaultEyePoint_(projection.eyePoint());
		displayModel.open();

		return displayModel;
	}

	/**
	 * Zoom in/out.
	 * 
	 * @param factor double
	 * @category manipulating
	 */
	public void zoom_(double factor) {
		double zoomFactor = factor;
		if (StInputState.Default().shiftDown()) {
			if (factor >= 1.0) {
				zoomFactor = 1.0 + ((factor - 1.0) * 0.1);
			} else {
				zoomFactor = 1.0 - ((1.0 - factor) * 0.1);
			}
		}
		this.zoomHeight_(this.zoomHeight() * zoomFactor);
	}

	/**
	 * Answer the current eye point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category projection
	 */
	public Jun3dPoint eyePoint() {
		return new Jun3dPoint(this.sight().x(), this.sight().y(), (this.topPeak() + 1) * 2);
	}

	/**
	 * Answer my current sight.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category projection
	 */
	public Jun2dPoint sight() {
		if (sight == null) {
			sight = this.defaultSight();
		}
		return sight;
	}

	/**
	 * Set my new sight.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category projection
	 */
	public void sight_(Jun2dPoint aPoint) {
		sight = aPoint;
		this.updateProjection();
	}

	/**
	 * Answer my current sight point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category projection
	 */
	public Jun3dPoint sightPoint() {
		return new Jun3dPoint(this.sight().x(), this.sight().y(), 0.0d);
	}

	/**
	 * Answer the current value of 'zoomHeight'.
	 * 
	 * @return double
	 * @category projection
	 */
	public double zoomHeight() {
		if (Double.isNaN(zoomHeight)) {
			zoomHeight = this.defaultZoomHeight();
		}
		return zoomHeight;
	}

	/**
	 * Set the new zoomHeight value.
	 * 
	 * @param newZoomHeight double
	 * @category projection
	 */
	public void zoomHeight_(double newZoomHeight) {
		zoomHeight = newZoomHeight;
		this.updateProjection();
	}

	/**
	 * Fit the 3d object to the current view.
	 * 
	 * @category projection
	 */
	public void fitSilently() {
		// Avoid to fit when opening.
	}

	/**
	 * Answer my menu bar.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenuBar
	 * @see jp.co.sra.smalltalk.StApplicationModel#_menuBar()
	 * @category resources
	 */
	public StMenuBar _menuBar() {
		if (_menuBar == null) {
			_menuBar = new StMenuBar();

			StMenu contoursMenu = new StMenu($String("Contours"));
			contoursMenu.add(new StMenuItem($String("Grid") + "...", new MenuPerformer(this, "changeContourGrid")));
			contoursMenu.add(new StMenuItem($String("Step") + "...", new MenuPerformer(this, "changeContourStep")));
			_menuBar.add(contoursMenu);

			StMenu viewMenu = new StMenu($String("View"));
			viewMenu.add(new StMenuItem($String("Center"), new MenuPerformer(this, "sightCenter")));
			viewMenu.addSeparator();
			viewMenu.add(new StMenuItem($String("x 1"), new MenuPerformer(this, "zoomX1")));
			viewMenu.add(new StMenuItem($String("x 2"), new MenuPerformer(this, "zoomX2")));
			viewMenu.add(new StMenuItem($String("x 4"), new MenuPerformer(this, "zoomX4")));
			viewMenu.add(new StMenuItem($String("x 8"), new MenuPerformer(this, "zoomX8")));
			_menuBar.add(viewMenu);

			StMenu miscMenu = new StMenu($String("Misc"));
			miscMenu.add(new StMenuItem($String("Show terrain contour"), new MenuPerformer(this, "showContour")));
			miscMenu.add(new StMenuItem($String("Show terrain field"), new MenuPerformer(this, "showTerrainField")));
			_menuBar.add(miscMenu);

		}
		return _menuBar;
	}

	/**
	 * Change my contour grid.
	 * 
	 * @category menu messages
	 */
	public void changeContourGrid() {
		String numberString = JunDialog.Request_($String("Input a contour grid."), String.valueOf(this.contourGrid()));
		if (numberString == null) {
			return;
		}

		try {
			double grid = Double.parseDouble(numberString);
			this.contourGrid_(grid);
		} catch (NumberFormatException e) {
			JunDialog.Warn_(numberString + $String(" is invalid value."));
		}
	}

	/**
	 * Change my contour step.
	 * 
	 * @category menu messages
	 */
	public void changeContourStep() {
		String numberString = JunDialog.Request_($String("Input a contour step."), String.valueOf(this.contourStep()));
		if (numberString == null) {
			return;
		}

		try {
			double step = Double.parseDouble(numberString);
			this.contourStep_(step);
		} catch (NumberFormatException e) {
			JunDialog.Warn_(numberString + $String(" is invalid value."));
		}
	}

	/**
	 * Show the contour.
	 * 
	 * @category menu messages
	 */
	public void showContour() {
		this.displayObject().show();
	}

	/**
	 * Show the terrain field.
	 * 
	 * @category menu messages
	 */
	public void showTerrainField() {
		JunOpenGL3dObject anObject = this.terrainField().asJunOpenGL3dObject();
		anObject.paint_(Color.green);
		anObject.show();
	}

	/**
	 * Align the sight to center.
	 * 
	 * @category menu messages
	 */
	public void sightCenter() {
		this.sight_(this.mapBounds().center());
	}

	/**
	 * Zoom x1.
	 * 
	 * @category menu messages
	 */
	public void zoomX1() {
		this.zoomHeight_(this.mapBounds().height() * 1.1d);
	}

	/**
	 * Zoom x2.
	 * 
	 * @category menu messages
	 */
	public void zoomX2() {
		this.zoomHeight_(this.mapBounds().height() / 2.0d * 1.1d);
	}

	/**
	 * Zoom x4.
	 * 
	 * @category menu messages
	 */
	public void zoomX4() {
		this.zoomHeight_(this.mapBounds().height() / 4.0d * 1.1d);
	}

	/**
	 * Zoom x8.
	 * 
	 * @category menu messages
	 */
	public void zoomX8() {
		this.zoomHeight_(this.mapBounds().height() / 8.0d * 1.1d);
	}

	/**
	 * Update the menu indications.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#updateMenuIndication()
	 * @category menu messages
	 */
	public void updateMenuIndication() {
		// do nothing.
	}

	/**
	 * Convert the mouse point to a Jun3dPoint with the current projection.
	 * 
	 * @param aController jp.co.sra.jun.terrain.editor.JunTerrainEditController
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category private
	 */
	protected Jun3dPoint mousePointFrom_(JunTerrainEditController aController) {
		return this.displayProjection().translateTo3dPointFromPoint_(aController.mouse2dPoint());
	}

	/**
	 * Normalize the mouse point.
	 * 
	 * @param aPoint java.awt.Point
	 * @param aRectangle java.awt.Rectangle
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category private
	 */
	protected Jun2dPoint regularizePoint_in_(Point aPoint, Rectangle aRectangle) {
		double x = (aPoint.x - aRectangle.width / 2.0d) / (aRectangle.height / 2.0d);
		double y = (aRectangle.height / 2.0d - aPoint.y) / (aRectangle.height / 2.0d);
		return new Jun2dPoint(x, y);
	}

	/**
	 * Create the current display object.
	 * 
	 * @category private-display
	 */
	protected void computeDisplayObject() {
		ArrayList objects = new ArrayList();
		objects.add(this.mapBorderLine());

		Jun3dPoint[] peaks = this.peaks();
		for (int i = 0; i < peaks.length; i++) {
			objects.add(this.peakMarkAt_(peaks[i]));
		}

		Jun3dPoint[][] contours = this.contours();
		for (int i = 0; i < contours.length; i++) {
			objects.add(new JunOpenGL3dPolyline(contours[i], Color.red));
		}

		this.openGL3dObject_(new JunOpenGL3dCompoundObject(objects));
	}

	/**
	 * Create a map border line.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dPolylineLoop
	 * @category private-display
	 */
	protected JunOpenGL3dPolylineLoop mapBorderLine() {
		Jun2dBoundingBox bounds = this.mapBounds();
		Jun2dPoint mapOrigin = bounds.origin();
		Jun2dPoint mapCorner = bounds.corner();
		return new JunOpenGL3dPolylineLoop(new Jun3dPoint[] {
				new Jun3dPoint(mapOrigin.x(), mapOrigin.y(), 0.0d),
				new Jun3dPoint(mapCorner.x(), mapOrigin.y(), 0.0d),
				new Jun3dPoint(mapCorner.x(), mapCorner.y(), 0.0d),
				new Jun3dPoint(mapOrigin.x(), mapCorner.y(), 0.0d) }, Color.black);
	}

	/**
	 * Create a new peak mark at the specified point.
	 * 
	 * @param aJun3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category private-display
	 */
	protected JunOpenGL3dObject peakMarkAt_(Jun3dPoint aJun3dPoint) {
		double scale = this.mapBounds().height() / 20.0d;
		JunOpenGL3dObject peakMark = new JunOpenGL3dPolylineLoop(new Jun3dPoint[] { new Jun3dPoint(0.0d, 1.0d, 0.0d), new Jun3dPoint(-0.5d, 0.0d, 0.0d), new Jun3dPoint(0.5d, 0.0d, 0.0d) }, Color.blue);
		peakMark = peakMark.scaledBy_(scale);
		return peakMark.translatedBy_(aJun3dPoint);
	}

	/**
	 * Update the current display object.
	 * 
	 * @category private-display
	 */
	protected void updateDisplayObject() {
		this.flushDisplayObject();
		this.changed_($("object"));
	}

	/**
	 * Update the projection.
	 * 
	 * @category private-projection
	 */
	protected void updateProjection() {
		this.displayProjector().sightPoint_(this.sightPoint());
		this.displayProjector().eyePoint_(this.eyePoint());
		this.displayProjector().upVector_(this.defaultUpVector());
		this.displayProjector().viewFactor_(this.defaultViewFactor());
		this.displayProjector().zoomHeight_(this.zoomHeight());
		this.changed_($("projection"));
	}

	/**
	 * Answer the basic 3D point which corresponds to the specified 2D point.
	 * 
	 * @param aJun2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category private-terrain
	 */
	protected Jun3dPoint basic3dPointAt_(Jun2dPoint aJun2dPoint) {
		return new Jun3dPoint(aJun2dPoint.x(), aJun2dPoint.y(), this.basicHeightAt_(aJun2dPoint));
	}

	/**
	 * Answer the basic height at the specified 2D point.
	 * 
	 * @param aJun2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return double
	 * @category private-terrain
	 */
	protected double basicHeightAt_(Jun2dPoint aJun2dPoint) {
		if (peaks.isEmpty()) {
			return 0.0d;
		}

		double maxDistance = this.mapBounds().width() + this.mapBounds().height();
		double distance1 = maxDistance;
		double distance2 = maxDistance;
		double height1 = Double.NaN;
		double height2 = Double.NaN;

		Jun3dPoint[] peaks = this.peaks();
		for (int i = 0; i < peaks.length; i++) {
			Jun3dPoint peak = peaks[i];
			double distance = aJun2dPoint.distance_(new Jun2dPoint(peak.x(), peak.y()));

			if (distance < distance1) {
				distance2 = distance1;
				height2 = height1;
				distance1 = distance;
				height1 = this.heightWithPeak_distance_(peak, distance);
			} else if (distance < distance2) {
				distance2 = distance;
				height2 = this.heightWithPeak_distance_(peak, distance);
			}
		}

		return Double.isNaN(height2) ? height1 : (distance2 * height1 + distance1 * height2) / (distance1 + distance2);
	}

	/**
	 * Create the current contours.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[][]
	 * @category private-terrain
	 */
	protected Jun3dPoint[][] computeContours() {
		final StBlockClosure evaluationBlock = new StBlockClosure() {
			public Object value_value_(Object objX, Object objY) {
				double x = ((Number) objX).doubleValue();
				double y = ((Number) objY).doubleValue();
				return new Double(basicHeightAt_(new Jun2dPoint(x, y)));
			}
		};

		final ArrayList contourStream = new ArrayList();
		JunCursorAnimator.ClockCursors().showWhile_(new StBlockClosure() {
			public Object value() {
				final Jun2dPoint mapOrigin = mapBounds().origin();
				final Jun2dPoint mapCorner = mapBounds().corner();
				double topPeak = topPeak();
				double contourStep = contourStep();
				for (double z = 0.0d; z <= topPeak; z += contourStep) {
					double contourGrid = contourGrid();
					Jun2dPoint[][] polylines = JunBorderGenerator.ContourPolylinesFrom_at_xInterval_yInterval_interim_(evaluationBlock, z, new StInterval(mapOrigin.x(), mapCorner.x() + (contourGrid / 2.0d), contourGrid), new StInterval(mapOrigin.y(),
							mapCorner.y() + (contourGrid / 2.0d), contourGrid), null);
					for (int i = 0; i < polylines.length; i++) {
						Jun3dPoint[] points = new Jun3dPoint[polylines[i].length];
						for (int j = 0; j < points.length; j++) {
							Jun2dPoint point = polylines[i][j];
							points[j] = new Jun3dPoint(point.x(), point.y(), z);
						}
						contourStream.add(points);
					}
				}
				return null;
			}
		});

		return (Jun3dPoint[][]) contourStream.toArray(new Jun3dPoint[contourStream.size()][]);
	}

	/**
	 * Create my terrain field.
	 * 
	 * @return jp.co.sra.jun.terrain.support.JunTerrainField
	 * @category private-terrain
	 */
	protected JunTerrainField computeTerrainField() {
		Jun2dDelaunayProcessor delaunayProcessor = new Jun2dDelaunayProcessor(this.mapBounds().expandedBy_(this.mapBounds().extent().dividedBy_(4.0d)));
		delaunayProcessor.insertPoint_(this.basic3dPointAt_(new Jun2dPoint(this.mapBounds().origin().x(), this.mapBounds().origin().x())));
		delaunayProcessor.insertPoint_(this.basic3dPointAt_(new Jun2dPoint(this.mapBounds().corner().x(), this.mapBounds().origin().x())));
		delaunayProcessor.insertPoint_(this.basic3dPointAt_(new Jun2dPoint(this.mapBounds().corner().x(), this.mapBounds().corner().x())));
		delaunayProcessor.insertPoint_(this.basic3dPointAt_(new Jun2dPoint(this.mapBounds().origin().x(), this.mapBounds().corner().x())));

		Jun3dPoint[][] contours = this.contours();
		for (int i = 0; i < contours.length; i++) {
			for (int j = 0; j < contours[i].length; j++) {
				delaunayProcessor.insertPoint_(contours[i][j]);
			}
		}

		final ArrayList triangles = new ArrayList();
		delaunayProcessor.convexHullTrianglesDo_(new StBlockClosure() {
			public Object value_value_value_(Object p1, Object p2, Object p3) {
				triangles.add(new JunTerrainTriangle((Jun3dPoint) p1, (Jun3dPoint) p2, (Jun3dPoint) p3));
				return null;
			}
		});

		return new JunTerrainField((JunTerrainTriangle[]) triangles.toArray(new JunTerrainTriangle[triangles.size()]));
	}

	/**
	 * Flush the contours information.
	 * 
	 * @category private-terrain
	 */
	protected void flushContours() {
		if (contours != null) {
			contours = this.computeContours();
		}
		this.flushTerrainField();
		this.flushDisplayObject();
	}

	/**
	 * Flush the current terrain field.
	 * 
	 * @category private-terrain
	 */
	protected void flushTerrainField() {
		terrainField = null;
	}

	/**
	 * Answer the height calculated with the specified peak and distance.
	 * 
	 * @param aJun3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber double
	 * @return double
	 * @category private-terrain
	 */
	protected double heightWithPeak_distance_(Jun3dPoint aJun3dPoint, double aNumber) {
		return Math.max(aJun3dPoint.z() - (aNumber / 3.0d), 0.0d);
	}

	/**
	 * Answer the value of the top peak.
	 * 
	 * @return double
	 * @category private-terrain
	 */
	protected double topPeak() {
		double top = 0.0d;
		Jun3dPoint[] peaks = this.peaks();
		for (int i = 0; i < peaks.length; i++) {
			Jun3dPoint peak = peaks[i];
			if (peak.z() > top) {
				top = peak.z();
			}
		}
		return top;
	}

}
