package jp.co.sra.jun.topology.elements;

import java.awt.Color;
import java.awt.Cursor;
import java.awt.Frame;
import java.awt.TextArea;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.IOException;
import java.io.Reader;
import java.io.Writer;
import java.util.Date;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StObject;
import jp.co.sra.smalltalk.StValueHolder;

import jp.co.sra.jun.geometry.abstracts.JunCurve;
import jp.co.sra.jun.geometry.abstracts.JunGeometry;
import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox;
import jp.co.sra.jun.geometry.curves.Jun3dLine;
import jp.co.sra.jun.geometry.surfaces.Jun3dPolygon;
import jp.co.sra.jun.geometry.surfaces.JunPlane;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolygon;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dVertexesObject;
import jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext;
import jp.co.sra.jun.topology.abstracts.JunEulerOperator;
import jp.co.sra.jun.topology.abstracts.JunTopologicalElement;
import jp.co.sra.jun.topology.abstracts.JunTopologicalElementProxy;
import jp.co.sra.jun.topology.euleroperators.JunKDEV;
import jp.co.sra.jun.topology.euleroperators.JunKEL;
import jp.co.sra.jun.topology.euleroperators.JunKVE;
import jp.co.sra.jun.topology.euleroperators.JunMEKL;
import jp.co.sra.jun.topology.euleroperators.JunMEL;
import jp.co.sra.jun.topology.euleroperators.JunMEV;
import jp.co.sra.jun.topology.euleroperators.JunMEVVL;
import jp.co.sra.jun.topology.euleroperators.JunMVE;
import jp.co.sra.jun.topology.support.JunSolidModelingEngine;

/**
 * JunBody class
 * 
 *  @author    ASTI Shanghai
 *  @created   UNKNOWN
 *  @updated   1999/07/28 (by nisinaka)
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunBody.java,v 8.13 2008/02/20 06:33:01 nisinaka Exp $
 */
public class JunBody extends JunTopologicalElement {
	/** A name of the JunBody. */
	protected String name = null;

	/** A hashtable of the loops. */
	protected Hashtable loops = null;

	/** A hashtable of the edges. */
	protected Hashtable edges = null;

	/** A hashtable of the vertexes. */
	protected Hashtable vertexes = null;

	/** A hashtable of the proxies. */
	protected Hashtable proxies = null;

	/**
	 * Create a new instance of <code>JunBody</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunBody() {
		super();
	}

	/**
	 * Create a thin polygon from an array of Jun3dPoint and add it into a JunBody. 
	 * The array contains the followings.<br>
	 * 0 : <code>JunLoop</code><br>
	 * 1 : <code>JunLoop</code><br>
	 * 2 : <code>JunVertex[]</code><br>
	 * 
	 * @param anArrayOfPoint jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param aJunBody jp.co.sra.jun.topology.elements.JunBody
	 * @return java.lang.Object[]
	 * @category Private
	 */
	public static final Object[] AddThinPolygonFromPoints_into_(Jun3dPoint[] anArrayOfPoint, JunBody aJunBody) {
		Jun3dPoint[] arrayOfPoint = new Jun3dPoint[anArrayOfPoint.length];

		for (int index = 0; index < anArrayOfPoint.length; index++) {
			arrayOfPoint[index] = Jun3dPoint.Coerce_(anArrayOfPoint[index]);
		}

		JunVertex[] vertexes = new JunVertex[arrayOfPoint.length];
		JunMEVVL mevvl = JunMEVVL.Body_point_point_(aJunBody, arrayOfPoint[0], arrayOfPoint[1]);
		mevvl.doOperation();

		JunVertex firstVertex = mevvl.vertex1();
		JunVertex lastVertex = mevvl.vertex2();
		vertexes[0] = firstVertex;
		vertexes[1] = lastVertex;

		JunLoop loop = mevvl.loop();

		for (int index = 2; index < arrayOfPoint.length; index++) {
			JunMEV mev = JunMEV.Body_vertex_loop_point_(aJunBody, lastVertex, loop, arrayOfPoint[index]);
			mev.doOperation();
			lastVertex = mev.newVertex();
			vertexes[index] = lastVertex;
		}

		JunMEL mel = JunMEL.Body_loop_vertex_vertex_(aJunBody, loop, lastVertex, firstVertex);
		mel.doOperation();

		Object[] retArray = new Object[3];
		retArray[0] = mel.loop1();
		retArray[1] = mel.loop2();
		retArray[2] = vertexes;

		return retArray;
	}

	/**
	 * Create a basic skinning body from an array of <code>Jun3dPoint</code> array.
	 * 
	 * @param anArrayOfPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[][]
	 * @return jp.co.sra.jun.topology.elements.JunBody
	 * @category Private
	 */
	public static final JunBody BasicSkinning_(Jun3dPoint[][] anArrayOfPoints) {
		//	"(JunBody basicSkinning: ( #(
		//	#(#(0 0 0) #(1 0 0) #(1 1 0) #(0 1 0))
		//	#(#(0.5 0 1) #(1.5 0 1) #(1 1 1) #(0 1 1))
		//	#(#(0 0 2) #(1 0.5 2) #(1 1 2) #(0 1 2))
		//	#(#(0 0 3) #(1 0 3) #(1 1.5 3) #(0 1 3))) collect: [ :array1 | array1 collect: [ :array2 |
		//	Jun3dPoint fromArray: array2]])) show"
		JunBody body = new JunBody();
		Object[][] arrayOfLoopsAndVertexes = new Object[anArrayOfPoints.length][];

		for (int index = 0; index < anArrayOfPoints.length; index++) {
			arrayOfLoopsAndVertexes[index] = JunBody.AddThinPolygonFromPoints_into_(anArrayOfPoints[index], body);
		}

		for (int polygonIndex = 0; polygonIndex < (anArrayOfPoints.length - 1); polygonIndex++) {
			Object[] loopsAndVertexes1 = arrayOfLoopsAndVertexes[polygonIndex];
			Object[] loopsAndVertexes2 = arrayOfLoopsAndVertexes[polygonIndex + 1];
			JunLoop loop1 = (JunLoop) loopsAndVertexes1[0];
			JunVertex[] vertexes1 = (JunVertex[]) loopsAndVertexes1[2];
			JunLoop loop2 = (JunLoop) loopsAndVertexes2[1];
			JunVertex[] vertexes2 = (JunVertex[]) loopsAndVertexes2[2];
			JunMEKL mekl = JunMEKL.Body_loop_loop_vertex_vertex_(body, loop1, loop2, vertexes1[0], vertexes2[0]);
			mekl.doOperation();

			JunLoop aliveLoop = mekl.aliveLoop();

			for (int vertexIndex = 1; vertexIndex < vertexes1.length; vertexIndex++) {
				JunMEL mel;
				mel = JunMEL.Body_loop_vertex_vertex_(body, aliveLoop, vertexes2[vertexIndex], vertexes1[vertexIndex]);
				mel.doOperation();

				JunEdge newEdge = mel.newEdge();

				if (aliveLoop.numberOfEdges() == 4) {
					aliveLoop = newEdge.oppositeLoopOfLoop_(aliveLoop);
				}
			}
		}

		return body;
	}

	/**
	 * Typical bodies - cube.
	 * 
	 * @param aJun3dPoint1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aJun3dPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.topology.elements.JunBody
	 * @category Typical bodies
	 */
	public static final JunBody CubeOrigin_corner_(Jun3dPoint aJun3dPoint1, Jun3dPoint aJun3dPoint2) {
		double originX = Math.min(aJun3dPoint1.x(), aJun3dPoint2.x());
		double cornerX = Math.max(aJun3dPoint1.x(), aJun3dPoint2.x());
		double originY = Math.min(aJun3dPoint1.y(), aJun3dPoint2.y());
		double cornerY = Math.max(aJun3dPoint1.y(), aJun3dPoint2.y());
		double originZ = Math.min(aJun3dPoint1.z(), aJun3dPoint2.z());
		double cornerZ = Math.max(aJun3dPoint1.z(), aJun3dPoint2.z());
		Jun3dPoint pointOOO = new Jun3dPoint(originX, originY, originZ);
		Jun3dPoint pointOOC = new Jun3dPoint(originX, originY, cornerZ);
		Jun3dPoint pointOCO = new Jun3dPoint(originX, cornerY, originZ);
		Jun3dPoint pointOCC = new Jun3dPoint(originX, cornerY, cornerZ);
		Jun3dPoint pointCOO = new Jun3dPoint(cornerX, originY, originZ);
		Jun3dPoint pointCOC = new Jun3dPoint(cornerX, originY, cornerZ);
		Jun3dPoint pointCCO = new Jun3dPoint(cornerX, cornerY, originZ);
		Jun3dPoint pointCCC = new Jun3dPoint(cornerX, cornerY, cornerZ);
		JunBody cube = new JunBody();
		JunEulerOperator eop = JunMEVVL.Body_point_point_(cube, pointCCC, pointOCC);
		eop.doOperation();

		JunVertex vertexCCC = ((JunMEVVL) eop).newVertex1();
		JunVertex vertexOCC = ((JunMEVVL) eop).newVertex2();
		eop = JunMVE.Body_edge_point_(cube, cube.edgeForVertex_and_(vertexCCC, vertexOCC), pointOOC);
		eop.doOperation();

		JunVertex vertexOOC = ((JunMVE) eop).newVertex();
		eop = JunMVE.Body_edge_point_(cube, cube.edgeForVertex_and_(vertexCCC, vertexOOC), pointCOC);
		eop.doOperation();

		JunVertex vertexCOC = ((JunMVE) eop).newVertex();
		eop = JunMEL.Body_loop_vertex_vertex_(cube, cube.loopForVertex_and_(vertexCCC, vertexOCC), vertexCCC, vertexOCC);
		eop.doOperation();

		JunLoop branchLoop = cube.loopOnVertex_next_(vertexOCC, vertexCCC);
		eop = JunMEV.Body_vertex_loop_point_(cube, vertexCCC, branchLoop, pointCCO);
		eop.doOperation();

		JunVertex vertexCCO = ((JunMEV) eop).newVertex();
		eop = JunMEV.Body_vertex_loop_point_(cube, vertexCOC, branchLoop, pointCOO);
		eop.doOperation();

		JunVertex vertexCOO = ((JunMEV) eop).newVertex();
		eop = JunMEV.Body_vertex_loop_point_(cube, vertexOCC, branchLoop, pointOCO);
		eop.doOperation();

		JunVertex vertexOCO = ((JunMEV) eop).newVertex();
		eop = JunMEV.Body_vertex_loop_point_(cube, vertexOOC, branchLoop, pointOOO);
		eop.doOperation();

		JunVertex vertexOOO = ((JunMEV) eop).newVertex();
		eop = JunMEL.Body_loop_vertex_vertex_(cube, branchLoop, vertexCOO, vertexCCO);
		eop.doOperation();
		eop = JunMEL.Body_loop_vertex_vertex_(cube, branchLoop, vertexOOO, vertexCOO);
		eop.doOperation();
		eop = JunMEL.Body_loop_vertex_vertex_(cube, branchLoop, vertexOCO, vertexOOO);
		eop.doOperation();
		eop = JunMEL.Body_loop_vertex_vertex_(cube, branchLoop, vertexCCO, vertexOCO);
		eop.doOperation();

		return cube;
	}

	/**
	 * Answer the default center point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Defaults
	 */
	public static final Jun3dPoint DefaultCenter() {
		return Jun3dPoint.Zero();
	}

	/**
	 * Answer the default coordinate abs.
	 * 
	 * @return double
	 * @category Defaults
	 */
	public static final double DefaultCoordinateAbs() {
		return 1.0d;
	}

	/**
	 * Answer the default corner point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Defaults
	 */
	public static final Jun3dPoint DefaultCorner() {
		return new Jun3dPoint(1, 1, 1);
	}

	/**
	 * Answer the default origin point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Defaults
	 */
	public static final Jun3dPoint DefaultOrigin() {
		return new Jun3dPoint(-1.0, -1.0, -1.0);
	}

	/**
	 * Typical bodies - from JunOpenGL3dObject
	 * 
	 * @param aJunOpenGL3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @return jp.co.sra.jun.topology.elements.JunBody
	 * @category Typical bodies
	 */
	public static final JunBody FromJunOpenGL3dObject_(JunOpenGL3dObject aJunOpenGL3dObject) {
		JunSolidModelingEngine engine = new JunSolidModelingEngine();

		return JunBody.FromJunOpenGL3dObject_on_(aJunOpenGL3dObject, engine);
	}

	/**
	 * Typical bodies - from JunOpenGL3dObject and JunSolidModelingEngine
	 * 
	 * @param aJunOpenGL3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param aJunSolidModelingEngine jp.co.sra.jun.topology.support.JunSolidModelingEngine
	 * @return jp.co.sra.jun.topology.elements.JunBody
	 * @category Typical bodies
	 */
	public static final JunBody FromJunOpenGL3dObject_on_(JunOpenGL3dObject aJunOpenGL3dObject, JunSolidModelingEngine aJunSolidModelingEngine) {
		final JunSolidModelingEngine engine = aJunSolidModelingEngine;
		final JunBody body = engine.body();
		final Vector faces = new Vector();
		final StBlockClosure edgeUnificationBlock = new StBlockClosure() {
			public Object value_value_(Object anObject1, Object anObject2) {
				JunEdge edge1 = (JunEdge) anObject1;
				JunEdge edge2 = (JunEdge) anObject2;
				JunVertex vertex11 = edge1.startVertex();
				JunVertex vertex12 = edge1.endVertex();
				JunVertex vertex21;
				JunVertex vertex22;
				JunLoop loop1;
				JunLoop loop2;

				if (edge1.startVertex().point().equals(edge2.startVertex().point())) {
					vertex21 = edge2.startVertex();
					vertex22 = edge2.endVertex();
				} else {
					vertex21 = edge2.endVertex();
					vertex22 = edge2.startVertex();
				}

				if (faces.contains(edge1.leftLoop())) {
					loop1 = edge1.leftLoop();
				} else {
					loop1 = edge1.rightLoop();
				}

				if (faces.contains(edge2.leftLoop())) {
					loop2 = edge2.leftLoop();
				} else {
					loop2 = edge2.rightLoop();
				}

				JunLoop backLoop1 = edge1.oppositeLoopOfLoop_(loop1);
				JunLoop backLoop2 = edge2.oppositeLoopOfLoop_(loop2);
				JunEdge killEdge1 = null;

				if (vertex21 != vertex11) {
					if (backLoop1 == backLoop2) {
						JunMEL operator = JunMEL.Body_loop_vertex_vertex_(body, backLoop1, vertex21, vertex11);
						engine.operation_(operator);

						if (backLoop1.includesVertex_(vertex12) == false) {
							backLoop1 = operator.newLoop();

							if (backLoop1.includesVertex_(vertex12) && backLoop1.includesVertex_(vertex22)) {
								throw SmalltalkException.Error("Jun internal error");
							}
						}

						killEdge1 = operator.newEdge();
					} else {
						JunMEKL operator = JunMEKL.Body_killLoop_aliveLoop_vertex_vertex_(body, backLoop2, backLoop1, vertex21, vertex11);
						engine.operation_(operator);
						killEdge1 = operator.newEdge();
					}
				}

				JunEdge killEdge2 = null;

				if (vertex12 != vertex22) {
					JunMEL operator = JunMEL.Body_loop_vertex_vertex_(body, backLoop1, vertex12, vertex22);
					engine.operation_(operator);
					killEdge2 = operator.newEdge();
				}

				engine.operation_(JunKEL.Body_edge_aliveLoop_(body, edge2, loop2));

				if (killEdge1 != null) {
					if (vertex21.numberOfEdges() == 2) {
						engine.operation_(JunKVE.Body_vertex_killEdge_(body, vertex21, killEdge1));
					} else {
						engine.operation_(JunKDEV.Body_edge_(body, killEdge1));
					}
				}

				if (killEdge2 != null) {
					if (vertex22.numberOfEdges() == 2) {
						engine.operation_(JunKVE.Body_vertex_killEdge_(body, vertex22, killEdge2));
					} else {
						engine.operation_(JunKDEV.Body_edge_(body, killEdge2));
					}
				}

				return null;
			}
		};

		final StBlockClosure loopCreationBlock = new StBlockClosure() {
			public Object value_(Object anObject) {
				Jun3dPoint[] points = (Jun3dPoint[]) anObject;
				JunEulerOperator operator = JunMEVVL.Body_point_point_(body, points[0], points[1]);
				engine.operation_(operator);

				JunLoop loop = ((JunMEVVL) operator).loop();
				JunVertex firstVertex = ((JunMEVVL) operator).vertex1();
				JunVertex newVertex = ((JunMEVVL) operator).vertex2();

				for (int i = 2; i < points.length; i++) {
					operator = JunMEV.Body_vertex_loop_point_(body, newVertex, loop, points[i]);
					engine.operation_(operator);
					newVertex = ((JunMEV) operator).newVertex();
				}

				operator = JunMEL.Body_loop_vertex_vertex_(body, loop, newVertex, firstVertex);
				engine.operation_(operator);

				if (faces != null) {
					faces.addElement(((JunMEL) operator).newEdge().loopToVertex_(firstVertex));
				}

				return loop;
			}
		};

		final StBlockClosure twinEdgeBlock = new StBlockClosure() {
			public Object value_(Object anObject) {
				final JunEdge edge = (JunEdge) anObject;

				return body.detectEdge_ifNone_(new StBlockClosure() {
					public Object value_(Object anObject) {
						JunEdge twinEdge = (JunEdge) anObject;
						Jun3dPoint p11 = edge.startVertex().point();
						Jun3dPoint p12 = edge.endVertex().point();
						Jun3dPoint p21 = twinEdge.startVertex().point();
						Jun3dPoint p22 = twinEdge.endVertex().point();

						return new Boolean(((p11.equals(p21) && p12.equals(p22)) || (p11.equals(p22) && p12.equals(p21))) && (edge != twinEdge));
					}
				}, new StBlockClosure());
			}
		};

		final StBlockClosure addPolygonBlock = new StBlockClosure() {
			public Object value_(Object anObject) {
				JunOpenGL3dVertexesObject polygon = (JunOpenGL3dVertexesObject) anObject;
				Jun3dPolygon[] triangles = (Jun3dPolygon.Vertexes_(polygon.vertexes())).asArrayOfTrianglePolygons();

				for (int i = 0; i < triangles.length; i++) {
					JunLoop loop = (JunLoop) loopCreationBlock.value_(triangles[i].points());
					JunEdge[] edges = loop.edges();

					for (int j = 0; j < edges.length; j++) {
						JunEdge twinEdge = (JunEdge) twinEdgeBlock.value_(edges[j]);

						if (twinEdge != null) {
							edgeUnificationBlock.value_value_(twinEdge, edges[j]);
						}
					}
				}

				return null;
			}
		};

		//
		if (aJunOpenGL3dObject.isCompound()) {
			final StValueHolder size = new StValueHolder(0);
			aJunOpenGL3dObject.primitivesDo_(new StBlockClosure() {
				public Object value_(Object anObject) {
					size.value_(size._intValue() + 1);

					return null;
				}
			});
			faces.setSize(size._intValue());

			if (size._intValue() > 100) {
				// Show progress.
				aJunOpenGL3dObject.primitivesDo_(new StBlockClosure() {
					public Object value_(Object primitive) {
						addPolygonBlock.value_(primitive);

						return null;
					}
				});
			} else {
				aJunOpenGL3dObject.primitivesDo_(addPolygonBlock);
			}
		} else {
			addPolygonBlock.value_(aJunOpenGL3dObject);
		}

		return body;
	}

	/**
	 * Create a JunBody from a lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @return jp.co.sra.jun.topology.elements.JunBody
	 * @category Lisp support
	 */
	public static final JunBody FromLispList_(JunLispList aList) {
		if (aList.head() != $("JunBody")) {
			throw SmalltalkException.Error("unsupported format");
		}

		//
		JunLoop[] loopArray = null;
		JunEdge[] edgeArray = null;
		JunVertex[] vertexArray = null;
		JunLispList list = (JunLispList) aList.tail();

		while (list.nullList() == false) {
			JunLispCons tuple = (JunLispCons) list.head();

			if (tuple.head() == $("loops")) {
				int size = ((JunLispList) tuple.tail()).size();
				loopArray = new JunLoop[size];

				for (int i = 0; i < size; i++) {
					loopArray[i] = new JunLoop();
				}
			}

			if (tuple.head() == $("edges")) {
				int size = ((JunLispList) tuple.tail()).size();
				edgeArray = new JunEdge[size];

				for (int i = 0; i < size; i++) {
					edgeArray[i] = new JunEdge();
				}
			}

			if (tuple.head() == $("vertexes")) {
				int size = ((JunLispList) tuple.tail()).size();
				vertexArray = new JunVertex[size];

				for (int i = 0; i < size; i++) {
					vertexArray[i] = new JunVertex();
				}
			}

			list = (JunLispList) list.tail();
		}

		//
		list = (JunLispList) aList.tail();

		while (list.nullList() == false) {
			JunLispCons tuple = (JunLispCons) list.head();

			if (tuple.head() == $("loops")) {
				int size = ((JunLispList) tuple.tail()).size();
				Object[] listArray = ((JunLispList) tuple.tail()).asArray();

				for (int i = 0; i < size; i++) {
					loopArray[i].fromLispList_forLoops_edges_vertexes_((JunLispCons) listArray[i], loopArray, edgeArray, vertexArray);
				}
			}

			if (tuple.head() == $("edges")) {
				int size = ((JunLispList) tuple.tail()).size();
				Object[] listArray = ((JunLispList) tuple.tail()).asArray();

				for (int i = 0; i < size; i++) {
					edgeArray[i].fromLispList_forLoops_edges_vertexes_((JunLispCons) listArray[i], loopArray, edgeArray, vertexArray);
				}
			}

			if (tuple.head() == $("vertexes")) {
				int size = ((JunLispList) tuple.tail()).size();
				Object[] listArray = ((JunLispList) tuple.tail()).asArray();

				for (int i = 0; i < size; i++) {
					vertexArray[i].fromLispList_forLoops_edges_vertexes_((JunLispCons) listArray[i], loopArray, edgeArray, vertexArray);
				}
			}

			list = (JunLispList) list.tail();
		}

		//
		JunBody body = new JunBody();

		for (int i = 0; i < loopArray.length; i++) {
			body.addLoop_(loopArray[i]);
		}

		for (int i = 0; i < edgeArray.length; i++) {
			body.addEdge_(edgeArray[i]);
		}

		for (int i = 0; i < vertexArray.length; i++) {
			body.addVertex_(vertexArray[i]);
		}

		return body;
	}

	/**
	 * Read a JunBody from a file specified by a user.
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunBody
	 * @category Instance creation
	 */
	public static JunBody FromUser() {
		return null;
	}

	/**
	 * Typical bodies - globe
	 * 
	 * @param numberOfLatitudes int
	 * @param numberOfLongitudes int
	 * @param center jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param radius double
	 * @return jp.co.sra.jun.topology.elements.JunBody
	 * @category Typical bodies
	 */
	public static final JunBody GlobeLatitudes_longitudes_center_radius_(int numberOfLatitudes, int numberOfLongitudes, Jun3dPoint center, double radius) {
		final JunBody globe = new JunBody();
		JunVertex northPole = null;
		JunVertex southPole = null;
		JunVertex[][] arrayOfVertexes = new JunVertex[numberOfLongitudes][];
		JunEdge[] edges = new JunEdge[numberOfLongitudes];

		for (int longitudeIndex = 0; longitudeIndex < numberOfLongitudes; longitudeIndex++) {
			if (northPole == null) {
				JunMEVVL eop = JunMEVVL.Body_point_point_(globe, (Jun3dPoint) center.plus_(new Jun3dPoint(0, radius, 0)), (Jun3dPoint) center.minus_(new Jun3dPoint(0, radius, 0)));
				eop.doOperation();
				edges[longitudeIndex] = eop.newEdge();
				northPole = eop.newVertex1();
				southPole = eop.newVertex2();
			} else {
				JunMEL eop = JunMEL.Body_loop_vertex_vertex_(globe, edges[longitudeIndex - 1].loopFromVertex_(northPole), southPole, northPole);
				eop.doOperation();
				edges[longitudeIndex] = eop.newEdge();
			}
		}

		for (int longitudeIndex = 0; longitudeIndex < numberOfLongitudes; longitudeIndex++) {
			JunAngle longitude = JunAngle.FromRad_((Math.PI * 2 * longitudeIndex) / numberOfLongitudes);
			JunVertex[] vertexes = new JunVertex[numberOfLatitudes];

			for (int latitudeIndex = 0; latitudeIndex < numberOfLatitudes; latitudeIndex++) {
				JunAngle latitude = JunAngle.FromRad_((Math.PI * (latitudeIndex + 1)) / (numberOfLatitudes + 1));
				Jun3dPoint point = new Jun3dPoint(radius * latitude.sin() * longitude.sin(), radius * latitude.cos(), radius * latitude.sin() * longitude.cos());
				point = (Jun3dPoint) center.plus_(point);

				JunMVE eop = JunMVE.Body_edge_point_(globe, edges[longitudeIndex], point);
				eop.doOperation();
				vertexes[latitudeIndex] = eop.vertex();
			}

			arrayOfVertexes[longitudeIndex] = vertexes;
		}

		StBlockClosure latitudeBlock = new StBlockClosure() {
			public Object value_value_(Object anObject1, Object anObject2) {
				JunVertex[] longitudeVertexes1 = (JunVertex[]) anObject1;
				JunVertex[] longitudeVertexes2 = (JunVertex[]) anObject2;

				for (int i = 0; i < longitudeVertexes1.length; i++) {
					JunVertex from = longitudeVertexes1[i];
					JunVertex to = longitudeVertexes2[i];
					JunLoop commonLoop = globe.loopForVertex_and_(from, to);

					if (commonLoop == null) {
						throw SmalltalkException.Error("Euler operation error");
					}

					(JunMEL.Body_loop_vertex_vertex_(globe, commonLoop, from, to)).doOperation();
				}

				return null;
			}
		};

		for (int i = 1; i < arrayOfVertexes.length; i++) {
			latitudeBlock.value_value_(arrayOfVertexes[i - 1], arrayOfVertexes[i]);
		}

		latitudeBlock.value_value_(arrayOfVertexes[arrayOfVertexes.length - 1], arrayOfVertexes[0]);

		return globe;
	}

	/**
	 * Load a JunBody from an InputStream.
	 * 
	 * @param aReader java.io.Reader
	 * @return jp.co.sra.jun.topology.elements.JunBody
	 * @category Lisp support
	 */
	public static final JunBody LoadFrom_(Reader aReader) {
		return null;
	}

	/**
	 * Typical bodies - rotation sweep 2D points.
	 * 
	 * @param anArrayOf2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @param anInteger int
	 * @return jp.co.sra.jun.topology.elements.JunBody
	 * @category Typical bodies
	 */
	public static final JunBody RotationSweep2dPoints_divisions_(Jun2dPoint[] anArrayOf2dPoint, int anInteger) {
		int divisions = Math.abs(anInteger);
		Jun2dPoint[] points = anArrayOf2dPoint;
		int numberOfPoints = points.length;
		JunBody body = new JunBody();
		double accuracy = Jun2dPoint.ACCURACY;
		boolean closed = (points[0].distance_(points[numberOfPoints - 1]) < accuracy);

		if (closed) {
			numberOfPoints--;
		}

		JunVertex[] firstVertexes = null;
		JunLoop firstLoop = null;
		JunVertex[] lastVertexes = null;
		JunLoop lastLoop = null;

		//
		for (int angleIndex = 0; angleIndex <= (divisions - 1); angleIndex++) {
			JunAngle angle = JunAngle.FromDeg_((360 * angleIndex) / divisions);
			Jun3dPoint[] rotatedPoints = new Jun3dPoint[numberOfPoints];

			for (int i = 0; i < numberOfPoints; i++) {
				Jun2dPoint point = points[i];
				rotatedPoints[i] = new Jun3dPoint(point.x(), angle.cos() * point.y(), angle.sin() * point.y());
			}

			JunVertex[] vertexes = new JunVertex[rotatedPoints.length];
			JunMEVVL mevvl = JunMEVVL.Body_point_point_(body, rotatedPoints[0], rotatedPoints[1]);
			mevvl.doOperation();

			JunVertex firstVertex = mevvl.vertex1();
			JunVertex lastVertex = mevvl.vertex2();
			vertexes[0] = firstVertex;
			vertexes[1] = lastVertex;

			JunLoop loop = mevvl.loop();

			for (int i = 2; i < rotatedPoints.length; i++) {
				JunMEV mev = JunMEV.Body_vertex_loop_point_(body, lastVertex, loop, rotatedPoints[i]);
				mev.doOperation();
				lastVertex = mev.newVertex();
				vertexes[i] = lastVertex;
			}

			JunLoop newLoop = loop;

			if (closed) {
				JunMEL mel = JunMEL.Body_loop_vertex_vertex_(body, loop, lastVertex, firstVertex);
				mel.doOperation();
				newLoop = mel.newLoop();
			}

			if (firstLoop == null) {
				firstLoop = loop;
			}

			if (firstVertexes == null) {
				firstVertexes = vertexes;
			}

			if (lastVertexes != null) {
				JunMEKL mekl = JunMEKL.Body_killLoop_aliveLoop_vertex_vertex_(body, lastLoop, loop, lastVertexes[0], vertexes[0]);
				mekl.doOperation();

				if (mekl.vertex1().point().distance_(mekl.vertex2().point()) < accuracy) {
					JunKVE kve = JunKVE.Body_vertex_(body, mekl.vertex2());
					kve.doOperation();
					vertexes[0] = kve.vertex1();
				}

				for (int i = 1; i < (vertexes.length - 1); i++) {
					JunMEL mel = JunMEL.Body_loop_vertex_vertex_(body, loop, lastVertexes[i], vertexes[i]);
					mel.doOperation();

					if (loop.numberOfEdges() == 4) {
						loop = mel.newLoop();
					}
				}

				JunMEL mel = JunMEL.Body_loop_vertex_vertex_(body, loop, lastVertexes[lastVertexes.length - 1], vertexes[vertexes.length - 1]);
				mel.doOperation();

				if (mel.vertex1().point().distance_(mel.vertex2().point()) < accuracy) {
					JunKVE kve = JunKVE.Body_vertex_(body, mel.vertex2());
					kve.doOperation();
					vertexes[vertexes.length - 1] = kve.vertex1();
				}
			}

			lastLoop = newLoop;
			lastVertexes = vertexes;
		}

		if (firstLoop.isKilled()) {
			JunMEL mel = JunMEL.Body_loop_vertex_vertex_(body, lastLoop, lastVertexes[0], firstVertexes[0]);
			mel.doOperation();
			firstLoop = lastLoop;
		} else {
			JunMEKL mekl = JunMEKL.Body_killLoop_aliveLoop_vertex_vertex_(body, lastLoop, firstLoop, lastVertexes[0], firstVertexes[0]);
			mekl.doOperation();
		}

		for (int i = 1; i < firstVertexes.length; i++) {
			JunMEL mel = JunMEL.Body_loop_vertex_vertex_(body, firstLoop, lastVertexes[i], firstVertexes[i]);
			mel.doOperation();

			if (firstLoop.numberOfEdges() == 4) {
				firstLoop = mel.newLoop();
			}
		}

		return body;
	}

	/**
	 * Typical bodies - rotation sweep 3D curves.
	 * 
	 * @param anArrayOfJunCurve jp.co.sra.jun.geometry.abstracts.JunCurve[]
	 * @param aJun3dLine jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @return jp.co.sra.jun.topology.elements.JunBody
	 * @category Typical bodies
	 */
	public static final JunBody RotationSweep3dCurves_around_(JunCurve[] anArrayOfJunCurve, Jun3dLine aJun3dLine) {
		return null;
	}

	/**
	 * Popup a window which displays the Lisp List.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Lisp support
	 */
	public static JunLispList ShowLispList_(JunLispList aList) {
		TextArea textArea = new TextArea(5, 40);
		textArea.setEditable(false);
		textArea.append(aList.saveString());

		Frame aFrame = new Frame();
		aFrame.add("Center", textArea);
		aFrame.addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e) {
				e.getWindow().dispose();
			}
		});
		aFrame.setTitle("Lisp List");
		aFrame.pack();
		aFrame.setVisible(true);

		return aList;
	}

	/**
	 * Typical bodies - skinning 3D points.
	 * 
	 * @param anArrayOfPointArray jp.co.sra.jun.geometry.basic.Jun3dPoint[][]
	 * @return jp.co.sra.jun.topology.elements.JunBody
	 * @category Typical bodies
	 */
	public static final JunBody Skinning3dPoints_(Jun3dPoint[][] anArrayOfPointArray) {
		Vector arrayOfLoopsAndVertexes = new Vector();
		JunBody body = new JunBody();

		for (int index = 0; index < anArrayOfPointArray.length; index++) {
			Jun3dPoint[] points = anArrayOfPointArray[index];
			arrayOfLoopsAndVertexes.addElement(AddThinPolygonFromPoints_into_(points, body));
		}

		for (int polygonIndex = 0; polygonIndex < (anArrayOfPointArray.length - 1); polygonIndex++) {
			Object[] loopsAndVertexes1 = (Object[]) arrayOfLoopsAndVertexes.elementAt(polygonIndex);
			Object[] loopsAndVertexes2 = (Object[]) arrayOfLoopsAndVertexes.elementAt(polygonIndex + 1);
			JunLoop loop1 = (JunLoop) loopsAndVertexes1[0];
			JunVertex[] vertexes1 = (JunVertex[]) loopsAndVertexes1[2];
			JunLoop loop2 = (JunLoop) loopsAndVertexes2[1];
			JunVertex[] vertexes2 = (JunVertex[]) loopsAndVertexes2[2];
			JunMEKL mekl = JunMEKL.Body_loop_loop_vertex_vertex_(body, loop1, loop2, vertexes1[0], vertexes2[0]);
			mekl.doOperation();

			JunLoop aliveLoop = mekl.aliveLoop();
			JunVertex[] lessVertexes;
			JunVertex[] moreVertexes;

			if (vertexes1.length < vertexes2.length) {
				lessVertexes = vertexes1;
				moreVertexes = vertexes2;
			} else {
				lessVertexes = vertexes2;
				moreVertexes = vertexes1;
			}

			int lessSize = lessVertexes.length;
			int moreSize = moreVertexes.length;
			int next = (moreSize / 2) - lessSize;
			int lessIndex = 0;

			for (int moreIndex = 1; moreIndex < moreSize; moreIndex++) {
				if (next <= 0) {
					lessIndex++;
					next += moreSize;
				}

				JunMEL mel = JunMEL.Body_loop_vertex_vertex_(body, aliveLoop, moreVertexes[moreIndex], lessVertexes[lessIndex]);
				mel.doOperation();

				JunEdge newEdge = mel.newEdge();

				if (aliveLoop.numberOfEdges() <= 4) {
					aliveLoop = newEdge.oppositeLoopOfLoop_(aliveLoop);
				}

				if ((newEdge.oppositeLoopOfLoop_(aliveLoop).numberOfEdges()) > 4) {
					(new JunBody()).error_("MEL error");
				}

				next -= lessSize;
			}
		}

		return body;
	}

	/**
	 * Typical bodies - tetrahedron
	 * 
	 * @param anArrayOfJun3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @return jp.co.sra.jun.topology.elements.JunBody
	 * @category Typical bodies
	 */
	public static final JunBody Tetrahedron_(Jun3dPoint[] anArrayOfJun3dPoint) {
		//	"JunBody tetrahedron: (Array with: 0, 0, 0 with: 1, 0, 0 with: 0, 1, 0 with: 0, 0, 1)"
		//	"(JunBody tetrahedron: (Array with: 0, 0, 0 with: 1, 0, 0 with: 0, 1, 0 with: 0, 0, 1)) show"
		Jun3dPoint point1 = anArrayOfJun3dPoint[0];
		Jun3dPoint point2 = anArrayOfJun3dPoint[1];
		Jun3dPoint point3 = anArrayOfJun3dPoint[2];
		Jun3dPoint point4 = anArrayOfJun3dPoint[3];
		Jun3dPoint faceVector = ((Jun3dPoint) point2.minus_(point1)).product_((Jun3dPoint) point3.minus_(point2));

		if (faceVector.innerProduct_(point4.minus_(point1)) < 0) {
			Jun3dPoint tmpPoint = point1;
			point1 = point2;
			point2 = tmpPoint;
		}

		JunBody tetra = new JunBody();
		JunEulerOperator eop = JunMEVVL.Body_point_point_(tetra, point1, point2);
		eop.doOperation();

		JunVertex v1 = ((JunMEVVL) eop).vertex1();
		JunVertex v2 = ((JunMEVVL) eop).vertex2();
		eop = JunMEV.Body_vertex_loop_point_(tetra, v2, tetra.loopForVertex_(v2), point3);
		eop.doOperation();

		JunVertex v3 = ((JunMEV) eop).vertex1();
		eop = JunMEV.Body_vertex_loop_point_(tetra, v3, tetra.loopForVertex_(v3), point4);
		eop.doOperation();

		JunVertex v4 = ((JunMEV) eop).vertex1();
		eop = JunMEL.Body_loop_vertex_vertex_(tetra, tetra.loopOnVertex_next_(v3, v4), v4, v2);
		eop.doOperation();
		eop = JunMEL.Body_loop_vertex_vertex_(tetra, tetra.loopForVertex_and_(v2, v1), v1, v4);
		eop.doOperation();
		eop = JunMEL.Body_loop_vertex_vertex_(tetra, tetra.loopForVertex_and_(v1, v3), v3, v1);
		eop.doOperation();

		return tetra;
	}

	/**
	 * Typical unit bodies -  unit corn array.
	 * The array contains the followings.<br>
	 * 0 : JunBody<br>
	 * 1 : JunLoop<br>
	 * 2 : JunVertex[]<br>
	 * 3 : JunVertex
	 * 
	 * @param divisions int
	 * @return java.lang.Object[]
	 * @category Typical unit bodies
	 */
	public static final Object[] UnitCornArrayDivisions_(final int divisions) {
		//	"(JunBody unitCornArrayDivisions: 50) first show"
		if (divisions < 3) {
			return null;
		}

		final Jun3dPoint origin = JunBody.DefaultOrigin();
		final Jun3dPoint corner = JunBody.DefaultCorner();
		StBlockClosure pointBlock = new StBlockClosure() {
			public Object value_value_(Object index, Object z) {
				JunAngle angle = JunAngle.FromDeg_(((((Integer) index).intValue() - 1) * 360) / divisions);

				return new Jun3dPoint(angle.cos() * ((corner.x() - origin.x()) / 2), angle.sin() * ((corner.y() - origin.y()) / 2), ((Double) z).doubleValue());
			}
		};

		JunVertex[] bottomVertexes = new JunVertex[divisions];
		JunBody corn = new JunBody();
		JunEulerOperator eop = JunMEVVL.Body_point_point_(corn, (Jun3dPoint) pointBlock.value_value_(new Integer(1), new Double(origin.z())), (Jun3dPoint) pointBlock.value_value_(new Integer(2), new Double(origin.z())));
		eop.doOperation();
		bottomVertexes[0] = ((JunMEVVL) eop).vertex1();
		bottomVertexes[1] = ((JunMEVVL) eop).vertex2();

		JunLoop bottomLoop = ((JunMEVVL) eop).loop();

		for (int index = 2; index < divisions; index++) {
			eop = JunMEV.Body_vertex_loop_point_(corn, bottomVertexes[index - 1], bottomLoop, (Jun3dPoint) pointBlock.value_value_(new Integer(index), new Double(origin.z())));
			eop.doOperation();
			bottomVertexes[index] = ((JunMEV) eop).newVertex();
		}

		eop = JunMEL.Body_loop_vertex_vertex_(corn, bottomLoop, bottomVertexes[divisions - 1], bottomVertexes[0]);
		eop.doOperation();

		JunLoop sideLoop = ((JunMEL) eop).newLoop();

		eop = JunMEV.Body_vertex_loop_point_(corn, bottomVertexes[0], sideLoop, new Jun3dPoint((origin.x() + corner.x()) / 2, (origin.y() + corner.y()) / 2, corner.z()));
		eop.doOperation();

		JunVertex topVertex = ((JunMEV) eop).newVertex();

		for (int index = 1; index < divisions; index++) {
			eop = JunMEL.Body_loop_vertex_vertex_(corn, sideLoop, bottomVertexes[index], topVertex);
			eop.doOperation();
		}

		Object[] unitCornArray = new Object[5];
		unitCornArray[0] = corn;
		unitCornArray[1] = bottomLoop;
		unitCornArray[2] = bottomVertexes;
		unitCornArray[3] = topVertex;

		return unitCornArray;
	}

	/**
	 * Typical unit bodies - unit cube
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunBody
	 * @category Typical unit bodies
	 */
	public static final JunBody UnitCube() {
		//	"(JunBody unitCube) show"
		JunEulerOperator eop = null;
		JunLoop branchLoop;
		JunVertex vertexCCC = null;
		JunVertex vertexOCC = null;
		JunVertex vertexOOC;
		JunVertex vertexCOC;
		JunVertex vertexCCO;
		JunVertex vertexCOO;
		JunVertex vertexOCO;
		JunVertex vertexOOO;
		Jun3dPoint origin;
		Jun3dPoint corner;
		Jun3dPoint pointOOO = null;
		Jun3dPoint pointOOC = null;
		Jun3dPoint pointOCO = null;
		Jun3dPoint pointOCC = null;
		Jun3dPoint pointCOO = null;
		Jun3dPoint pointCOC = null;
		Jun3dPoint pointCCO = null;
		Jun3dPoint pointCCC = null;
		JunBody cube = new JunBody();
		origin = JunBody.DefaultOrigin();
		corner = JunBody.DefaultCorner();
		pointOOO = origin;
		pointOOC = new Jun3dPoint(origin.x(), origin.y(), corner.z());
		pointOCO = new Jun3dPoint(origin.x(), corner.y(), origin.z());
		pointOCC = new Jun3dPoint(origin.x(), corner.y(), corner.z());
		pointCOO = new Jun3dPoint(corner.x(), origin.y(), origin.z());
		pointCOC = new Jun3dPoint(corner.x(), origin.y(), corner.z());
		pointCCO = new Jun3dPoint(corner.x(), corner.y(), origin.z());
		pointCCC = new Jun3dPoint(corner.x(), corner.y(), corner.z());
		eop = JunMEVVL.Body_point_point_(cube, pointCCC, pointOCC);
		eop.doOperation();
		vertexCCC = ((JunMEVVL) eop).newVertex1();
		vertexOCC = ((JunMEVVL) eop).newVertex2();
		eop = JunMVE.Body_edge_point_(cube, cube.edgeForVertex_and_(vertexCCC, vertexOCC), pointOOC);
		eop.doOperation();
		vertexOOC = ((JunMVE) eop).newVertex();
		eop = JunMVE.Body_edge_point_(cube, cube.edgeForVertex_and_(vertexCCC, vertexOOC), pointCOC);
		eop.doOperation();
		vertexCOC = ((JunMVE) eop).newVertex();
		eop = JunMEL.Body_loop_vertex_vertex_(cube, cube.loopForVertex_and_(vertexCCC, vertexOCC), vertexCCC, vertexOCC);
		eop.doOperation();
		branchLoop = cube.loopOnVertex_next_(vertexOCC, vertexCCC);
		eop = JunMEV.Body_vertex_loop_point_(cube, vertexCCC, branchLoop, pointCCO);
		eop.doOperation();
		vertexCCO = ((JunMEV) eop).newVertex();
		eop = JunMEV.Body_vertex_loop_point_(cube, vertexCOC, branchLoop, pointCOO);
		eop.doOperation();
		vertexCOO = ((JunMEV) eop).newVertex();
		eop = JunMEV.Body_vertex_loop_point_(cube, vertexOCC, branchLoop, pointOCO);
		eop.doOperation();
		vertexOCO = ((JunMEV) eop).newVertex();
		eop = JunMEV.Body_vertex_loop_point_(cube, vertexOOC, branchLoop, pointOOO);
		eop.doOperation();
		vertexOOO = ((JunMEV) eop).newVertex();
		eop = JunMEL.Body_loop_vertex_vertex_(cube, branchLoop, vertexCOO, vertexCCO);
		eop.doOperation();
		eop = JunMEL.Body_loop_vertex_vertex_(cube, branchLoop, vertexOOO, vertexCOO);
		eop.doOperation();
		eop = JunMEL.Body_loop_vertex_vertex_(cube, branchLoop, vertexOCO, vertexOOO);
		eop.doOperation();
		eop = JunMEL.Body_loop_vertex_vertex_(cube, branchLoop, vertexCCO, vertexOCO);
		eop.doOperation();

		return cube;
	}

	/**
	 * Typical unit bodies - unit cylinder array.
	 * The array contains the followings.<br>
	 * 0 : JunBody<br>
	 * 1 : JunLoop<br>
	 * 2 : JunLoop<br>
	 * 3 : JunVertex[]<br>
	 * 4 : JunVertex[]
	 * 
	 * @param divisions int
	 * @return java.lang.Object[]
	 * @category Typical unit bodies
	 */
	public static final Object[] UnitCylinderArrayDivisions_(final int divisions) {
		//	"(JunBody unitCylinderArrayDivisions: 5) first show"
		if (divisions < 3) {
			return null;
		}

		final Jun3dPoint origin = JunBody.DefaultOrigin();
		final Jun3dPoint corner = JunBody.DefaultCorner();
		StBlockClosure pointBlock = new StBlockClosure() {
			public Object value_value_(Object index, Object z) {
				JunAngle angle = JunAngle.FromDeg_(((((Integer) index).intValue() - 1) * 360) / divisions);

				return new Jun3dPoint(angle.cos() * ((corner.x() - origin.x()) / 2), angle.sin() * ((corner.y() - origin.y()) / 2), ((Double) z).doubleValue());
			}
		};

		JunVertex[] coverVertexes = new JunVertex[divisions];
		JunVertex[] bottomVertexes = new JunVertex[divisions];
		JunBody cylinder = new JunBody();
		JunEulerOperator eop = JunMEVVL.Body_point_point_(cylinder, (Jun3dPoint) pointBlock.value_value_(new Integer(1), new Double(origin.z())), (Jun3dPoint) pointBlock.value_value_(new Integer(2), new Double(origin.z())));
		eop.doOperation();
		bottomVertexes[0] = ((JunMEVVL) eop).vertex1();
		bottomVertexes[1] = ((JunMEVVL) eop).vertex2();

		JunLoop bottomLoop = ((JunMEVVL) eop).loop();

		for (int index = 2; index < divisions; index++) {
			eop = JunMEV.Body_vertex_loop_point_(cylinder, bottomVertexes[index - 1], bottomLoop, (Jun3dPoint) pointBlock.value_value_(new Integer(index), new Double(origin.z())));
			eop.doOperation();
			bottomVertexes[index] = ((JunMEV) eop).newVertex();
		}

		eop = JunMEL.Body_loop_vertex_vertex_(cylinder, bottomLoop, bottomVertexes[divisions - 1], bottomVertexes[0]);
		eop.doOperation();

		JunLoop sideLoop = ((JunMEL) eop).newLoop();

		for (int index = 0; index < divisions; index++) {
			eop = JunMEV.Body_vertex_loop_point_(cylinder, bottomVertexes[index], sideLoop, (Jun3dPoint) pointBlock.value_value_(new Integer(index), new Double(corner.z())));
			eop.doOperation();
			coverVertexes[index] = ((JunMEV) eop).newVertex();
		}

		for (int index = 0; index < divisions; index++) {
			eop = JunMEL.Body_loop_vertex_vertex_(cylinder, sideLoop, coverVertexes[(index + 1) % divisions], coverVertexes[index]);
			eop.doOperation();
		}

		JunLoop coverLoop = ((JunMEL) eop).originalLoop();
		Object[] unitCylinderArray = new Object[5];
		unitCylinderArray[0] = cylinder;
		unitCylinderArray[1] = coverLoop;
		unitCylinderArray[2] = bottomLoop;
		unitCylinderArray[3] = coverVertexes;
		unitCylinderArray[4] = bottomVertexes;

		return unitCylinderArray;
	}

	/**
	 * Typical unit bodies - unit prism array 2.
	 * The array contains the followings.<br>
	 * 0 : JunBody<br>
	 * 1 : JunLoop<br>
	 * 2 : JunLoop<br>
	 * 3 : JunVertex[]<br>
	 * 4 : JunVertex[]
	 * 
	 * @param divisions int
	 * @return java.lang.Object[]
	 * @category Typical unit bodies
	 */
	public static final Object[] UnitPrismArray2Divisions_(final int divisions) {
		//	"(JunBody unitPrismArray2Divisions: 5) first show"
		if (divisions < 3) {
			return null;
		}

		final Jun3dPoint origin = JunBody.DefaultOrigin();
		final Jun3dPoint corner = JunBody.DefaultCorner();
		StBlockClosure pointBlock = new StBlockClosure() {
			public Object value_value_(Object index, Object z) {
				JunAngle angle = JunAngle.FromDeg_(((((Integer) index).intValue() - 1) * 360) / divisions);

				return new Jun3dPoint(angle.cos() * ((corner.x() - origin.x()) / 2), angle.sin() * ((corner.y() - origin.y()) / 2), ((Double) z).doubleValue());
			}
		};

		JunVertex[] coverVertexes = new JunVertex[divisions];
		JunVertex[] bottomVertexes = new JunVertex[divisions];
		JunBody prism = new JunBody();
		JunEulerOperator eop = JunMEVVL.Body_point_point_(prism, (Jun3dPoint) pointBlock.value_value_(new Integer(1), new Double(origin.z())), (Jun3dPoint) pointBlock.value_value_(new Integer(2), new Double(origin.z())));
		eop.doOperation();
		bottomVertexes[0] = ((JunMEVVL) eop).vertex1();
		bottomVertexes[1] = ((JunMEVVL) eop).vertex2();

		JunLoop bottomLoop = ((JunMEVVL) eop).loop();

		for (int index = 2; index < divisions; index++) {
			eop = JunMEV.Body_vertex_loop_point_(prism, bottomVertexes[index - 1], bottomLoop, (Jun3dPoint) pointBlock.value_value_(new Integer(index), new Double(origin.z())));
			eop.doOperation();
			bottomVertexes[index] = ((JunMEV) eop).newVertex();
		}

		eop = JunMEL.Body_loop_vertex_vertex_(prism, bottomLoop, bottomVertexes[divisions - 1], bottomVertexes[0]);
		eop.doOperation();

		JunLoop sideLoop = ((JunMEL) eop).newLoop();

		for (int index = 0; index < divisions; index++) {
			eop = JunMEV.Body_vertex_loop_point_(prism, bottomVertexes[index], sideLoop, (Jun3dPoint) pointBlock.value_value_(new Integer(index), new Double(corner.z())));
			eop.doOperation();
			coverVertexes[index] = ((JunMEV) eop).newVertex();
		}

		for (int index = 0; index < divisions; index++) {
			eop = JunMEL.Body_loop_vertex_vertex_(prism, sideLoop, coverVertexes[(index + 1) % divisions], coverVertexes[index]);
			eop.doOperation();
		}

		JunLoop coverLoop = ((JunMEL) eop).originalLoop();
		Object[] unitPrismArray = new Object[5];
		unitPrismArray[0] = prism;
		unitPrismArray[1] = coverLoop;
		unitPrismArray[2] = bottomLoop;
		unitPrismArray[3] = coverVertexes;
		unitPrismArray[4] = bottomVertexes;

		return unitPrismArray;
	}

	/**
	 * Typical unit bodies - unit prism array.
	 * The array contains the followings.<br>
	 * 0 : JunBody<br>
	 * 1 : JunLoop<br>
	 * 2 : JunLoop<br>
	 * 3 : JunVertex[]<br>
	 * 4 : JunVertex[]
	 * 
	 * @param divisions int
	 * @return java.lang.Object[]
	 * @category Typical unit bodies
	 */
	public static final Object[] UnitPrismArrayDivisions_(final int divisions) {
		//	"(JunBody unitPrismArrayDivisions: 3) first show"
		if (divisions < 3) {
			return null;
		}

		final Jun3dPoint origin = JunBody.DefaultOrigin();
		final Jun3dPoint corner = JunBody.DefaultCorner();
		JunAngle singleAngle = JunAngle.FromDeg_(360 / divisions);
		final double scale = Math.sqrt((double) 1 / ((1 - singleAngle.cos()) * (1 - singleAngle.cos()) + (singleAngle.sin() * singleAngle.sin())));
		StBlockClosure pointBlock = new StBlockClosure() {
			public Object value_value_(Object index, Object z) {
				JunAngle angle;
				angle = JunAngle.FromDeg_(((((Integer) index).intValue() - 1) * 360) / divisions);

				return new Jun3dPoint(angle.cos() * scale * ((corner.x() - origin.x()) / 2), angle.sin() * scale * ((corner.y() - origin.y()) / 2), ((Double) z).doubleValue());
			}
		};

		JunVertex[] coverVertexes = new JunVertex[divisions];
		JunVertex[] bottomVertexes = new JunVertex[divisions];
		JunBody prism = new JunBody();
		JunEulerOperator eop = JunMEVVL.Body_point_point_(prism, (Jun3dPoint) pointBlock.value_value_(new Integer(1), new Double(origin.z())), (Jun3dPoint) pointBlock.value_value_(new Integer(2), new Double(origin.z())));
		eop.doOperation();
		bottomVertexes[0] = ((JunMEVVL) eop).vertex1();
		bottomVertexes[1] = ((JunMEVVL) eop).vertex2();

		JunLoop bottomLoop = ((JunMEVVL) eop).loop();

		for (int index = 2; index < divisions; index++) {
			eop = JunMEV.Body_vertex_loop_point_(prism, bottomVertexes[index - 1], bottomLoop, (Jun3dPoint) pointBlock.value_value_(new Integer(index), new Double(origin.z())));
			eop.doOperation();
			bottomVertexes[index] = ((JunMEV) eop).newVertex();
		}

		eop = JunMEL.Body_loop_vertex_vertex_(prism, bottomLoop, bottomVertexes[divisions - 1], bottomVertexes[0]);
		eop.doOperation();

		JunLoop sideLoop = ((JunMEL) eop).newLoop();

		for (int index = 0; index < divisions; index++) {
			eop = JunMEV.Body_vertex_loop_point_(prism, bottomVertexes[index], sideLoop, (Jun3dPoint) pointBlock.value_value_(new Integer(index), new Double(corner.z())));
			eop.doOperation();
			coverVertexes[index] = ((JunMEV) eop).newVertex();
		}

		for (int index = 0; index < divisions; index++) {
			eop = JunMEL.Body_loop_vertex_vertex_(prism, sideLoop, coverVertexes[(index + 1) % divisions], coverVertexes[index]);
			eop.doOperation();
		}

		JunLoop coverLoop = ((JunMEL) eop).originalLoop();
		Object[] unitPrismArray = new Object[5];
		unitPrismArray[0] = prism;
		unitPrismArray[1] = coverLoop;
		unitPrismArray[2] = bottomLoop;
		unitPrismArray[3] = coverVertexes;
		unitPrismArray[4] = bottomVertexes;

		return unitPrismArray;
	}

	/**
	 * Typical unit bodies -  unit pyramid array 2.
	 * The array contains the followings.<br>
	 * 0 : JunBody<br>
	 * 1 : JunLoop<br>
	 * 2 : JunLoop<br>
	 * 3 : JunVertex[]<br>
	 * 4 : JunVertex[]
	 * 
	 * @param divisions int
	 * @return java.lang.Object[]
	 * @category Typical unit bodies
	 */
	public static final Object[] UnitPyramidArray2Divisions_(final int divisions) {
		//	"(JunBody unitPyramidArray2Divisions: 5) first show"
		if (divisions < 3) {
			return null;
		}

		final Jun3dPoint origin = JunBody.DefaultOrigin();
		final Jun3dPoint corner = JunBody.DefaultCorner();
		StBlockClosure pointBlock = new StBlockClosure() {
			public Object value_value_(Object index, Object z) {
				JunAngle angle;
				angle = JunAngle.FromDeg_(((((Integer) index).intValue() - 1) * 360) / divisions);

				return new Jun3dPoint(angle.cos() * ((corner.x() - origin.x()) / 2), angle.sin() * ((corner.y() - origin.y()) / 2), ((Double) z).doubleValue());
			}
		};

		JunVertex[] bottomVertexes = new JunVertex[divisions];
		JunBody pyramid = new JunBody();
		JunEulerOperator eop = JunMEVVL.Body_point_point_(pyramid, (Jun3dPoint) pointBlock.value_value_(new Integer(1), new Double(origin.z())), (Jun3dPoint) pointBlock.value_value_(new Integer(2), new Double(origin.z())));
		eop.doOperation();
		bottomVertexes[0] = ((JunMEVVL) eop).vertex1();
		bottomVertexes[1] = ((JunMEVVL) eop).vertex2();

		JunLoop bottomLoop = ((JunMEVVL) eop).loop();

		for (int index = 2; index < divisions; index++) {
			eop = JunMEV.Body_vertex_loop_point_(pyramid, bottomVertexes[index - 1], bottomLoop, (Jun3dPoint) pointBlock.value_value_(new Integer(index), new Double(origin.z())));
			eop.doOperation();
			bottomVertexes[index] = ((JunMEV) eop).newVertex();
		}

		eop = JunMEL.Body_loop_vertex_vertex_(pyramid, bottomLoop, bottomVertexes[divisions - 1], bottomVertexes[0]);
		eop.doOperation();

		JunLoop sideLoop = ((JunMEL) eop).newLoop();

		eop = JunMEV.Body_vertex_loop_point_(pyramid, bottomVertexes[0], sideLoop, new Jun3dPoint((origin.x() + corner.x()) / 2, (origin.y() + corner.y()) / 2, corner.z()));
		eop.doOperation();

		JunVertex topVertex = ((JunMEV) eop).newVertex();

		for (int index = 1; index < divisions; index++) {
			eop = JunMEL.Body_loop_vertex_vertex_(pyramid, sideLoop, bottomVertexes[index], topVertex);
			eop.doOperation();
		}

		Object[] unitPyramidArray = new Object[5];
		unitPyramidArray[0] = pyramid;
		unitPyramidArray[1] = bottomLoop;
		unitPyramidArray[2] = bottomVertexes;
		unitPyramidArray[3] = topVertex;

		return unitPyramidArray;
	}

	/**
	 * Typical unit bodies -  unit pyramid array.
	 * The array contains the followings.<br>
	 * 0 : JunBody<br>
	 * 1 : JunLoop<br>
	 * 2 : JunLoop<br>
	 * 3 : JunVertex[]<br>
	 * 4 : JunVertex[]
	 * 
	 * @param divisions int
	 * @return java.lang.Object[]
	 * @category Typical unit bodies
	 */
	public static final Object[] UnitPyramidArrayDivisions_(final int divisions) {
		// ((JunBody) JunBody.UnitPyramidArrayDivisions_(5)[0]).show();
		if (divisions < 3) {
			return null;
		}

		final Jun3dPoint origin = JunBody.DefaultOrigin();
		final Jun3dPoint corner = JunBody.DefaultCorner();
		JunAngle singleAngle = JunAngle.FromDeg_(360 / divisions);
		final double scale = Math.sqrt(1.0d / ((1.0d - singleAngle.cos()) * (1.0d - singleAngle.cos()) + (singleAngle.sin() * singleAngle.sin())));
		StBlockClosure pointBlock = new StBlockClosure() {
			public Object value_value_(Object index, Object z) {
				JunAngle angle = JunAngle.FromDeg_(((((Number) index).intValue() - 1) * 360) / divisions);
				return new Jun3dPoint(angle.cos() * scale * ((corner.x() - origin.x()) / 2), angle.sin() * scale * ((corner.y() - origin.y()) / 2), ((Number) z).doubleValue());
			}
		};

		JunVertex[] bottomVertexes = new JunVertex[divisions];
		JunBody pyramid = new JunBody();
		JunEulerOperator eop = JunMEVVL.Body_point_point_(pyramid, (Jun3dPoint) pointBlock.value_value_(new Integer(1), new Double(origin.z())), (Jun3dPoint) pointBlock.value_value_(new Integer(2), new Double(origin.z())));
		eop.doOperation();
		bottomVertexes[0] = ((JunMEVVL) eop).vertex1();
		bottomVertexes[1] = ((JunMEVVL) eop).vertex2();

		JunLoop bottomLoop = ((JunMEVVL) eop).loop();

		for (int index = 2; index < divisions; index++) {
			eop = JunMEV.Body_vertex_loop_point_(pyramid, bottomVertexes[index - 1], bottomLoop, (Jun3dPoint) pointBlock.value_value_(new Integer(index), new Double(origin.z())));
			eop.doOperation();
			bottomVertexes[index] = ((JunMEV) eop).newVertex();
		}

		eop = JunMEL.Body_loop_vertex_vertex_(pyramid, bottomLoop, bottomVertexes[divisions - 1], bottomVertexes[0]);
		eop.doOperation();

		JunLoop sideLoop = ((JunMEL) eop).newLoop();

		eop = JunMEV.Body_vertex_loop_point_(pyramid, bottomVertexes[0], sideLoop, new Jun3dPoint((origin.x() + corner.x()) / 2, (origin.y() + corner.y()) / 2, corner.z()));
		eop.doOperation();

		JunVertex topVertex = ((JunMEV) eop).newVertex();

		for (int index = 0; index < divisions; index++) {
			eop = JunMEL.Body_loop_vertex_vertex_(pyramid, sideLoop, bottomVertexes[index], topVertex);
			eop.doOperation();
		}

		Object[] unitPyramidArray = new Object[5];
		unitPyramidArray[0] = pyramid;
		unitPyramidArray[1] = bottomLoop;
		unitPyramidArray[2] = bottomVertexes;
		unitPyramidArray[3] = topVertex;

		return unitPyramidArray;
	}

	/**
	 * Add the edge to the receiver.
	 * 
	 * @param anEdge jp.co.sra.jun.topology.elements.JunEdge
	 * @return jp.co.sra.jun.topology.elements.JunEdgeProxy
	 * @category adding
	 */
	public JunEdgeProxy addEdge_(JunEdge anEdge) {
		return this.addEdge_as_(anEdge, null);
	}

	/**
	 * Add the edge to the receiver with the edge proxy.
	 * 
	 * @param anEdge jp.co.sra.jun.topology.elements.JunEdge
	 * @param anEdgeProxy jp.co.sra.jun.topology.elements.JunEdgeProxy
	 * @return jp.co.sra.jun.topology.elements.JunEdgeProxy
	 * @category adding
	 */
	public JunEdgeProxy addEdge_as_(JunEdge anEdge, JunEdgeProxy anEdgeProxy) {
		JunEdgeProxy proxy;

		if (anEdgeProxy == null) {
			proxy = (JunEdgeProxy) proxies.get(anEdge);

			if (proxy == null) {
				proxy = new JunEdgeProxy();
			}
		} else {
			proxy = anEdgeProxy;
		}

		edges.put(proxy, anEdge);
		proxies.put(anEdge, proxy);

		return proxy;
	}

	/**
	 * Add the loop to the receiver.
	 * 
	 * @param aLoop jp.co.sra.jun.topology.elements.JunLoop
	 * @return jp.co.sra.jun.topology.elements.JunLoopProxy
	 * @category adding
	 */
	public JunLoopProxy addLoop_(JunLoop aLoop) {
		return this.addLoop_as_(aLoop, null);
	}

	/**
	 * Add the loop to the receiver with the loop proxy.
	 * 
	 * @param aLoop jp.co.sra.jun.topology.elements.JunLoop
	 * @param aLoopProxy jp.co.sra.jun.topology.elements.JunLoopProxy
	 * @return jp.co.sra.jun.topology.elements.JunLoopProxy
	 * @category adding
	 */
	public JunLoopProxy addLoop_as_(JunLoop aLoop, JunLoopProxy aLoopProxy) {
		JunLoopProxy proxy;

		if (aLoopProxy == null) {
			proxy = (JunLoopProxy) proxies.get(aLoop);

			if (proxy == null) {
				proxy = new JunLoopProxy();
			}
		} else {
			proxy = aLoopProxy;
		}

		loops.put(proxy, aLoop);
		proxies.put(aLoop, proxy);

		return proxy;
	}

	/**
	 * Add the vertex to the receiver.
	 * 
	 * @param aVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @return jp.co.sra.jun.topology.elements.JunVertexProxy
	 * @category adding
	 */
	public JunVertexProxy addVertex_(JunVertex aVertex) {
		return this.addVertex_as_(aVertex, null);
	}

	/**
	 * Add the vertex to the receiver with the vertex proxy.
	 * 
	 * @param aVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @param aVertexProxy jp.co.sra.jun.topology.elements.JunVertexProxy
	 * @return jp.co.sra.jun.topology.elements.JunVertexProxy
	 * @category adding
	 */
	public JunVertexProxy addVertex_as_(JunVertex aVertex, JunVertexProxy aVertexProxy) {
		JunVertexProxy proxy;

		if (aVertexProxy == null) {
			proxy = (JunVertexProxy) proxies.get(aVertex);

			if (proxy == null) {
				proxy = new JunVertexProxy();
			}
		} else {
			proxy = aVertexProxy;
		}

		vertexes.put(proxy, aVertex);
		proxies.put(aVertex, proxy);

		return proxy;
	}

	/**
	 * Convert the receiver as a <code>JunOpenGL3dObject</code>.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.topology.abstracts.JunTopologicalElement#asJunOpenGL3dObject()
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		final JunOpenGL3dCompoundObject surface = new JunOpenGL3dCompoundObject();
		this.loopsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunLoop loop = (JunLoop) anObject;

				if (loop.isChildLoop() == false) {
					JunOpenGL3dObject s = loop.asJunOpenGL3dObject();

					if (s != null) {
						surface.add_(s);
					}
				}

				return null;
			}
		});
		surface.paint_(Color.blue);

		return surface;
	}

	/**
	 * Convert the receiver as a proxy in aBody.
	 * 
	 * @param aBody jp.co.sra.jun.topology.elements.JunBody
	 * @param aTopologicalElementProxy jp.co.sra.jun.topology.abstracts.JunTopologicalElementProxy
	 * @return jp.co.sra.jun.topology.abstracts.JunTopologicalElementProxy
	 * @see jp.co.sra.jun.topology.abstracts.JunTopologicalElementOrProxy#asProxyIn_advise_(jp.co.sra.jun.topology.elements.JunBody, jp.co.sra.jun.topology.abstracts.JunTopologicalElementProxy)
	 * @category converting
	 */
	public JunTopologicalElementProxy asProxyIn_advise_(JunBody aBody, JunTopologicalElementProxy aTopologicalElementProxy) {
		throw SmalltalkException.ShouldNotImplement();
	}

	/**
	 * Convert the receiver as a wireframed <code>JunOpenGL3dObject</code>.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.topology.abstracts.JunTopologicalElement#asWireframedOpenGL3dObject()
	 * @category converting
	 */
	public JunOpenGL3dObject asWireframedOpenGL3dObject() {
		final JunOpenGL3dCompoundObject wireframe = new JunOpenGL3dCompoundObject();
		this.loopsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunLoop loop = (JunLoop) anObject;
				wireframe.add_(loop.asWireframedOpenGL3dObject());

				return null;
			}
		});
		wireframe.paint_(Color.blue);

		return wireframe;
	}

	/**
	 * Answer the Jun3dBoundingBox of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dBoundingBox
	 * @category bounds accessing
	 */
	public Jun3dBoundingBox boundingBox() {
		double minX = Double.POSITIVE_INFINITY;
		double minY = Double.POSITIVE_INFINITY;
		double minZ = Double.POSITIVE_INFINITY;
		double maxX = Double.NEGATIVE_INFINITY;
		double maxY = Double.NEGATIVE_INFINITY;
		double maxZ = Double.NEGATIVE_INFINITY;
		JunVertex[] anArrayOfVertex = this.vertexes();

		for (int i = 0; i < anArrayOfVertex.length; i++) {
			Jun3dPoint p = anArrayOfVertex[i].point();
			minX = Math.min(minX, p.x());
			minY = Math.min(minY, p.y());
			minZ = Math.min(minZ, p.z());
			maxX = Math.max(maxX, p.x());
			maxY = Math.max(maxY, p.y());
			maxZ = Math.max(maxZ, p.z());
		}

		return Jun3dBoundingBox.Origin_corner_(new Jun3dPoint(minX, minY, minZ), new Jun3dPoint(maxX, maxY, maxZ));
	}

	/**
	 * Answer the center point of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category bounds accessing
	 */
	public Jun3dPoint center() {
		return (Jun3dPoint) this.boundingBox().center();
	}

	/**
	 * Answer true if the receiver contains the Jun3dPoint, otherwise false.
	 * 
	 * @param aJun3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return boolean
	 * @category testing
	 */
	public boolean containsPoint_(final Jun3dPoint aJun3dPoint) {
		Jun3dBoundingBox bounds = this.boundingBox();

		if (!bounds.containsOrTouchesPoint_(aJun3dPoint)) {
			return false;
		}

		final double accuracy = JunGeometry.ACCURACY;
		JunEdge edge = (JunEdge) edges.elements().nextElement();
		Jun3dPoint referencePoint = (Jun3dPoint) edge.startVertex().point().plus_(edge.endVertex().point()).dividedBy_(2.0d);
		final Jun3dLine theLine = new Jun3dLine(aJun3dPoint, referencePoint);
		final Jun3dPoint vector = referencePoint.minus_(aJun3dPoint);
		final StValueHolder referenceSurface = new StValueHolder(null);
		final StValueHolder minDistance = new StValueHolder(Double.MAX_VALUE);
		this.loopsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunLoop loop = (JunLoop) anObject;
				JunPlane surface = (JunPlane) loop.surface();

				if (surface != null) {
					Jun3dPoint intersectingPoint = surface.intersectingPointWithLine_(theLine);

					if (intersectingPoint != null) {
						double distance = aJun3dPoint.distance_(intersectingPoint);

						if ((surface.normalVector().dotProduct_(vector) > accuracy) && (distance > accuracy) && (minDistance._doubleValue() > distance)) {
							referenceSurface.value_(surface);
							minDistance.value_(distance);
						}
					}
				}

				return null;
			}
		});

		if (referenceSurface.value() == null) {
			return false;
		}

		Jun3dPoint normalVector = ((JunPlane) referenceSurface.value()).normalUnitVector();
		return (normalVector.dotProduct_(vector) > 0);
	}

	/**
	 * Create a copy of the receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StObject
	 * @see jp.co.sra.smalltalk.StObject#copy()
	 * @category copying
	 */
	public StObject copy() {
		return JunBody.FromLispList_(this.toLispList());
	}

	/**
	 * Initialize the JunBody as a copy of the receiver.
	 * 
	 * @param aBody jp.co.sra.jun.topology.elements.JunBody
	 * @return jp.co.sra.jun.topology.elements.JunBody
	 * @category copying
	 */
	public JunBody copyTo_(JunBody aBody) {
		return aBody.fromLispList_(this.toLispList());
	}

	/**
	 * Answer the corner point of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category bounds accessing
	 */
	public Jun3dPoint corner() {
		return (Jun3dPoint) this.boundingBox().corner();
	}

	/**
	 * Enumerate the curves and evaluate the Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating
	 */
	public void curvesDo_(final StBlockClosure aBlock) {
		this.edgesDo_(new StBlockClosure() {
			public Object value_(Object edge) {
				return aBlock.value_(((JunEdge) edge).curve());
			}
		});
	}

	/**
	 * Detect an edge which meets the condition specified with aBlock. If not
	 * found, evaluate errorBlock.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param errorBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.jun.topology.elements.JunEdge
	 * @category enumerating
	 */
	public JunEdge detectEdge_ifNone_(final StBlockClosure aBlock, StBlockClosure errorBlock) {
		Object result = this.edgesDo_(new StBlockClosure() {
			public Object value_(Object edge) {
				if (((Boolean) aBlock.value_(edge)).booleanValue() == true) {
					return edge;
				}

				return null;
			}
		});

		return (JunEdge) ((result != null) ? result : errorBlock.value());
	}

	/**
	 * Detect an Loop which meets the condition specified with aBlock. If not
	 * found, evaluate errorBlock.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param errorBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.jun.topology.elements.JunLoop
	 * @category enumerating
	 */
	public JunLoop detectLoop_ifNone_(final StBlockClosure aBlock, StBlockClosure errorBlock) {
		Object result = this.loopsDo_(new StBlockClosure() {
			public Object value_(Object loop) {
				if (((Boolean) aBlock.value_(loop)).booleanValue() == true) {
					return loop;
				}

				return null;
			}
		});

		return (JunLoop) ((result != null) ? result : errorBlock.value());
	}

	/**
	 * Detect an Vertex which meets the condition specified with aBlock. If not
	 * found, evaluate errorBlock.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param errorBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.jun.topology.elements.JunVertex
	 * @category enumerating
	 */
	public JunVertex detectVertex_ifNone_(final StBlockClosure aBlock, StBlockClosure errorBlock) {
		Object result = this.vertexesDo_(new StBlockClosure() {
			public Object value_(Object vertex) {
				if (((Boolean) aBlock.value_(vertex)).booleanValue() == true) {
					return vertex;
				}

				return null;
			}
		});

		return (JunVertex) ((result != null) ? result : errorBlock.value());
	}

	/**
	 * Answer the JunEdge specified with the JunEdgeProxy.
	 * 
	 * @param anEdgeProxy jp.co.sra.jun.topology.elements.JunEdgeProxy
	 * @return jp.co.sra.jun.topology.elements.JunEdge
	 * @category accessing
	 */
	public JunEdge edgeAt_(JunEdgeProxy anEdgeProxy) {
		return (JunEdge) edges.get(anEdgeProxy);
	}

	/**
	 * Answer an edge which contains aVertex.
	 * 
	 * @param aVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @return jp.co.sra.jun.topology.elements.JunEdge
	 * @category accessing
	 */
	public JunEdge edgeForVertex_(JunVertex aVertex) {
		JunEdge[] edgesArray = this.edges();
		JunVertex s;
		JunVertex e;

		for (int i = 0; i < edgesArray.length; i++) {
			s = edgesArray[i].startVertex();
			e = edgesArray[i].endVertex();

			if ((s.equals(aVertex) == true) || (e.equals(aVertex) == true)) {
				return edgesArray[i];
			}
		}

		return null;
	}

	/**
	 * Answer an edge which contains aVertex1 and aVertex2.
	 * 
	 * @param aVertex1 jp.co.sra.jun.topology.elements.JunVertex
	 * @param aVertex2 jp.co.sra.jun.topology.elements.JunVertex
	 * @return jp.co.sra.jun.topology.elements.JunEdge
	 * @category accessing
	 */
	public JunEdge edgeForVertex_and_(final JunVertex aVertex1, final JunVertex aVertex2) {
		JunEdge[] edgesArray = this.edges();
		JunVertex s;
		JunVertex e;

		for (int i = 0; i < edgesArray.length; i++) {
			s = edgesArray[i].startVertex();
			e = edgesArray[i].endVertex();

			if (((s.equals(aVertex1) == true) && (e.equals(aVertex2) == true)) || ((s.equals(aVertex2) == true) && (e.equals(aVertex1) == true))) {
				return edgesArray[i];
			}
		}

		return null;
	}

	/**
	 * Answer the array of the edge proxies.
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunEdgeProxy[]
	 * @category private
	 */
	public JunEdgeProxy[] edgeProxies() {
		JunEdgeProxy[] anArrayOfEdgeProxy = new JunEdgeProxy[edges.size()];
		int index = 0;
		Enumeration enumeration = edges.keys();

		while (enumeration.hasMoreElements()) {
			anArrayOfEdgeProxy[index++] = (JunEdgeProxy) enumeration.nextElement();
		}

		return anArrayOfEdgeProxy;
	}

	/**
	 * Answer all edges contained in the receiver.
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunEdge[]
	 * @category accessing
	 */
	public JunEdge[] edges() {
		JunEdge[] anArrayOfEdge = new JunEdge[edges.size()];
		int index = 0;
		Enumeration enumeration = edges.elements();

		while (enumeration.hasMoreElements()) {
			anArrayOfEdge[index++] = (JunEdge) enumeration.nextElement();
		}

		return anArrayOfEdge;
	}

	/**
	 * Enumerate the edges and evaluate the Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category enumerating
	 */
	public Object edgesDo_(StBlockClosure aBlock) {
		JunEdge[] anArrayOfEdge = this.edges();

		for (int i = 0; i < anArrayOfEdge.length; i++) {
			Object result = aBlock.value_(anArrayOfEdge[i]);

			if (result != null) {
				return result;
			}
		}

		return null;
	}

	/**
	 * Answer all vertexes which contains aVertex.
	 * 
	 * @param aVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @return jp.co.sra.jun.topology.elements.JunEdge[]
	 * @category accessing
	 */
	public JunEdge[] edgesForVertex_(JunVertex aVertex) {
		Vector edgesVector = new Vector();
		JunEdge[] edgesArray = this.edges();
		JunVertex s;
		JunVertex e;

		for (int i = 0; i < edgesArray.length; i++) {
			s = edgesArray[i].startVertex();
			e = edgesArray[i].endVertex();

			if ((s == aVertex) || (e == aVertex)) {
				edgesVector.addElement(edgesArray[i]);
			}
		}

		JunEdge[] edgesForVertexArray = new JunEdge[edgesVector.capacity()];
		int count = 0;

		for (Enumeration element = edgesVector.elements(); element.hasMoreElements();) {
			edgesForVertexArray[count++] = (JunEdge) element.nextElement();
		}

		return edgesForVertexArray;
	}

	/**
	 * Answer the lisp list which represents the edges of the receiver.
	 * 
	 * @param loopArray jp.co.sra.jun.topology.elements.JunLoop[]
	 * @param edgeArray jp.co.sra.jun.topology.elements.JunEdge[]
	 * @param vertexArray jp.co.sra.jun.topology.elements.JunVertex[]
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	public JunLispCons edgesToLispListForLoops_edges_vertexes_(final JunLoop[] loopArray, final JunEdge[] edgeArray, final JunVertex[] vertexArray) {
		final JunLispCons list = this.lispCons();
		list.head_($("edges"));
		this.edgesDo_(new StBlockClosure() {
			public Object value_(Object edge) {
				list.add_(((JunEdge) edge).toLispListForLoops_edges_vertexes_(loopArray, edgeArray, vertexArray));

				return null;
			}
		});

		return list;
	}

	/**
	 * Answer the extent point of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category bounds accessing
	 */
	public Jun3dPoint extent() {
		return (Jun3dPoint) this.boundingBox().extent();
	}

	/**
	 * Forget every instance variables.
	 * 
	 * @category private
	 */
	public void forget() {
		name = null;
		loops = null;
		edges = null;
		vertexes = null;
	}

	/**
	 * Initialize the receiver with the information in a lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @return jp.co.sra.jun.topology.elements.JunBody
	 * @category lisp support
	 */
	public JunBody fromLispList_(JunLispList aList) {
		if (aList.head() != this._className()) {
			throw SmalltalkException.Error("unsupported format");
		}

		//
		JunLoop[] loopArray = null;
		JunEdge[] edgeArray = null;
		JunVertex[] vertexArray = null;
		JunLispList list = (JunLispList) aList.tail();

		while (list.nullList() == false) {
			JunLispCons tuple = (JunLispCons) list.head();

			if (tuple.head() == $("loops")) {
				int size = ((JunLispList) tuple.tail()).size();
				loopArray = new JunLoop[size];

				for (int i = 0; i < size; i++) {
					loopArray[i] = new JunLoop();
				}
			}

			if (tuple.head() == $("edges")) {
				int size = ((JunLispList) tuple.tail()).size();
				edgeArray = new JunEdge[size];

				for (int i = 0; i < size; i++) {
					edgeArray[i] = new JunEdge();
				}
			}

			if (tuple.head() == $("vertexes")) {
				int size = ((JunLispList) tuple.tail()).size();
				vertexArray = new JunVertex[size];

				for (int i = 0; i < size; i++) {
					vertexArray[i] = new JunVertex();
				}
			}

			list = (JunLispList) list.tail();
		}

		//
		list = (JunLispList) aList.tail();

		while (list.nullList() == false) {
			JunLispCons tuple = (JunLispCons) list.head();

			if (tuple.head() == $("loops")) {
				int size = ((JunLispList) tuple.tail()).size();
				Object[] listArray = ((JunLispList) tuple.tail()).asArray();

				for (int i = 0; i < size; i++) {
					loopArray[i].fromLispList_forLoops_edges_vertexes_((JunLispCons) listArray[i], loopArray, edgeArray, vertexArray);
				}
			}

			if (tuple.head() == $("edges")) {
				int size = ((JunLispList) tuple.tail()).size();
				Object[] listArray = ((JunLispList) tuple.tail()).asArray();

				for (int i = 0; i < size; i++) {
					edgeArray[i].fromLispList_forLoops_edges_vertexes_((JunLispCons) listArray[i], loopArray, edgeArray, vertexArray);
				}
			}

			if (tuple.head() == $("vertexes")) {
				int size = ((JunLispList) tuple.tail()).size();
				Object[] listArray = ((JunLispList) tuple.tail()).asArray();

				for (int i = 0; i < size; i++) {
					vertexArray[i].fromLispList_forLoops_edges_vertexes_((JunLispCons) listArray[i], loopArray, edgeArray, vertexArray);
				}
			}

			list = (JunLispList) list.tail();
		}

		this.initialize();
		JunBody body = this;

		for (int i = 0; i < loopArray.length; i++) {
			body.addLoop_(loopArray[i]);
		}

		for (int i = 0; i < edgeArray.length; i++) {
			body.addEdge_(edgeArray[i]);
		}

		for (int i = 0; i < vertexArray.length; i++) {
			body.addVertex_(vertexArray[i]);
		}

		return body;
	}

	/**
	 * Answer the height of the receiver.
	 * 
	 * @return double
	 * @category bounds accessing
	 */
	public double height() {
		return this.boundingBox().height();
	}

	/**
	 * Answer true if the receiver contains the JunEdge.
	 * 
	 * @param anEdge jp.co.sra.jun.topology.elements.JunEdge
	 * @return boolean
	 * @category testing
	 */
	public boolean includesEdge_(JunEdge anEdge) {
		return edges.contains(anEdge);
	}

	/**
	 * Answer true if the receiver contains the JunLoop.
	 * 
	 * @param aLoop jp.co.sra.jun.topology.elements.JunLoop
	 * @return boolean
	 * @category testing
	 */
	public boolean includesLoop_(JunLoop aLoop) {
		return loops.contains(aLoop);
	}

	/**
	 * Answer true if the receiver contains the JunVertex.
	 * 
	 * @param aVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @return boolean
	 * @category testing
	 */
	public boolean includesVertex_(JunVertex aVertex) {
		return vertexes.contains(aVertex);
	}

	/**
	 * Initialize the instance of JunBody.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunAbstractObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		name = (new Date()).toString();
		loops = new Hashtable();
		edges = new Hashtable();
		vertexes = new Hashtable();
		proxies = new Hashtable();
	}

	/**
	 * Answer true if the receiver is consistent, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isConsistent() {
		return this.isConsistentWithElements();
	}

	/**
	 * Answer true if the receiver is consistent with the elements, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isConsistentWithElements() {
		Enumeration keys = loops.keys();

		while (keys.hasMoreElements()) {
			Object loop = loops.get(keys.nextElement());

			if (proxies.containsKey(loop) == false) {
				return false;
			}
		}

		keys = edges.keys();

		while (keys.hasMoreElements()) {
			Object edge = edges.get(keys.nextElement());

			if (proxies.containsKey(edge) == false) {
				return false;
			}
		}

		keys = vertexes.keys();

		while (keys.hasMoreElements()) {
			Object vertex = vertexes.get(keys.nextElement());

			if (proxies.containsKey(vertex) == false) {
				return false;
			}
		}

		Object result = this.edgesDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunEdge edge = (JunEdge) anObject;

				if (edge.leftLoop.includesEdge_(edge) == false) {
					return Boolean.FALSE;
				}

				if (edge.rightLoop.includesEdge_(edge) == false) {
					return Boolean.FALSE;
				}

				return null;
			}
		});

		if (result != null) {
			return false;
		}

		result = this.vertexesDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunVertex vertex = (JunVertex) anObject;

				if ((vertex.edge().startVertex() != vertex) && (vertex.edge().endVertex() != vertex)) {
					return Boolean.FALSE;
				}

				return null;
			}
		});

		if (result != null) {
			return false;
		}

		result = this.loopsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunLoop loop = (JunLoop) anObject;

				if ((loop.edge().leftLoop() != loop) && (loop.edge.rightLoop() != loop)) {
					return Boolean.FALSE;
				}

				return null;
			}
		});

		if (result != null) {
			return false;
		}

		return true;
	}

	/**
	 * Answer true if the receiver is consistent with an Euler equation, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isConsistentWithEulerEquation() {
		return ((vertexes.size() - edges.size() + loops.size()) == 2) || ((vertexes.size() + edges.size() + loops.size()) == 0);
	}

	/**
	 * Answer the JunLoop specified with a JunLoopProxy.
	 * 
	 * @param aJunLoopProxy jp.co.sra.jun.topology.elements.JunLoopProxy
	 * @return jp.co.sra.jun.topology.elements.JunLoop
	 * @category accessing
	 */
	public JunLoop loopAt_(JunLoopProxy aJunLoopProxy) {
		return (JunLoop) loops.get(aJunLoopProxy);
	}

	/**
	 * Answer a JunLoop which contains the JunVertex.
	 * 
	 * @param aVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @return jp.co.sra.jun.topology.elements.JunLoop
	 * @category accessing
	 */
	public JunLoop loopForVertex_(JunVertex aVertex) {
		JunEdge edge;
		JunLoop loop;
		edge = this.edgeForVertex_(aVertex);

		if (edge == null) {
			return null;
		}

		loop = edge.leftLoop();

		if (loops.contains(loop) == true) {
			return loop;
		}

		loop = edge.rightLoop();

		if (loops.contains(loop) == true) {
			return loop;
		}

		return null;
	}

	/**
	 * Answer a JunLoop which contains aVertex1 and aVertex2.
	 * 
	 * @param aVertex1 jp.co.sra.jun.topology.elements.JunVertex
	 * @param aVertex2 jp.co.sra.jun.topology.elements.JunVertex
	 * @return jp.co.sra.jun.topology.elements.JunLoop
	 * @category accessing
	 */
	public JunLoop loopForVertex_and_(final JunVertex aVertex1, final JunVertex aVertex2) {
		return (JunLoop) this.loopsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunLoop loop = (JunLoop) anObject;

				if (loop.includesVertex_(aVertex1) && loop.includesVertex_(aVertex2)) {
					return loop;
				}

				return null;
			}
		});
	}

	/**
	 * Answer a JunLoop which contains aVertex1 and its next vertex is aVertex2.
	 * 
	 * @param aVertex1 jp.co.sra.jun.topology.elements.JunVertex
	 * @param aVertex2 jp.co.sra.jun.topology.elements.JunVertex
	 * @return jp.co.sra.jun.topology.elements.JunLoop
	 * @category accessing
	 */
	public JunLoop loopOnVertex_next_(final JunVertex aVertex1, final JunVertex aVertex2) {
		return (JunLoop) this.loopsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunLoop loop = (JunLoop) anObject;

				if (loop.includesPathFrom_to_(aVertex1, aVertex2)) {
					return loop;
				}

				return null;
			}
		});
	}

	/**
	 * Answer the array of the loop proxies.
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunLoopProxy[]
	 * @category private
	 */
	public JunLoopProxy[] loopProxies() {
		JunLoopProxy[] anArrayOfLoopProxy = new JunLoopProxy[loops.size()];
		int index = 0;
		Enumeration enumeration = loops.keys();

		while (enumeration.hasMoreElements()) {
			anArrayOfLoopProxy[index++] = (JunLoopProxy) enumeration.nextElement();
		}

		return anArrayOfLoopProxy;
	}

	/**
	 * Answer all loops of the receiver.
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunLoop[]
	 * @category accessing
	 */
	public JunLoop[] loops() {
		JunLoop[] anArrayOfLoop = new JunLoop[loops.size()];
		int index = 0;
		Enumeration enumeration = loops.elements();

		while (enumeration.hasMoreElements()) {
			anArrayOfLoop[index++] = (JunLoop) enumeration.nextElement();
		}

		return anArrayOfLoop;
	}

	/**
	 * Enumerate the loops and evaluate the Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category enumerating
	 */
	public Object loopsDo_(StBlockClosure aBlock) {
		JunLoop[] anArrayOfLoop = this.loops();

		for (int i = 0; i < anArrayOfLoop.length; i++) {
			Object result = aBlock.value_(anArrayOfLoop[i]);

			if (result != null) {
				return result;
			}
		}

		return null;
	}

	/**
	 * Answer the lisp list which represents the loops of the receiver.
	 * 
	 * @param loopArray jp.co.sra.jun.topology.elements.JunLoop[]
	 * @param edgeArray jp.co.sra.jun.topology.elements.JunEdge[]
	 * @param vertexArray jp.co.sra.jun.topology.elements.JunVertex[]
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	public JunLispCons loopsToLispListForLoops_edges_vertexes_(final JunLoop[] loopArray, final JunEdge[] edgeArray, final JunVertex[] vertexArray) {
		final JunLispCons list = this.lispCons();
		list.head_($("loops"));
		this.loopsDo_(new StBlockClosure() {
			public Object value_(Object loop) {
				list.add_(((JunLoop) loop).toLispListForLoops_edges_vertexes_(loopArray, edgeArray, vertexArray));

				return null;
			}
		});

		return list;
	}

	/**
	 * Answer the name of the receiver.
	 * 
	 * @return java.lang.String
	 * @category accessing
	 */
	public String name() {
		return name;
	}

	/**
	 * Set the name of the receiver.
	 * 
	 * @param aString java.lang.String
	 * @category accessing
	 */
	public void name_(String aString) {
		name = aString;
	}

	/**
	 * Answer the number of the edges.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int numberOfEdges() {
		return edges.size();
	}

	/**
	 * Answer the number of the loops.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int numberOfLoops() {
		return loops.size();
	}

	/**
	 * Answer the number of the vertexes.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int numberOfVertexes() {
		return vertexes.size();
	}

	/**
	 * Answer the origin point of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category bounds accessing
	 */
	public Jun3dPoint origin() {
		return (Jun3dPoint) this.boundingBox().origin();
	}

	/**
	 * Enumerate the points and evaluate the Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating
	 */
	public void pointsDo_(StBlockClosure aBlock) {
		final StBlockClosure aBlock_ = aBlock;
		this.vertexesDo_(new StBlockClosure() {
			public Object value_(Object vertex) {
				return aBlock_.value_(((JunVertex) vertex).point());
			}
		});
	}

	/**
	 * Print my string representation on the Writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write("Body (");
		aWriter.write("L: " + this.numberOfLoops());
		aWriter.write(" E: " + this.numberOfEdges());
		aWriter.write(" V: " + this.numberOfVertexes());
		aWriter.write(")");
	}

	/**
	 * Enumerate the proxy and the edge pairs and evaluate the Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category private
	 */
	public void proxiesAndEdgesDo_(StBlockClosure aBlock) {
		JunEdgeProxy[] anArrayOfEdgeProxy = this.edgeProxies();

		for (int i = 0; i < anArrayOfEdgeProxy.length; i++) {
			JunEdgeProxy proxy = anArrayOfEdgeProxy[i];
			JunEdge edge = (JunEdge) edges.get(proxy);
			aBlock.value_value_(proxy, edge);
		}
	}

	/**
	 * Enumerate the proxy and the loop pairs and evaluate the Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category private
	 */
	public void proxiesAndLoopsDo_(StBlockClosure aBlock) {
		JunLoopProxy[] anArrayOfLoopProxy = this.loopProxies();

		for (int i = 0; i < anArrayOfLoopProxy.length; i++) {
			JunLoopProxy proxy = anArrayOfLoopProxy[i];
			JunLoop loop = (JunLoop) loops.get(proxy);
			aBlock.value_value_(proxy, loop);
		}
	}

	/**
	 * Enumerate the proxy and the vertex pairs and evaluate the Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category private
	 */
	public void proxiesAndVertexesDo_(StBlockClosure aBlock) {
		JunVertexProxy[] anArrayOfVertexProxy = this.vertexProxies();

		for (int i = 0; i < anArrayOfVertexProxy.length; i++) {
			JunVertexProxy proxy = anArrayOfVertexProxy[i];
			JunVertex vertex = (JunVertex) vertexes.get(proxy);
			aBlock.value_value_(proxy, vertex);
		}
	}

	/**
	 * Answer the proxy for the edge.
	 * 
	 * @param anEdge jp.co.sra.jun.topology.elements.JunEdge
	 * @return jp.co.sra.jun.topology.abstracts.JunTopologicalElementProxy
	 * @category accessing
	 */
	public JunTopologicalElementProxy proxyForEdge_(JunEdge anEdge) {
		return (JunTopologicalElementProxy) proxies.get(anEdge);
	}

	/**
	 * Answer the proxy for the loop.
	 * 
	 * @param aLoop jp.co.sra.jun.topology.elements.JunLoop
	 * @return jp.co.sra.jun.topology.abstracts.JunTopologicalElementProxy
	 * @category accessing
	 */
	public JunTopologicalElementProxy proxyForLoop_(JunLoop aLoop) {
		return (JunTopologicalElementProxy) proxies.get(aLoop);
	}

	/**
	 * Answer the proxy for the vertex.
	 * 
	 * @param aVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @return jp.co.sra.jun.topology.abstracts.JunTopologicalElementProxy
	 * @category accessing
	 */
	public JunTopologicalElementProxy proxyForVertex_(JunVertex aVertex) {
		return (JunTopologicalElementProxy) proxies.get(aVertex);
	}

	/**
	 * Remove the JunEdge from the receiver.
	 * 
	 * @param anEdge jp.co.sra.jun.topology.elements.JunEdge
	 * @category removing
	 */
	public void removeEdge_(JunEdge anEdge) {
		JunEdgeProxy proxy = (JunEdgeProxy) proxies.get(anEdge);
		edges.remove(proxy);
		proxies.remove(anEdge);
	}

	/**
	 * Remove the JunLoop from the receiver.
	 * 
	 * @param aLoop jp.co.sra.jun.topology.elements.JunLoop
	 * @category removing
	 */
	public void removeLoop_(JunLoop aLoop) {
		JunLoopProxy proxy = (JunLoopProxy) proxies.get(aLoop);
		loops.remove(proxy);
		proxies.remove(aLoop);
	}

	/**
	 * Remove the JunVertex from the receiver.
	 * 
	 * @param aVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @category removing
	 */
	public void removeVertex_(JunVertex aVertex) {
		JunVertexProxy proxy = (JunVertexProxy) proxies.get(aVertex);
		vertexes.remove(proxy);
		proxies.remove(aVertex);
	}

	/**
	 * Render the receiver on the rendering context.
	 * 
	 * @param aRenderingContext jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * @category displaying
	 */
	public void renderOn_(final JunOpenGLRenderingContext aRenderingContext) {
		this.surfacesDo_(new StBlockClosure() {
			public Object value_(Object surface) {
				((JunOpenGL3dPolygon) surface).renderOn_(aRenderingContext);

				return null;
			}
		});
	}

	/**
	 * Save the lisp representation of the receiver on a Writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @category lisp support
	 */
	public void saveOn_(Writer aWriter) {
		JunLispList aList;
		JunCursors cursor = new JunCursors(new Cursor(Cursor.WAIT_CURSOR));
		try {
			cursor._show();
			aList = this.toLispList();
		} finally {
			cursor._restore();
		}
		try {
			aList.saveOn_(aWriter);
		} catch (IOException e) {
			System.err.println(e.getMessage());
			e.printStackTrace();
		}
	}

	/**
	 * Select edges which meets the condition specified with aBlock1. Enumerate
	 * the selected edges and evaluate aBlock2.
	 * 
	 * @param aBlock1 jp.co.sra.smalltalk.StBlockClosure
	 * @param aBlock2 jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating
	 */
	public void selectEdges_do_(final StBlockClosure aBlock1, final StBlockClosure aBlock2) {
		StBlockClosure b = new StBlockClosure() {
			public Object value_(Object edge) {
				if (aBlock1.value_(edge) == Boolean.TRUE) {
					return aBlock2.value_(edge);
				}

				return null;
			}
		};

		this.edgesDo_(b);
	}

	/**
	 * Select loops which meets the condition specified with aBlock1. Enumerate
	 * the selected loops and evaluate aBlock2.
	 * 
	 * @param aBlock1 jp.co.sra.smalltalk.StBlockClosure
	 * @param aBlock2 jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating
	 */
	public void selectLoops_do_(final StBlockClosure aBlock1, final StBlockClosure aBlock2) {
		StBlockClosure b = new StBlockClosure() {
			public Object value_(Object edge) {
				if (aBlock1.value_(edge) == Boolean.TRUE) {
					return aBlock2.value_(edge);
				}

				return null;
			}
		};

		this.loopsDo_(b);
	}

	/**
	 * Select vertexes which meets the condition specified with aBlock1.
	 * Enumerate the selected vertexes and evaluate aBlock2.
	 * 
	 * @param aBlock1 jp.co.sra.smalltalk.StBlockClosure
	 * @param aBlock2 jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating
	 */
	public void selectVertexes_do_(final StBlockClosure aBlock1, final StBlockClosure aBlock2) {
		StBlockClosure b = new StBlockClosure() {
			public Object value_(Object edge) {
				if (aBlock1.value_(edge) == Boolean.TRUE) {
					return aBlock2.value_(edge);
				}

				return null;
			}
		};

		this.vertexesDo_(b);
	}

	/**
	 * Show the receiver created from a lisp list on a window.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	public JunLispList showLispList() {
		return this.showLispList_(this.toLispList());
	}

	/**
	 * Show the receiver created from a lisp list on a window.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	public JunLispList showLispList_(JunLispList aList) {
		return JunBody.ShowLispList_(aList);
	}

	/**
	 * Enumerate the surfaces and evaluate the Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating
	 */
	public void surfacesDo_(final StBlockClosure aBlock) {
		this.loopsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				aBlock.value_(((JunLoop) anObject).surface());

				return null;
			}
		});
	}

	/**
	 * Convert the receiver as a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	public JunLispList toLispList() {
		return this.toLispListForLoops_edges_vertexes_(this.loops(), this.edges(), this.vertexes());
	}

	/**
	 * Answer the lisp list which represents the receiver.
	 * 
	 * @param loopArray jp.co.sra.jun.topology.elements.JunLoop[]
	 * @param edgeArray jp.co.sra.jun.topology.elements.JunEdge[]
	 * @param vertexArray jp.co.sra.jun.topology.elements.JunVertex[]
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @see jp.co.sra.jun.topology.abstracts.JunTopologicalElement#toLispListForLoops_edges_vertexes_(jp.co.sra.jun.topology.elements.JunLoop[], jp.co.sra.jun.topology.elements.JunEdge[], jp.co.sra.jun.topology.elements.JunVertex[])
	 * @category lisp support
	 */
	public JunLispList toLispListForLoops_edges_vertexes_(JunLoop[] loopArray, JunEdge[] edgeArray, JunVertex[] vertexArray) {
		JunLispCons list = this.lispCons();
		list.head_(this.kindName());
		list.add_(this.loopsToLispListForLoops_edges_vertexes_(loopArray, edgeArray, vertexArray));
		list.add_(this.edgesToLispListForLoops_edges_vertexes_(loopArray, edgeArray, vertexArray));
		list.add_(this.vertexesToLispListForLoops_edges_vertexes_(loopArray, edgeArray, vertexArray));

		return list;
	}

	/**
	 * Answer true if the receiver touches the Jun3dPoint, otherwise false.
	 * 
	 * @param aJun3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return boolean
	 * @category testing
	 */
	public boolean touchesPoint_(final Jun3dPoint aJun3dPoint) {
		Jun3dBoundingBox bounds = this.boundingBox();

		if (bounds.containsOrTouchesPoint_(aJun3dPoint) == false) {
			return false;
		}

		Object result = this.loopsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunLoop loop = (JunLoop) anObject;

				if (((JunPlane) loop.surface()).containsPoint_(aJun3dPoint)) {
					return Boolean.TRUE;
				}

				return null;
			}
		});

		return (result != null);
	}

	/**
	 * Answer a JunVertex specified with the JunVertexProxy.
	 * 
	 * @param aVertexProxy jp.co.sra.jun.topology.elements.JunVertexProxy
	 * @return jp.co.sra.jun.topology.elements.JunVertex
	 * @category accessing
	 */
	public JunVertex vertexAt_(JunVertexProxy aVertexProxy) {
		return (JunVertex) vertexes.get(aVertexProxy);
	}

	/**
	 * Answer all vertexes of the receiver.
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunVertex[]
	 * @category accessing
	 */
	public JunVertex[] vertexes() {
		JunVertex[] anArrayOfVertex = new JunVertex[vertexes.size()];
		int index = 0;
		Enumeration enumeration = vertexes.elements();

		while (enumeration.hasMoreElements()) {
			anArrayOfVertex[index++] = (JunVertex) enumeration.nextElement();
		}

		return anArrayOfVertex;
	}

	/**
	 * Enumerate the vertexes and evaluate the Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category enumerating
	 */
	public Object vertexesDo_(StBlockClosure aBlock) {
		JunVertex[] anArrayOfVertex = this.vertexes();

		for (int i = 0; i < anArrayOfVertex.length; i++) {
			Object result = aBlock.value_(anArrayOfVertex[i]);

			if (result != null) {
				return result;
			}
		}

		return null;
	}

	/**
	 * Answer the lisp list which represents the vertexes of the receiver.
	 * 
	 * @param loopArray jp.co.sra.jun.topology.elements.JunLoop[]
	 * @param edgeArray jp.co.sra.jun.topology.elements.JunEdge[]
	 * @param vertexArray jp.co.sra.jun.topology.elements.JunVertex[]
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	public JunLispCons vertexesToLispListForLoops_edges_vertexes_(final JunLoop[] loopArray, final JunEdge[] edgeArray, final JunVertex[] vertexArray) {
		final JunLispCons list = this.lispCons();
		list.head_($("vertexes"));
		this.vertexesDo_(new StBlockClosure() {
			public Object value_(Object vertex) {
				list.add_(((JunVertex) vertex).toLispListForLoops_edges_vertexes_(loopArray, edgeArray, vertexArray));

				return null;
			}
		});

		return list;
	}

	/**
	 * Answer the array of the vertex proxies.
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunVertexProxy[]
	 * @category private
	 */
	public JunVertexProxy[] vertexProxies() {
		JunVertexProxy[] anArrayOfVertexProxy = new JunVertexProxy[vertexes.size()];
		int index = 0;
		Enumeration enumeration = vertexes.keys();

		while (enumeration.hasMoreElements()) {
			anArrayOfVertexProxy[index++] = (JunVertexProxy) enumeration.nextElement();
		}

		return anArrayOfVertexProxy;
	}

	/**
	 * Answer my volume.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double volume() {
		double v = 0;
		JunLoop[] anArrayOfLoop = this.loops();

		for (int i = 0; i < anArrayOfLoop.length; i++) {
			v += anArrayOfLoop[i].volume();
		}

		return v;
	}

	/**
	 * Answer the width of the receiver.
	 * 
	 * @return double
	 * @category bounds accessing
	 */
	public double width() {
		return this.boundingBox().width();
	}
}
