package jp.co.sra.jun.topology.graph;

import java.awt.Font;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;
import java.io.IOException;
import java.io.Writer;

import jp.co.sra.smalltalk.StComposedText;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StSymbol;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.goodies.lisp.JunLispList;
import jp.co.sra.jun.goodies.tables.JunAttributeTable;

/**
 * JunElementalNode class
 * 
 *  @author    nisinaka
 *  @created   2006/04/05 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun597 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunElementalNode.java,v 8.10 2008/02/20 06:33:02 nisinaka Exp $
 */
public class JunElementalNode extends JunElementalStuff {

	protected static JunAttributeTable DefaultAttributes = new JunAttributeTable();
	protected static StSymbol[] AttributeSymbolsToReset = new StSymbol[] { $("labelString"), $("locationPoint"), $("foregroundColor"), $("backgroundColor"), $("borderColor"), $("borderWidth") };

	protected StImage visualObject;
	protected StRectangle boundingBox;

	/**
	 * Create a new instance of JunElementalNode and initialize it.
	 *
	 * @category Instance creation
	 */
	public JunElementalNode() {
		super();
	}

	/**
	 * Create a new instance of JunElementalNode and initialize it.
	 *
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunElementalNode(JunLispList aList) {
		super(aList);
	}

	/**
	 * Create a new instance of JunElementalNode and initialize it.
	 *
	 * @param labelString java.lang.String
	 * @category Instance creation
	 */
	public JunElementalNode(String labelString) {
		this();
		this.setLabelString_(labelString);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.topology.graph.JunElementalStuff#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		this.flushVisualObject();
	}

	/**
	 * Answer my current visual object.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category accessing
	 */
	public StImage visualObject() {
		if (visualObject == null) {
			Font aFont = StComposedText.DefaultFont();
			if (this.labelEmphasis()) {
				aFont = aFont.deriveFont(Font.BOLD);
			}
			StComposedText aComposedText = new StComposedText(this.labelString(), aFont);
			StRectangle aBox = new StRectangle(aComposedText.bounds());
			aBox = aBox.expandedBy_(new StRectangle(1, 1, 2, 2));
			aBox = aBox.expandedBy_(this.borderWidth());
			aBox = new StRectangle(0, 0, aBox.width(), aBox.height());
			StImage anImage = new StImage(aBox.width(), aBox.height());

			Graphics aGraphics = null;
			try {
				aGraphics = anImage.image().getGraphics();
				aGraphics.setColor(this.borderColor());
				aGraphics.fillRect(aBox.x(), aBox.y(), aBox.width(), aBox.height());

				aBox = aBox.insetBy_(this.borderWidth());
				aGraphics.setColor(this.backgroundColor());
				aGraphics.fillRect(aBox.x(), aBox.y(), aBox.width(), aBox.height());

				StRectangle textBox = new StRectangle(aComposedText.bounds());
				textBox = textBox.align_with_(textBox.center(), aBox.center());
				aGraphics.setColor(this.foregroundColor());
				aComposedText.displayOn_at_(aGraphics, textBox.origin());
			} finally {
				if (aGraphics != null) {
					aGraphics.dispose();
				}
			}

			visualObject = anImage;
		}
		return visualObject;
	}

	/**
	 * Answer my current bounding box.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @see jp.co.sra.jun.topology.graph.JunElementalStuff#boundingBox()
	 * @category accessing
	 */
	public StRectangle boundingBox() {
		if (boundingBox == null) {
			StRectangle aBox = new StRectangle(this.visualObject().bounds());
			aBox = aBox.align_with_(aBox.center(), this.locationPoint());
			boundingBox = aBox;
		}
		return boundingBox;
	}

	/**
	 * Answer my width.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int width() {
		return this.boundingBox().width();
	}

	/**
	 * Answer my height.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int height() {
		return this.boundingBox().height();
	}

	/**
	 * Answer my origin.
	 * 
	 * @return java.awt.Point
	 * @category accessing
	 */
	public Point origin() {
		return this.boundingBox().origin();
	}

	/**
	 * Set my origin.
	 * 
	 * @param aPoint java.awt.Point
	 * @category accessing
	 */
	public void origin_(Point aPoint) {
		StRectangle aBox = this.boundingBox();
		this.locationPoint_(aBox.align_with_(aBox.origin(), aPoint).center());
	}

	/**
	 * Answer my center.
	 * 
	 * @return java.awt.Point
	 * @category accessing
	 */
	public Point center() {
		return this.boundingBox().center();
	}

	/**
	 * Set my center.
	 * 
	 * @param aPoint java.awt.Point
	 * @category accessing
	 */
	public void center_(Point aPoint) {
		StRectangle aBox = this.boundingBox();
		this.locationPoint_(aBox.align_with_(aBox.center(), aPoint).center());
	}

	/**
	 * Answer the attribute symbols to reset.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol[]
	 * @see jp.co.sra.jun.topology.graph.JunElementalStuff#attributeSymbolsToReset()
	 * @category attributes
	 */
	protected StSymbol[] attributeSymbolsToReset() {
		return AttributeSymbolsToReset;
	}

	/**
	 * Flush the visual object.
	 * 
	 * @see jp.co.sra.jun.topology.graph.JunElementalStuff#flushVisualObject()
	 * @category flushing
	 */
	public void flushVisualObject() {
		visualObject = null;
		this.flushBoundingBox();
	}

	/**
	 * Flush the bounding box.
	 * 
	 * @see jp.co.sra.jun.topology.graph.JunElementalStuff#flushBoundingBox()
	 * @category flushing
	 */
	protected void flushBoundingBox() {
		boundingBox = null;
	}

	/**
	 * Answer the default attribute table.
	 * 
	 * @return jp.co.sra.jun.goodies.tables.JunAttributeTable
	 * @see jp.co.sra.jun.topology.graph.JunElementalStuff#defaultAttributes()
	 * @category defaults
	 */
	protected JunAttributeTable defaultAttributes() {
		return DefaultAttributes;
	}

	/**
	 * Print my string representation on aWriter.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException if failed.
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write("node(");
		aWriter.write(this.labelString());
		aWriter.write(')');
	}

	/**
	 * Display the receiver on the graphics at the specified point.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @category displaying
	 */
	public void displayOn_at_(Graphics aGraphics, Point aPoint) {
		StRectangle displayBox = this.boundingBox().translatedBy_(aPoint.x, aPoint.y);
		Rectangle clippingBox = aGraphics.getClipBounds();
		if (clippingBox == null || displayBox.toRectangle().intersects(clippingBox)) {
			this.visualObject().displayOn_at_(aGraphics, displayBox.origin());
		}
	}

	/**
	 * Display the receiver on the graphics at the specified point with the specified scale factor.
	 *
	 * @param aGraphics java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @param scaleFactor jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category displaying
	 */
	public void displayOn_at_scaledBy_(Graphics aGraphics, Point aPoint, Jun2dPoint scaleFactor) {
		StRectangle displayBox = this.boundingBox().translatedBy_(aPoint.x, aPoint.y);
		displayBox = displayBox.scaledBy_(scaleFactor.x(), scaleFactor.y());
		if (displayBox.area() <= 0) {
			displayBox = displayBox.expandedBy_(new StRectangle(0, 0, 1, 1));
		}

		Rectangle clippingBox = aGraphics.getClipBounds();
		if (clippingBox == null || displayBox.toRectangle().intersects(clippingBox)) {

			aGraphics.setColor(this.backgroundColor());
			aGraphics.fillRect(displayBox.x(), displayBox.y(), displayBox.width(), displayBox.height());

			displayBox = displayBox.insetBy_(new StRectangle(0, 0, 1, 1));
			aGraphics.setColor(this.borderColor());
			aGraphics.drawRect(displayBox.x(), displayBox.y(), displayBox.width(), displayBox.height());
		}
	}

}
