package jp.co.sra.jun.voronoi.twoD.diagram;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.geom.Line2D;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StValueHolder;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.surfaces.Jun2dCircle;
import jp.co.sra.jun.geometry.surfaces.Jun2dPolygon;
import jp.co.sra.jun.geometry.surfaces.Jun2dTriangle;

/**
 * JunDelaunay2dDiagram class
 * 
 *  @author    NISHIHARA Satoshi
 *  @created   2000/02/07 (by NISHIHARA Satoshi)
 *  @updated   2006/11/09 (by nisinaka)
 *  @updated   2007/05/13 (by m-asada)
 *  @version   699 (with StPL8.9) based on Jun662 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunDelaunay2dDiagram.java,v 8.12 2008/02/20 06:33:14 nisinaka Exp $
 */
public class JunDelaunay2dDiagram extends JunVoronoi2dDiagram {
	class LineSegment extends Line2D.Double {
		/**
		 * Create a new instance of <code>LineSegment</code> and initialize it.
		 * 
		 * @param fromPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
		 * @param toPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
		 * @category Instance creation
		 */
		public LineSegment(Jun2dPoint fromPoint, Jun2dPoint toPoint) {
			super(fromPoint.x(), fromPoint.y(), toPoint.x(), toPoint.y());
		}

		/**
		 * Answer the receiver's start point.
		 * 
		 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
		 * @category accessing
		 */
		public Jun2dPoint start() {
			return new Jun2dPoint(this.getX1(), this.getY1());
		}

		/**
		 * Answer the receiver's end point.
		 * 
		 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
		 * @category accessing
		 */
		public Jun2dPoint end() {
			return new Jun2dPoint(this.getX2(), this.getY2());
		}

		/**
		 * Indicates whether some other object is "equal to" this one.
		 * 
		 * @param anObject
		 * @return boolean
		 * @see java.lang.Object#equals(java.lang.Object)
		 * @category comparing
		 */
		public boolean equals(Object anObject) {
			if (anObject instanceof LineSegment == false) {
				return false;
			}
			LineSegment otherLine = (LineSegment) anObject;
			return (this.x1 == otherLine.x1 && this.y1 == otherLine.y1 && this.x2 == otherLine.x2 && this.y2 == otherLine.y2);
		}

		/**
		 * Returns a string representation of the object.
		 * 
		 * @return java.lang.String
		 * @see java.lang.Object#toString()
		 * @category printing
		 */
		public String toString() {
			return "LineSegment((" + this.x1 + ", " + this.y1 + ")->(" + this.x2 + ", " + this.y2 + "))";
		}
	}

	/**
	 * Create a new instance of JunDelaunay2dDiagram and initialize it.
	 *
	 * @param width int
	 * @param height int
	 * @category Instance creation
	 */
	public JunDelaunay2dDiagram(int width, int height) {
		super(width, height);
	}

	/**
	 * Create a new instance of JunDelaunay2dDiagram and initialize it.
	 *
	 * @param extent java.awt.Dimension
	 * @category Instance creation
	 */
	public JunDelaunay2dDiagram(Dimension extent) {
		super(extent);
	}

	/**
	 * Convert the receiver as JunVoronoi2dDiagram.
	 * 
	 * @return jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dDiagram
	 * @see jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dDiagram#asVoronoiDiagram()
	 * @category converting
	 */
	public JunVoronoi2dDiagram asVoronoiDiagram() {
		JunVoronoi2dDiagram diagram = new JunVoronoi2dDiagram(this.extent());
		diagram.addAll_(this.points());
		return diagram;
	}

	/**
	 * Answer the convex hull.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dPolygon
	 * @category accessing
	 */
	public Jun2dPolygon convexHull() {
		int numOfPoints = this.points().length;
		if (numOfPoints >= 3) {
			LineSegment[] edgeSet = this.convexHullEdges();
			Jun2dPoint[] vertexArray = this.lineupVertexSet_(edgeSet);
			return new Jun2dPolygon(vertexArray);
		}
		if (numOfPoints > 0) {
			return new Jun2dPolygon(this.points());
		}
		return null;
	}

	/**
	 * Answer the minimal enclosing circle.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dCircle
	 * @category accessing
	 */
	public Jun2dCircle minimalEnclosingCircle() {
		Jun2dPoint[] points = this.points();
		if (points.length >= 3) {
			return this.mecByChrystal();
		}
		if (points.length == 2) {
			Jun2dPoint point1 = points[0];
			Jun2dPoint point2 = points[1];
			Jun2dPoint center = point1.plus_(point2).dividedBy_(2);
			double radius = point2.distance_(point1) / 2;
			return new Jun2dCircle(center, radius);
		}
		if (points.length == 1) {
			Jun2dPoint center = points[0];
			return new Jun2dCircle(center, 0);
		}
		return null;
	}

	/**
	 * Convert the receiver as JunDelaunay2dDiagram.
	 * 
	 * @return jp.co.sra.jun.voronoi.twoD.diagram.JunDelaunay2dDiagram
	 * @see jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dDiagram#asDelaunayDiagram()
	 * @category converting
	 */
	public JunDelaunay2dDiagram asDelaunayDiagram() {
		return this;
	}

	/**
	 * Answer my label string for showing.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dDiagram#labelString()
	 * @category utilities
	 */
	protected String labelString() {
		return "Delaunay Diagram";
	}

	/**
	 * Answer the assure enclosure of the specified points in the approximate circle.
	 * 
	 * @param hullVertexes jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param approximateCircle jp.co.sra.jun.geometry.surfaces.Jun2dCircle
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dCircle
	 * @category private
	 */
	protected Jun2dCircle assureEnclosureOf_in_(Jun2dPoint[] hullVertexes, Jun2dCircle approximateCircle) {
		Jun2dPoint center = approximateCircle.center();
		double radius = 0.0d;
		for (int i = 0; i < hullVertexes.length; i++) {
			double destination = hullVertexes[i].distance_(center);
			radius = Math.max(radius, destination);
		}
		return new Jun2dCircle(center, radius);
	}

	/**
	 * Answer my border color.
	 * 
	 * @return java.awt.Color
	 * @see jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dDiagram#borderColor()
	 * @category private
	 */
	protected Color borderColor() {
		return Color.cyan;
	}

	/**
	 * Answer the convex h ull edges.
	 * 
	 * @return jp.co.sra.jun.voronoi.twoD.diagram.JunDelaunay2dDiagram.LineSegment[]
	 * @category private
	 */
	protected LineSegment[] convexHullEdges() {
		Jun2dPoint[][] triangles = this.processor().triangles();
		Collection edges = new ArrayList();
		for (int i = 0; i < triangles.length; i++) {
			Jun2dPoint p1 = triangles[i][0];
			Jun2dPoint p2 = triangles[i][1];
			Jun2dPoint p3 = triangles[i][2];
			LineSegment e1 = this._createLineSegmentWith_with_(p1, p2);
			LineSegment e2 = this._createLineSegmentWith_with_(p2, p3);
			LineSegment e3 = this._createLineSegmentWith_with_(p3, p1);
			if (edges.contains(e1)) {
				edges.remove(e1);
			} else {
				edges.add(e1);
			}
			if (edges.contains(e2)) {
				edges.remove(e2);
			} else {
				edges.add(e2);
			}
			if (edges.contains(e3)) {
				edges.remove(e3);
			} else {
				edges.add(e3);
			}
		}
		return (LineSegment[]) edges.toArray(new LineSegment[edges.size()]);
	}

	/**
	 * Line up isolated edges of a polygon along the connection of the edges.
	 * Make an array that contains vertexes in the order and return it.
	 * 
	 * @param aSet jp.co.sra.jun.voronoi.twoD.diagram.JunDelaunay2dDiagram.LineSegment[]
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category private
	 */
	protected Jun2dPoint[] lineupVertexSet_(LineSegment[] aSet) {
		Collection vertexArray = new ArrayList(aSet.length);
		Map vertexXDict = new HashMap();
		for (int i = 0; i < aSet.length; i++) {
			Jun2dPoint[] points = new Jun2dPoint[] { aSet[i].start(), aSet[i].end() };
			for (int j = 0; j < points.length; j++) {
				Set edgesAtX = (Set) vertexXDict.get(new Double(points[j].x()));
				if (edgesAtX == null) {
					edgesAtX = new HashSet();
					vertexXDict.put(new Double(points[j].x()), edgesAtX);
				}
				edgesAtX.add(aSet[i]);
			}
		}
		LineSegment startVector = aSet[0];
		LineSegment nextVector = startVector;
		vertexArray.add(nextVector.start());
		for (int i = 1; i < aSet.length - 1; i++) {
			Set branches = (Set) vertexXDict.get(new Double(nextVector.getX2()));
			nextVector = this.polylineGrowFrom_branches_(nextVector, branches);
			vertexArray.add(nextVector.start());
		}
		vertexArray.add(nextVector.end());
		return (Jun2dPoint[]) vertexArray.toArray(new Jun2dPoint[vertexArray.size()]);
	}

	/**
	 * Create my current image.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @see jp.co.sra.jun.voronoi.twoD.diagram.JunVoronoi2dDiagram#makeImage()
	 * @category private
	 */
	protected StImage makeImage() {
		voronoiProcessor = new JunVoronoi2dProcessor(this.points());
		processingCondition = voronoiProcessor.compute();

		StImage anImage = new StImage(this.extent());
		Graphics2D aGraphics = null;
		try {
			aGraphics = (Graphics2D) anImage.image().getGraphics();
			aGraphics.setBackground(Color.white);

			Rectangle bounds = this.bounds();
			double factorX = (double) bounds.width / (bounds.width + 1);
			double factorY = (double) bounds.height / (bounds.height + 1);
			Jun2dPoint factor = new Jun2dPoint(factorX, factorY);

			// Draw the border.
			aGraphics.setColor(this.borderColor());
			aGraphics.drawRect(bounds.x, bounds.y, bounds.width - 1, bounds.height - 1);

			// Draw triangles.
			aGraphics.setColor(this.triangleColor());
			Jun2dPoint[][] triangles = voronoiProcessor.triangles();
			if (triangles.length == 0) {
				Jun2dPoint[] trivialDots = voronoiProcessor.dots();
				if (trivialDots.length == 2) {
					voronoiProcessor.displayLineOn_from_to_clip_(aGraphics, trivialDots[0]._toPoint(), trivialDots[1]._toPoint(), bounds);
				}
			}
			for (int i = 0; i < triangles.length; i++) {
				Jun2dPoint[] triangle = triangles[i];
				Point[] points = new Point[] { triangle[0].translatedBy_(factor)._toPoint(), triangle[1].translatedBy_(factor)._toPoint(), triangle[2].translatedBy_(factor)._toPoint() };
				voronoiProcessor.displayLoopOn_points_clip_(aGraphics, points, bounds);
			}
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
			}
		}

		return anImage;
	}

	/**
	 * Answer the polyline from the specified line branches the collection.
	 * 
	 * @param directedLineSegment jp.co.sra.jun.voronoi.twoD.diagram.JunDelaunay2dDiagram.LineSegment
	 * @param aCollection java.util.Set
	 * @return jp.co.sra.jun.voronoi.twoD.diagram.JunDelaunay2dDiagram.LineSegment
	 * @category private
	 */
	protected LineSegment polylineGrowFrom_branches_(LineSegment directedLineSegment, Set aCollection) {
		Jun2dPoint nextOrigin = directedLineSegment.end();
		LineSegment[] edges = (LineSegment[]) aCollection.toArray(new LineSegment[aCollection.size()]);
		for (int i = 0; i < edges.length; i++) {
			Jun2dPoint point1 = edges[i].start();
			Jun2dPoint point2 = edges[i].end();
			;
			if (point1.equals(nextOrigin)) {
				if (point2.equals(directedLineSegment.start()) == false) {
					return new LineSegment(point1, point2);
				}
			} else {
				if (point2.equals(nextOrigin)) {
					if (point1.equals(directedLineSegment.start()) == false) {
						return new LineSegment(point2, point1);
					}
				}
			}
		}
		return null;
	}

	/**
	 * Compute minimal enclosing circle.
	 * This algorithm was presented by Pr. Chrystal in the proceedings of the third meeting of the Edinburgh Mathematical Society.
	 * http://www.personal.kent.edu/~rmuhamma/Compgeometry/MyCG/CG-Applets/Center/centercli.htm
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dCircle
	 * @category private
	 */
	protected Jun2dCircle mecByChrystal() {
		final Jun2dPoint[] hullVertexes = this.convexHull().points();
		final StBlockClosure scissorsCosBlock = new StBlockClosure() {
			public Object value_value_value_(Object obj1, Object obj2, Object obj3) {
				Jun2dPoint pivot = (Jun2dPoint) obj1;
				Jun2dPoint point1 = (Jun2dPoint) obj2;
				Jun2dPoint point2 = (Jun2dPoint) obj3;
				Jun2dPoint vector1 = point1.minus_(pivot);
				Jun2dPoint vector2 = point2.minus_(pivot);
				double dotProduct = vector1.dotProduct_(vector2);
				double d1squared = vector1.dotProduct_(vector1);
				double d2squared = vector2.dotProduct_(vector2);
				return new Double(dotProduct / Math.sqrt(d1squared * d2squared));
			}
		};
		final StValueHolder maxCosHolder = new StValueHolder();
		final StValueHolder maxCosVertexHolder = new StValueHolder();
		StBlockClosure maximumViewCosBlock = new StBlockClosure() {
			public Object value_value_(Object obj2, Object obj3) {
				Jun2dPoint point1 = (Jun2dPoint) obj2;
				Jun2dPoint point2 = (Jun2dPoint) obj3;
				maxCosHolder.value_(-99.99d);
				for (int i = 0; i < hullVertexes.length; i++) {
					Jun2dPoint viewPoint = hullVertexes[i];
					if (viewPoint.equals(point1) == false && viewPoint.equals(point2) == false) {
						double cos = ((Number) scissorsCosBlock.value_value_value_(viewPoint, point1, point2)).doubleValue();
						if (cos > maxCosHolder._doubleValue()) {
							maxCosHolder.value_(cos);
							maxCosVertexHolder.value_(viewPoint);
						}
					}
				}
				return null;
			}
		};
		Jun2dPoint s1 = hullVertexes[0];
		Jun2dPoint s2 = hullVertexes[1];
		Collection inscribe = null;
		while (inscribe == null) {
			maximumViewCosBlock.value_value_(s1, s2);
			if (maxCosHolder._doubleValue() <= 0.0) {
				inscribe = new ArrayList();
				inscribe.add(s1);
				inscribe.add(s2);
			} else {
				double cos = ((Number) scissorsCosBlock.value_value_value_(s1, (Jun2dPoint) maxCosVertexHolder.value(), s2)).doubleValue();
				if (cos >= 0) {
					cos = ((Number) scissorsCosBlock.value_value_value_(s2, (Jun2dPoint) maxCosVertexHolder.value(), s1)).doubleValue();
					if (cos >= 0) {
						inscribe = new ArrayList();
						inscribe.add(maxCosVertexHolder.value());
						inscribe.add(s1);
						inscribe.add(s2);
					} else {
						s2 = (Jun2dPoint) maxCosVertexHolder.value();
					}
				} else {
					s1 = (Jun2dPoint) maxCosVertexHolder.value();
				}
			}
		}

		Jun2dCircle approximateCircle = null;
		if (inscribe.size() == 2) {
			Jun2dPoint center = s1.plus_(s2).dividedBy_(2);
			double radius = s2.distance_(s1) / 2;
			approximateCircle = new Jun2dCircle(center, radius);
		} else {
			Jun2dPoint[] points = (Jun2dPoint[]) inscribe.toArray(new Jun2dPoint[inscribe.size()]);
			Jun2dTriangle inscribeTriangle = new Jun2dTriangle(points[0], points[1], points[2]);
			approximateCircle = inscribeTriangle.circumcircle();
		}
		return this.assureEnclosureOf_in_(hullVertexes, approximateCircle);
	}

	/**
	 * Answer my triangle color.
	 * 
	 * @return java.awt.Color
	 * @category private
	 */
	protected Color triangleColor() {
		return Color.magenta;
	}

	/**
	 * Create new <code>LineSegment</code> object with specified points.
	 * 
	 * @param p1 jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param p2 jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.voronoi.twoD.diagram.JunDelaunay2dDiagram.LineSegment
	 * @category private
	 */
	protected LineSegment _createLineSegmentWith_with_(Jun2dPoint p1, Jun2dPoint p2) {
		Jun2dPoint start = p1;
		Jun2dPoint end = p2;
		if (start.x() > end.x() || (start.x() == end.x() && start.y() > end.y())) {
			start = p2;
			end = p1;
		}
		return new LineSegment(start, end);
	}
}
