package jp.co.sra.smalltalk;

import java.io.IOException;
import java.io.Writer;

/**
 * StByteArray class
 * 
 *  @author    He Weijie
 *  @created   1998/09/20 (by xiao lai)
 *  @updated   1998/10/19 (by nisinaka)
 *  @updated   2002/04/17 (by nisinaka)
 *  @version   8.9
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: StByteArray.java,v 8.10 2008/02/20 06:33:18 nisinaka Exp $
 */
public class StByteArray extends StObject {
	/** The array. */
	protected byte[] byteArray = null;

	/**
	 * Create a new instance of <code>StByteArray</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public StByteArray() {
		this(0);
	}

	/**
	 * Create a new instance of <code>StByteArray</code> and initialize it with the array of bytes.
	 * 
	 * @param bytes byte[]
	 * @category Instance creation
	 */
	public StByteArray(byte[] bytes) {
		byteArray = new byte[bytes.length];
		System.arraycopy(bytes, 0, byteArray, 0, bytes.length);
	}

	/**
	 * Create a new instance of <code>StByteArray</code> and initialize it with the array of ints.
	 * 
	 * @param ints int[]
	 * @category Instance creation
	 */
	public StByteArray(int[] ints) {
		byteArray = new byte[ints.length];

		for (int i = 0; i < ints.length; i++) {
			byteArray[i] = (byte) ints[i];
		}
	}

	/**
	 * Create a new instance of <code>StByteArray</code> and initialize it.
	 * 
	 * @param size int
	 * @category Instance creation
	 */
	public StByteArray(int size) {
		byteArray = new byte[size];

		for (int i = 0; i < size; i++) {
			byteArray[i] = 0;
		}
	}

	/**
	 * Create a new instance of <code>StByteArray</code> from a packed string.
	 * 
	 * @param packedString jp.co.sra.smalltalk.StString
	 * @return jp.co.sra.smalltalk.StByteArray
	 * @category Instance creation
	 */
	public static StByteArray FromPackedString_(String packedString) {
		int size = packedString.length();

		if (size == 0) {
			return new StByteArray();
		}

		int last = packedString.charAt(size - 1);
		int resultSize = Math.round((float) size / 4) * 3;
		// int extra;

		if (last >= 96) {
			resultSize = (resultSize - 3 + last) - 96;
			// extra = 1;
		} else {
			// extra = 0;
		}

		StByteArray result = new StByteArray(resultSize);

		if (size > 400) {
			// The fast RasterOp algorithm is not implemented yet.
			// Do it with the slower algorithm.
			SlowDecodeFromFrom_Starting_Into_(1, packedString, 1, result);
			// FastDecodeFrom_Into_(packedString, result);
			// SlowDecodeFromFrom_Starting_Into_(size - Math.round((float) extra / 16) * 12 + 1, packedString, size - Math.round(((float) extra) / 16) * 16 + 1, result);
		} else {
			SlowDecodeFromFrom_Starting_Into_(1, packedString, 1, result);
		}

		return result;
	}

	/**
	 * Decode a string into a <code>StByteArray</code>.
	 * 
	 * @param start start
	 * @param source jp.co.sra.smalltalk.StString
	 * @param index index
	 * @param dest jp.co.sra.smalltalk.StByteArray
	 * @category Private
	 */
	private static void SlowDecodeFromFrom_Starting_Into_(int start, String source, int index, StByteArray dest) {
		int w;
		int to = start;
		int from = index;
		int stop = dest.size();

		while (to <= stop) {
			w = ((source.charAt(from - 1)) & 63) << 18;
			w += (((source.charAt(from)) & 63) << 12);
			w += (((source.charAt(from + 1)) & 63) << 6);
			w += ((source.charAt(from + 2)) & 63);
			from += 4;
			dest.at_put_(to, (byte) (w >> 16));

			if (to < stop) {
				dest.at_put_(to + 1, (byte) ((w >> 8) & 255));

				if ((to + 1) < stop) {
					dest.at_put_(to + 2, (byte) (w & 255));
				}
			}

			to = to + 3;
		}
	}

	/**
	 * Convert this <code>StByteArray</code> as an array of <code>byte</code>.
	 * 
	 * @return byte[]
	 * @category converting
	 */
	public byte[] _asBytes() {
		byte[] newArray = new byte[byteArray.length];

		System.arraycopy(byteArray, 0, newArray, 0, byteArray.length);

		return newArray;
	}

	/**
	 * Convert this <code>StByteArray</code> as an array of <code>int</code>.
	 * 
	 * @return int[]
	 * @category converting
	 */
	public int[] _asInts() {
		int[] ints = new int[byteArray.length];

		for (int i = 0; i < byteArray.length; i++) {
			ints[i] = (int) byteArray[i] & 0xff;
		}

		return ints;
	}

	/**
	 * Encode the receiver into a printable string.
	 * 
	 * @return java.lang.String
	 * @category converting
	 */
	public String asPackedString() {
		int size = this.size();
		StByteArray result = new StByteArray((size + 2) / 3 * 4);

		if (size > 300) {
			// fast RasterOp algorithm is not supported.
			// use slower algorithm.
			this.slowEncodeFrom_into_startingAt_(1, result, 1);
		} else {
			this.slowEncodeFrom_into_startingAt_(1, result, 1);
		}

		byte[] bytes = result._asBytes();
		StringBuffer buffer = new StringBuffer(bytes.length);

		for (int i = 0; i < bytes.length; i++) {
			char ch = (char) bytes[i];

			if (ch == '\'') {
				buffer.append(ch);
			}

			buffer.append(ch);
		}

		return buffer.toString();
	}

	/**
	 * Answer the byte value at the specified place.
	 * 
	 * @param index int
	 * @return byte
	 * @category accessing
	 */
	public byte at_(int index) {
		return byteArray[index - 1];
	}

	/**
	 * Put the byte value at the specified place.
	 * 
	 * @param index int
	 * @param value byte
	 * @category accessing
	 */
	public void at_put_(int index, byte value) {
		byteArray[index - 1] = value;
	}

	/**
	 * Answer the size of the byte array.
	 * 
	 * @return int
	 * @category accessing
	 */
	public final int size() {
		return byteArray.length;
	}

	/**
	 * Store my string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.smalltalk.StObject#storeOn_(java.io.Writer)
	 * @category printing
	 */
	public void storeOn_(Writer aWriter) throws IOException {
		if (byteArray.length > 200) {
			aWriter.write("(ByteArray fromPackedString: '");
			aWriter.write(this.asPackedString());
			aWriter.write("')");
		} else {
			aWriter.write("#[");

			if (byteArray.length > 0) {
				aWriter.write(String.valueOf(byteArray[0] & 0xff));

				for (int i = 1; i < byteArray.length; i++) {
					aWriter.write(' ');
					aWriter.write(String.valueOf(byteArray[i] & 0xff));
				}
			}

			aWriter.write(']');
		}
	}

	/**
	 * Encode the receiver into printable characters.
	 * 
	 * @param start int
	 * @param dest jp.co.sra.smalltalk.StByteArray
	 * @param index int
	 * @category private
	 */
	private void slowEncodeFrom_into_startingAt_(int start, StByteArray dest, int index) {
		int from = start;
		int to = index;
		int stop = this.size();
		int w;

		while (from <= stop) {
			w = (this.at_(from) & 0xff) << 16;

			if (from < stop) {
				w += ((this.at_(from + 1) & 0xff) << 8);

				if ((from + 1) < stop) {
					w += (this.at_(from + 2) & 0xff);
				}
			}

			w = w ^ 8521760;
			from += 3;
			dest.at_put_(to, (byte) ((w >> 18) + 32));
			dest.at_put_(to + 1, (byte) (((w >> 12) & 63) + 32));
			dest.at_put_(to + 2, (byte) (((w >> 6) & 63) + 32));
			dest.at_put_(to + 3, (byte) ((w & 63) + 32));
			to += 4;
		}

		if ((stop % 3) != 0) {
			dest.at_put_(to - 1, (byte) (stop % 3 + 96));
		}
	}
}
