package jp.co.sra.smalltalk;

import java.awt.Point;
import java.awt.Rectangle;

/**
 * SystemInterface class
 * 
 *  @author    Hoshi Takanori
 *  @created   2001/11/30 (by Hoshi Takanori)
 *  @updated   N/A
 *  @version   8.9
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: SystemInterface.java,v 8.9 2008/02/20 06:33:18 nisinaka Exp $
 */
public final class SystemInterface {
	public static final String libraryName = "StPL";
	public static final String requiredVersion = "2.0";
	public static final int requiredBuildNo = 6;

	private static SystemInterface Current = new SystemInterface();

	static {
		try {
			System.loadLibrary(libraryName);

			String findVersion = getVersion();
			String buildPrefix = requiredVersion + " build ";
			int buildNo;

			try {
				buildNo = Integer.parseInt(findVersion.substring(buildPrefix.length()));
			} catch (NumberFormatException e) {
				buildNo = 0;
			}

			if (findVersion.startsWith(buildPrefix) && (buildNo >= requiredBuildNo)) {
				System.out.println("JNI library '" + libraryName + "' version " + findVersion);
			} else {
				System.out.println("JNI library '" + libraryName + "' version error.");
				System.out.println("Found library is version " + findVersion);
				System.out.println("Required library is version " + requiredVersion + ", build " + requiredBuildNo + " or later");
				throw new Error("JNI library '" + libraryName + "' version error");
			}
		} catch (UnsatisfiedLinkError e) {
			System.out.println("Can't find JNI library '" + libraryName + "'.");
			throw e;
		}
	}

	/**
	 * Constainer Method.
	 */
	protected SystemInterface() {
		// Singleton pattern
		super();
	}

	/**
	 * Answer the only instance of SystemInterface. Call native methods in a
	 * block of synchronized. sample:
	 * <pre>
	 *   SystemInterface StPLInterface = SystemInterface.Current();
	 *   synchronized (StPLInterface) {
	 *     pixels = StPLInterface.utilImageFromUser(dimension);
	 *   }
	 * </pre>
	 * 
	 * @return jp.co.sra.smalltalk.SystemInterface
	 */
	public static SystemInterface Current() {
		return Current;
	}

	/**
	 * Answer a rectangle from user.
	 * 
	 * @return java.awt.Rectangle
	 */
	public static Rectangle _RectangleFromUser() {
		SystemInterface StPLInterface = SystemInterface.Current();
		int[] rect;
		synchronized (StPLInterface) {
			rect = StPLInterface.utilRectangleFromUser();
		}

		if (rect != null) {
			return new Rectangle(rect[0], rect[1], rect[2], rect[3]);
		} else {
			return null;
		}
	}

	/**
	 * Answer the current mouse position.
	 * 
	 * @return java.awt.Point
	 */
	public static Point _MousePoint() {
		SystemInterface StPLInterface = SystemInterface.Current();
		int[] point;
		synchronized (StPLInterface) {
			point = StPLInterface.utilMousePoint();
		}

		if (point != null) {
			return new Point(point[0], point[1]);
		} else {
			return null;
		}
	}

	/**
	 * Set the mouse position.
	 * 
	 * @param x int
	 * @param y int
	 */
	public static void _SetMousePoint(int x, int y) {
		SystemInterface StPLInterface = SystemInterface.Current();
		synchronized (StPLInterface) {
			StPLInterface.utilSetMousePoint(x, y);
		}
	}

	/**
	 * Native Methods.
	 */
	public static native String getVersion();
	public native int[] utilImageFromUser(int[] dimension);
	public native int[] utilImageOfArea(int x, int y, int width, int height);
	public native int[] utilImageOfWholeScreen(int[] dimension);
	public native int[] utilRectangleFromUser();
	public native int utilColorSpuit();
	public native int[] utilMousePoint();
	public native void utilSetMousePoint(int x, int y);
	public native void utilBrowse(String url);
}
