/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_COLOR
#define __GLIB_COLOR

#include "glib/primitives/GString.h"
#include "glib/util/GVector.h"

/**
 * Class of which to represent a single Red-Green-Blue (RGB) color.
 *
 * It also contains a set of predefined constant static color objects
 * that should be used instead of the system define RGB-color
 * macros (if any).
 *
 * @author  Leif Erik Larsen
 * @since   1999.11.04
 */
class GColor : public GObject
{
   private:

      int rgb;

   public:

      static const GColor BLACK;
      static const GColor DBLUE;
      static const GColor DGREEN;
      static const GColor DCYAN;
      static const GColor DRED;
      static const GColor DMAGENTA;
      static const GColor DYELLOW;
      static const GColor GRAY;
      static const GColor DGRAY;
      static const GColor BLUE;
      static const GColor GREEN;
      static const GColor CYAN;
      static const GColor RED;
      static const GColor MAGENTA;
      static const GColor YELLOW;
      static const GColor WHITE;

      static GVector<int> Std16Pal;
      static GVector<int> Std256Pal;

   public:

      GColor ();
      GColor ( int red, int green, int blue );
      GColor ( int rgb );
      GColor ( const GColor& c );

      virtual ~GColor ();

   public:

      int getBlue () const;
      int getGreen () const;
      int getRed () const;
      int getRGB () const;

      /**
       * Return truye if and only if the color is a graytone color.
       * That is if all color components (red, green and blue) are 
       * exactly the same.
       *
       * @author  Leif Erik Larsen
       * @since   2004.09.14
       */
      bool isGrayTone () const;

      GColor& setBlue ( int blue );
      GColor& setGreen ( int green );
      GColor& setRed ( int red );
      GColor& setRGB ( int rgb );

      GColor& operator= ( int c );
      GColor& operator= ( const GColor& c );
      bool operator== ( int c ) const;
      bool operator== ( const GColor& c ) const;
      bool operator!= ( int c ) const;
      bool operator!= ( const GColor& c ) const;
      operator int () const;

      /**
       * Get a new color object with a darker color than this one.
       *
       * @param  step  Color intensity step value, of which to subtract from
       *               each color element.
       * @param  spin  True if we shall automatically spin to a lighter
       *               color in case the current color is black and can not
       *               get any darker, or else false if we shall ignore this
       *               and simply return a black color that is not any darker
       *               than the source. Giving true in this parameter will
       *               ensure that the returned color is always somewhat
       *               different from the source, even if the source
       *               color is black.
       */
      GColor getDarker ( int step = 0x50, bool spin = true ) const;

      /**
       * Same as {@link #getDarker}, except that this method returns
       * a lighter instead of a darker color.
       */
      GColor getLighter ( int step = 0x50, bool spin = true ) const;

      static const GVector<int>& GetStd16ColorPalette ();
      static const GVector<int>& GetStd256ColorPalette ();

      GString toString () const;

      static GColor ParseColor ( const GString& strColor, const GColor& def );
};

#endif

