/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_COMMANDPALETTE
#define __GLIB_COMMANDPALETTE

#include "glib/gui/GWindow.h"

/**
 * This class is used by <i>GDialogPanel</i> to represent a menu and a
 * toolbar in one component on screen in a compact and sentralized way.
 *
 * It sentralizes the management of a set of menu commands that logically
 * belongs to the Window Panel it self, in an an automated way.
 * This way, all Window Panels (including child panels) can have its own menu
 * bar and tool bar. In addition, all of them can also have its own default
 * Popup Menu, which will be automatically displayed by <i>GDialogPanel</i>
 * when the user right-clicks with the mouse on the area of the Window Panel.
 *
 * Since all these three types of menues are now managed by the same panel
 * (the Command Palette panel) it is much easier for the program to toggle,
 * enable and disable a specific command regardless if the command is in the
 * menu bar, the tool bar, the popup menu or all of them. The program just
 * uses {@link no.cba.ica.gui.IWindowPanel#enableCommand}
 * and {@link no.cba.ica.gui.IWindowPanel#toggleCommand} and ICA takes care of
 * the rest. This way, a command ID can actually be seen by the program code
 * as a true command ID. It really doesn't matter to the program how it is
 * represented to the user, wheter it is in a menu bar or a tool bar.
 *
 * When the program disables a command then it will automatically be disabled
 * both from the menu bar, the tool bar and the popup menu (if any). In
 * addition, if we are talking about Dialog Panels, any Push Buttons with the
 * same command ID will be automatically disabled as well.
 *
 * The program can always choose if both the menu and the toolbar should be
 * visible, or just one of them. The program can also choose to have a
 * right-click popup menu on the Command Palette. Typically, Top Level Window
 * Panels will have a menu bar, while sub-panels and Dialog Controls will
 * eventually have a popup menu.
 *
 * Another good reason for using this clas is that then we will have a
 * common code implementation of the various program menues and toolbars so
 * that we can easily change and extend them in the future. For instance,
 * maybe wee will need support for floating toolbars some time in the future.
 * Floating toolbars can easilty be implemented here without even letting the
 * program code "know" about it.
 *
 * @author  Leif Erik Larsen
 * @since   2000.01.23
 */
class GCommandPalette : public GWindow
{
   public:

      /**
       * @author  Leif Erik Larsen
       * @since   2000.01.23
       * @throws  GAddComponentException in case of any error constructing
       *                                 the new command palette window.
       */
      GCommandPalette ( GWindow& parent, 
                        const GString& name, 
                        const GString& constraints );

      virtual ~GCommandPalette ();

   private:

      /** Disable the copy constructor. */
      GCommandPalette ( const GCommandPalette& src ) {}

      /** Disable the assignment operator. */
      GCommandPalette& operator= ( const GCommandPalette& ) { return *this; }

   public:

      GMenu* getMenubar ();
      void setMenubar ( class GMenuResource& newMenu, bool useFancy, GResourceTable* res, bool topMnemonicsOn );
      void setToolbar ( class GToolbarResource& tb, GResourceTable* res = null );
};

#endif
