/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/GDialogFrame.h"
#include "glib/gui/GDialogPanel.h"
#include "glib/gui/GFocusManager.h"
#include "glib/gui/GPushButton.h"
#include "glib/gui/GExecuteDialogException.h"
#include "glib/gui/event/GKeyMessage.h"
#include "glib/gui/event/GUserMessage.h"
#include "glib/GProgram.h"
#include "glib/sys/GSystem.h"
#include "glib/exceptions/GIllegalStateException.h"

const GString GDialogFrame::DefaultFont = "9.WarpSans";
const GString GDialogFrame::DefaultMonoFont = "8.Courier";

GDialogFrame::GDialogFrame ( GDialogMessageHandler* msgProc,
                             const GDialogResource& params )
             :GFrameWindow(params.getIDString() == "" ? GString("UnnamedDialog") : params.getIDString(),
                           GString::Empty, // titleStr
                           null, // ownerWin (null, because we provide it upon dialog execution instead.)
                           0, // winStyle
                           WS2_OS2Y, // winStyle2
                           true), // isdlg
              logicalOwnerWin(null),
              topLevelPanel(null),
              currentFocus(null),
              recentFocus(null),
              focusGained_counter(0),
              dlgDismissWasCalled(false),
              wasActiveUponDismiss(false),
              modal(false),
              isIn_DoExecute(false),
              pos(params.getPos()),
              size(params.getSize()),
              breakMsgLoop(true),
              hasBeenDismissed(false),
              autoDeleteFocusMngr(false),
              focusMngr(null),
              stayInvisible(false)
{
   // OS/2 dialogs doesn't have a separate client window.
   // Therefore we must manually respect the title bar and dialog frame.
   int frameWidth = GSystem::GetSystemMetrics(GSystem::SMID_CXDLGFRAME);
   int frameHeight = GSystem::GetSystemMetrics(GSystem::SMID_CYDLGFRAME);
   int titleHeight = GSystem::GetSystemMetrics(GSystem::SMID_CYTITLEBAR);
   GInsets ins(frameHeight + titleHeight, frameWidth, frameHeight, frameWidth);
   setInsets(&ins, false);

   // We must subclass the window so that our {@link #handleWindowMessage}
   // gets called the same way as for other window classes.
   HWND hDlg = getFrame().getHWND();
   GWindow::WindowProc supMsgProc = GWindow::GenericWindowProc;
   GWindow::WindowProc defMsgProc = GWindow::SubclassWindow(hDlg, supMsgProc);
   setDefaultMsgProc(defMsgProc);

   GFocusManager* fm = new GFocusManager();
   setFocusManager(fm, true);

   // Create the top level dialog panel that will fill the
   // client area of this dialog frame window.
   GString name = params.getIDString();
   topLevelPanel = new GDialogPanel(*this, name, GString::Empty, msgProc, params);
   setAutoLayoutClientWin(topLevelPanel);
}

GDialogFrame::~GDialogFrame ()
{
   if (autoDeleteFocusMngr)
      delete focusMngr;
   focusMngr = null;
   GDialogPanel* dlg = topLevelPanel;
   topLevelPanel = null;
   delete dlg;
}

void GDialogFrame::trackContainedComponentFocusGain ( GWindow& win )
{
   // If the dialog has been dismissed then we are better ignoring this
   // event. In that case this event is probably occuring while the
   // window handle is about to be destroyed by the system.
   if (!isIn_DoExecute)
      return;

   // If the newly activated component is the same as (from our
   // point of view) the one that already has the input focus, 
   // then do nothing but return.
   if (currentFocus == &win)
      return;

   // Increment the counter, and keep a copy of the new value of the
   // counter so that we can possibly get to know if the focus has been
   // requested to change while we are still in here.
   int oldCounterValue = ++focusGained_counter;

   // Find the dialog panel that contains the low level control that is
   // about to receive focus.
   GWindow* ctrlReceive = &win;
   GDialogPanel* ctrlReceiveOwnerPanel = ctrlReceive->getOwnerDialogPanel();

   // Find the dialog panel that contains the low level control that is
   // about to loose focus. This panel will receive the first GM_FOCUSLOSS
   // message.
   GWindow* ctrlLoose = currentFocus;
   GDialogPanel* ctrlLooseOwnerPanel = null;
   if (ctrlLoose != null)
      ctrlLooseOwnerPanel = ctrlLoose->getOwnerDialogPanel();

   // In some cases when the program has manually removed a Dialog Panel
   // or a Dialog Component (e.g. from a tabbed panel) it might
   // happen that there are no component to receive CM_FOCUSLOSS. In that
   // case, this variable will be false.
   bool sendToLooser = true;

   // Find the Dialog Panel that contains both the focus-loose control and
   // the focus-set control.
   GDialogPanel* commonOwner;
   if (ctrlLooseOwnerPanel == null)
   {
      commonOwner = ctrlReceiveOwnerPanel;
   }
   else
   {
      if (ctrlReceiveOwnerPanel->isDescendant(*ctrlLooseOwnerPanel))
      {
         commonOwner = ctrlLooseOwnerPanel;
      }
      else
      if (ctrlLooseOwnerPanel->isDescendant(*ctrlReceiveOwnerPanel))
      {
         commonOwner = ctrlReceiveOwnerPanel;
      }
      else
      {
         commonOwner = ctrlReceiveOwnerPanel->getOwnerDialogPanel();
         while (commonOwner != null && !ctrlLooseOwnerPanel->isDescendant(*commonOwner))
            commonOwner = commonOwner->getOwnerDialogPanel();
         // In cases when the program has manually removed a Dialog Panel
         // or a Dialog Component (e.g. from a tabbed panel) it might
         // happen that we will find no <i>commonOwner</i>.
         if (commonOwner == null)
         {
            commonOwner = ctrlReceiveOwnerPanel;
            sendToLooser = false;
         }
      }
   }

   // We have to set the "current" Component reference here, event
   // if this should actually have been done _after_ the CM_FOCUSLOSS
   // and the DM_FOCUSLOSS messages has all been sent and handled.
   // This is because the focus messages are actually saying
   // that "the Component is _about_ to loose focus". But, if we don't
   // do this now then there are certain focus handling code that will
   // not work (?). Thus, if the program calls IDialogPanel's
   // getFocusComponentID() while handling the focus loss message then
   // the returned ID will be the ID of the Component that is about to
   // receive the focus (but it has not neccessarily actually received it
   // yet). This is probably no problem. If it tends to be a problem then
   // we have to come back here and find a solution.
   currentFocus = &win;
   if (currentFocus == recentFocus)
      return;

   // --------------------------------------------------------------------
   // First, send all messages that have to do with notification about
   // focus loss.

   // GM_FOCUSLOSS
   // ~~~~~~~~~~~~
   // The chain starts with the originally focused control it self and
   // walks upward in the nested dialog panels until we reach the panel
   // that contains both the focus-lost and the focus-set control.
   if (ctrlLoose != null && sendToLooser)
   {
      GWindow* loose = ctrlLoose;
      GDialogPanel* owner = loose->getOwnerDialogPanel();
      while (owner != null)
      {
         GWindow* p = ctrlReceive;
         for (;;)
         {
            GDialogPanel* pp = p->getOwnerDialogPanel();
            if (pp == null || pp == owner)
               break;
            p = pp;
         }

         if (p == loose)
            break;

         GString name2 = GDialogFrame::GetChildComponentNameFor(*owner, p);
         GString name1 = GDialogFrame::GetChildComponentNameFor(*owner, loose);
         if (name1 != name2)
            owner->sendDialogMessage(GDialogMessageHandler::GM_FOCUSLOSS, name1, name2);
         if (oldCounterValue != focusGained_counter)
            return; // Event handler forced the focus away. Respect it!

         loose = owner;
         owner = owner->getOwnerDialogPanel();
         if (loose == commonOwner)
            break;
      }
   }

   // --------------------------------------------------------------------
   // Then, send all messages that have to do with notification about
   // focus received.

   recentFocus = currentFocus;

   // Update the 'recentFocus' parameter in all the stacked Dialog Panels.
   {
      GWindow* child = ctrlReceive;
      GDialogPanel* parent = child->getOwnerDialogPanel();
      while (parent != null)
      {
         parent->recentFocus = child;
         child = parent;
         parent = child->getOwnerDialogPanel();
      }
   }

   // Create the stack of Dialog Panels of which to use when sending
   // the GM_FOCUSSET messages.
   int numLevels = 1;
   GArray<GWindow> panelStack(16);
   {
      GWindow* panel = ctrlReceive;
      while (commonOwner != panel)
      {
         panel = panel->getOwnerDialogPanel();
         if (panel == null)
            break;
         numLevels += 1;
      }

      // Fill the array with "dummy" items, so we can loop backward
      // when actually setting the item values.
      for (int i=0; i<numLevels; i++)
      {
         GWindow* dummy = ctrlReceive; // Must use a panel != null.
         panelStack.add(dummy, false);
      }

      panel = ctrlReceive;
      for (int i=numLevels-1; i>=0; i--)
      {
         panelStack.set(i, panel, false);
         panel = panel->getOwnerDialogPanel();
      }
   }

   // GM_FOCUSSET
   // ~~~~~~~~~~~
   // The chain starts with the panel that contains both the focus-lost and
   // the focus-set control and walks downward in the sub-nested dialog
   // panels until we reach the requested focused control it self.
   for (int i=0; i<numLevels-1; i++)
   {
      GDialogPanel* dlg = dynamic_cast<GDialogPanel*>(&panelStack[i]);
      if (dlg != null)
      {
         GString name2;
         if (ctrlLoose != null)
         {
            GWindow* p = ctrlLoose;
            GWindow* owner = &panelStack[i];
            for (;;)
            {
               GDialogPanel* pp = p->getOwnerDialogPanel();
               if (pp == null || pp == owner)
                  break;
               p = pp;
            }

            if (p == &panelStack[i+1])
               continue;

            if (owner == p->getOwnerDialogPanel())
               name2 = GDialogFrame::GetChildComponentNameFor(*dlg, p);
         }
         GString name1 = GDialogFrame::GetChildComponentNameFor(*dlg, &panelStack[i+1]);
         if (name1 != name2)
            dlg->sendDialogMessage(GDialogMessageHandler::GM_FOCUSSET, name1, name2);
      }
      if (oldCounterValue != focusGained_counter)
         return; // Event handler forced the focus away. Respect it!
   }
}

GString GDialogFrame::GetChildComponentNameFor ( GDialogPanel& owner, GWindow* child )
{
   GWindow& ca = owner.getClientArea();
   while (child != null)
   {
      GWindow* p = child->getParentWindow();
      if (p == &ca)
         return child->getName();
      child = p;
   }
   return GString::Empty;
}

GWindowMessage::Answer EXPENTRY GDialogFrame::DefaultDlgProc ( HWND hDlg,
                                                               GWindowMessage::IDType msg,
                                                               GWindowMessage::Param1 mp1,
                                                               GWindowMessage::Param2 mp2 )
{
   return ::WinDefDlgProc(hDlg, msg, mp1, mp2);
}

/**
 * To be used locally by {@link GDialogFrame#modalMessageLoop()} only.
 * @author  Leif Erik Larsen
 * @since   2006.10.28
 */
class ModalMessageLoopFinaliser
{
   public:
   GDialogFrame& dlgf;

   ModalMessageLoopFinaliser ( GDialogFrame& dlgf ) : dlgf(dlgf)
   {
      // Make the owner window force activation of the modal dialog while
      // the dialog executes. That is, if the user e.g. clicks on the 
      // owner window in an attemt to activate it, the owner window will
      // transfer the activation to the owned modal dialog box instead.
      GWindow* owner = dlgf.getOwnerWindow();
      if (owner != null)
      {
         owner->addOwnedModalDialog(dlgf);
         owner->setEnabled(false);
      }
   }

   ~ModalMessageLoopFinaliser () 
   { 
      GWindow* owner = dlgf.getOwnerWindow();
      if (owner != null)
      {
         owner->removeOwnedModalDialog(dlgf);
         owner->setEnabled(true, false);
      }
      dlgf.setVisible(false);
      if (dlgf.wasActiveUponDismiss && owner != null)
         owner->setActive(true);
      // Some applications on OS/2 have problems with a "living" sub-dialog
      // that is not visible (dismissed, but which HWND is still valid).
      // This is true e.g. for the XPager component of XWorkplace, and for 
      // the "screen capture" feature of PMView. Thus, "disconnect" our 
      // dialog window handle from its owner so these applications don't 
      // "see" it. We will reset the correct owner when the dialog 
      // is re-executed. I have not seen that this is actually needed on 
      // Windows, but it should not hurt since we always explicitly sets 
      // the owner window upon {@link #execute} anyway. 
      // ---
      // dlgf.setOwnerWindow(null, false);
      // ---
      // LEL 2006.12.13: The above statement was commented out because 
      // it caused some problems with dialog frame window being deactivated
      // after a modal subdialog was dismissed and hidden. I tested if 
      // the above mentioned problems are still true for the newest 
      // versions of PMView and XPager. XPager seems to have been fixed,
      // but not PMView. However, for PMView it is only a problem when 
      // screen capturing the window content of Larsen Commander. Other 
      // program works fine, and capturing the whole screen works fine.
      // However, doing the statement asynchronously 
      // seems to work OK...
      dlgf.postUserMessage("GClearOwnerWindow");
   }
};

bool GDialogFrame::onUserMessage ( GUserMessage& msg )
{
   GString param1 = msg.getParam1String();
   if (param1 = "GClearOwnerWindow")
   {
      setOwnerWindow(null, false);
      return true;
   }
   else
   {
      return GFrameWindow::onUserMessage(msg);
   }
}

void GDialogFrame::modalMessageLoop ()
{
   ModalMessageLoopFinaliser finaliser(*this);

   QMSG msg;
   while (!breakMsgLoop)
   {
      if (!::WinGetMsg(GProgram::hAB, &msg, 0, 0, 0))
         break;
      ::WinDispatchMsg(GProgram::hAB, &msg);
   }

   // The finiliser code gets executed here, by the destructor of 
   // local class "ModalMessageLoopFinaliser".
}

GDialogPanel& GDialogFrame::getDialogPanel () 
{ 
   if (topLevelPanel == null)
      gthrow_(GIllegalStateException("No top-level panel"));
   return *topLevelPanel; 
}

GWindow* GDialogFrame::getFocusedWindow () const 
{ 
   return currentFocus; 
}

bool GDialogFrame::isAlive () const
{
   return !hasBeenDismissed;
}

bool GDialogFrame::isExecutingAsDialog () const
{
   return isIn_DoExecute;
}

void GDialogFrame::setFocusManager ( GFocusManager* fm, bool autoDelete )
{
   // Delete the previous focus manager (if any)
   if (autoDeleteFocusMngr)
      delete focusMngr;

   // Activate the new focus manager
   this->focusMngr = fm;
   this->autoDeleteFocusMngr = autoDelete;
}

bool GDialogFrame::onKeyDown ( const GKeyMessage& key )
{
   switch (key.getCode())
   {
      case GKey::KEY_ESCAPE:
           onClose();
           return true;

      default:
           return GFrameWindow::onKeyDown(key);
   }
}

bool GDialogFrame::onClose ()
{
   if (topLevelPanel->sendDialogMessage(GDialogMessageHandler::GM_CLOSEWIN))
      return true;

   GPushButton* esc = topLevelPanel->getEscapePushButton();
   if (esc != null)
      esc->doClick();

   return true;
}

const GString& GDialogFrame::getDismissArgument ()
{
   return dismissArg;
}

void GDialogFrame::setEnableCloseButton ( bool flag )
{
   HWND hwnd = getHWND();
   HWND hwndSysMenu = ::WinWindowFromID(hwnd, FID_SYSMENU);
   ::WinSendMsg(hwndSysMenu, MM_SETITEMATTR, MPFROM2SHORT(SC_CLOSE, true), MPFROM2SHORT(flag?0:MIA_DISABLED, flag?0:MIA_DISABLED));
}

void GDialogFrame::setDialogSize ( double dlgWidth, double dlgHeight )
{
   size.setSize(dlgWidth, dlgHeight);
   int winW = (int) (dlgWidth * topLevelPanel->getFontWidth());
   int winH = (int) (dlgHeight * topLevelPanel->getFontHeight());
   int frameWidth = GSystem::GetSystemMetrics(GSystem::SMID_CXDLGFRAME);
   int frameHeight = GSystem::GetSystemMetrics(GSystem::SMID_CYDLGFRAME);
   int titleHeight = GSystem::GetSystemMetrics(GSystem::SMID_CYTITLEBAR);
   int w = winW + 2*frameWidth;
   int h = winH + 2*frameHeight + titleHeight;
   if (isToolbarVisible())
      h += getPreferredToolbarHeight();
   if (isKeybarVisible())
      h += getPreferredKeybarHeight();
   if (isStatusbarVisible())
      h += getPreferredStatusbarHeight();
   setWindowSize(w, h);
   layout();
}

const GComponentSize& GDialogFrame::getDialogSize ()
{
   return size;
}

void GDialogFrame::setDialogPosition ( double xpos, double ypos )
{
   pos.setPos(xpos, ypos);
}

void GDialogFrame::setStayInvisible ( bool yes )
{
   stayInvisible = yes;
}

bool GDialogFrame::isModal () const
{
   return modal;
}

void GDialogFrame::dismiss ( const GString& arg )
{
   dlgDismissWasCalled = true; // Remember that this method has been called
   if (breakMsgLoop)
      return; // This dialog has already been dismissed

   dismissArg = arg;
   breakMsgLoop = true;
   isInDismiss = true;
   hasBeenDismissed = true;
   wasActiveUponDismiss = isActive();

   // Make sure that there is at least one more message in the message queue,
   // so that the message loop get a chance to see that the dialog has been
   // dismissed and {@link #modalMessageLoop} will therefore break.
   postMessage(WM_NULL);

   // ---
   topLevelPanel->dismiss(arg);
   recentFocus = currentFocus;
   
   // Send the DM_WRITEPROFILE message to top level dialog panels only.
   topLevelPanel->hasSent_GM_INITDIALOG = true; // Temporarily set it back on.
   topLevelPanel->sendDialogMessage(GDialogMessageHandler::GM_WRITEPROFILE);
   topLevelPanel->hasSent_GM_INITDIALOG = false;

   // ---
   isIn_DoExecute = false;
   isInDismiss = false;

   if (!isModal())
      setVisible(false);

   modal = false;
}

void GDialogFrame::adjustWindowPosOnScreen ( int xpos, int ypos )
{
   GPoint opos;
   GWindow* ownerw = logicalOwnerWin;
   if (ownerw != null)
      opos = ownerw->getWindowPosOnScreen();

   if (xpos == 0 && ypos == 0)
   {
      // Center the dialog within its owner window.
      GDimension pixsize = getWindowSize(); // Get size in pixels.
      GDimension osize;
      if (ownerw != null)
         osize = ownerw->getWindowSize();
      else
         osize = GSystem::GetScreenSize();
      xpos = opos.x + (osize.width/2) - (pixsize.width/2);
      ypos = opos.y + (osize.height/2) - (pixsize.height/2);
   }
   else
   {
      // Position the dialog with respect to its owner window.
      xpos += opos.x;
      ypos += opos.y;
   }

   setWindowPosOnScreen(xpos, ypos, true);
   layout();
}

void GDialogFrame::setVisible ( bool flag )
{
   GFrameWindow::setVisible(flag);
   if (flag)
      updateWindow();
}

void GDialogFrame::execute ( GWindow* ownerWin, bool modal, GObject* initArg )
{
   dismissArg = "";
   dlgDismissWasCalled = false;
   breakMsgLoop = false;
   isIn_DoExecute = true;
   hasBeenDismissed = false;
   wasActiveUponDismiss = false;
   this->modal = modal;

   // Remember which window was specified by the caller to be the
   // owner window.
   logicalOwnerWin = ownerWin;

   // Set the owner of the dialog box.
   if (ownerWin != null)
   {
      GWindow* ownerTop = &ownerWin->getTopLevelWindow();
      GFrameWindow* frame = dynamic_cast<GFrameWindow*>(ownerTop);
      if (frame != null)
         ownerTop = &frame->getFrame();
      setOwnerWindow(ownerTop, false);
   }

   // Place the Dialog with respect to the Parent Window (if any).
   int xpos = int(pos.x * topLevelPanel->getFontWidth());
   int ypos = int(pos.y * topLevelPanel->getFontHeight());
   setDialogSize(size.width, size.height); // Set size in "dialog-characters".
   adjustWindowPosOnScreen(xpos, ypos);

   // Prepare the panel, but don't show it yet.
   if (!topLevelPanel->prepareForExecution(initArg, &breakMsgLoop))
   {
      isIn_DoExecute = false;
      gthrow_(GExecuteDialogException("Failed to execute the Dialog"));
   }

   // Send the DM_QUERYPROFILE message to top level dialog panels only.
   topLevelPanel->sendDialogMessage(GDialogMessageHandler::GM_QUERYPROFILE);
   if (breakMsgLoop)
      return; // Program did call dismiss() during GM_QUERYPROFILE.

   // Return without showing the dialog if dismiss() was called
   // during the handling of GM_INIDIALOG for the top level dialog panel.
   if (breakMsgLoop)
   {
      isIn_DoExecute = false;
      return;
   }

   // Now its finally time to show the dialog window on screen, but only if
   // program has not called {@link #setStayInvisible} with a true argument.
   if (!stayInvisible)
      setVisible(true);

   // The frame is now visible.
   // Make sure that the keyboard focus is initially correct.
   if (recentFocus == null)
   {
      if (!topLevelPanel->activateNextEditableComponent(true)) // First, try to skip any OILY components
         topLevelPanel->activateNextEditableComponent(false);  // In an emergency, accept OILY components
   }
   else
   {
      recentFocus->grabFocus();
   }

   // Clear the "dismissed" state of the dialog frame window, in case
   // this is not the first time that this dialog is executed.
   // This is needed in order to make sure that the method
   // {@link #modalMessageLoop} does not return until the dialog
   // is actually dismissed again.
   USHORT flags = ::WinQueryWindowUShort(getHWND(), QWS_FLAGS);
   flags &= ~FF_DLGDISMISSED;
   ::WinSetWindowUShort(getHWND(), QWS_FLAGS, flags);

   if (modal)
   {
      modalMessageLoop();
      isIn_DoExecute = false;
   }
}

const GString& GDialogFrame::executeModal ( GWindow* ownerWin, GObject* initArg )
{
   execute(ownerWin, true, initArg);
   return getDismissArgument();
}

GDialogPanel& GDialogFrame::executeModeless ( GWindow* ownerWin, GObject* initArg )
{
   execute(ownerWin, false, initArg);
   return *topLevelPanel;
}
