/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/GFocusManager.h"
#include "glib/gui/GDialogPanel.h"

void GFocusManager::focusNextComponent ( GWindow* aComponent, bool skipOily )
{
   if (aComponent != null)
   {
      GWindow* fc = aComponent;
      GWindow* root = getFocusRoot(fc);
      if (root != null)
      {
         GWindow* nc = getFocusableComponentAfter(fc, root, true, skipOily);
         if (nc != null)
            nc->grabFocus();
      }
   }
}

void GFocusManager::focusPreviousComponent ( GWindow* aComponent, bool skipOily )
{
   if (aComponent != null)
   {
      GWindow* fc = aComponent;
      GWindow* root = getFocusRoot(fc);

      if (root != null)
      {
         GWindow* nc = getFocusableComponentAfter(fc, root, false, skipOily);
         if (nc != null)
            nc->grabFocus();
      }
   }
}

GWindow* GFocusManager::getFocusableComponentAfter ( GWindow* focusedComponent, GWindow* rootContainer, bool moveForward, bool skipOily )
{
   bool canGoDown = true;
   GWindow* nextComponent = focusedComponent;
   GWindow* initialComponent = focusedComponent;

   for (;;)
   {
      if (moveForward)
         nextComponent = getNextComponent(nextComponent, rootContainer, canGoDown);
      else
         nextComponent = getPreviousComponent(nextComponent, rootContainer, skipOily);

      if (nextComponent == null)
         break;

      if (nextComponent == initialComponent)
         break;

      canGoDown = true;
      if (nextComponent->isVisible() &&
          nextComponent->isFocusable() &&
          nextComponent->isEnabled())
      {
         if (skipOily)
         {
            if (nextComponent->isOily())
            {
               canGoDown = false;
               continue;
            }
            GDialogPanel* pane = nextComponent->getOwnerDialogPanel();
            if (pane != null && pane->isOily())
            {
               canGoDown = false;
               continue;
            }
         }
         break;
      }
   }

   if (nextComponent == focusedComponent)
      return null;
   else
      return nextComponent;
}

GWindow* GFocusManager::getNextComponent ( GWindow* component, GWindow* root, bool canGoDown )
{
   if (component == null)
      return null;

   if (canGoDown && component->isVisible() &&
       component->getChildWindowCount() > 0)
   {
      return getFirstComponent(component);
   }
   else
   {
      GWindow* parent = component->getParentWindow();
      if (parent == null)
         return null;
      GWindow* nsv = getComponentAfter(parent, component);
      if (nsv != null)
         return nsv;
      if (parent == root)
         return root;
      else
         return getNextComponent(parent, root, false);
   }
}

GWindow* GFocusManager::getPreviousComponent ( GWindow* component,
                                               GWindow* root,
                                               bool skipOily )
{
   if (component == null)
      return null;

   GWindow* parent = component->getParentWindow();
   if (parent == null)
      return null;

   if (component == root)
   {
      return getDeepestLastComponent(root);
   }
   else
   {
      GWindow* nsv = getComponentBefore(parent, component, skipOily);
      if (nsv != null)
         return getDeepestLastComponent(nsv);
      else
         return parent;
   }
}

GWindow* GFocusManager::getDeepestLastComponent ( GWindow* component )
{
   if (component == null)
      return null;

   if (component->isVisible() &&
       component->getChildWindowCount() > 0)
   {
      return getDeepestLastComponent(getLastComponent(component));
   }
   else
      return component;
}

GWindow* GFocusManager::getFocusRoot ( GWindow* c )
{
   for (GWindow* p=c->getParentWindow(); p != null; p=p->getParentWindow())
      if (p->isFocusCycleRoot())
         return p;
   return null;
}

GWindow* GFocusManager::getFirstComponent ( GWindow* aContainer )
{
   if (aContainer == null)
      return null;
   int num = aContainer->getChildWindowCount();
   if (num > 0)
      return &aContainer->getChildWindow(0);
   else
      return null;
}

GWindow* GFocusManager::getLastComponent ( GWindow* aContainer )
{
   if (aContainer == null)
      return null;
   int num = aContainer->getChildWindowCount();
   if (num > 0)
      return &aContainer->getChildWindow(num - 1);
   else
      return null;
}

GWindow* GFocusManager::getComponentBefore ( GWindow* aContainer,
                                             GWindow* aComponent,
                                             bool skipOily )
{
   if (aContainer == null)
      return null;

   // ---
   int num = aContainer->getChildWindowCount();
   for (int i=1; i<num; i++)
   {
      if (&aContainer->getChildWindow(i) == aComponent)
      {
         if (!skipOily)
            return &aContainer->getChildWindow(i-1);
         else
         for (;;)
         {
            if (i <= 0)
               break;
            GWindow& win = aContainer->getChildWindow(--i);
            if (win.isOily())
               continue;
            else
               return &win;
         }
         break;
      }
   }

   // ---
   return null;
}

GWindow* GFocusManager::getComponentAfter ( GWindow* aContainer, GWindow* aComponent )
{
   if (aContainer == null)
      return null;

   // ---
   int num = aContainer->getChildWindowCount();
   for (int i=0; i<num-1; i++)
      if (&aContainer->getChildWindow(i) == aComponent)
         return &aContainer->getChildWindow(i+1);

   // ---
   return null;
}
