/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/GSpinner.h"
#include "glib/gui/GDialogPanel.h"
#include "glib/gui/event/GKeyMessage.h"
#include "glib/gui/layout/GBorderLayout.h"
#include "glib/util/GLog.h"
#include "glib/sys/GSystem.h"

GSpinner::GSpinner ( const GString& name,
                     const GString& constraints,
                     GWindow& parentWin,
                     long winStyle,
                     long winStyle2,
                     ADJUST adjust,
                     int min,
                     int max,
                     int step,
                     const GArray<GString>* stringMap )
         :GWindow(name,
                  constraints, 
                  &parentWin, 
                  &parentWin,
                  winStyle,
                  winStyle2 | WS2_AUTO_PAINT_BACKGROUND | WS2_OS2Y | WS2_NOT_STATIC),
          peer(*this, adjust),
          minValue(min),
          maxValue(max),
          stepValue(step),
          stringMap(stringMap == null ? null : new GArray<GString>(*stringMap))
{
   peer.setFocusable(true);
   if (stepValue != 1 && stringMap == null)
   {
      // We can support stepValue != 1 on OS/2 only if there are no strings
      // to display instead of values. In order for the stepValue to apply
      // we must do our own handling of SPBN_UPARROW and SPBN_DOWNARROW. This
      // is achieved by setting the "Spin Button Master" to NULLHANDLE.
      peer.sendMessage(SPBM_SETMASTER, GWindowMessage::Param1(NULLHANDLE), 0);
   }
   else
   {
      stepValue = 1;
   }
   peer.sendMessage(SPBM_SETLIMITS, GWindowMessage::Param1(max), GWindowMessage::Param2(min));
   int numStrings = (stringMap != null ? stringMap->getCount() : 0);
   if (numStrings > 0 && numStrings == (max - min + 1))
   {
      const char** arr = new const char*[numStrings];
      for (int i=0; i<numStrings; i++)
         arr[i] = stringMap->get(i).cstring();
      peer.sendMessage(SPBM_SETARRAY, MPARAM(&arr[0]), MPARAM(numStrings));
      delete [] arr;
   }
   setLayoutManager(new GBorderLayout(), true);
}

GSpinner::~GSpinner ()
{
}

GSpinner::Peer::Peer ( GSpinner& owner, GSpinner::ADJUST adjust )
               :GWindow(GString::Empty,
                        GBorderLayout::CENTER, 
                        &owner, 
                        &owner,
                        WS_VISIBLE | SPBS_MASTER | SPBS_ALLCHARACTERS | SPBS_FASTSPIN |
                           ((adjust == GSpinner::LEFT) ? SPBS_JUSTLEFT : 
                           ((adjust == GSpinner::CENTER) ? SPBS_JUSTCENTER : SPBS_JUSTRIGHT)),
                        WS2_DEFAULTPAINT | WS2_OS2Y | WS2_NOT_STATIC,
                        GWindowClass::SPINNER),
                owner(owner)
{
}

GSpinner::Peer::~Peer ()
{
}

bool GSpinner::onNotify ( int ctrlID, int notifyID, int data, int& sysAnswerToReturn )
{
   if (ctrlID != peer.getWindowID())
      return false;

   switch (notifyID)
   {
      case SPBN_CHANGE: // The Spinner Value has been changed.
         fireValueChangeListeners();
         break;

      case SPBN_UPARROW:
         if (stepValue != 1)
         {
            // When stepValue != 1 on OS/2 we have set up the spin control 
            // so that we must handle the SPBN_UPARROW and SPBN_DOWNARROW
            // notifications our self. Else OS/2 does this for us.
            spinUp();
         }
         break;

      case SPBN_DOWNARROW:
         if (stepValue != 1)
         {
            // When stepValue != 1 on OS/2 we have set up the spin control 
            // so that we must handle the SPBN_UPARROW and SPBN_DOWNARROW
            // notifications our self. Else OS/2 does this for us.
            spinDown();
         }
         break;
   }

   return true;
}

void GSpinner::grabFocus ( bool force )
{
   peer.grabFocus(force);
}

bool GSpinner::onKeyDown ( const GKeyMessage& key )
{
   switch (key.getCode())
   {
      case GKey::KEY_CTRL_UP:
         spinUp();
         return true;

      case GKey::KEY_CTRL_DOWN:
         spinDown();
         return true;

      default:
         return GWindow::onKeyDown(key);
   }
}

void GSpinner::setEnabled ( bool flag, bool repaint )
{
   peer.setEnabled(flag, false);
   GWindow::setEnabled(flag, repaint);
}

void GSpinner::setOily ( bool flag )
{
   peer.setOily(flag);
   GWindow::setOily(flag);
}

int GSpinner::spinUp ()
{
   int val = getCurrentValue();
   val += stepValue;
   int mod = val % stepValue;
   val -= mod;
   if (val > getMaximumValue())
      val = getMinimumValue();
   setValue(val);
   return val;
}

int GSpinner::spinDown ()
{
   int val = getCurrentValue();
   val -= stepValue;
   int mod = val % stepValue;
   if (mod != 0)
      val += stepValue - mod;
   if (val < getMinimumValue())
      val = getMaximumValue();
   setValue(val);
   return val;
}

void GSpinner::setMaximumValue ( int max )
{
   if (max == maxValue)
      return;
   maxValue = max;
   peer.sendMessage(SPBM_SETLIMITS, MPARAM(max), MPARAM(minValue));
}

void GSpinner::setMinimumValue ( int min )
{
   if (min == minValue)
      return;
   minValue = min;
   peer.sendMessage(SPBM_SETLIMITS, MPARAM(maxValue), MPARAM(min));
}

int GSpinner::getMaximumValue () const 
{ 
   return maxValue; 
}

int GSpinner::getMinimumValue () const 
{ 
   return minValue; 
}

int GSpinner::getCurrentValue () const
{
   LONG val = getMinimumValue() - 1;
   peer.sendMessage(SPBM_QUERYVALUE, MPFROMP(&val), MPFROM2SHORT(0, SPBQ_ALWAYSUPDATE));
   if (stringMap != null)
   {
      // OS/2 always uses zero-based text array index as the "current value"
      // for spinners that uses a text array as its display values. Thus,
      // we must convert it into the user specified minValue-based index.
      val += minValue;
   }
   return val;
}

bool GSpinner::isEmpty () const
{
   return false;
}

void GSpinner::changeValue ( const GString& newValue, bool notify )
{
   int newVal;
   try {
      newVal = GInteger::ParseInt(newValue);
   } catch (GNumberFormatException& /*e*/) {
      return; // Not a number!
   }
   enterCtrlChangedFilter();
   setValue(newVal);
   exitCtrlChangedFilter();
   if (notify)
      fireValueChangeListeners();
}

GString GSpinner::queryValue () const
{
   int val = getCurrentValue();
   return GInteger::ToString(val);
}

void GSpinner::setValue ( int value, bool notify )
{
   if (value < minValue || value > maxValue)
      return;
   enterCtrlChangedFilter();
   LONG val = value;
   peer.sendMessage(SPBM_SETCURRENTVALUE, MPARAM(val), MPARAM(0));
   exitCtrlChangedFilter();
   if (notify)
      fireValueChangeListeners();
}

bool GSpinner::valueIsLegal ( int value )
{
   if (value >= minValue && value <= maxValue)
      return true;
   else
      return false;
}
