/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/event/GKeyMessage.h"
#include "glib/primitives/GCharacter.h"
#include "glib/exceptions/GIllegalArgumentException.h"
#include "glib/sys/GSystem.h"

GKeyMessage::GKeyMessage ( const GWindowMessage& msg )
            :up(false),
             alt(false),
             ctrl(false),
             shift(false),
             numKeypadKey(false),
             repeat(0),
             scanCode(0),
             chr('\0'),
             vk(0),
             keyCode(GKey::KEY_UNKNOWN),
             nameIsDirty(true),
             keyCodeIsDirty(true),
             originalMsg(msg)
{
   int id = msg.getID();

   if (id != WM_CHAR)
      gthrow_(GIllegalArgumentException(""));
   int flags = LOUSHORT(msg.getParam1Int());
   up = ((flags & KC_KEYUP) != 0);
   alt = ((flags & KC_ALT) != 0);
   ctrl = ((flags & KC_CTRL) != 0);
   shift = ((flags & KC_SHIFT) != 0);
   repeat = LOBYTE(HIUSHORT(msg.getParam1Int()));
   scanCode = HIBYTE(HIUSHORT(msg.getParam1Int()));
   if (((flags & KC_VIRTUALKEY) != 0))
      vk = HIUSHORT(msg.getParam2Int());
   else
      chr = LOUSHORT(msg.getParam2Int());
   numKeypadKey = scanCode == 93  || // PrintScreen
          scanCode == 70  || // Scroll Lock
          scanCode == 95  || // Pause
          scanCode == 104 || // Insert
          scanCode == 96  || // Home
          scanCode == 98  || // PageUp
          scanCode == 105 || // Delete
          scanCode == 101 || // End
          scanCode == 103 || // PageDown
          scanCode == 99  || // Left
          scanCode == 97  || // Up
          scanCode == 102 || // Down
          scanCode == 100 || // Right
          scanCode == 69  || // NumLock
          scanCode == 92  || // /
          scanCode == 55  || // *
          scanCode == 74  || // -
          scanCode == 78  || // +
          scanCode == 90  || // Enter
          scanCode == 83  || // ,
          scanCode == 82  || // 0
          scanCode == 79  || // 1
          scanCode == 80  || // 2
          scanCode == 81  || // 3
          scanCode == 75  || // 4
          scanCode == 76  || // 5
          scanCode == 77  || // 6
          scanCode == 71  || // 7
          scanCode == 72  || // 8
          scanCode == 73;    // 9
   // If numlock is on, handle key as numeric character instead of arrow.
   bool numLock = GSystem::IsShiftKeyToggledOn(GSystem::SK_NUM_LOCK);
   if (numLock && numKeypadKey)
   {
      if (scanCode == 83  || // ,
          scanCode == 82  || // 0
          scanCode == 79  || // 1
          scanCode == 80  || // 2
          scanCode == 81  || // 3
          scanCode == 75  || // 4
          scanCode == 76  || // 5
          scanCode == 77  || // 6
          scanCode == 71  || // 7
          scanCode == 72  || // 8
          scanCode == 73)    // 9
      {
         vk = 0;
         chr = LOUSHORT(msg.getParam2Int());
      }
   }

   keyCode = getCode();
   keyCodeIsDirty = false;

   name = getName();
   nameIsDirty = false;
}

GKeyMessage::~GKeyMessage ()
{
}

GKey::KeyCode GKeyMessage::getCode () const
{
   if (!keyCodeIsDirty)
      return keyCode;

   if (!alt && !shift && !ctrl)
   {
      switch (vk)
      {
         case VK_F1: return KEY_F1;
         case VK_F2: return KEY_F2;
         case VK_F3: return KEY_F3;
         case VK_F4: return KEY_F4;
         case VK_F5: return KEY_F5;
         case VK_F6: return KEY_F6;
         case VK_F7: return KEY_F7;
         case VK_F8: return KEY_F8;
         case VK_F9: return KEY_F9;
         case VK_F10: return KEY_F10;
         case VK_F11: return KEY_F11;
         case VK_F12: return KEY_F12;
         case VK_DOWN: return KEY_DOWN;
         case VK_UP: return KEY_UP;
         case VK_LEFT: return KEY_LEFT;
         case VK_RIGHT: return KEY_RIGHT;
         case VK_PAGEUP: return KEY_PAGEUP;
         case VK_PAGEDOWN: return KEY_PAGEDOWN;
         case VK_HOME: return KEY_HOME;
         case VK_END: return KEY_END;
         case VK_INSERT: return KEY_INSERT;
         case VK_DELETE: return KEY_DELETE;
         case VK_BACKSPACE: return KEY_BACKSPACE;
         case VK_ESC: return KEY_ESCAPE;
         case VK_SPACE: return KEY_SPACE;
         case VK_TAB: return KEY_TABULATOR;
         case VK_ENTER: return KEY_ENTER;
         case VK_BACKTAB: return KEY_SHIFT_TABULATOR;
         case VK_NEWLINE: return KEY_ENTER;
      }
      switch (GCharacter::ToUpperCase(chr))
      {
         case 'A': return KEY_A;
         case 'B': return KEY_B;
         case 'C': return KEY_C;
         case 'D': return KEY_D;
         case 'E': return KEY_E;
         case 'F': return KEY_F;
         case 'G': return KEY_G;
         case 'H': return KEY_H;
         case 'I': return KEY_I;
         case 'J': return KEY_J;
         case 'K': return KEY_K;
         case 'L': return KEY_L;
         case 'M': return KEY_M;
         case 'N': return KEY_N;
         case 'O': return KEY_O;
         case 'P': return KEY_P;
         case 'Q': return KEY_Q;
         case 'R': return KEY_R;
         case 'S': return KEY_S;
         case 'T': return KEY_T;
         case 'U': return KEY_U;
         case 'V': return KEY_V;
         case 'W': return KEY_W;
         case 'X': return KEY_X;
         case 'Y': return KEY_Y;
         case 'Z': return KEY_Z;
         case '0': return KEY_0;
         case '1': return KEY_1;
         case '2': return KEY_2;
         case '3': return KEY_3;
         case '4': return KEY_4;
         case '5': return KEY_5;
         case '6': return KEY_6;
         case '7': return KEY_7;
         case '8': return KEY_8;
         case '9': return KEY_9;
      }
      return KEY_UNKNOWN;
   }
   else
   if (alt && !shift && !ctrl)
   {
      switch (vk)
      {
         case VK_F1: return KEY_ALT_F1;
         case VK_F2: return KEY_ALT_F2;
         case VK_F3: return KEY_ALT_F3;
         case VK_F4: return KEY_ALT_F4;
         case VK_F5: return KEY_ALT_F5;
         case VK_F6: return KEY_ALT_F6;
         case VK_F7: return KEY_ALT_F7;
         case VK_F8: return KEY_ALT_F8;
         case VK_F9: return KEY_ALT_F9;
         case VK_F10: return KEY_ALT_F10;
         case VK_F11: return KEY_ALT_F11;
         case VK_F12: return KEY_ALT_F12;
         case VK_DOWN: return KEY_ALT_DOWN;
         case VK_UP: return KEY_ALT_UP;
         case VK_LEFT: return KEY_ALT_LEFT;
         case VK_RIGHT: return KEY_ALT_RIGHT;
         case VK_PAGEUP: return KEY_ALT_PAGEUP;
         case VK_PAGEDOWN: return KEY_ALT_PAGEDOWN;
         case VK_HOME: return KEY_ALT_HOME;
         case VK_END: return KEY_ALT_END;
         case VK_INSERT: return KEY_ALT_INSERT;
         case VK_DELETE: return KEY_ALT_DELETE;
         case VK_BACKSPACE: return KEY_ALT_BACKSPACE;
         case VK_ESC: return KEY_ALT_ESCAPE;
         case VK_SPACE: return KEY_ALT_SPACE;
         case VK_TAB: return KEY_ALT_TABULATOR;
         case VK_ENTER: return KEY_ALT_ENTER;
         case VK_NEWLINE: return KEY_ALT_ENTER;
      }
      switch (GCharacter::ToUpperCase(chr))
      {
         case 'A': return KEY_ALT_A;
         case 'B': return KEY_ALT_B;
         case 'C': return KEY_ALT_C;
         case 'D': return KEY_ALT_D;
         case 'E': return KEY_ALT_E;
         case 'F': return KEY_ALT_F;
         case 'G': return KEY_ALT_G;
         case 'H': return KEY_ALT_H;
         case 'I': return KEY_ALT_I;
         case 'J': return KEY_ALT_J;
         case 'K': return KEY_ALT_K;
         case 'L': return KEY_ALT_L;
         case 'M': return KEY_ALT_M;
         case 'N': return KEY_ALT_N;
         case 'O': return KEY_ALT_O;
         case 'P': return KEY_ALT_P;
         case 'Q': return KEY_ALT_Q;
         case 'R': return KEY_ALT_R;
         case 'S': return KEY_ALT_S;
         case 'T': return KEY_ALT_T;
         case 'U': return KEY_ALT_U;
         case 'V': return KEY_ALT_V;
         case 'W': return KEY_ALT_W;
         case 'X': return KEY_ALT_X;
         case 'Y': return KEY_ALT_Y;
         case 'Z': return KEY_ALT_Z;
         case '0': return KEY_ALT_0;
         case '1': return KEY_ALT_1;
         case '2': return KEY_ALT_2;
         case '3': return KEY_ALT_3;
         case '4': return KEY_ALT_4;
         case '5': return KEY_ALT_5;
         case '6': return KEY_ALT_6;
         case '7': return KEY_ALT_7;
         case '8': return KEY_ALT_8;
         case '9': return KEY_ALT_9;
      }
      return KEY_UNKNOWN;
   }
   else
   if (!alt && shift && !ctrl)
   {
      switch (vk)
      {
         case VK_F1: return KEY_SHIFT_F1;
         case VK_F2: return KEY_SHIFT_F2;
         case VK_F3: return KEY_SHIFT_F3;
         case VK_F4: return KEY_SHIFT_F4;
         case VK_F5: return KEY_SHIFT_F5;
         case VK_F6: return KEY_SHIFT_F6;
         case VK_F7: return KEY_SHIFT_F7;
         case VK_F8: return KEY_SHIFT_F8;
         case VK_F9: return KEY_SHIFT_F9;
         case VK_F10: return KEY_SHIFT_F10;
         case VK_F11: return KEY_SHIFT_F11;
         case VK_F12: return KEY_SHIFT_F12;
         case VK_DOWN: return KEY_SHIFT_DOWN;
         case VK_UP: return KEY_SHIFT_UP;
         case VK_LEFT: return KEY_SHIFT_LEFT;
         case VK_RIGHT: return KEY_SHIFT_RIGHT;
         case VK_PAGEUP: return KEY_SHIFT_PAGEUP;
         case VK_PAGEDOWN: return KEY_SHIFT_PAGEDOWN;
         case VK_HOME: return KEY_SHIFT_HOME;
         case VK_END: return KEY_SHIFT_END;
         case VK_INSERT: return KEY_SHIFT_INSERT;
         case VK_DELETE: return KEY_SHIFT_DELETE;
         case VK_BACKSPACE: return KEY_SHIFT_BACKSPACE;
         case VK_ESC: return KEY_SHIFT_ESCAPE;
         case VK_SPACE: return KEY_SHIFT_SPACE;
         case VK_TAB: return KEY_SHIFT_TABULATOR;
         case VK_BACKTAB: return KEY_SHIFT_TABULATOR;
         case VK_ENTER: return KEY_SHIFT_ENTER;
         case VK_NEWLINE: return KEY_SHIFT_ENTER;
      }
      switch (GCharacter::ToUpperCase(chr))
      {
         case 'A': return KEY_SHIFT_A;
         case 'B': return KEY_SHIFT_B;
         case 'C': return KEY_SHIFT_C;
         case 'D': return KEY_SHIFT_D;
         case 'E': return KEY_SHIFT_E;
         case 'F': return KEY_SHIFT_F;
         case 'G': return KEY_SHIFT_G;
         case 'H': return KEY_SHIFT_H;
         case 'I': return KEY_SHIFT_I;
         case 'J': return KEY_SHIFT_J;
         case 'K': return KEY_SHIFT_K;
         case 'L': return KEY_SHIFT_L;
         case 'M': return KEY_SHIFT_M;
         case 'N': return KEY_SHIFT_N;
         case 'O': return KEY_SHIFT_O;
         case 'P': return KEY_SHIFT_P;
         case 'Q': return KEY_SHIFT_Q;
         case 'R': return KEY_SHIFT_R;
         case 'S': return KEY_SHIFT_S;
         case 'T': return KEY_SHIFT_T;
         case 'U': return KEY_SHIFT_U;
         case 'V': return KEY_SHIFT_V;
         case 'W': return KEY_SHIFT_W;
         case 'X': return KEY_SHIFT_X;
         case 'Y': return KEY_SHIFT_Y;
         case 'Z': return KEY_SHIFT_Z;
         case '0': return KEY_SHIFT_0;
         case '1': return KEY_SHIFT_1;
         case '2': return KEY_SHIFT_2;
         case '3': return KEY_SHIFT_3;
         case '4': return KEY_SHIFT_4;
         case '5': return KEY_SHIFT_5;
         case '6': return KEY_SHIFT_6;
         case '7': return KEY_SHIFT_7;
         case '8': return KEY_SHIFT_8;
         case '9': return KEY_SHIFT_9;
      }
      return KEY_UNKNOWN;
   }
   else
   if (!alt && !shift && ctrl)
   {
      switch (vk)
      {
         case VK_F1: return KEY_CTRL_F1;
         case VK_F2: return KEY_CTRL_F2;
         case VK_F3: return KEY_CTRL_F3;
         case VK_F4: return KEY_CTRL_F4;
         case VK_F5: return KEY_CTRL_F5;
         case VK_F6: return KEY_CTRL_F6;
         case VK_F7: return KEY_CTRL_F7;
         case VK_F8: return KEY_CTRL_F8;
         case VK_F9: return KEY_CTRL_F9;
         case VK_F10: return KEY_CTRL_F10;
         case VK_F11: return KEY_CTRL_F11;
         case VK_F12: return KEY_CTRL_F12;
         case VK_DOWN: return KEY_CTRL_DOWN;
         case VK_UP: return KEY_CTRL_UP;
         case VK_LEFT: return KEY_CTRL_LEFT;
         case VK_RIGHT: return KEY_CTRL_RIGHT;
         case VK_PAGEUP: return KEY_CTRL_PAGEUP;
         case VK_PAGEDOWN: return KEY_CTRL_PAGEDOWN;
         case VK_HOME: return KEY_CTRL_HOME;
         case VK_END: return KEY_CTRL_END;
         case VK_INSERT: return KEY_CTRL_INSERT;
         case VK_DELETE: return KEY_CTRL_DELETE;
         case VK_BACKSPACE: return KEY_CTRL_BACKSPACE;
         case VK_ESC: return KEY_CTRL_ESCAPE;
         case VK_SPACE: return KEY_CTRL_SPACE;
         case VK_TAB: return KEY_CTRL_TABULATOR;
         case VK_ENTER: return KEY_CTRL_ENTER;
         case VK_NEWLINE: return KEY_CTRL_ENTER;
      }
      switch (GCharacter::ToUpperCase(chr))
      {
         case 'A': return KEY_CTRL_A;
         case 'B': return KEY_CTRL_B;
         case 'C': return KEY_CTRL_C;
         case 'D': return KEY_CTRL_D;
         case 'E': return KEY_CTRL_E;
         case 'F': return KEY_CTRL_F;
         case 'G': return KEY_CTRL_G;
         case 'H': return KEY_CTRL_H;
         case 'I': return KEY_CTRL_I;
         case 'J': return KEY_CTRL_J;
         case 'K': return KEY_CTRL_K;
         case 'L': return KEY_CTRL_L;
         case 'M': return KEY_CTRL_M;
         case 'N': return KEY_CTRL_N;
         case 'O': return KEY_CTRL_O;
         case 'P': return KEY_CTRL_P;
         case 'Q': return KEY_CTRL_Q;
         case 'R': return KEY_CTRL_R;
         case 'S': return KEY_CTRL_S;
         case 'T': return KEY_CTRL_T;
         case 'U': return KEY_CTRL_U;
         case 'V': return KEY_CTRL_V;
         case 'W': return KEY_CTRL_W;
         case 'X': return KEY_CTRL_X;
         case 'Y': return KEY_CTRL_Y;
         case 'Z': return KEY_CTRL_Z;
         case '0': return KEY_CTRL_0;
         case '1': return KEY_CTRL_1;
         case '2': return KEY_CTRL_2;
         case '3': return KEY_CTRL_3;
         case '4': return KEY_CTRL_4;
         case '5': return KEY_CTRL_5;
         case '6': return KEY_CTRL_6;
         case '7': return KEY_CTRL_7;
         case '8': return KEY_CTRL_8;
         case '9': return KEY_CTRL_9;
      }
      return KEY_UNKNOWN;
   }
   else
   if (!alt && shift && ctrl)
   {
      switch (vk)
      {
         case VK_F1: return KEY_SHIFT_CTRL_F1;
         case VK_F2: return KEY_SHIFT_CTRL_F2;
         case VK_F3: return KEY_SHIFT_CTRL_F3;
         case VK_F4: return KEY_SHIFT_CTRL_F4;
         case VK_F5: return KEY_SHIFT_CTRL_F5;
         case VK_F6: return KEY_SHIFT_CTRL_F6;
         case VK_F7: return KEY_SHIFT_CTRL_F7;
         case VK_F8: return KEY_SHIFT_CTRL_F8;
         case VK_F9: return KEY_SHIFT_CTRL_F9;
         case VK_F10: return KEY_SHIFT_CTRL_F10;
         case VK_F11: return KEY_SHIFT_CTRL_F11;
         case VK_F12: return KEY_SHIFT_CTRL_F12;
         case VK_DOWN: return KEY_SHIFT_CTRL_DOWN;
         case VK_UP: return KEY_SHIFT_CTRL_UP;
         case VK_LEFT: return KEY_SHIFT_CTRL_LEFT;
         case VK_RIGHT: return KEY_SHIFT_CTRL_RIGHT;
         case VK_PAGEUP: return KEY_SHIFT_CTRL_PAGEUP;
         case VK_PAGEDOWN: return KEY_SHIFT_CTRL_PAGEDOWN;
         case VK_HOME: return KEY_SHIFT_CTRL_HOME;
         case VK_END: return KEY_SHIFT_CTRL_END;
         case VK_INSERT: return KEY_SHIFT_CTRL_INSERT;
         case VK_DELETE: return KEY_SHIFT_CTRL_DELETE;
         case VK_BACKSPACE: return KEY_SHIFT_CTRL_BACKSPACE;
         case VK_ESC: return KEY_SHIFT_CTRL_ESCAPE;
         case VK_SPACE: return KEY_SHIFT_CTRL_SPACE;
         case VK_TAB: return KEY_SHIFT_CTRL_TABULATOR;
         case VK_ENTER: return KEY_SHIFT_CTRL_ENTER;
         case VK_NEWLINE: return KEY_SHIFT_CTRL_ENTER;
      }
      switch (GCharacter::ToUpperCase(chr))
      {
         case 'A': return KEY_SHIFT_CTRL_A;
         case 'B': return KEY_SHIFT_CTRL_B;
         case 'C': return KEY_SHIFT_CTRL_C;
         case 'D': return KEY_SHIFT_CTRL_D;
         case 'E': return KEY_SHIFT_CTRL_E;
         case 'F': return KEY_SHIFT_CTRL_F;
         case 'G': return KEY_SHIFT_CTRL_G;
         case 'H': return KEY_SHIFT_CTRL_H;
         case 'I': return KEY_SHIFT_CTRL_I;
         case 'J': return KEY_SHIFT_CTRL_J;
         case 'K': return KEY_SHIFT_CTRL_K;
         case 'L': return KEY_SHIFT_CTRL_L;
         case 'M': return KEY_SHIFT_CTRL_M;
         case 'N': return KEY_SHIFT_CTRL_N;
         case 'O': return KEY_SHIFT_CTRL_O;
         case 'P': return KEY_SHIFT_CTRL_P;
         case 'Q': return KEY_SHIFT_CTRL_Q;
         case 'R': return KEY_SHIFT_CTRL_R;
         case 'S': return KEY_SHIFT_CTRL_S;
         case 'T': return KEY_SHIFT_CTRL_T;
         case 'U': return KEY_SHIFT_CTRL_U;
         case 'V': return KEY_SHIFT_CTRL_V;
         case 'W': return KEY_SHIFT_CTRL_W;
         case 'X': return KEY_SHIFT_CTRL_X;
         case 'Y': return KEY_SHIFT_CTRL_Y;
         case 'Z': return KEY_SHIFT_CTRL_Z;
         case '0': return KEY_SHIFT_CTRL_0;
         case '1': return KEY_SHIFT_CTRL_1;
         case '2': return KEY_SHIFT_CTRL_2;
         case '3': return KEY_SHIFT_CTRL_3;
         case '4': return KEY_SHIFT_CTRL_4;
         case '5': return KEY_SHIFT_CTRL_5;
         case '6': return KEY_SHIFT_CTRL_6;
         case '7': return KEY_SHIFT_CTRL_7;
         case '8': return KEY_SHIFT_CTRL_8;
         case '9': return KEY_SHIFT_CTRL_9;
      }
      return KEY_UNKNOWN;
   }
   else
   {
      return KEY_UNKNOWN;
   }
}

const GWindowMessage& GKeyMessage::getOriginalMessage () const
{
   return originalMsg;
}

bool GKeyMessage::isAlphaChar ( bool evenIfShift ) const
{
   if (!evenIfShift)
      if (isAltDown() || isCtrlDown())
         return false;
   if (isalpha(chr))
      return true;
   else
      return false;
}

bool GKeyMessage::isDigitChar ( bool evenIfShift ) const
{
   if (!evenIfShift)
      if (isAltDown() || isCtrlDown())
         return false;
   if (isdigit(chr))
      return true;
   else
      return false;
}

bool GKeyMessage::isNumericKeypadKey () const
{
   return numKeypadKey;
}

bool GKeyMessage::isPureCharacter () const
{
   return chr != '\0' && !alt && !ctrl;
}

char GKeyMessage::getCharacter () const 
{ 
   return chr; 
}

int GKeyMessage::getRepeat () const 
{ 
   return repeat; 
}

int GKeyMessage::getScanCode () const 
{ 
   return scanCode; 
}

int GKeyMessage::getVirtualKey () const 
{ 
   return vk; 
}

bool GKeyMessage::isAnyShiftKeyDown () const
{
   return isAltDown() || isCtrlDown() || isShiftDown();
}

bool GKeyMessage::isAltDown () const 
{ 
   return alt; 
}

bool GKeyMessage::isCtrlDown () const 
{ 
   return ctrl; 
}

bool GKeyMessage::isKeyDown () const 
{ 
   return !up; 
}

bool GKeyMessage::isKeyUp () const 
{ 
   return up; 
}

bool GKeyMessage::isShiftDown () const 
{ 
   return shift; 
}

const GString& GKeyMessage::getName () const
{
   if (!nameIsDirty)
      return name;

   switch (vk)
   {
      case VK_F1: name = "VK_F1"; break;
      case VK_F2: name = "VK_F2"; break;
      case VK_F3: name = "VK_F3"; break;
      case VK_F4: name = "VK_F4"; break;
      case VK_F5: name = "VK_F5"; break;
      case VK_F6: name = "VK_F6"; break;
      case VK_F7: name = "VK_F7"; break;
      case VK_F8: name = "VK_F8"; break;
      case VK_F9: name = "VK_F9"; break;
      case VK_F10: name = "VK_F10"; break;
      case VK_F11: name = "VK_F11"; break;
      case VK_F12: name = "VK_F12"; break;
      case VK_DOWN: name = "VK_DOWN"; break;
      case VK_UP: name = "VK_UP"; break;
      case VK_LEFT: name = "VK_LEFT"; break;
      case VK_RIGHT: name = "VK_RIGHT"; break;
      case VK_PAGEUP: name = "VK_PAGEUP"; break;
      case VK_PAGEDOWN: name = "VK_PAGEDOWN"; break;
      case VK_HOME: name = "VK_HOME"; break;
      case VK_END: name = "VK_END"; break;
      case VK_INSERT: name = "VK_INSERT"; break;
      case VK_DELETE: name = "VK_DELETE"; break;
      case VK_BACKSPACE: name = "VK_BACKSPACE"; break;
      case VK_ESC: name = "VK_ESC"; break;
      case VK_SPACE: name = "VK_SPACE"; break;
      case VK_TAB: name = "VK_TAB"; break;
      case VK_NUMLOCK: name = "VK_NUMLOCK"; break;

      case VK_BACKTAB:
         name = "VK_TAB";
         break;

      case VK_NEWLINE:
      case VK_ENTER:
         name = "VK_ENTER";
         break;

      case VK_PRINTSCRN: name = "VK_PRINTSCREEN"; break;
      case VK_SCRLLOCK: name = "VK_SCROLLLOCK"; break;
      case VK_SYSRQ: name = "VK_PAUSE"; break;
   }

   if (name == "" && chr != '\0')
      name += GCharacter::ToUpperCase(chr);

   if (isCtrlDown())
      name += "_C";

   if (isAltDown())
      name += "_A";

   if (isShiftDown())
      name += "_S";

   if (isNumericKeypadKey())
      name += "_N";

   if (isKeyUp())
      name += "_U";

   return name;
}
