/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include <limits.h>
#include "glib/gui/layout/GBorderLayout.h"
#include "glib/gui/GWindow.h"
#include "glib/primitives/GInteger.h"
#include "glib/util/GMath.h"

const GString GBorderLayout::CENTER = "Center";
const GString GBorderLayout::NORTH = "North";
const GString GBorderLayout::SOUTH = "South";
const GString GBorderLayout::WEST = "West";
const GString GBorderLayout::EAST = "East";

GBorderLayout::GBorderLayout ()
{
}

GBorderLayout::~GBorderLayout ()
{
}

GDimension GBorderLayout::preferredLayoutSize ( GWindow& parent )
{
   int w = preferredLayoutWidth(parent);
   int h = preferredLayoutHeight(parent);
   return GDimension(w, h);
}

int GBorderLayout::preferredLayoutWidth ( GWindow& parent )
{
   GWindow* center = parent.getChildWindowByLayoutConstraints(CENTER);
   GWindow* north = parent.getChildWindowByLayoutConstraints(NORTH);
   GWindow* south = parent.getChildWindowByLayoutConstraints(SOUTH);
   GWindow* west = parent.getChildWindowByLayoutConstraints(WEST);
   GWindow* east = parent.getChildWindowByLayoutConstraints(EAST);

   double widthNorth = 0;
   if (north != null)
      widthNorth = north->getPreferredWidth();

   double widthSouth = 0;
   if (south != null)
      widthSouth = south->getPreferredWidth();

   double widthInter = 0;
   if (west != null)
      widthInter += west->getPreferredWidth();
   if (east != null)
      widthInter += east->getPreferredWidth();
   if (center != null)
      widthInter += center->getPreferredWidth();
   if (widthInter > INT_MAX)
      widthInter = INT_MAX;

   return (int) GMath::Max(widthNorth, GMath::Max(widthSouth, widthInter));
}

int GBorderLayout::preferredLayoutHeight ( GWindow& parent )
{
   GWindow* center = parent.getChildWindowByLayoutConstraints(CENTER);
   GWindow* north = parent.getChildWindowByLayoutConstraints(NORTH);
   GWindow* south = parent.getChildWindowByLayoutConstraints(SOUTH);
   GWindow* west = parent.getChildWindowByLayoutConstraints(WEST);
   GWindow* east = parent.getChildWindowByLayoutConstraints(EAST);

   double heightWest = 0;
   if (west != null)
      heightWest = west->getPreferredHeight();

   double heightEast = 0;
   if (east != null)
      heightEast = east->getPreferredHeight();

   double heightInter = 0;
   if (north != null)
      heightInter += north->getPreferredHeight();
   if (south != null)
      heightInter += south->getPreferredHeight();
   if (center != null)
      heightInter += center->getPreferredHeight();
   if (heightInter > GInteger::MAX_VALUE)
      heightInter = GInteger::MAX_VALUE;

   return (int) GMath::Max(heightWest, GMath::Max(heightEast, heightInter));
}

void GBorderLayout::layoutWindow ( GWindow& parent )
{
   GWindow* center = parent.getChildWindowByLayoutConstraints(CENTER);
   GWindow* north = parent.getChildWindowByLayoutConstraints(NORTH);
   GWindow* south = parent.getChildWindowByLayoutConstraints(SOUTH);
   GWindow* west = parent.getChildWindowByLayoutConstraints(WEST);
   GWindow* east = parent.getChildWindowByLayoutConstraints(EAST);

   GDimension parentSize = parent.getWindowSize();
   const GInsets& ins = parent.getInsets();
   int top = parentSize.height - 1 - ins.top;
   int bottom = ins.bottom;
   int left = ins.left;
   int right = parentSize.width - 1 - ins.right;
   int countWindows = 0;

   if (north != null)
   {
      int h = north->getPreferredHeight();
      top -= h;
      north->setWindowBounds(left, top + 1, right - left + 1, h);
      countWindows++;
   }

   if (south != null)
   {
      int h = south->getPreferredHeight();
      south->setWindowBounds(left, bottom, right - left + 1, h);
      bottom += h;
      countWindows++;
   }

   if (east != null)
   {
      int w = east->getPreferredWidth();
      east->setWindowBounds(right - w + 1, bottom, w, top - bottom + 1);
      right -= w;
      countWindows++;
   }

   if (west != null)
   {
      int w = west->getPreferredWidth();
      west->setWindowBounds(left, bottom, w, top - bottom + 1);
      left += w;
      countWindows++;
   }

   // Use the first child, if no child was found with any of the
   // supported constraints ID's.
   if (center == null && countWindows == 0 && parent.getChildWindowCount() >= 1)
      center = &parent.getChildWindow(0);

   // Layout the center window.
   if (center != null)
      center->setWindowBounds(left, bottom, right - left + 1, top - bottom + 1);
}
