/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/tree/GDefaultTreeModel.h"
#include "glib/gui/tree/GMutableTreeNode.h"
#include "glib/gui/tree/GTreeModelEvent.h"
#include "glib/gui/tree/GTreePath.h"
#include "glib/exceptions/GIllegalArgumentException.h"

GDefaultTreeModel::GDefaultTreeModel ( GMutableTreeNode* root, bool autoDeleteRoot )
                  :root(root),
                   autoDeleteRoot(autoDeleteRoot),
                   asksAllowsChildren(false),
                   listenerList(4)
{
   if (root == null)
      gthrow_(GIllegalArgumentException("Root node cannot be null!"));
}

GDefaultTreeModel::~GDefaultTreeModel ()
{
   if (autoDeleteRoot)
      delete root;
}

void GDefaultTreeModel::setAsksAllowsChildren ( bool newValue ) 
{
   asksAllowsChildren = newValue;
}

bool GDefaultTreeModel::getAsksAllowsChildren () const 
{
   return asksAllowsChildren;
}

void GDefaultTreeModel::setRoot ( GMutableTreeNode* root, bool autoDeleteRoot ) 
{
   GTreeNode* oldRoot = this->root;
   bool oldAutoDeleteRoot = this->autoDeleteRoot;
   this->autoDeleteRoot = autoDeleteRoot;
   this->root = root;
   if (root == oldRoot)
      return;
   if (oldAutoDeleteRoot)
      delete oldRoot;
}

GTreeNode* GDefaultTreeModel::getRoot () 
{
   return root;
}

int GDefaultTreeModel::getIndexOfChild ( const GTreeNode& parent, const GTreeNode& child ) const
{
   return parent.getIndex(child);
}

GTreeNode& GDefaultTreeModel::getChild ( GTreeNode& parent, int index ) 
{
   return parent.getChildAt(index);
}

int GDefaultTreeModel::getChildCount ( const GTreeNode& parent ) const
{
   return parent.getChildCount();
}

bool GDefaultTreeModel::isLeaf ( const GTreeNode& node ) const
{
   if (asksAllowsChildren)
      return !node.getAllowsChildren();
   return node.isLeaf();
}

void GDefaultTreeModel::valueForPathChanged ( GTreePath& path, GObject* newValue, bool autoDelete ) 
{
	GMutableTreeNode& node = dynamic_cast<GMutableTreeNode&>(path.getLastPathComponent());
   node.setUserObject(newValue, autoDelete);
   nodeChanged(node);
}

void GDefaultTreeModel::insertNodeInto ( GMutableTreeNode* newChild, GMutableTreeNode& parent, int index, bool autoDeleteChild )
{
   parent.insert(newChild, index, autoDeleteChild);
   GVector<int> newIndexs(1);
   newIndexs.add(index);
   nodesWereInserted(parent, newIndexs);
}

void GDefaultTreeModel::removeNodeFromParent ( GTreeNode& node, bool doDestroyNode ) 
{
   GMutableTreeNode* parent = dynamic_cast<GMutableTreeNode*>(node.getParent());
   if (parent == null)
      gthrow_(GIllegalArgumentException("node does not have a parent."));
   GVector<int> childIndex(1);
   int index = parent->getIndex(node);
   childIndex.add(index);
   bool autoDelete = parent->isAutoDeleteChildAt(index);
   parent->remove(index, false);
   GArray<GTreeNode> removedArray(1);
   removedArray.add(node);
   nodesWereRemoved(*parent, childIndex, removedArray);
   if (doDestroyNode && autoDelete)
      delete &node;
}

void GDefaultTreeModel::nodeChanged ( const GTreeNode& node ) 
{
   const GTreeNode* parent = const_cast<GTreeNode&>(node).getParent();
   if (parent != null) 
   {
      int anIndex = parent->getIndex(node);
      if (anIndex >= 0) 
      {
         GVector<int> cIndexs(1);
         cIndexs.add(anIndex);
         nodesChanged(*parent, cIndexs);
      }
   }
	else 
   if (&node == getRoot()) 
   {
      GVector<int> empty(0);
		nodesChanged(node, empty);
	}
}

void GDefaultTreeModel::nodesWereInserted ( GTreeNode& parent, const GVector<int>& childIndexes ) 
{
   int cCount = childIndexes.getCount();
   if (cCount > 0) 
   {
      GArray<GTreeNode> newChildren(cCount);
      for (int counter = 0; counter < cCount; counter++)
      {
         GTreeNode& child = parent.getChildAt(childIndexes[counter]);
         newChildren.add(&child, false);
      }
      const GTreePath& path = parent.getPath();
      fireTreeNodesInserted(path, childIndexes, newChildren);
   }
}
    
void GDefaultTreeModel::nodesWereRemoved ( const GTreeNode& node, 
                                           const GVector<int>& childIndexes, 
                                           GArray<GTreeNode>& removedChildren ) 
{
   const GTreePath& path = node.getPath();
   fireTreeNodesRemoved(path, childIndexes, removedChildren);
}

void GDefaultTreeModel::nodesChanged ( const GTreeNode& node, const GVector<int>& childIndexes ) 
{
	int cCount = childIndexes.getCount();
   GArray<GTreeNode> cChildren(cCount);
   for (int counter = 0; counter < cCount; counter++)
   {
      GTreeNode& child = node.getChildAt(childIndexes[counter]);
      cChildren.add(&child, false);
   }
   const GTreePath& path = node.getPath();
   fireTreeNodesChanged(path, childIndexes, cChildren);
}

void GDefaultTreeModel::addTreeModelListener ( GTreeModelListener* l ) 
{
   if (!listenerList.contains(l)) // Don't add if it is already there!
      listenerList.add(l);
}

void GDefaultTreeModel::removeTreeModelListener ( GTreeModelListener* l ) 
{
   int idx = listenerList.indexOf(l);
   if (idx >= 0)
      listenerList.remove(idx);
}

void GDefaultTreeModel::fireTreeNodesChanged ( const GTreePath& path, 
                                               const GVector<int>& childIndexes,  
                                               const GArray<GTreeNode>& children ) 
{
   // Get a working copy of the listener array, in case some listers are 
   // added or removed by the code being notifyed.
   int count = listenerList.getCount();
   GVector<GTreeModelListener*> llist(count);
   for (int i=0; i<count; i++)
      llist.add(listenerList[i]);

   // Invoke the listeners.
   const GTreeModelEvent ev(&path, childIndexes, children);
   for (int i=0; i<count; i++) 
   {
      GTreeModelListener* l = llist[i];
      l->treeNodesChanged(ev);
   }
}

void GDefaultTreeModel::fireTreeNodesInserted ( const GTreePath& path, 
                                                const GVector<int>& childIndexes,  
                                                const GArray<GTreeNode>& children ) 
{
   // Get a working copy of the listener array, in case some listers are 
   // added or removed by the code being notifyed.
   int count = listenerList.getCount();
   GVector<GTreeModelListener*> llist(count);
   for (int i=0; i<count; i++)
      llist.add(listenerList[i]);

   // Invoke the listeners.
   const GTreeModelEvent ev(&path, childIndexes, children);
   for (int i=0; i<count; i++) 
   {
      GTreeModelListener* l = llist[i];
      l->treeNodesInserted(ev);
   }
}

void GDefaultTreeModel::fireTreeNodesRemoved ( const GTreePath& path, 
                                               const GVector<int>& childIndexes,  
                                               const GArray<GTreeNode>& children )
{
   // Get a working copy of the listener array, in case some listers are 
   // added or removed by the code being notifyed.
   int count = listenerList.getCount();
   GVector<GTreeModelListener*> llist(count);
   for (int i=0; i<count; i++)
      llist.add(listenerList[i]);

   // Invoke the listeners.
   const GTreeModelEvent ev(&path, childIndexes, children);
   for (int i=0; i<count; i++) 
   {
      GTreeModelListener* l = llist[i];
      l->treeNodesRemoved(ev);
   }
}
