/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include <stdio.h>
#include "glib/io/GFileOutputStream.h"
#include "glib/util/GLog.h"
#include "glib/sys/GSystem.h"

const int GFileOutputStream::BuffSize = 512;

GFileOutputStream::GFileOutputStream ( GVfs& vfs,
                                       const GString& name, 
                                       bool createIfNew, 
                                       bool replaceIfExist,
                                       bool textMode )
                  :vfs(vfs),
                   path(name),
                   hfile(null),
                   autoClose(true),
                   textMode(textMode),
                   buffPos(0),
                   myBuff(null)
{
   myBuff = new BYTE[BuffSize];
   init(createIfNew, replaceIfExist);
}

GFileOutputStream::GFileOutputStream ( GVfs& vfs,
                                       const GFile& file, 
                                       bool createIfNew, 
                                       bool replaceIfExist,
                                       bool textMode )
                  :vfs(vfs),
                   path(file.getFullPath()),
                   hfile(null),
                   autoClose(true),
                   textMode(textMode),
                   buffPos(0),
                   myBuff(null)
{
   myBuff = new BYTE[BuffSize];
   init(createIfNew, replaceIfExist);
}

GFileOutputStream::GFileOutputStream ( GVfs& vfs,
                                       GVfs::FileHandle hfile, 
                                       bool autoClose, 
                                       bool textMode,
                                       const GString& path )
                  :vfs(vfs),
                   path(path),
                   hfile(hfile),
                   autoClose(autoClose),
                   textMode(textMode),
                   buffPos(0),
                   myBuff(null)
{
   myBuff = new BYTE[BuffSize];
}

GFileOutputStream::~GFileOutputStream ()
{
   flush();
   delete [] myBuff;
   close();
}

void GFileOutputStream::init ( bool createIfNew, bool replaceIfExist )
{
   if (!createIfNew)
      if (!vfs.existFile(path))
         gthrow_(GFileNotFoundException("Filename: " + path));
   GError err;
   hfile = vfs.openFile(path, &err, GVfs::Mode_ReadWrite, GVfs::Create_Always);
   if (hfile == null)
   {
      GString sysMsg = err.getErrorMessage();
      GString msg("%s\n\nFilename:\n%s", GVArgs(sysMsg).add(path));
      gthrow_(GOpenFileException(msg));
   }
}

void GFileOutputStream::close ()
{
   if (autoClose && hfile != null)
   {
      flush();
      vfs.closeFile(hfile);
      hfile = null;
   }
}

GVfs::FileHandle GFileOutputStream::getFileHandle ()
{
   return hfile;
}

void GFileOutputStream::flush () const
{
   if (buffPos > 0)
   {
      GError err = vfs.writeToFile(hfile, myBuff, buffPos);
      if (err != GError::Ok)
         gthrow_(GIOException(err.getErrorMessage()));
      buffPos = 0;
   }
}

void GFileOutputStream::writeByte ( int b ) const
{
   BYTE byte_ = BYTE(b);
   if (buffPos >= BuffSize)
      flush();
   myBuff[buffPos++] = byte_;
}

void GFileOutputStream::write ( const void* buff, int size, int count ) const
{
   if (textMode)
   {
      BYTE* byteBuff = (BYTE*) buff;
      for (int i=0; i<count; i++)
      {
         for (int i2=0; i2<size; i2++)
         {
            BYTE next = *byteBuff++;
            if (next == '\n')
            {
               writeByte('\r');
               writeByte('\n');
            }
            else
            {
               writeByte(next);
            }
         }
      }
   }
   else
   {
      int bytesToWrite = size * count;
      if (buffPos + bytesToWrite > BuffSize)
      {
         flush();
         int written = 0;
         GError err = vfs.writeToFile(hfile, buff, bytesToWrite, &written);
         if (written != bytesToWrite)
            gthrow_(GIOException(err.getErrorMessage()));
      }
      else
      {
         memcpy(&myBuff[buffPos], buff, bytesToWrite);
         buffPos += bytesToWrite;
      }
   }
}

int GFileOutputStream::print ( const GString& str ) const
{
   int len = str.length();
   const char* buff = str.cstring();
   write(buff, len, 1);
   return len;
}

int GFileOutputStream::printf ( const char* str, const GVArgs& args ) const
{
   GString data(str, args);
   int len = data.length();
   const char* buff = data.cstring();
   write(buff, len, 1);
   return len;
}
