/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_OUTPUTSTREAM
#define __GLIB_OUTPUTSTREAM

#include "glib/primitives/GVArgs.h"
#include "glib/io/GIOException.h"

/**
 * This is an abstract base class for all classes that provides
 * writing to an output stream.
 *
 * Only one method is pure virtual in this class, and that is the method
 * for writing one single byte to the stream ({@link #writeByte}).
 * This method is the only one that is absolutely required to be
 * implemented by the subclass.
 *
 * We provide a default implementation for all the other virtual methods
 * of this class. These default implementations dependends on the
 * {@link #writeByte} method only. Because of this, be aware that the
 * default implementations probably have relatively poor speed performance.
 * Thus, it makes sense for most subclasses to override every method from
 * this class, so that the concrete implementation can run at a maximized
 * speed.
 *
 * @author  Leif Erik Larsen
 * @since   2000.07.29
 * @see     GInputStream
 * @see     GRandomAccessFile
 */
class GOutputStream : public GObject
{
   public:

      /**
       * Flush the buffered output stream bytes, if any.
       *
       * @author  Leif Erik Larsen
       * @since   2004.12.10
       * @throws  GIOException In case of any error flushing the buffer.
       */
      virtual void flush () const = 0;

      /**
       * Write the specified integer as single byte.
       *
       * For most systems, this is the same as converting the specified
       * integer to an unsigned character, and write that character only.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @throws  GIOException In case of any error writing the byte.
       */
      virtual void writeByte ( int b ) const = 0;

      /**
       * This method has the same meaning as the ANSI C fwrite() function.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @throws  GIOException  In case of any error writing the buffer.
       */
      virtual void write ( const void* buff, int size, int count ) const;

      /**
       * Print the specified string to the file, not including the
       * terminating null character.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @return  The number of bytes written (always the string length).
       * @throws  GIOException  In case of any error writing the string.
       */
      virtual int print ( const GString& str ) const;

      /**
       * Print a formatted string, just as the ANSI C fprintf() function.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @return  The number of bytes actually written.
       * @throws  GIOException  In case of any error writing the
       *                        formatted string.
       */
      virtual int printf ( const char* str, const GVArgs& args = GVArgs() ) const;
};

#endif
