/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_PIPEINPUTSTREAM
#define __GLIB_PIPEINPUTSTREAM

#include "glib/io/GFileInputStream.h"
#include "glib/vfs/GVfsLocal.h"

/**
 * @author  Leif Erik Larsen
 * @since   2004.04.28
 */
class GPipeInputStream : public GFileInputStream
{
   public:

      typedef HPIPE HPipe;

   private:

      /** Needed by our super constructor. */
      static GVfsLocal LocalVfs;

   public:

      /**
       * Create a pipe input stream that wraps the specified system 
       * dependent pipe handle. The pipe handle must represent the 
       * read end of the pipe.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.28
       * @param   hpipe     The system dependent pipe handle of which to wrap.
       * @param   autoClose True if we shall close the handle automatically.
       */
      GPipeInputStream ( HPipe hpipe, bool autoClose );

      /**
       * Close the pipe handle stream.
       */
      virtual ~GPipeInputStream ();

   private:

      /** Disable the copy constructor. */
      GPipeInputStream ( const GPipeInputStream& src );

      /** Disable the assignment operator. */
      const GPipeInputStream& operator= ( const GPipeInputStream& src ) const { return *this; }

   public:

      /**
       * Check the number of bytes that can currently be read without
       * blocking from the pipe. That is, the number of bytes currently 
       * contained in the pipe buffer.
       * 
       * @author  Leif Erik Larsen
       * @since   2004.11.24
       * @return  Number of bytes currently available to be read from the 
       *          pipe without blocking, or zero if the pipe currently 
       *          contains no buffered bytes.
       */
      unsigned peekBytesAvailable () const;

      /**
       * Read data from the pipe, without blocking, into the specified text 
       * string. If the buffer is empty when this method is called then we 
       * will do nothing but return immediately with an empty text. Else we 
       * will read some buffered data and put in into the specified text 
       * before we return.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.28
       * @param   text       The buffer of where to store the read text.
       *                     It is always cleared by this method before we 
       *                     starts to read, so if this method reads no data 
       *                     the text will be empty upon return.
       * @param   maxLength  Maximum characters to read before return, or 0
       *                     to read with no maximum length. If 0 is 
       *                     specified then we will read until the pipe 
       *                     buffer is empty, and the text can theoretically
       *                     be very long.
       */
      void readUnblocked ( GString& text, int maxLength = 0 );
};

#endif

