/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_DOUBLE
#define __GLIB_DOUBLE

#include "glib/primitives/GFloatNumber.h"

/**
 * A general double representation class.
 *
 * @author  Leif Erik Larsen
 * @since   2000.07.29
 */
class GDouble : public GFloatNumber
{
   private:

      double value;

   public:

      static const double MIN_VALUE;
      static const double MAX_VALUE;

   public:

      GDouble () : value(0.0) {}
      GDouble ( const GDouble& val ) : value(val.value) {}
      GDouble ( double val ) : value(val) {}

      /**
       * @throws GNumberFormatException if the specified string is not
       *                                possible to parse into a double.
       */
      explicit GDouble ( const char* str );

      virtual ~GDouble ();

   public:

      operator double () const { return value; }

   public:

      virtual bool boolValue () const { return value == 0.0 ? false : true; }
      virtual char charValue () const { return char(int(value)); }
      virtual short shortValue () const { return short(value); }
      virtual int intValue () const { return int(value); }
      virtual longlong longValue () const { return longlong(value); }
      virtual unsigned short ushortValue () const { return (unsigned short) value; }
      virtual unsigned int uintValue () const { return (unsigned int) value; }
      virtual ulonglong ulongValue () const { return ulonglong(value); }
      virtual float floatValue () const { return float(value); }
      virtual double doubleValue () const { return value; }

      virtual int hashCode () const;
      virtual bool equals ( const GObject& obj ) const;

   public:

      /**
       * Get the parsed double value of the specified string.
       *
       * @param  str    String of which to parse.
       * @return The parsed double value.
       * @throws GNumberFormatException if the specified string is not
       *                                possible to parse into a double.
       */
      static double ParseDouble ( const GString& str );

      /**
       * Return a <i>GString</i> representation of the current
       * double value.
       */
      virtual GString toString () const;

      /**
       * Convert the specified double value to a string.
       *
       * @since  2000.07.29
       * @author Leif Erik Larsen
       * @param  val       Double value of which to convert.
       * @param  expForm   True if we shall produce a string in the
       *                   exponential form, or else false. If true is
       *                   specified then the E will always be upper case.
       */
      static GString ToString ( double val, bool expForm = false );
};

#endif


