/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_LONG
#define __GLIB_LONG

#include "glib/primitives/GIntegerNumber.h"

/**
 * A general long integer representation class.
 *
 * @author  Leif Erik Larsen
 * @since   2000.10.23
 */
class GLong : public GIntegerNumber<longlong>
{
   private:

      longlong value;

   public:

      static const longlong MIN_VALUE;
      static const longlong MAX_VALUE;

   public:

      GLong ();
      GLong ( const GLong& val );
      GLong ( longlong val );

      virtual ~GLong ();

   public:

      operator longlong () const { return value; }

   public:

      virtual bool boolValue () const { return value == 0 ? false : true; }
      virtual char charValue () const { return char(value); }
      virtual short shortValue () const { return short(value); }
      virtual int intValue () const { return int(value); }
      virtual longlong longValue () const { return value; }
      virtual unsigned short ushortValue () const { return (unsigned short) value; }
      virtual unsigned int uintValue () const { return (unsigned int) value; }
      virtual ulonglong ulongValue () const { return ulonglong(value); }
      virtual float floatValue () const { return float(value); }
      virtual double doubleValue () const { return double(value); }

      virtual int hashCode () const;
      virtual bool equals ( const GObject& obj ) const;

   public:

      /**
       * Return a <i>GString</i> representation of the current
       * long integer value.
       */
      virtual GString toString () const;

      /**
       * Return a <i>GString</i> representation of the specified
       * long integer value.
       */
      static GString ToString ( longlong val, int radix = 10 );

      /**
       * Get the parsed long integer value of the specified string,
       * assuming the string is specifying a long integer with the
       * specified radix.
       *
       * @param   str    String of which to parse.
       * @param   radix  0, 2, 8, 10 or 16. If 0 is specified then we
       *                 will use the default radix as with respect to
       *                 the format of the string; assuming that the
       *                 string is specifying an integer number with
       *                 radix 10, or radix 16 if the string starts
       *                 with the character sequence "0x".
       * @return  The parsed long integer value.
       * @throws  GNumberFormatException if the specified string is not
       *                                 possible to parse into a long.
       */
      static longlong ParseLong ( const GString& str, int radix = 0 );
};

#endif
