/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_OBJECT
#define __GLIB_OBJECT

#include "glib/sys/GLibSys.h"

/**
 * This class is intended to form the base class of all non-interface
 * classes that are part of the <b><i>Genious Library</i></b>.
 *
 * It is nice to have such a common base in case of any container
 * that needs to see all objects as abstract as possible. The virtual
 * destructor is another good reason for such a common base, as well as the
 * virtual {@link #toString} method which is inspired by the Java language
 * and is very useful when parsing parameter objects of unknown type.
 *
 * Finally, all objects of this type will have access to a flexible and 
 * very easy-to-use thread synchronization mechanism that is very 
 * inspired by the synchronixed-keyword of the Java language. See 
 * the documentation of the inner class {@link #Synchronizer} for more 
 * information and some samples.
 *
 * @author  Leif Erik Larsen
 * @since   1999.04.10
 * @see     #toString
 */
class GObject
{
   private:

      /** The synchronization lock for this object. */
      mutable class GCriticalSection* lock;

      /** The monitor for this object. */
      mutable class GEventSemaphore* monitor;

   public:
 
      /**
       * This inner class can be used for extremely easy and very powerful
       * synchronization of any object of type GObject, ala Java's 
       * synchronized-keyword. Consider this sample:
       *
       * <pre>
       *    GObject* myObject = ...;
       *    GObject::Synchronizer synch(myObject);
       *    myObject->doSomeThreadSynchronizedCode();
       * </pre>
       *
       * The above code is a cleaner and safer way to do the same 
       * as in the below code, because C++ gurantees that the destructor
       * of synch gets called upon scope-exit regardless of how the scope
       * is exited. That is; even in case of some C++ exception. The below 
       * code can possibly leave the synchronization lock in a locked state
       * in case of some C++ exception in doSomeThreadSynchronizedCode().
       *
       * <pre>
       *    GObject* myObject = ...;
       *    myObject->enterSynchronizationLock();
       *    myObject->doSomeThreadSynchronizedCode();
       *    myObject->exitSynchronizationLock();
       * </pre>
       *
       * In order to achive the same level of robustness without 
       * using this inner class you would have to write something like:
       *
       * <pre>
       *    GObject* myObject = ...;
       *    myObject->enterSynchronizationLock();
       *    bool ok = false;
       *    try {
       *       myObject->doSomeThreadSynchronizedCode();
       *       ok = true;
       *       myObject->exitSynchronizationLock();
       *    } catch (...) {
       *       if (!ok)
       *          myObject->exitSynchronizationLock();
       *       throw;
       *    }
       * </pre>
       *
       * This last code is both tedious and error prone so I will
       * indeed recommend using this inner class as shown in the 
       * first of the above samples.
       *
       * @author  Leif Erik Larsen
       * @since   2004.01.11
       */
      class Synchronizer
      {
         private:

            /** The object which is synchronized by us. */
            const GObject* obj;

         public:

            /** Create a synchronizer for the specified object. */
            Synchronizer ( const GObject& obj )
               :obj(&obj)
            {
               obj.enterSynchronizationLock();
            }

            /** Create a synchronizer for the specified object. */
            Synchronizer ( const GObject* obj )
               :obj(obj)
            {
               if (obj != null)
                  obj->enterSynchronizationLock();
            }

         private:

            /** Disable the copy constructor for this class. */
            Synchronizer ( const Synchronizer& src ) 
               :obj(null) 
            {
            }

         public:

            ~Synchronizer ()
            {
               if (obj != null)
                  obj->exitSynchronizationLock();
            }
      };

   private:

      /** Used to synchronize access object locks and monitors. */
      static class GCriticalSection& GetSynchLock ();

      /** Make sure that {@link #lock} is created, in a thread safe way. */
      void ensureLockObjectIsCreated () const;

      /** Make sure that {@link #monitor} is created, in a thread safe way. */
      void ensureMonitorObjectIsCreated () const;

   public:

      /** Default constructor. */
      GObject ();

      /** Copy constructor does not copy the synchronization lock. */
      GObject ( const GObject& src );

      /** Destroy the object. */
      virtual ~GObject ();

      /** Assignment operator does not affect the contained synchronization lock. */
      virtual GObject& operator= ( const GObject& src );

      /**
       * This method is used by e.g. {@link GArrayImpl#compare2Objects}
       * to compare two objects for equality, "less than" or "greater than" 
       * in one single method call. 
       *
       * The default implementation alphabetically compares the string 
       * values as returned by {@link #toString}. Most primitive classes 
       * that is typically stored in containers should override this 
       * method, for a more type-specific and optimized comparization 
       * operation.
       *
       * @author  Leif Erik Larsen
       * @since   2004.03.14
       * @return  Less than zero if "this" object is considered to be 
       *          "less than" the specified object, greater than zero 
       *          if "this" object is considered to be "greater than"
       *          the specified object, or zero if the two objects 
       *          are considered to be exactly equal.
       * @see     #equals
       */
      virtual int compareObj ( const GObject& obj ) const;

      /**
       * Enter the synchronization lock of this object.
       * It is extremely important that {@link #exitSynchronizationLock}
       * is called by the same thread that called this method, as soon as 
       * possible after the code needing synchronization has finished.
       *
       * @author  Leif Erik Larsen
       * @since   2004.01.11
       * @see     #Synchronizer
       * @see     #exitSynchronizationLock
       */
      void enterSynchronizationLock () const;

      /**
       * Exit the synchronization lock of this object.
       *
       * @author  Leif Erik Larsen
       * @since   2004.01.11
       * @see     #enterSynchronizationLock
       */
      void exitSynchronizationLock () const;

      /** 
       * This default implementation returns true only if the specified
       * object is a reference to *this.
       *
       * @author  Leif Erik Larsen
       * @see     #hashCode
       * @see     #compareObj
       */
      virtual bool equals ( const GObject& obj ) const;

      /**
       * Called my the garbage collector when the object is about to be 
       * finilized. Most objects overriding this method should not depend 
       * on any manual destruction. Instead, this method is called.
       * <p>
       * The default implementation of this method is to implicitly call 
       * the destructor of the overriding class.
       *
       * @author  Leif Erik Larsen
       * @since   2005.01.29
       */
      virtual void finalize ();

      /**
       * Returns a hash code value for the object. This method is 
       * supported for the benefit of hashtables such as those provided 
       * by {@link GHashtable}.
       * <p>
       * The general contract of <code>hashCode()</code> is: 
       * <ul>
       * <li>Whenever it is invoked on the same object more than once during 
       *     an execution of an application, the <tt>hashCode()</tt> method 
       *     must consistently return the same integer, provided no 
       *     information used in comparisons on the object is modified. 
       *     This integer need not remain consistent between different 
       *     instances (sessions) of the same application.
       * <li>If two objects are equal according to their {@link #equals}
       *     method, then calling the <tt>hashCode()</tt> method 
       *     on each of the two objects must produce the same integer 
       *     result. 
       * <li>It is <em>not</em> required that if two objects are unequal 
       *     according to their {@link #equals} method, then calling the 
       *     <tt>hashCode()</tt> method on each of the two objects must 
       *     produce distinct integer results. However, the programmer 
       *     should be aware that producing distinct integer results for
       *     unequal objects may improve the performance of hashtables.
       * </ul>
       * <p>
       * As much as is reasonably practical, the <tt>hashCode()</tt> method 
       * defined by class {@link GObject} does return distinct integers for 
       * distinct objects.
       *
       * @author  Leif Erik Larsen
       * @since   2004.03.11
       * @return  A hash code value for this object.
       * @see     GHashtable
       */
      virtual int hashCode () const;

      /**
       * TODO: Not fully implemented yet!
       * Check the monitor facilities in Java, and make GLib work the same.
       *
       * @author  Leif Erik Larsen
       * @since   2004.02.02
       */
      void notifyAll () const;

      /**
       * Return a <i>GString</i> representation of the object.
       *
       * This method is frequently overridden and the meaning of the
       * returned string therefore depends very much on which class it is
       * about.
       *
       * The default implementation returns a string that contains the
       * hexadecimal memory address of the object. This is just a default
       * implementation of this very "abstract" method. Noone should depend
       * on the string returned by this default implementation. It might
       * change in the future as well.
       */
      virtual class GString toString () const;

      /**
       * TODO: Not fully implemented yet!
       * Check the monitor facilities in Java, and make GLib work the same.
       *
       * @author  Leif Erik Larsen
       * @since   2004.02.02
       */
      void wait () const;

      /**
       * TODO: Not fully implemented yet!
       * Check the monitor facilities in Java, and make GLib work the same.
       *
       * @author  Leif Erik Larsen
       * @since   2004.02.02
       */
      void wait ( int timeout ) const;
};

/**
 * You may use the synchronization macros like this:
 * <p>
 * <pre>
 * GObject& guardedObj = *this;
 * synchronized(guardedObj)
 * {
 *    ...
 *
 * } end_synchronized
 *
 * @author Leif Erik Larsen
 * @since  2005.01.29
 */
#define synchronized(guardedObj)                     \
        {                                            \
           GObject::Synchronizer guard(guardedObj);

#define synchronized_end                             \
        }

#define end_synchronized synchronized_end

#endif
