/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_STRING
#define __GLIB_STRING

#include <string.h>
#include "glib/primitives/GObject.h"

/**
 * A general text string representation class that is used almost
 * everywhere in both the class library it self and in user code.
 *
 * By many means this string class is very similar to the <i>String</i> class
 * of the Java language. The main difference is that <i>GString</i> is not
 * read only. Thus, this string class can be seen as a hybrid between the
 * Java <i>String</i> and the C++ <i>string</i>.
 *
 * @author  Leif Erik Larsen
 * @since   1999.09.05
 */
class GString : public GObject
{
   private:

      /** The string buffer it self. Dynamically allocated from the heap. */
      char* theString;

      /** The current number of characters in the string, as seen by the user code. */
      int currentLength;

      /** The number of bytes actually allocated to the string buffer. */
      int allocatedSize;

      /** The hashcode of the string, if calculated and string not changed. */
      mutable int cachedHashCode;

   public:

      /**
       * Defines the various string trim/strip methods that are supported.
       * @see #trim
       */
      enum TrimType
      {
         /** Trim/strip all from start and end only. */
         TrimEndsOnly,

         /** Trim/strip all from start and end as well as repeating. */
         TrimEndsAndRepeating,

         /** Trim/strip all occurenses. */
         TrimAll
      };

   public:

      /** This static constant can be used in place of any blank (" ") constant string. */
      static const GString Blank;

      /** This static constant can be used in place of any empty ("") constant string. */
      static const GString Empty;

      /** This static string contains the platform dependent EOL-string (End-Of-Line) for usage in text files. */
      static const GString EOL;

      /** This static string is the default argument for {@link #trim}. */
      static const GString WhiteChars;

      /** This static string contains three dots only Usable e.g. with {@link GGraphics#getShortenedText}. */
      static const GString ThreeDots;

   public:

      /**
       * Create a new and empty string with a default initial buffer size of
       * zero characters and a default grow size factor of 1/3.
       */
      GString ();

      /**
       * Create a new and empty string with the specified initial string
       * buffer size and the specified grow size.
       *
       * @param  allocs  The number of bytes of which to initially allocate.
       */
      explicit GString ( int allocs );

      /**
       * Create a new string that initially contains the specified
       * character only.
       *
       * @author  Leif Erik Larsen
       * @since   2000.11.20
       */
      explicit GString ( char chr );

      /**
       * Create a new string that is initially equal to the specified
       * null-terminated C-string. The initial buffer size will be the length
       * of the specified source string. The grow size will be a factor of 1/3.
       *
       * @param str    The C-string of which to use as the initial
       *               string value.
       */
      GString ( const char* str );

      /**
       * Create a new string that is initially equal to the specified string.
       * The initial buffer size will be the length of the specified source
       * string. The grow size will be a factor of 1/3.
       *
       * @param str    The string of which to use as the initial string value.
       */
      GString ( const GString& str );

      /**
       * Create a new string that is initially equal to a subset of the
       * specified string. The initial buffer size will be the same as the
       * actual number of characters of the subset. The grow size will be a
       * factor of 1/3.
       *
       * @param str       The string of which to copy from.
       * @param fromIndex Start to copy from, and including, the character at
       *                  this index.
       * @param count     Copy max this number of characters.
       *                  A value of -1 means no count limit.
       */
      GString ( const GString& str, int fromIndex, int count = -1 );

      /**
       * Use this string constructor instead of using the ANCI C function
       * <i>sprintf()</i> to create a string from a variable argument list.
       *
       * The parameter list formatting is done in the same
       * way as with <i>sprintf()</i>, but is is faster and more reliable
       * because there is no practical limit to the length of the
       * destination string.
       *
       * @author  Leif Erik Larsen
       * @since   2000.10.30
       * @see     #GString(const GString&, const GVArgs&)
       */
      GString ( const char* format, const class GVArgs& args );

      /**
       * Use this string constructor to format a variable argument list
       * with a source string containing optional format tags almost like
       * the classic <i>fprintf()</i> set of functions in ANSI C.
       *
       * The implementation of this constructor can also be used as an
       * example of how the <i>GVArgs</i> class it self can be used.
       *
       * This constrcutor accepts a series of arguments, where each applies
       * to a format specifier contained in the format string pointed to by
       * <i>format</i>, and outputs the formatted data to the new string that
       * is constructed. There must be the same number of format specifiers
       * as arguments or else the result is unpredictable.
       *
       * The following format tags are supported:
       *
       * <table border cellspacing=0 cellpadding=5 width="100%" bgcolor="#CCFFFF">
       * <tr><td><b>%d</b></td><td>Signed decimal integer (Byte, Short, Integer, Long, Float, Double or "Double.parseLong(arg.toString())").</td></tr>
       * <tr><td><b>%i</b></td><td>Ditto.</td></tr>
       * <tr><td><b>%b</b></td><td>Ditto, but binary.</td></tr>
       * <tr><td><b>%o</b></td><td>Ditto, but octal.</td></tr>
       * <tr><td><b>%x</b></td><td>Ditto, but hexadecimal with lowercase a...f.</td></tr>
       * <tr><td><b>%X</b></td><td>Ditto, but uppercase A...F.</td></tr>
       * <tr><td><b>%f</b></td><td>Signed floating point (Byte, Short, Integer, Long, Float, Double or "Double.parseDouble(arg.toString())").</td></tr>
       * <tr><td><b>%g</b></td><td>Ditto.</td></tr>
       * <tr><td><b>%G</b></td><td>Ditto.</td></tr>
       * <tr><td><b>%e</b></td><td>Ditto, but in exponential form with lowercase e.</td></tr>
       * <tr><td><b>%E</b></td><td>Ditto, but uppercase E.</td></tr>
       * <tr><td><b>%c</b></td><td>Single character (Character or "arg.toString().charAt(0)").</td></tr>
       * <tr><td><b>%s</b></td><td>Character string (String or "arg.toString()").</td></tr>
       * <tr><td><b>%%</b></td><td>The % character it self.</td></tr>
       * </table>
       *
       * <b><u>Example</u></b>
       *
       * <pre>
       *    GString str("Variable %s equals %d, which is %d%% of %d",
       *                 GVArgs("X").add(11).add(10).add(110));
       * </pre>
       *
       * The above code will produce the following string in "str":
       *
       * <pre>
       *    "Variable X equals 11, which is 10% of 110"
       * </pre>
       *
       * @author  Leif Erik Larsen
       * @since   2000.10.30
       * @param   str   The string that will be used to format the
       *                variable argument list.
       * @param   args  The list of variable arguments that are to be
       *                joined into the format string in a formatted way.
       * @see     #GString(const char*, const GVArgs&)
       */
      GString ( const GString& format, const GVArgs& args );

      /**
       * Destroy the string and free all resources associated to it,
       * including the dynamic string buffer.
       */
      virtual ~GString ();

   private:

      /**
       * Common construction code for the two constructors that takes
       * a variable argument list as an instance of {@link GVArgs}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.11.01
       */
      void init ( const char* format, const int formatLen, const class GVArgs& args );
      
      /** 
       * Enlarge the string buffer with room for some more characters 
       * than the specified length.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.02
       */
      void enlargeTo ( int length );

   public:

      /**
       * Set the string value to the specified long value.
       * @return A reference to our self.
       */
      GString& operator= ( long strval );

      /**
       * Set the string value to the specified single character.
       * @return A reference to our self.
       */
      GString& operator= ( char chr );

      /**
       * Set the string value to be equal to the specified source C-string.
       * @return A reference to our self.
       */
      GString& operator= ( const char* str );

      /**
       * Set the string value to be equal to the specified source string.
       * @return A reference to our self.
       */
      GString& operator= ( const GString& str );

      /**
       * Append the specified character.
       *
       * The character can be a null-character as well, so that this method
       * can be used to build up a character buffer that contains more
       * null-characters than the terminating null-character. This is needed
       * for example on OS/2 when building up an environment variable buffer
       * that is to be inherited by a child process.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.02
       */
      GString& operator+= ( char chr );
      GString& operator+= ( signed char chr );
      GString& operator+= ( unsigned char chr );
      GString& operator+= ( float val );
      GString& operator+= ( double val );
      GString& operator+= ( int val );
      GString& operator+= ( const char* str );
      GString& operator+= ( const GString& str );
      GString& operator+= ( bool val );
      GString& operator+= ( unsigned val );
      GString& operator+= ( longlong val );
      GString& operator+= ( ulonglong val );

      /**
       * Append the string as is returned by the <i>toString()</i> method of
       * the specified object, to the string.
       *
       * @return A reference to our self.
       */
      GString& operator+= ( const GObject& obj );

      GString operator+ ( int val ) const;
      GString operator+ ( float val ) const;
      GString operator+ ( char chr ) const;
      GString operator+ ( const char* str ) const;
      GString operator+ ( const GString& str ) const;
      GString operator+ ( bool val ) const;
      GString operator+ ( unsigned val ) const;
      GString operator+ ( double val ) const;
      GString operator+ ( longlong val ) const;
      GString operator+ ( ulonglong val ) const;

      /**
       * Create a new string object that contains the current string value
       * with the string as is returned by the <i>toString()</i> method of
       * the specified object appended to it.
       * @return  A new concatenated string object.
       */
      GString operator+ ( const GObject& obj ) const;

      /**
       * Return true if and only if the specified C-string is 100% equal
       * to "this" string.
       */
      bool operator== ( const char* str ) const;

      /**
       * Return true if and only if the specified string is 100% equal
       * to "this" string.
       */
      bool operator== ( const GString& str ) const;

      /**
       * Return true if and only if the specified C-string is not 100% equal
       * to "this" string.
       */
      bool operator!= ( const char* str ) const;

      /**
       * Return true if and only if the specified string is not 100% equal
       * to "this" string.
       */
      bool operator!= ( const GString& str ) const;

      /**
       * Return the indexed character, without bounds-checking.
       *
       * @see #charAt
       */
      char operator[] ( int index ) const;

      /**
       * Return a pointer to a null terminated C-string of "this" string,
       * that can be safely used with the traditional ANSI C functions.
       *
       * In the current implementation this is a pointer to the internal
       * character buffer that is used to contain the characters of the
       * string during the lifetime of the string object.
       * But this might very well be changed in the future, so don't assume
       * this in your code.
       *
       * @see #cstring
       */
      operator const char* () const;

      /**
       * Append <i>count</i> number of the specified character.
       */
      GString& append ( char chr, int count );

      /**
       * Append the specified string, but no more than the specified
       * number of characters.
       */
      GString& appendNum ( const char *str, int num );

      /** @see #endsWith */
      bool beginsWith ( char chr, bool ignoreCase = false ) const;
      bool beginsWith ( const char* str, bool ignoreCase = false ) const;
      bool beginsWith ( const GString& str, bool ignoreCase = false ) const;

      /**
       * Return the indexed character, with bounds-checking.
       *
       * This method allows access to the character at index==currentLength,
       * in which case the terminating '\0' character is always returned.
       *
       * @throws GArrayIndexOutOfBoundsException if the specified index
       *                           is outside legal range. Legal range is
       *                           index >= 0 && index <= currentLength.
       */
      char charAt ( int index ) const;

      /**
       * Make the string empty without changing its preallocated buffer size.
       */
      GString& clear ();

      int compare ( const char* str ) const;
      int compare ( const GString& str ) const;
      int compare ( const char* str, bool ignoreCase ) const;
      int compare ( const GString& str, bool ignoreCase ) const;
      int compareIgnoreCase ( const char* str ) const;
      int compareIgnoreCase ( const GString& str ) const;

      virtual int compareObj ( const GObject& obj ) const;

      /**
       * Return true if and only if the current string value contains at least
       * one character that is equal to the specified character.
       * @see #indexOf
       */
      bool contains ( char chr ) const;

      /**
       * Return true if and only if the string contains any white space
       * characters such as spaces, linefeeds or tabs.
       * Else we will return false.
       */
      bool containsWhiteSpace () const;

      /**
       * This is a static version of the method, that can be used by any
       * code to test if a given C-string contains any whitespace characters.
       */
      static bool ContainsWhiteSpace ( const char* str );

      /**
       * Return a pointer to a null terminated C-string of "this" string,
       * that can be safely used with the traditional ANSI C functions.
       *
       * In the current implementation this is a pointer to the internal
       * character buffer that is used to contain the characters of the
       * string during the lifetime of the string object.
       * But this might very well be changed in the future, so don't assume
       * this in your code.
       */
      const char* cstring () const;

      /**
       * Remove all the characters from, and including, the indexed
       * character.
       */
      GString& cutTailFrom ( int index );

      /** @see #beginsWith */
      bool endsWith ( char chr, bool ignoreCase = false ) const;
      bool endsWith ( const char *str, bool ignoreCase = false ) const;
      bool endsWith ( const GString& str, bool ignoreCase = false ) const;

      /**
       * Test if the string ends with one or more occurences of some platform
       * dependent EOL (end-of-line) character(s). We will test for all known
       * EOL character combinations that are valid on any known operating 
       * system. Thus, this method returns true regardless if the string 
       * ends with "\r\n" (Windows/OS2), "\n" (Unix) or just "\r" (Mac).
       *
       * @author  Leif Erik Larsen
       * @since   2005.02.02
       * @see     #ensureCompatibleEolFormat
       * @see     #stripTrailingEol
       * @see     #EOL
       */
      bool endsWithEol () const;
      
      GString& ensureAllocatedSize ( int length );

      virtual bool equals ( const GObject& obj ) const;
      
      bool equalsNum ( const char *str, int num ) const;
      bool equalsNum ( const GString& str, int num ) const;
      bool equalsIgnoreCase ( const char *str ) const;
      bool equalsIgnoreCase ( const GString& str ) const;
      bool equalsIgnoreCaseNum ( const char *str, int num ) const;
      bool equalsIgnoreCaseNum ( const GString& str, int num ) const;
      bool equalsString ( const char *str, bool ignoreCase ) const;
      bool equalsString ( const GString& str, bool ignoreCase ) const;

      /**
       * Return a copy of the very first character in the string,
       * or '\0' if the string is empty.
       */
      char firstChar () const;

      /**
       * Fetch out an indexed substring part of "this" string.
       *
       * Each part must be separated by one or more of the characters
       * specified by <i>sep</i> to be identified as a substring part.
       *
       * @since  1992.04.07
       * @author Leif Erik Larsen
       * @param  index  Index of the string part to fetch out (0..).
       * @param  sep    List of separating characters.
       * @return The indexed substring part, or an empty string if the
       *         indexed substring part does not exist.
       */
      GString getIndexedPart ( int index, const char* sep ) const;

      virtual int hashCode () const;

      int indexOf ( const char *str, int startIndex = 0 ) const;
      int indexOf ( const GString& str, int startIndex = 0 ) const;
      int indexOf ( char chr, int startIndex = 0 ) const;
      int indexOfAnyChar ( char chr1, char chr2, int startIndex = 0 ) const;

      /**
       * @throws GArrayIndexOutOfBoundsException if the specified index
       *                           is outside legal range. Legal range is
       *                           index >= 0 && index <= currentLength.
       */
      GString& insert ( char chr, int index = 0 );
      GString& insert ( const char* str, int index = 0 );
      GString& insert ( const GString& str, int index = 0 );

      bool isEmpty () const;

      /**
       * Return true if and only if the string contains mixed
       * lowercase/uppercase characters (testing A...Z range only).
       */
      bool isMixedCase () const;

      /**
       * Return a copy of the very last character in the string,
       * or '\0' if the string is empty.
       */
      char lastChar () const;

      int lastIndexOf ( char chr ) const;
      int lastIndexOfAnyChar ( const GString& chrStr ) const;
      int length () const;
      GString& remove ( int index, int count );

      /**
       * @throws GArrayIndexOutOfBoundsException if the specified index
       *                           is outside legal range. Legal range is
       *                           index >= 0 && index < currentLength.
       */
      GString& removeCharAt ( int index );

      GString& removeFirstChar ();
      GString& removeLastChar ();

      /**
       * Replace all occurences of the <i>oldchr</i> character with
       * the <i>newchr</i> character.
       */
      GString& replaceAll ( char oldchr, char newchr );

      /**
       * Replace all occurences of all the characters in <i>oldchars</i>
       * with the <i>newchr</i> character.
       */
      GString& replaceAll ( const char* oldchars, char newchr );

      /**
       * Replace all occurences of all the characters in <i>oldchars</i>
       * with the <i>newchr</i> character.
       */
      GString& replaceAll ( const GString& oldchars, char newchr );

      GString& reverse ();

      /**
       *
       * @throws GArrayIndexOutOfBoundsException if the specified index
       *                           is outside legal range. Legal range is
       *                           index >= 0 && index < currentLength.
       */
      GString& setCharAt ( int index, char chr );

      /**
       * Make sure the trailing EOL (if any) is compatible with the 
       * operation system of where the application instance runs.
       * 
       * Test if the string ends with one or more occurences of some platform
       * dependent EOL (end-of-line) character(s). We will test for all known
       * EOL character combinations that are valid on any known operating 
       * system. Thus, this method returns true regardless if the string 
       * ends with "\r\n" (Windows/OS2), "\n" (Unix) or just "\r" (Mac).
       *
       * @author  Leif Erik Larsen
       * @since   2005.02.02
       * @return  Number of EOL's that this string actually ends with. 
       *          This is the number of system compatible linefeeds, where 
       *          each is possibly a pair of "\r" and/or "\n".
       *          Note that we will return the correct count even if we 
       *          did not actually change anythig (e.g. if the original 
       *          EOLs in the string already was compatible with the 
       *          underlying system).
       * @see     #endsWithEol
       * @see     #stripTrailingEol
       * @see     #EOL
       */
      int ensureCompatibleEolFormat ();

      /**
       * Strip all the trailing linefeed and carriage return characters
       * from the string.
       *
       * One example of usage is to remove the trailing linefeed from
       * a string returned by {@link GInputStream#readString} or one of
       * its concrete implementations.
       *
       * @author  Leif Erik Larsen
       * @since   2000.08.09
       * @return  Number of EOL's actually stripped. This is the 
       *          number of system compatible linefeeds, where each 
       *          is possibly a pair of "\r" and/or "\n".
       */
      int stripTrailingEol ();

      /**
       * Get a copy of the tail of the string, starting with and
       * including the specified start index.
       */
      GString substring ( int startIndex ) const;

      /**
       * Get a copy of the string between, and including, the specified start
       * index and, but not including, the specified end index.
       */
      GString substring ( int startIndex, int endIndex ) const;

      /**
       * Return a copy of our self.
       */
      virtual GString toString () const;

      GString& toLowerCase ();

      GString& toUpperCase ();

      /**
       * Trim/strip specified characters from the string, 
       * using the trim method specified by <code>strip</code>.
       * 
       * The default arguments will make this method trim all white 
       * characters (blanks, linefeeds, etc.) from both ends of the string.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.08
       * @param   strip  Flag to tell function the stripping method.
       * @param   chars  Set of characters of which to strip.
       * @return  A reference to our self.
       */
      GString& trim ( TrimType tt = TrimEndsOnly, 
                      const GString& chars = WhiteChars );
};

GString operator+ ( const char* str1, const GString& str2 );
GString operator+ ( const GObject& str1, const GString& str2 );
GString operator+ ( int str1, const GString& str2 );
GString operator+ ( float str1, const GString& str2 );

/**
 * Use this class to automatically load a language dependent text from the
 * Resource Table of the application.
 *
 * If the specified text is not the ID of an existing Text Resource then
 * we will initiate the new string object with the specified text as is.
 * Only texts that is specified with a % (percent) prefix character are
 * attemted to be loaded from the Resource Table. All other texts are used
 * as is without attempting to load it from the Resource Table.
 *
 * In addition, the constructor of this class takes a variable argument list
 * so that the string can easily be set up with runtime parameters. The
 * prameter list formatting is done in the same way as with <i>sprintf()</i>.
 *
 * The parameter list is parsed <i>after</i> the Text Resource has been
 * loaded. This means that it is the language dependent text string that will
 * actually be used to format the parameters.
 *
 * @author  Leif Erik Larsen
 * @see     GProgram#loadText
 */
class GStringl : public GString
{
   public:

      GStringl ( const char* format );
      GStringl ( const char* format, const GVArgs& args );
};

#endif // #ifndef __GLIB_STRING


