/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/resource/GRcCompiler.h"
#include "glib/resource/GRcException.h"
#include "glib/util/GLog.h"
#include "glib/sys/GSystem.h"

aptr<GResourceTable> GRcCompiler::ParseResources ( GRcTokenizer& tokenizer )
{
   aptr<GResourceTable> table(new GResourceTable());
   for (;;)
   {
      GAbstractResource* resource = ParseResource(tokenizer, *table);
      if (resource == null)
         break;
   }
   return table;
}

GAbstractResource* GRcCompiler::ParseResource ( GRcTokenizer& tokenizer, GResourceTable& table )
{
   const GRcToken* token = tokenizer.nextPreCompiledToken(true);
   if (token->isEmpty())
      return null;

   GString id = token->toString(); // All resources must start with an ID.
   token = tokenizer.queryArgValue();
   if (*token == GRcTokenizer::Token_text)
   {
      if (table.getTextBag().containsKey(id))
         gthrow_(GRcException(tokenizer, GRcException::ERRMENTXTDEFD, GVArgs(id))); // Specified TEXT ID already defined: %s
      GTextResource* newText = new GTextResource(id);
      aptr<GTextResource> text(newText);
      text->loadFromScript(tokenizer, &table);
      table.getTextBag().put(id, text.release());
      return newText;
   }
   else
   if (*token == GRcTokenizer::Token_icon)
   {
      if (table.getIconBag().containsKey(id))
         gthrow_(GRcException(tokenizer, GRcException::ERRICONDEFD, GVArgs(id))); // Specified ICON ID already defined: %s
      GIconResource* newIcon = new GIconResource(id);
      aptr<GIconResource> icon(newIcon);
      icon->loadFromScript(tokenizer, &table);
      table.getIconBag().put(id, icon.release());
      return newIcon;
   }
   else
   if (*token == GRcTokenizer::Token_menu)
   {
      if (table.getMenuBag().containsKey(id))
         gthrow_(GRcException(tokenizer, GRcException::ERRMENUDEFD, GVArgs(id))); // Specified ICON ID already defined: %s
      GMenuResource* newMenu = new GMenuResource(id);
      aptr<GMenuResource> menu(newMenu);
      menu->loadFromScript(tokenizer, &table);
      table.getMenuBag().put(id, menu.release());
      return newMenu;
   }
   else
   if (*token == GRcTokenizer::Token_toolbar)
   {
      if (table.getToolbarBag().containsKey(id))
         gthrow_(GRcException(tokenizer, GRcException::ERRTBARDEFD, GVArgs(id))); // Specified TOOLBAR ID already defined: %s
      GToolbarResource* newTB = new GToolbarResource(id);
      aptr<GToolbarResource> tb(newTB);
      tb->loadFromScript(tokenizer, &table);
      table.getToolbarBag().put(id, tb.release());
      return newTB;
   }
   else
   if (*token == GRcTokenizer::Token_dialog)
   {
      if (table.getDialogBag().containsKey(id))
         gthrow_(GRcException(tokenizer, GRcException::ERRDIALOGDEFD, GVArgs(id))); // Specified DIALOG ID already defined: %s
      GDialogResource* newDlg = new GDialogResource(id);
      aptr<GDialogResource> dlg(newDlg);
      dlg->loadFromScript(tokenizer, &table);
      table.getDialogBag().put(id, dlg.release());
      return newDlg;
   }
   else
   if (*token == GRcTokenizer::Token_acceltable)
   {
      if (table.getAcceleratorBag().containsKey(id))
         gthrow_(GRcException(tokenizer, GRcException::ERRACCELDEFD, GVArgs(id))); // Specified ACCELTABLE ID already defined: %s
      GAccelResource* newAccel = new GAccelResource(id);
      aptr<GAccelResource> accel(newAccel);
      accel->loadFromScript(tokenizer, &table);
      table.getAcceleratorBag().put(id, accel.release());
      return newAccel;
   }
   else
      gthrow_(GRcException(tokenizer, GRcException::ERRWUNKNOWN, GVArgs(id))); // Unknown token: %s
}

