/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/resource/GResourceTable.h"
#include "glib/util/GLog.h"

GResourceTable::GResourceTable ()
               :texts(64),
                icons(16),
                menus(4),
                tbars(4),
                dialogs(8),
                accels(4),
                loadText_Counter(0)
{
}

GResourceTable::~GResourceTable ()
{
}

GKeyBag<GTextResource>& GResourceTable::getTextBag () 
{ 
   return texts; 
}

GKeyBag<GIconResource>& GResourceTable::getIconBag () 
{ 
   return icons; 
}

GKeyBag<GMenuResource>& GResourceTable::getMenuBag () 
{ 
   return menus; 
}

GKeyBag<GToolbarResource>& GResourceTable::getToolbarBag () 
{ 
   return tbars; 
}

GKeyBag<GDialogResource>& GResourceTable::getDialogBag () 
{ 
   return dialogs; 
}

GKeyBag<GAccelResource>& GResourceTable::getAcceleratorBag () 
{ 
   return accels; 
}

GTextResource* GResourceTable::getTextResource ( const GString& name ) const 
{ 
   return texts.get(name); 
}

GIconResource* GResourceTable::getIconResource ( const GString& name ) const 
{ 
   return icons.get(name); 
}

GMenuResource* GResourceTable::getMenuResource ( const GString& name ) const 
{ 
   return menus.get(name); 
}

GToolbarResource* GResourceTable::getToolbarResource ( const GString& name ) const 
{ 
   return tbars.get(name); 
}

GDialogResource* GResourceTable::getDialogResource ( const GString& name ) const 
{ 
   return dialogs.get(name); 
}

GAccelResource* GResourceTable::getAcceleratorResource ( const GString& name ) const 
{ 
   return accels.get(name); 
}

bool GResourceTable::IsLoadableText ( const GString& textID ) 
{ 
   if (textID.length() <= 1)
      return false;
   else
   if (textID[0] == '%' && textID[1] != '%')
      return true;
   else
      return false;
}

const GString& GResourceTable::GetBestMatch ( int maxLength, 
                                              const GString& s1, 
                                              const GString& s2, 
                                              const GString& s3, 
                                              const GString& s4, 
                                              const GString& s5,
                                              const GString& s6,
                                              const GString& s7,
                                              const GString& s8,
                                              const GString& s9,
                                              const GString& s10,
                                              const GString& s11 )
{
   int indexLongest = 0;
   int indexClosest = 0;
   const GString* altt[11] = { &s1, &s2, &s3, &s4, &s5, &s6, &s7, &s8, &s9, &s10, &s11 };
   for (int i=1; i<11; i++)
   {
      int len = altt[i]->length();
      if (len <= 0)
         continue; // Ignore empty strings
      if (len > altt[indexLongest]->length())
         indexLongest = i;
      if (maxLength <= 0)
         indexClosest = indexLongest;
      else
      {
         int diffClosest = maxLength - altt[indexClosest]->length();
         int diffCurrent = maxLength - altt[i]->length();
         if (diffClosest > 0 && diffCurrent > 0)
         {
            if (diffCurrent < diffClosest)
               indexClosest = i;
         }
         else
         if (diffClosest < 0 && diffCurrent < 0)
         {
            if (diffCurrent > diffClosest)
               indexClosest = i;
         }
         else
         if (diffClosest < 0 && diffCurrent >= 0)
         {
            indexClosest = i;
         }
      }
   }

   return *altt[indexClosest];
}

GString GResourceTable::loadText ( const char* sourceStr, 
                                   LoadTextFlags flags, 
                                   int maxLength ) const
{
   const int MAX_NESTED_REF = 32; // Max 32 levels of nested text references (I guess this will be enough ;-)
   if (loadText_Counter > MAX_NESTED_REF)
   {
      GLog::Log(this, "%s: To many levels of nested text references. Max is %d.", GVArgs(__FUNCTION__).add(MAX_NESTED_REF));
      return sourceStr;
   }

   loadText_Counter++;

   GString ret = sourceStr;
   if (ret[0] == '%' && ret[1] != '%') // If this text is an ID of another text resource
   {
      GTextResource* text = texts.get(ret.cstring() + 1); // Skip the %-prefix
      if (text == null)
      {
         ;
      }
      else
      if (flags & LT_PREFER_TEXT)
      {
         if (text->textID != "")
         {
            ret = loadText(text->textID, LT_PREFER_TEXT, maxLength);
         }
         else
         {
            ret = GetBestMatch(maxLength,
                               loadText(text->hintID, LT_PREFER_TEXT, maxLength),
                               loadText(text->alt1, LT_PREFER_TEXT, maxLength),
                               loadText(text->alt2, LT_PREFER_TEXT, maxLength),
                               loadText(text->alt3, LT_PREFER_TEXT, maxLength),
                               loadText(text->alt4, LT_PREFER_TEXT, maxLength),
                               loadText(text->alt5, LT_PREFER_TEXT, maxLength),
                               loadText(text->alt6, LT_PREFER_TEXT, maxLength),
                               loadText(text->alt7, LT_PREFER_TEXT, maxLength),
                               loadText(text->alt8, LT_PREFER_TEXT, maxLength),
                               loadText(text->alt9, LT_PREFER_TEXT, maxLength));
         }
      }
      else
      if (flags & LT_PREFER_HINT)
      {
         if (text->hintID != "")
         {
            ret = loadText(text->hintID, LT_PREFER_HINT, maxLength);
         }
         else
         {
            ret = GetBestMatch(maxLength,
                               loadText(text->textID, LT_PREFER_HINT, maxLength),
                               loadText(text->alt1, LT_PREFER_HINT, maxLength),
                               loadText(text->alt2, LT_PREFER_HINT, maxLength),
                               loadText(text->alt3, LT_PREFER_HINT, maxLength),
                               loadText(text->alt4, LT_PREFER_HINT, maxLength),
                               loadText(text->alt5, LT_PREFER_HINT, maxLength),
                               loadText(text->alt6, LT_PREFER_HINT, maxLength),
                               loadText(text->alt7, LT_PREFER_HINT, maxLength),
                               loadText(text->alt8, LT_PREFER_HINT, maxLength),
                               loadText(text->alt9, LT_PREFER_HINT, maxLength));
         }
      }
      else
      if (flags & LT_PREFER_ALT1)
      {
         if (text->alt1 != "")
         {
            ret = loadText(text->alt1, LT_PREFER_ALT1, maxLength);
         }
         else
         {
            ret = GetBestMatch(maxLength,
                               loadText(text->alt2, LT_PREFER_ALT1, maxLength),
                               loadText(text->alt3, LT_PREFER_ALT1, maxLength),
                               loadText(text->alt4, LT_PREFER_ALT1, maxLength),
                               loadText(text->alt5, LT_PREFER_ALT1, maxLength),
                               loadText(text->alt6, LT_PREFER_ALT1, maxLength),
                               loadText(text->alt7, LT_PREFER_ALT1, maxLength),
                               loadText(text->alt8, LT_PREFER_ALT1, maxLength),
                               loadText(text->alt9, LT_PREFER_ALT1, maxLength));
            if (ret == "")
            {
               ret = GetBestMatch(maxLength,
                                  loadText(text->textID, LT_PREFER_ALT1, maxLength),
                                  loadText(text->hintID, LT_PREFER_ALT1, maxLength));
            }
         }
      }
      else
      if (flags & LT_PREFER_ALT2)
      {
         if (text->alt2 != "")
         {
            ret = loadText(text->alt2, LT_PREFER_ALT2, maxLength);
         }
         else
         {
            ret = GetBestMatch(maxLength,
                               loadText(text->alt1, LT_PREFER_ALT2, maxLength),
                               loadText(text->alt3, LT_PREFER_ALT2, maxLength),
                               loadText(text->alt4, LT_PREFER_ALT2, maxLength),
                               loadText(text->alt5, LT_PREFER_ALT2, maxLength),
                               loadText(text->alt6, LT_PREFER_ALT2, maxLength),
                               loadText(text->alt7, LT_PREFER_ALT2, maxLength),
                               loadText(text->alt8, LT_PREFER_ALT2, maxLength),
                               loadText(text->alt9, LT_PREFER_ALT2, maxLength));
            if (ret == "")
            {
               ret = GetBestMatch(maxLength,
                                  loadText(text->textID, LT_PREFER_ALT2, maxLength),
                                  loadText(text->hintID, LT_PREFER_ALT2, maxLength));
            }
         }
      }
      else
      if (flags & LT_PREFER_ALT3)
      {
         if (text->alt3 != "")
         {
            ret = loadText(text->alt3, LT_PREFER_ALT3, maxLength);
         }
         else
         {
            ret = GetBestMatch(maxLength,
                               loadText(text->alt1, LT_PREFER_ALT3, maxLength),
                               loadText(text->alt2, LT_PREFER_ALT3, maxLength),
                               loadText(text->alt4, LT_PREFER_ALT3, maxLength),
                               loadText(text->alt5, LT_PREFER_ALT3, maxLength),
                               loadText(text->alt6, LT_PREFER_ALT3, maxLength),
                               loadText(text->alt7, LT_PREFER_ALT3, maxLength),
                               loadText(text->alt8, LT_PREFER_ALT3, maxLength),
                               loadText(text->alt9, LT_PREFER_ALT3, maxLength));
            if (ret == "")
            {
               ret = GetBestMatch(maxLength,
                                  loadText(text->textID, LT_PREFER_ALT3, maxLength),
                                  loadText(text->hintID, LT_PREFER_ALT3, maxLength));
            }
         }
      }
      else
      if (flags & LT_PREFER_ALTERNATIVE)
      {
         if (text->alt1 != "" || text->alt2 != "" || text->alt3 != "")
         {
            ret = GetBestMatch(maxLength,
                               loadText(text->alt1, LT_PREFER_ALTERNATIVE, maxLength),
                               loadText(text->alt2, LT_PREFER_ALTERNATIVE, maxLength),
                               loadText(text->alt3, LT_PREFER_ALTERNATIVE, maxLength),
                               loadText(text->alt4, LT_PREFER_ALTERNATIVE, maxLength),
                               loadText(text->alt5, LT_PREFER_ALTERNATIVE, maxLength),
                               loadText(text->alt6, LT_PREFER_ALTERNATIVE, maxLength),
                               loadText(text->alt7, LT_PREFER_ALTERNATIVE, maxLength),
                               loadText(text->alt8, LT_PREFER_ALTERNATIVE, maxLength),
                               loadText(text->alt9, LT_PREFER_ALTERNATIVE, maxLength));
         }
         else
         {
            ret = GetBestMatch(maxLength,
                               loadText(text->textID, LT_PREFER_ALTERNATIVE, maxLength),
                               loadText(text->hintID, LT_PREFER_ALTERNATIVE, maxLength));
         }
         if (ret == "")
         {
            ret = GetBestMatch(maxLength,
                               loadText(text->textID, LT_PREFER_TEXT, maxLength),
                               loadText(text->hintID, LT_PREFER_TEXT, maxLength));
         }
      }
      else // LT_PREFER_NONE
      {
         ret = GetBestMatch(maxLength,
                            loadText(text->textID, LT_PREFER_NONE, maxLength),
                            loadText(text->hintID, LT_PREFER_NONE, maxLength),
                            loadText(text->alt1, LT_PREFER_NONE, maxLength),
                            loadText(text->alt2, LT_PREFER_NONE, maxLength),
                            loadText(text->alt3, LT_PREFER_NONE, maxLength),
                            loadText(text->alt4, LT_PREFER_NONE, maxLength),
                            loadText(text->alt5, LT_PREFER_NONE, maxLength),
                            loadText(text->alt6, LT_PREFER_NONE, maxLength),
                            loadText(text->alt7, LT_PREFER_NONE, maxLength),
                            loadText(text->alt8, LT_PREFER_NONE, maxLength),
                            loadText(text->alt9, LT_PREFER_NONE, maxLength));
      }
   }

   if (flags & LT_CLIP_IF_NEEDED)
   {
      if (maxLength > 0 && ret.length() > maxLength)
         ret.cutTailFrom(maxLength);
   }

   loadText_Counter--;

   return ret;
}
