/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_CRITICALSECTION
#define __GLIB_CRITICALSECTION

#include "glib/primitives/GObject.h"

/**
 * This class is used to protect critical sections of the code
 * from multithreaded activities. That is, by using this class 
 * one can easily make sure that only one thread gets access 
 * to a certain scope of code at a time.
 *
 * <b>Note</b> that here the term "critical section" means what 
 * it is defined on the Win32 platform and not what it is defined 
 * on OS/2. That is, here a "critical section" means that only one 
 * thread can pass into the protected portion of code at once and 
 * any other thread will wait until the first thread is finished.
 * On OS/2 this is known as an "anonymous mutex semaphore".
 *
 * <b>Mark</b> that this class should be used with extreme care.
 * It can easily produce deadlock situations, if not used with the
 * greatest caution.
 *
 * @author  Leif Erik Larsen
 * @since   2000.10.06
 * @see     GObject#Synchronizer
 * @see     GObject#enterSynchronizationLock
 */
class GCriticalSection : public GObject
{
   private:

      /** 
       * On OS2 we uses a "anonymous mutex semaphore" to achieve the same 
       * as is provided by a Win32 "critical section".
       */
      HMTX hmtx;

   public:

      /**
       * Class used to automatically exit the critical section 
       * upon scope exit.
       *
       * @author  Leif Erik Larsen
       * @since   2005.10.23
       */
      class LocalLock : public GObject
      {
         private:
            GCriticalSection& lock;
         public:
            LocalLock ( GCriticalSection& lock );
            virtual ~LocalLock ();
      };

   public:

      GCriticalSection ();
      virtual ~GCriticalSection ();

   private:

      /** Prevent the assignment operator. */
      GCriticalSection& operator= ( const GCriticalSection& ) { return *this; }

   public:

      void enter ();
      void exit ();
};

#endif


