/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_STACK
#define __GLIB_STACK

#include "glib/util/GArray.h"

/**
 * Dynamic stack.
 */
template <class T> class GStack : public GArray<T>
{
   public:

      /**
       * Construct a new Stack Object with the specified initial size.
       */
      explicit GStack ( int initial = 64 ) : GArray<T>(initial, -3)
      {
      }

      /**
       * Destruct the Stack Object and all of its elements.
       */
      virtual ~GStack ()
      {
         while (!GArray<T>::isEmpty())
            pop(true);
      }

   public:

      /**
       * Push a new item to the top of the stack.
       */
      void push ( T* item, bool autoDelete = true )
      {
         GArray<T>::add(item, autoDelete);
      }

      /**
       * Pop the topmost element from the stack.
       *
       * If there are no more items in the stack then we will return null,
       * else we will return a pointer to the popped element.
       *
       * @param  destroy  True if we shall destroy the topmost stack item and
       *                  return null, or else false if we shall return the
       *                  topmost stack item and let the caller decide if the
       *                  item will be destroyed or not.
       */
      T* pop ( bool destroy = false )
      {
         int count = GArray<T>::getCount();
         if (count <= 0)
            return null;

         if (destroy)
         {
            GArray<T>::remove(count - 1);
            return null;
         }
         else
         {
            T* item = &GArray<T>::get(count - 1);
            GArray<T>::remove(count - 1, 1, false);
            return item;
         }
      }

      /**
       * Get a pointer to the topmost element of the stack.
       * That is the last element pushed.
       *
       * If there are no items in the stack then we will return null.
       */
      T* top () const
      {
         if (num >= 1)
            return dynamic_cast<T*>(items[num-1]->data);
         else
            return null;
      }

      /**
       * Get a const reference to the topmost element of the stack.
       * That is the last element pushed.
       *
       * @throws  GArrayIndexOutOfBoundsException  if there are no items in
       *                                           the stack.
       */
      const T& peek () const throw(GArrayIndexOutOfBoundsException)
      {
         if (GArray<T>::isEmpty())
            gthrow_(GArrayIndexOutOfBoundsException());
         return *top();
      }

      /**
       * Get a non-const reference to the topmost element of the stack.
       * That is the last element pushed.
       *
       * @throws  GArrayIndexOutOfBoundsException  if there are no items in
       *                                           the stack.
       */
      T& peek () throw(GArrayIndexOutOfBoundsException)
      {
         if (GArray<T>::isEmpty())
            gthrow_(GArrayIndexOutOfBoundsException());
         return *top();
      }
};

#endif


