/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include <stdlib.h>
#include <string.h>
#include "glib/vfs/GExtendedAttributes.h"
#include "glib/GProgram.h"

#define GetInfoLevel1 1 // Get info from SFT.
// #define GetInfoLevel2 2 // Get size of FEAlist.
// #define GetInfoLevel3 3 // Get FEAlist given the GEAlist.
// #define GetInfoLevel4 4 // Get whole FEAlist.
// #define GetInfoLevel5 5 // Get FSDname.

// #define SetInfoLevel1 1 // Set info in SFT.
#define SetInfoLevel2 2 // Set FEAlist.

#define MAX_GEA (4096L * 2L) // Max size for a GEA List
#define MEM_SAFETY_EXTRA_SPACE 128 // Remove this when EA support is stable!

GExtendedAttributes::GExtendedAttributes ()
{
   this->dlist = new GKeyBag<GString>(4, -3, true);
   this->eas = GenMakeEA();
}

GExtendedAttributes::GExtendedAttributes ( GSysFileHandle hfile )
{
   this->dlist = new GKeyBag<GString>(4, -3, true);
   this->eas = QueryEAs(false, (void*) &hfile);
   if (this->eas == null)
      this->eas = GenMakeEA();
}

GExtendedAttributes::GExtendedAttributes ( const GString& fname )
{
   this->dlist = new GKeyBag<GString>(4, -3, true);
   this->eas = QueryEAs(true, (void*) fname.cstring());
   if (this->eas == null)
      this->eas = GenMakeEA();
}

GExtendedAttributes::GExtendedAttributes ( const GFile& fname )
{
   GString path = fname.getFullPath();
   this->dlist = new GKeyBag<GString>(4, -3, true);
   this->eas = QueryEAs(true, (void*) path.cstring());
   if (this->eas == null)
      this->eas = GenMakeEA();
}

GExtendedAttributes::~GExtendedAttributes ()
{
   GenFreeEA(eas);
   delete dlist;
}

bool GExtendedAttributes::operator== ( const GExtendedAttributes& eas2 ) const
{
   int count = getEACount();
   if (count != eas2.getEACount())
      return false;
   for (int i=0; i<count; i++)
   {
      GString name = getIndexedEAName(i);
      GString ea1 = getEAString(name);
      GString ea2 = eas2.getEAString(name);
      if (ea1 != ea2)
         return false;
   }
   return true;
}

bool GExtendedAttributes::operator!= ( const GExtendedAttributes& eas2 ) const
{
   return !operator==(eas2);
}

int GExtendedAttributes::getEACount () const
{
   int count = 0;

   for (HOLDFEA *next = eas;
        next != null && next->cbName > 0;
        next = next->next)
   {
      int nameLen = next->cbName;
      GString name(nameLen);
      name.appendNum(next->szName, nameLen);
      if (!dlist->containsKey(name)) // Don't count the EAs that will be deleted
         count++;
   }

   return count;
}

GExtendedAttributes::HOLDFEA* GExtendedAttributes::getEAByName ( const GString& name ) const
{
   for (HOLDFEA* next = eas;
        next != null && next->cbName > 0;
        next = next->next)
   {
      int nameLen = next->cbName;
      GString n(nameLen);
      n.appendNum(next->szName, nameLen);
      if (n == name)
         return next;
   }

   return null;
}

GExtendedAttributes::HOLDFEA* GExtendedAttributes::getIndexedEA ( int index ) const
{
   for (HOLDFEA* next = eas;
        next != null && next->cbName > 0 && index >= 0;
        next = next->next)
   {
      int nameLen = next->cbName;
      GString name(nameLen);
      name.appendNum(next->szName, nameLen);
      if (dlist->containsKey(name)) // Don't count the EAs that will be deleted.
         continue;
      if (index == 0)
         return next;
      index--;
   }

   return null;
}

GString GExtendedAttributes::getIndexedEAName ( int index ) const
{
   HOLDFEA* ea = getIndexedEA(index);
   if (ea != null)
   {
      int nameLen = ea->cbName;
      GString name(nameLen);
      name.appendNum(ea->szName, nameLen);
      return name;
   }

   return "";
}

GExtendedAttributes::EAType GExtendedAttributes::getIndexedEAType ( int index ) const
{
   HOLDFEA* ea = getIndexedEA(index);
   if (ea != null)
   {
      char* ptr = ea->aValue;
      USHORT type = *((USHORT*) ptr);
      switch (type)
      {
         case EAT_ASCII:
              return TYPE_STRING;
      }
   }

   return TYPE_UNKNOWN;
}

GString GExtendedAttributes::getEAString ( const GString& name ) const
{
   if (dlist->containsKey(name))
      return "";
   else
      return GenGetEAStringValue(eas, name);
}

bool GExtendedAttributes::isEADefined ( const GString& name ) const
{
   if (dlist->containsKey(name))
      return false;
   else
      return GenIsEADefined(eas, name);
}

void GExtendedAttributes::removeEA ( const GString& name )
{
   dlist->put(name, new GString());
}

void GExtendedAttributes::setEAString ( const GString& name, const GString& value )
{
   if (dlist->containsKey(name))
      dlist->remove(name);
   GenSetEA(eas, name, value);
}

void GExtendedAttributes::discardRemovedEAs ( bool isPath, void* file )
{
   int num = dlist->getCount();
   for (int i=0; i<num; i++) // Clean out all the deleted EA names
   {
      GString name = dlist->getKey(i);
      GenDiscardEA(isPath, file, name);
   }
   dlist->removeAll();
}

APIRET GExtendedAttributes::writeEAs ( GSysFileHandle hfile )
{
   APIRET rc = NO_ERROR;

   if (getEACount() > 0)
      return WriteEAs(eas, null, false, (void *) &hfile);
   if (rc == NO_ERROR)
      discardRemovedEAs(false, (void *) &hfile);

   return rc;
}

APIRET GExtendedAttributes::writeEAs ( const GString& fname )
{
   APIRET rc = NO_ERROR;

   if (getEACount() > 0)
      rc = WriteEAs(eas, null, true, (void *) fname.cstring());
   if (rc == NO_ERROR)
      discardRemovedEAs(true, (void *) fname.cstring());

   return rc;
}

bool GExtendedAttributes::DiscardEA ( GSysFileHandle hfile,
                                      const GString& eaname )
{
   return GenDiscardEA(false, (void*) &hfile, eaname);
}

bool GExtendedAttributes::DiscardEA ( const GString& fname, const GString& eaname )
{
   return GenDiscardEA(true, (void*) fname.cstring(), eaname);
}

void GExtendedAttributes::GenFreeEA ( HOLDFEA* pHFEA )
{
   if (pHFEA != null)
   {
      delete [] pHFEA->szName;
      delete [] pHFEA->aValue;
      delete pHFEA;
   }
}

void GExtendedAttributes::FreeEAs ( HOLDFEA* pHFEA )
{
   while (pHFEA)
   {
      HOLDFEA *next = pHFEA->next;
      GenFreeEA(pHFEA);
      pHFEA = next;
   }
}

GExtendedAttributes::HOLDFEA* GExtendedAttributes::GenMakeEA ()
{
   HOLDFEA *ret = new HOLDFEA;
   memset(ret, 0, sizeof(*ret));
   return ret;
}

void GExtendedAttributes::SetAsciiEAValue ( HOLDFEA *pHFEA, const GString& eavalue )
{
   if (pHFEA->aValue != null)
   {
      delete [] pHFEA->aValue;
      pHFEA->aValue = null;
   }

   pHFEA->cbValue = (USHORT) (eavalue.length() + (2 * sizeof(USHORT)));
   pHFEA->aValue = new char[pHFEA->cbValue + 1 + MEM_SAFETY_EXTRA_SPACE]; // +1 for the '\0' character
   memset(pHFEA->aValue, 0, pHFEA->cbValue + 1 + MEM_SAFETY_EXTRA_SPACE);
   char *ptr = pHFEA->aValue;
   USHORT *type = (USHORT *) ptr;
   *type = EAT_ASCII;
   ptr += sizeof(USHORT);
   USHORT *len = (USHORT *) ptr;
   *len = (USHORT) eavalue.length();
   ptr += sizeof(USHORT);
   memcpy(ptr, eavalue.cstring(), eavalue.length()+1);
}

void GExtendedAttributes::GenSetEA ( HOLDFEA *pHFEA,
                                     const GString& eaname,
                                     const GString& eavalue )
{
   if (pHFEA == null)
      return;

   if (pHFEA->cbName <= 0)
   {
      // The EA list is empty, so add the specified EA as the one and only
      // in the list.
      pHFEA->fEA = 0;
      pHFEA->cbName = (USHORT) eaname.length();
      pHFEA->szName = new char[pHFEA->cbName + 1 + MEM_SAFETY_EXTRA_SPACE];
      strcpy(pHFEA->szName, eaname.cstring());
      SetAsciiEAValue(pHFEA, eavalue);
      pHFEA->next = null;
      return;
   }

   // If the EA already exist then update it.
   HOLDFEA *last = pHFEA;
   for (HOLDFEA *next = pHFEA; next != null; next = next->next)
   {
      last = next;
      int nameLen = next->cbName;
      if (eaname.length() == nameLen &&
          eaname.equalsIgnoreCaseNum(next->szName, nameLen))
      {
         SetAsciiEAValue(next, eavalue);
         return;
      }
   }

   // Add a new EA with the specified name
   HOLDFEA *newea = GenMakeEA();
   last->next = newea;
   newea->fEA = 0;
   newea->cbName = (USHORT) eaname.length();
   newea->szName = new char[pHFEA->cbName + 1 + MEM_SAFETY_EXTRA_SPACE];
   strcpy(newea->szName, eaname.cstring());
   SetAsciiEAValue(newea, eavalue);
   newea->next = null;
}

bool GExtendedAttributes::GenIsEADefined ( HOLDFEA *pHFEA, const GString& eaname )
{
   for (HOLDFEA *next = pHFEA; next != null && next->cbName > 0; next = next->next)
      if (eaname.length() == next->cbName && eaname.equalsIgnoreCaseNum(next->szName, next->cbName))
         return true;
   return false;
}

GString GExtendedAttributes::GenGetEAStringValue ( HOLDFEA *pHFEA, const GString& eaname )
{
   for (HOLDFEA *next = pHFEA; next != null && next->cbName > 0; next = next->next)
   {
      int nameLen = next->cbName;
      if (eaname.length() == nameLen &&
          eaname.equalsIgnoreCaseNum(next->szName, nameLen))
      {
         if (next->cbValue <= (2 * sizeof(USHORT)))
            return "";
         char *ptr = next->aValue;
         USHORT *type = (USHORT *) ptr;
         if (*type != EAT_ASCII)
            return "";
         ptr += sizeof(USHORT);
         USHORT *lenp = (USHORT *) ptr;
         USHORT len = *lenp;
         if (lenp <= 0)
            return "";
         ptr += sizeof(USHORT);
         GString ret(len);
         ret.appendNum(ptr, len);
         return ret;
      }
   }

   return "";
}

GExtendedAttributes::HOLDFEA* GExtendedAttributes::QueryEAs ( bool bIsPath, void* pvFile )
{
   HOLDFEA* pRet = null;             /* The pointer to the buffer to return */
   HOLDFEA* pLastIn = null;  /* Points to last EA added, so new EA can link */

   /* Allocate enough room for any GEA List */
   char* pAllocc = new char[MAX_GEA + 4096 + MEM_SAFETY_EXTRA_SPACE];
   memset(pAllocc, 0, MAX_GEA + 4096 + MEM_SAFETY_EXTRA_SPACE);
   ULONG ulEntryNum = 1;        /* count of current EA to read (1-relative) */

   for (;;)                   /* Loop continues until there are no more EAs */
   {
      ULONG ulEnumCnt = 1;             /* Only want to get one EA at a time */

      /* Read into pAlloc Buffer. Note that this does not get the aValue
         field, so DosQueryPathInfo must be called to get it. */
      if (DosEnumAttribute(bIsPath ? ENUMEA_REFTYPE_PATH : ENUMEA_REFTYPE_FHANDLE,
                           pvFile, ulEntryNum, (void*) pAllocc,
                           MAX_GEA, &ulEnumCnt, GetInfoLevel1))
         break; /* There was some sort of error */

      if (ulEnumCnt != 1)                     /* All the EAs have been read */
         break;

      ulEntryNum++;
      FEA2* pFEA = (FEA2 *) pAllocc;

      /* Struct to build the new EA in */
      HOLDFEA* pNewFEA = new GExtendedAttributes::HOLDFEA;
      memset(pNewFEA, 0, sizeof(*pNewFEA));
      pNewFEA->cbName = pFEA->cbName;
      pNewFEA->cbValue = pFEA->cbValue;
      pNewFEA->szName = new char[pFEA->cbName + 1 + MEM_SAFETY_EXTRA_SPACE];
      memset(pNewFEA->szName, 0, pFEA->cbName + 1 + MEM_SAFETY_EXTRA_SPACE);
      pNewFEA->aValue = new char[pFEA->cbValue + MEM_SAFETY_EXTRA_SPACE];
      memset(pNewFEA->aValue, 0, pFEA->cbValue + MEM_SAFETY_EXTRA_SPACE);
      pNewFEA->fEA = pFEA->fEA;
      pNewFEA->next = null;

      /* Copy in EA Name */
      strcpy(pNewFEA->szName, pFEA->szName);

      /* Temp buffer to hold each EA as it is read in */
      int cbBigAlloc = sizeof(FEA2LIST) + pNewFEA->cbName+1 + pNewFEA->cbValue;
      char *pBigAlloc = new char[cbBigAlloc + 4096 + MEM_SAFETY_EXTRA_SPACE];
      memset(pBigAlloc, 0, cbBigAlloc + 4096 + MEM_SAFETY_EXTRA_SPACE);

      GEA2LIST *pGEAList = (GEA2LIST *) pAllocc; /* Set up GEAList structure */
      pGEAList->cbList = sizeof(GEA2LIST) + pNewFEA->cbName;
      pGEAList->list[0].oNextEntryOffset = 0L;
      pGEAList->list[0].cbName = pNewFEA->cbName;
      strcpy(pGEAList->list[0].szName, pNewFEA->szName);

      /* Get the complete EA info */
      EAOP2 eaopGet; /* Used to call DosQueryPathInfo */
      eaopGet.fpGEA2List = (GEA2LIST *) pGEAList;
      eaopGet.fpFEA2List = (FEA2LIST *) pBigAlloc;
      eaopGet.fpFEA2List->cbList = cbBigAlloc;

      if (bIsPath)
      {
         const char* filename = (char *) pvFile;
         DosQueryPathInfo (filename, FIL_QUERYEASFROMLIST, (void*) &eaopGet, sizeof(EAOP2));
      }
      else
      {
         GSysFileHandle hfile = *((GSysFileHandle *)pvFile);
         DosQueryFileInfo (hfile, FIL_QUERYEASFROMLIST, (void*) &eaopGet, sizeof(EAOP2));
      }

      memcpy(pNewFEA->aValue,                  /* Copy the value to HoldFEA */
             pBigAlloc + sizeof(FEA2LIST) + pNewFEA->cbName,
             pNewFEA->cbValue);

      delete [] pBigAlloc;                  /* Release the temp Enum buffer */

      if(pRet == null)                         /* If first EA, set pHoldFEA */
         pRet = pNewFEA;
      else                                 /* Otherwise, add to end of list */
         pLastIn->next = pNewFEA;

      pLastIn = pNewFEA;                      /* Update the end of the list */
   }

   delete [] pAllocc;                    /* Free up the GEA buf for DosEnum */

   return pRet;
}

APIRET GExtendedAttributes::WriteEAs ( HOLDFEA *pHFEA,
                                       DELETELIST* pDL,
                                       bool bIsPath,
                                       void* pvFile )
{
   APIRET      rc;
   EAOP2       eaopWrite;
   CHAR        aBuf[MAX_GEA],*aPtr=null;
   FEA2       *pFEA = (FEA2 *) &aBuf[sizeof(ULONG)];
   USHORT      usMemNeeded;
   ULONG      *pulPtr=(ULONG *) aBuf;  /* Initally points to top of FEALIST */

   eaopWrite.fpFEA2List = (FEA2LIST *) aBuf;     /* Setup fields that won't */
   pFEA->fEA     = 0;                            /* change for the delete   */
   pFEA->cbValue = 0;                            /* calls to DosSetPathInfo */

   while (pDL)                        /* Clean out all the deleted EA names */
   {
      pFEA->cbName = (UCHAR) strlen(pDL->EAName);
      *pulPtr      = sizeof(FEA2LIST) + pFEA->cbName;
      strcpy(pFEA->szName, pDL->EAName);
      pFEA->oNextEntryOffset = 0L;

      /* Delete EA's by saying cbValue=0 */
      if (bIsPath)
         DosSetPathInfo ((char *) pvFile, SetInfoLevel2, (void*) &eaopWrite, sizeof(EAOP2), DSPI_WRTTHRU);
      else
         DosSetFileInfo (*((GSysFileHandle *) pvFile), SetInfoLevel2, (void*) &eaopWrite, sizeof(EAOP2));

      pDL = pDL->next;
   }

   while (pHFEA && pHFEA->cbName > 0)            /* Go through each HoldFEA */
   {
      usMemNeeded = (USHORT) (sizeof (FEA2LIST) + pHFEA->cbName+1 + pHFEA->cbValue);
      aPtr = new char[usMemNeeded + MEM_SAFETY_EXTRA_SPACE];
      memset(aPtr, 0, usMemNeeded + MEM_SAFETY_EXTRA_SPACE);

      eaopWrite.fpFEA2List = (FEA2LIST *) aPtr;  /* Fill in eaop struct */
      eaopWrite.fpFEA2List->cbList = usMemNeeded;

      eaopWrite.fpFEA2List->list[0].fEA     = pHFEA->fEA;
      eaopWrite.fpFEA2List->list[0].cbName  = pHFEA->cbName;
      eaopWrite.fpFEA2List->list[0].cbValue = pHFEA->cbValue;

      strcpy(eaopWrite.fpFEA2List->list[0].szName, pHFEA->szName);
      memcpy(eaopWrite.fpFEA2List->list[0].szName + pHFEA->cbName+1, pHFEA->aValue, pHFEA->cbValue);

      /* Write out the EA */
      if (bIsPath)
         rc = DosSetPathInfo((char *) pvFile, FIL_QUERYEASIZE, (void*) &eaopWrite, sizeof (EAOP2), DSPI_WRTTHRU);
      else
         rc = DosSetFileInfo(*((GSysFileHandle *) pvFile), FIL_QUERYEASIZE, (void*) &eaopWrite, sizeof (EAOP2));

      delete [] aPtr;
      if (rc)
         return rc;

      pHFEA = pHFEA->next;
   }

   return NO_ERROR;
}

bool GExtendedAttributes::GenDiscardEA ( bool bIsPath, void* fname, const GString& eaname )
{
   bool success = false;
   ULONG Quantity = ULONG(-1);

   ULONG Size = 1024;
   char* Descriptions = new char[Size + MEM_SAFETY_EXTRA_SPACE];
   memset(Descriptions, 0, Size + MEM_SAFETY_EXTRA_SPACE);
   DosEnumAttribute(bIsPath ? ENUMEA_REFTYPE_PATH : ENUMEA_REFTYPE_FHANDLE,
                   (void *) fname, 1, Descriptions, Size, &Quantity, ENUMEA_LEVEL_NO_VALUE);

   if (Quantity > 0)
   {
      bool ok = false;
      for (int i=0; i<int(Size-eaname.length()-1); i++)
      {
         if (eaname.equalsIgnoreCase(&Descriptions[i]))
         {
            ok = true;
            break;
         }
      }

      if (ok)
      {
         Size = (64 * 1024) + 1024;
         char* Data = new char[Size + MEM_SAFETY_EXTRA_SPACE];
         memset(Data, 0, Size + MEM_SAFETY_EXTRA_SPACE);

         *((PLONG) Data) = Size;

         {
            char* Pointer = Descriptions + sizeof(LONG);
            *((PLONG) Pointer) = 0;
            Pointer += sizeof(LONG);

            *Pointer = (BYTE) eaname.length();
            Pointer += sizeof(BYTE);
            strcpy(Pointer, eaname.cstring());
            Pointer += eaname.length() + 1;

            *((PLONG) Descriptions) = Pointer - Descriptions;
         }

         {
            EAOP2 Reference = { (PGEA2LIST) Descriptions, (PFEA2LIST) Data, 0 };
            if (bIsPath)
               DosQueryPathInfo((char *) fname, FIL_QUERYEASFROMLIST, &Reference, sizeof(Reference) );
            else
               DosQueryFileInfo(*((GSysFileHandle *) fname), FIL_QUERYEASFROMLIST, &Reference, sizeof(Reference) );
         }

         *((PSHORT) (Data + 10)) = 0;

         {
            EAOP2 Reference = { (PGEA2LIST) Descriptions, (PFEA2LIST) Data, 0 };
            if (bIsPath)
               DosSetPathInfo((char *) fname, FIL_QUERYEASIZE, &Reference, sizeof(Reference), 0);
            else
               DosSetFileInfo(*((GSysFileHandle *) fname), FIL_QUERYEASIZE, &Reference, sizeof(Reference));

            success = true;
         }

         delete [] Data;
      }
   }

   delete [] Descriptions;

   return success;
}
