/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/vfs/GFile.h"
#include "glib/vfs/GExtendedAttributes.h"
#include "glib/io/GRandomAccessFile.h"
#include "glib/util/GTokenizer.h"
#include "glib/util/GBuffer.h"
#include "glib/gui/GFrameWindow.h"
#include "glib/exceptions/GIllegalArgumentException.h"
#include "glib/vfs/GVfsLocal.h"

GFile::CurDirs::CurDirs ()
               :GArray<GString>(26)
{
   for (int i=0; i<26; i++)
   {
      GString* empty = new GString();
      add(empty);
   }
}

GFile::CurDirs::~CurDirs ()
{
}

GString GFile::CurDirs::GetSystemDefaultCurrentDriveAndDir ( int drive )
{
   char cd[GFile::MAXPATH];
   ULONG ulLen = sizeof(cd);
   ::DosError(FERR_DISABLEHARDERR);
   APIRET rc = ::DosQueryCurrentDir(drive+1, (BYTE*) cd, &ulLen);
   if (rc != NO_ERROR)
      gthrow_(rc);
   // Include drive name in "current directory" (as of Win32).
   GString dir("%c:%c%s", GVArgs(char('A' + drive)).add(GFile::SlashChar).add(cd));
   // Walk the directory in order to always get the correct case of each 
   // character in all parts of the path. On OS/2 it seems like 
   // ::DosGetCurrentDir() sometimes gives the directory with all uppercase 
   // characters. Even if they are not actually uppercase on disk. And on 
   // Win32 it seems like ::GetCurrentDirectory() sometimes gives the 
   // directory with all lowercase characters.
   return GFile::GetCorrectFilenameCase(dir);
}

GFile::GFile () 
{
}

GFile::GFile ( const GFile& path )
      :drive(path.getDrive()),
       dir(path.getDirectory()),
       fname(path.getName()),
       fext(path.getExtension())
{
}

GFile::GFile ( const GString& path )
{
   GFile::SplitPath(path, &this->drive, &this->dir, &this->fname, &this->fext);
}

GFile::GFile ( const GString& dir_, const GString& fileName )
      :dir(dir_),
       fname(fileName)
{
   if (dir.length() >= 2 && isalpha(dir[0]) && dir[1] == ':')
   {
      drive = dir.substring(0, 2);
      dir.remove(0, 2);
   }

   if (dir.length() > 0 && !GFile::IsSlash(dir.lastChar()))
      dir += GFile::SlashChar;

   if (fname == "." || fname == "..")
      return;

   // Find the index of the first character of the extension.
   int pos = fname.lastIndexOf('.');
   if (pos < 0)
      return;

   // Fetch out the extension part from the filename, and put it into fext.
   fext = fname.substring(pos);
   fname.cutTailFrom(pos);
}

GFile::~GFile ()
{
}

const GFile& GFile::operator= ( const GFile& src )
{
   if (&src == this)
      return *this;

   this->drive = src.drive;
   this->dir = src.dir;
   this->fname = src.fname;
   this->fext = src.fext;
   return *this;
}

const GFile& GFile::operator= ( const GString& src )
{
   return operator=(GFile(src));
}

const bool GFile::operator== ( const GFile& src ) const
{
   if (&src == this)
      return true;

   if (GFile::IsCaseSensitiveFS())
   {
      if (drive == src.drive &&
          dir == src.dir &&
          fname == src.fname &&
          fext == src.fext)
      {
         return true;
      }
      else
      {
         return false;
      }
   }
   else
   {
      if (drive.equalsIgnoreCase(src.drive) &&
          dir.equalsIgnoreCase(src.dir) &&
          fname.equalsIgnoreCase(src.fname) &&
          fext.equalsIgnoreCase(src.fext))
      {
         return true;
      }
      else
      {
         return false;
      }
   }
}

const bool GFile::operator== ( const GString& src ) const
{
   GString fp = getFullPath();
   if (GFile::IsCaseSensitiveFS())
   {
      if (src == fp)
         return true;
      else
         return false;
   }
   else
   {
      if (src.equalsIgnoreCase(fp))
         return true;
      else
         return false;
   }
}

bool GFile::IsCaseSensitiveFS ()
{
   return false;
}

bool GFile::IsSlash ( char chr )
{
   if (chr == '\\' || chr == '/')
      return true;
   else
      return false;
}

bool GFile::IsSlashed ( const GString& dir )
{
   return dir.length() > 0 && IsSlash(dir.lastChar());
}

GString& GFile::Slash ( GString& path )
{
   if (path != "")
      if (!IsSlash(path.lastChar()))
       path += SlashChar;
   return path;
}

APIRET GFile::SetCurrentDir ( const GString& dir )
{
   // Only one thread at a time please!
   GObject::Synchronizer lock(CurDirForEachDrive);
   if (GFile::ContainsDrive(dir))
   {
      char drive = GCharacter::ToUpperCase(dir[0]);
      APIRET rc = ::DosSetDefaultDisk(drive - 'A' + 1);
      if (rc != NO_ERROR)
         return rc;
   }
   APIRET rc = ::DosSetCurrentDir(dir);
   try {
      int drive = GFile::GetCurrentDrive();
      drive -= 1; // Turn into zero-based index.
      GString cdir = CurDirs::GetSystemDefaultCurrentDriveAndDir(drive);
      GString* item = new GString(cdir);
      CurDirForEachDrive.set(drive, item);
      return rc;
   } catch (APIRET& apiRet) {
      return apiRet;
   }
}

GString GFile::GetCurrentDir ( int drive )
{
   if (drive == 0)
      drive = GFile::GetCurrentDrive();
   else
   if (drive < 0 || drive > 26)
      gthrow_(GIllegalArgumentException(GString("drive=%d", GVArgs(drive))));
   // Only one thread at a time please!
   GObject::Synchronizer lock(CurDirForEachDrive);
   drive -= 1; // Turn into zero-based index.
   GString dir = CurDirForEachDrive.get(drive);
   if (dir == "") // If not initialized yet.
   {
      // Will throw the error code (as an "APIRET") in case of any error.
      dir = CurDirs::GetSystemDefaultCurrentDriveAndDir(drive);
      GString* item = new GString(dir);
      CurDirForEachDrive.set(drive, item);
   }
   return dir;
}

GString GFile::GetCorrectFilenameCase ( const GString& path )
{
   GString ret(path.length());
   GTokenizer tokenizer(path, "\\/", "", true);
   for (int i=0; true; i++)
   {
      const GToken* token = tokenizer.getNextToken();
      if (token->isEmpty())
         break;
      GString next = token->toString();
      if (i == 0)
      {
         // This is the first part of the directory, and on the underlying
         // system this is always the part that contains the drive.
         ret = next;
         ret.toUpperCase(); // Always represent the drive name in uppercase.
         GFile::Slash(ret);
         continue;
      }
      GFile::Slash(ret);
      GString pattern(ret.length() + 1 + next.length());
      pattern = ret;
      GFile::Slash(pattern);
      pattern += next;
      FILEFINDBUF3 ffb;
      HDIR hdir = HDIR(HDIR_CREATE);
      ULONG fcount = 1; // Read only one filename.
      APIRET rc = ::DosFindFirst(pattern, &hdir, GVfs::FAttrAll, &ffb, sizeof(ffb), &fcount, FIL_STANDARD);
      if (rc != NO_ERROR)
         return path; // Return the directory "as is".
      ::DosFindClose(hdir);
      ret += ffb.achName;
   }
   return ret;
}

int GFile::GetCurrentDrive () // throws APIRET
{
   ULONG curDrive, driveMap;
   APIRET rc = ::DosQueryCurrentDisk(&curDrive, &driveMap);
   if (rc != NO_ERROR)
      gthrow_(rc);
   return curDrive;
}

void GFile::SplitPath ( const GString& path, GString* drive, GString* dir, GString* file, GString* ext )
{
   if (path == "")
   {
      if (drive) *drive = "";
      if (dir) *dir = "";
      if (file) *file = "";
      if (ext) *ext = "";
      return;
   }
   else
   if (IsThisDir(path) || IsUpDir(path))
   {
      if (drive) *drive = "";
      if (dir) *dir = "";
      if (file) *file = path;
      if (ext) *ext = "";
      return;
   }

   GString _drive(2); // A drive can have only two characters.
   GString _dir(256);
   GString _file(128);
   GString _ext(128);

   int len = path.length();
   const char* pathptr = path.cstring();
   const char* fileptr = pathptr + len;
   if (!GFile::IsSlash(path.lastChar()) && // Filename if not an ending '\'.
       path.lastChar() != ':') // Path can be "X:" (no filename).
   {
      // Look for the first character of the filename.
      // Also look for the first character of the extension.
      const char* extptr = null; // No extension until opposite proven.
      for (;;) // Since we know the path is not empty we need no initial test.
      {
         fileptr--;
         if (*fileptr == '.')
         {
            if (extptr == null)
               extptr = fileptr; // This is the first character of the extension.
         }
         else
         if (GFile::IsSlash(*fileptr) || *fileptr == ':')
         {
            fileptr++; // Point to first character of the filename.
            break;
         }
         if (fileptr == pathptr) // If we have reached the start of the path.
            break;
      }

      // Copy name of file, but don't include the extention.
      for (const char* ptr = fileptr;
           *ptr && (extptr == null || ptr < extptr);
           _file += *ptr++);

      // Copy extention of file, if any.
      if (extptr != null)
         while (*extptr)
            _ext += *extptr++;
   }

   // ".." is to be treated as filname rather than extension.
   if (_file == "" && _ext == "..")
   {
      _file = _ext;
      _ext = "";
   }

   if (fileptr != pathptr) // If path consist of not only filename.
   {
      // Now, we are sure that 'fileptr' points to start of name-part (if
      // any, else 'fileptr' point to the null-terminating character
      // in path.

      if (len >= 2 && pathptr[1] == ':') // If path specify a drive.
      {
         _drive += *pathptr++;
         _drive += *pathptr++;
      }

      // Copy dir, if any.
      while (pathptr < fileptr)
         _dir += *pathptr++;
   }

   if (drive) *drive = _drive;
   if (dir) *dir = _dir;
   if (file) *file = _file;
   if (ext) *ext = _ext;
}

GString& GFile::CutFileName ( GString& path )
{
   int index = path.length() - 1;

   do {
      char chr = path.charAt(index);
      if (GFile::IsSlash(chr) || chr == ':')
         return path;
      path.removeLastChar();
   } while (--index >= 0);

   return path;
}

bool GFile::IsProgramFileName ( const GString& path )
{
   GFile file(path);
   if (file.containsExtension())
   {
      GString ext = file.getExtension();
      return ext.equalsIgnoreCase(".exe") ||
             ext.equalsIgnoreCase(".com") ||
             ext.equalsIgnoreCase(".cmd") ||
             ext.equalsIgnoreCase(".bat");
   }
   return false;
}

bool GFile::IsShellScriptFileName ( const GString& path )
{
   GFile file(path);
   if (file.containsExtension())
   {
      GString ext = file.getExtension();
      return ext.equalsIgnoreCase(".cmd") ||
             ext.equalsIgnoreCase(".bat");
   }
   return false;
}

bool GFile::LocateExecutableFileInDir ( const GString& dir, const GString& fileName, GFile& obj )
{
   GFile file(dir, fileName);
   if (file.containsExtension())
   {
      GVfsLocal localVfs;
      if (!localVfs.exist(file))
         return false;
      obj = file;
      return true;
   }

   // ---
   if (GFile::LocateExecutableFileInDir(dir, fileName + ".exe", obj))
      return true;
   if (GFile::LocateExecutableFileInDir(dir, fileName + ".com", obj))
      return true;
   if (GFile::LocateExecutableFileInDir(dir, fileName + ".cmd", obj))
      return true;
   if (GFile::LocateExecutableFileInDir(dir, fileName + ".bat", obj))
      return true;

   // ---
   return false;
}

bool GFile::FilesEquals ( GVfs& vfs1, const GString& path1, 
                          GVfs& vfs2, const GString& path2 )
{
   GRandomAccessFile file1(vfs1, path1, "r", false);
   GRandomAccessFile file2(vfs2, path2, "r", false);
   ulonglong len1 = file1.length();
   ulonglong len2 = file2.length();
   if (len1 != len2)
      return false;
   const int buffSize = 1024*64;
   GBuffer<byte> buff1(buffSize);
   GBuffer<byte> buff2(buffSize);
   for (;;)
   {
      int num1 = file1.read(buff1.theBuffer, buffSize);
      if (num1 <= 0)
         return true;
      int num2 = file2.read(buff2.theBuffer, num1);
      if (num1 != num2 || memcmp(buff1.theBuffer, buff2.theBuffer, num1) != 0)
         return false;
   }
}

bool GFile::IsFileNameInFilter ( const GString& fileName, const GString& filter )
{
   const char* fileNamePtr = fileName.cstring();
   const char* filterPtr = filter.cstring();
   const char* filterPtrLastStar = null;

   while (*filterPtr && *fileNamePtr)
   {
      if (*filterPtr == '?')
      {
         filterPtr++;
         fileNamePtr++;
      }

      else
      if (*filterPtr == '*')
      {
         filterPtrLastStar = filterPtr;

         while (*(++filterPtr) == '*');  // Jump over all '*'

         if (*filterPtr == '\0')
            return true;

         for (char next = GCharacter::ToUpperCase(*filterPtr);
              *fileNamePtr && next != GCharacter::ToUpperCase(*fileNamePtr);
              fileNamePtr++);
      }

      else
      if (GCharacter::ToUpperCase(*filterPtr) == GCharacter::ToUpperCase(*fileNamePtr))
      {
         filterPtr++;
         fileNamePtr++;
      }

      else
      {
         if (filterPtrLastStar != null)
         {
            filterPtr = filterPtrLastStar;
            filterPtrLastStar = null;
            continue;
         }
         return false;
      }
   }

   if (*filterPtr && *filterPtr != '*')
      return false;
   else
   if (*fileNamePtr == '\0')
      return true;
   else
      return false;
}

bool GFile::IsFileNameInFilterList ( const GString& fileName, const GString& filterList, char sep )
{
   const char* ptr = filterList.cstring();
   while (*ptr)
   {
      while (*ptr == sep || isspace(*ptr))
         ptr++;

      GString filter(128);
      bool isQuoted = (*ptr == '"');
      while (*ptr)
      {
         char chr = *ptr++;
         if (isQuoted && chr == '"')
            break;
         if (chr == sep)
            break;
         filter += chr;
      }

      // Cut blanks from start and end of the filter string.
      filter.trim();

      // Next filter item is now contained in filter
      if (GFile::IsFileNameInFilter(fileName, filter))
         return true;
   }

   return false;
}

bool GFile::removeEA ( const GString& eaname ) const
{
   return GExtendedAttributes::DiscardEA(getFullPath(), eaname);
}

GString GFile::getDirAndFileName ( bool skipInitialSlash ) const
{
   GString ret = getDirectory() + getFileName();
   if (skipInitialSlash && isDirectoryFromRoot())
      ret.removeFirstChar(); // Skip the initial slash.
   return ret;
}

GString GFile::getFullPath () const
{
   return drive + dir + fname + fext;
}

int GFile::getDriveNum () const
{
   int ret = 0;
   if (containsDrive())
      ret = toupper(drive[0]) - 'A' + 1;
   if (ret < 0 || ret > 26)
      ret = 0;
   return ret;
}

int GFile::GetDriveNum ( const GString& path )
{
   GString drv = GFile::GetDrive(path);
   if (drv == "")
      return 0;
   char drvChr = GCharacter::ToUpperCase(drv[0]);
   return drvChr - 'A' + 1;
}

void GFile::setDrive ( int drv )
{
   drive.clear();
   if (drv >= 1)
   {
      drive += char('A' + drv - 1);
      drive += ':';
   }
}

void GFile::setDrive ( const GString& drv )
{
   drive = drv;
   if (drive.length() > 2)
      drive.cutTailFrom(2);
}

void GFile::setDirectory ( const GString& newDir )
{
   if (newDir != "")
   {
      dir = newDir;
      if (dir.length() > 1 && !GFile::IsSlash(dir.lastChar()))
         dir += GFile::SlashChar;
   }
   else
   {
      dir = "";
   }
}

void GFile::setName ( const GString& fname )
{
   this->fname = fname;
}

void GFile::setExtension ( const GString& fext )
{
   this->fext = fext;
}

void GFile::setFileName ( const GString& path )
{
   GFile::SplitPath(path, null, null, &fname, &fext);
}

void GFile::setFullPath ( const GString& path )
{
   GFile::SplitPath(path, &drive, &dir, &fname, &fext);
}

const GString& GFile::getDrive () const
{
   return drive;
}

const GString& GFile::getDirectory () const
{
   return dir;
}

const GString& GFile::getName () const
{
   return fname;
}

const GString& GFile::getExtension () const
{
   return fext;
}

GString GFile::getFileName () const
{
   return fname + fext;
}

GString GFile::GetDrive ( const GString& path )
{
   GFile f(path);
   return f.getDrive();
}

GString GFile::GetDirectory ( const GString& path )
{
   GFile f(path);
   return f.getDirectory();
}

GString GFile::GetName ( const GString& path )
{
   GFile f(path);
   return f.getName();
}

GString GFile::GetExtension ( const GString& path )
{
   GFile f(path);
   return f.getExtension();
}

GString GFile::GetFileName ( const GString& path )
{
   GFile f(path);
   return f.getFileName();
}

bool GFile::containsDrive () const
{
   return drive != "";
}

bool GFile::containsDirectory () const
{
   return dir != "";
}

bool GFile::containsName () const
{
   return fname != "";
}

bool GFile::containsExtension () const
{
   return fext != "";
}

bool GFile::containsFileName () const
{
   return containsName() || containsExtension();
}

bool GFile::isDirectoryFromRoot () const
{
   return dir != "" && GFile::IsSlash(dir[0]);
}

bool GFile::IsThisDir ( const char* str )
{
   return str[0] == '.' &&
          str[1] == '\0';
}

bool GFile::IsThisDir ( const GString& str )
{
   return str[0] == '.' &&
          str[1] == '\0';
}

bool GFile::isThisDir () const
{
   return fext.isEmpty() &&
          fname[0] == '.' &&
          fname[1] == '\0';
}

bool GFile::IsUpDir ( const char* str )
{
   return str[0] == '.' &&
          str[1] == '.' &&
          str[2] == '\0';
}

bool GFile::IsUpDir ( const GString& str )
{
   return str[0] == '.' &&
          str[1] == '.' &&
          str[2] == '\0';
}

bool GFile::isUpDir () const
{
   return fext.isEmpty() &&
          fname[0] == '.' &&
          fname[1] == '.' &&
          fname[2] == '\0'; }

bool GFile::ContainsDrive ( const GString& path )
{
   GFile f(path);
   return f.containsDrive();
}

bool GFile::ContainsDirectory ( const GString& path )
{
   GFile f(path);
   return f.containsDirectory();
}

bool GFile::ContainsName ( const GString& path )
{
   GFile f(path);
   return f.containsName();
}

bool GFile::ContainsExtension ( const GString& path )
{
   GFile f(path);
   return f.containsExtension();
}

bool GFile::ContainsFileName ( const GString& path )
{
   GFile f(path);
   return f.containsFileName();
}

GFile::operator const GString () const 
{ 
   return toString(); 
}

GString GFile::toString () const
{
   return getFullPath();
}
