/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/vfs/GVfs.h"
#include "glib/vfs/GVfsLocal.h"
#include "glib/util/GLog.h"
#include "glib/exceptions/GIllegalStateException.h"
#include "glib/io/GFileInputStream.h"
#include "glib/io/GFileOutputStream.h"
#include "glib/util/GBuffer.h"

GVfs::GVfs ( class GVfs* parentVfs )
     :parentVfs(parentVfs)
{
}

GVfs::~GVfs ()
{
}

GVfs::AutoFileHandleCloser::AutoFileHandleCloser ( GVfs& fs, GVfs::FileHandle fh )
                           :fs(fs),
                            fh(fh)
{
}

GVfs::AutoFileHandleCloser::~AutoFileHandleCloser ()
{
   if (fh != 0)
      fs.closeFile(fh);
}

GVfs::AutoDeletionHandleCloser::AutoDeletionHandleCloser ( GVfs& fs, GVfs::DeletionHandle hdel, bool autoPerformDeletion )
                               :fs(fs),
                                hdel(hdel),
                                autoPerformDeletion(autoPerformDeletion)
{
}

GVfs::AutoDeletionHandleCloser::~AutoDeletionHandleCloser ()
{
   if (hdel != 0)
   {
      if (autoPerformDeletion)
         fs.performDeletion(hdel);
      fs.closeDeletion(hdel);
   }
}

GVfs::File::File ()
           :isdir(false),
            deletePhysicalFileOnDestroy(false)
{
}

GVfs::File::File ( const File& file )
           :virtualPath(file.virtualPath),
            physicalPath(file.physicalPath),
            isdir(file.isdir),
            deletePhysicalFileOnDestroy(file.deletePhysicalFileOnDestroy)
{
   file.deletePhysicalFileOnDestroy = false;
}

GVfs::File::File ( const GString& virtualPath,
                   const GString& physicalPath,
                   bool isdir, 
                   bool deletePhysicalFileOnDestroy )
           :virtualPath(virtualPath),
            physicalPath(physicalPath),
            isdir(isdir),
            deletePhysicalFileOnDestroy(deletePhysicalFileOnDestroy)
{
}

GVfs::File::~File ()
{
   if (deletePhysicalFileOnDestroy)
   {
      GVfsLocal vfs;
      DeletionHandle hdel = 0; // Local VFS supports zero-handle (delete directly).
      if (isdir)
         vfs.removeDirectory(hdel, physicalPath, false);
      else
         vfs.removeFile(hdel, physicalPath, false);
   }
}

const GVfs::File& GVfs::File::operator= ( const File& file )
{
   virtualPath = file.virtualPath;
   physicalPath = file.physicalPath;
   isdir = file.isdir;
   deletePhysicalFileOnDestroy = file.deletePhysicalFileOnDestroy;
   file.deletePhysicalFileOnDestroy = false;
   return *this;
}

GVfs::WorkStatus::WorkStatus ()
                 :cancelled(false)
{
}

GVfs::WorkStatus::~WorkStatus ()
{
}

longlong GVfs::WorkStatus::getFinishedCount () const
{
   return countFinished;
}

bool GVfs::WorkStatus::isCancelRequested () const
{
   return cancelled;
}

void GVfs::WorkStatus::requestCancel ()
{
   cancelled = true;
}

void GVfs::WorkStatus::setFinishedCount ( longlong count )
{
   countFinished = count;
}

GVfs::List::List ()
           :items(256)
{
}

GVfs::List::~List ()
{
}

void GVfs::List::clear ()
{
   items.removeAll();
}

int GVfs::List::size () const
{
   return items.getCount();
}

const GVfs::List::Item& GVfs::List::operator[] ( int index ) const
{
   return items[index];
}

GVfs::List::Item::Item ( const char* fname,
                         int flags,
                         ulonglong sizeIdeal,
                         ulonglong sizeAllocated )
                 :fname(fname),
                  flags(flags),
                  sizeIdeal(sizeIdeal),
                  sizeAllocated(sizeAllocated)
{
}

GVfs::List::Item::~Item ()
{
}

const GString& GVfs::List::Item::getFName () const
{
   return fname;
}

int GVfs::List::Item::getFlags () const
{
   return flags;
}

ulonglong GVfs::List::Item::getSizeIdeal () const
{
   return sizeIdeal;
}

ulonglong GVfs::List::Item::getSizeAllocated () const
{
   return sizeAllocated;
}

bool GVfs::List::Item::isDirectory () const
{
   return (flags & GVfs::FAttrDirectory) != 0;
}

bool GVfs::List::Item::isReadOnly () const
{
   return (flags & GVfs::FAttrReadOnly) != 0;
}

bool GVfs::List::Item::isHidden () const
{
   return (flags & GVfs::FAttrHidden) != 0;
}

bool GVfs::List::Item::isSystem () const
{
   return (flags & GVfs::FAttrSystem) != 0;
}

GError GVfs::getWalkedPath ( const GString& srcDir,
                             GString& walkedDir ) const
{
   if (srcDir == "")
   {
      walkedDir = getCurrentDirectory(true);
      return GError::Ok;
   }
   else
   try {
      GString fullSrcDir(256);
      if (srcDir.length() >= 2 && isalpha(srcDir[0]) && srcDir[1] == ':')
      {
         if (srcDir.length() >= 3 && isSlash(srcDir[2]))
         {
            fullSrcDir = srcDir;
         }
         else
         {
            GString curDir = getCurrentDirectory(true);
            fullSrcDir = GString("%c:%s%s", GVArgs(srcDir[0]).add(curDir).add(srcDir));
         }
      }
      else
      if (!isSlash(srcDir[0]))
      {
         GString curDir = getCurrentDirectory(true);
         fullSrcDir = GString("%s%s", GVArgs(curDir).add(srcDir));
      }
      else
      if (dynamic_cast<const GVfsLocal*>(this) != null)
      {
         int curDrive = GFile::GetCurrentDrive();
         fullSrcDir = GString("%c:%s", GVArgs(curDrive + 'A' - 1).add(srcDir));
      }
      else
      {
         fullSrcDir = srcDir;
      }

      // Now we have the full path in fullSrcDir.
      // It is time to walk though it to remove and translate any "." and ".."
      // sequences within the path.
      walkPath(fullSrcDir, GString::Empty);
      walkedDir = fullSrcDir;
      return GError::Ok;
   } catch (APIRET& rc) {
      return rc;
   }
}

bool GVfs::isSlashed ( const GString& path ) const
{
   char lastChr = path.lastChar();
   return isSlash(lastChr);
}

GError GVfs::TestDirectoryCircularity ( GVfs& srcVfs, const GString& srcDir, 
                                        GVfs& dstVfs, const GString& dstDir )
{
   GString walkedSrcDir(256);
   GString walkedDstDir(256);

   GError rc = srcVfs.getWalkedPath(srcDir, walkedSrcDir);
   if (rc != GError::Ok)
      return rc;

   rc = dstVfs.getWalkedPath(dstDir, walkedDstDir);
   if (rc != GError::Ok)
      return rc;

   int walkedSrcDirLen = walkedSrcDir.length();
   if (strnicmp(walkedSrcDir, walkedDstDir, walkedSrcDirLen) == 0)
   {
      if (GFile::IsSlash(walkedDstDir[walkedSrcDirLen]) ||
          walkedDstDir[walkedSrcDirLen] == '\0')
      {
         return GError::CircularityRequested;
      }
   }

   return GError::Ok;
}

bool GVfs::walkPath ( GString& path, const GString& walkDir ) const
{
   bool wasSlahed = isSlashed(walkDir);
   GString keepDrive(3);
   GString dir = path;
   if (dir.length() >= 2 && dir[1] == ':')
   {
      // Keep a copy of the drive part, and remove it from the dir.
      keepDrive += dir[0];
      keepDrive += dir[1];
      dir.removeFirstChar();
      dir.removeFirstChar();
   }
   if (walkDir.length() > 0)
   {
      int walkDirStartPos = 0;
      if (walkDir.length() >= 3 && walkDir[1] == ':')
      {
         keepDrive = "";
         keepDrive += walkDir[0];
         keepDrive += walkDir[1];
         walkDirStartPos = 2;
         dir = "";
      }
      if (isSlash(walkDir[walkDirStartPos]))
      {
         dir = walkDir.substring(walkDirStartPos); // Walk from root!
      }
      else
      {
         slash(dir);
         dir += walkDir.substring(walkDirStartPos);
      }
   }
   if (dir.length() > 0 && isSlash(dir[0]))
   {
      keepDrive += dir[0];
      dir.removeFirstChar(); // Remove initial slash.
   }

   // ---
   bool ok = true;
   GString buff(256);
   for (int i=0, len=dir.length(); i<len; i++)
   {
      if (dir[i] == '.' &&
          dir[i+1] == '.' &&
          (i == 0 || isSlash(dir[i-1])) &&
          (i+2 >= dir.length() || isSlash(dir[i+2])))
      {
         // Walk "upwards" in the destination buffer.
         buff.removeLastChar();
         if (buff.length() == 0)
            ok = false;
         while(buff.length() > 0 && !isSlash(buff.lastChar()))
            buff.removeLastChar();

         // Skip the following ".." sequence of characters.
         i += 2;
      }
      else
      if (dir[i] == '.' &&
          (i == 0 || isSlash(dir[i-1])) &&
          (i+1 >= dir.length() || isSlash(dir[i+1])))
      {
         // Skip the following "." character.
         i += 1;
      }
      else
      {
         buff += dir[i];
      }
   }

   // Make sure we leave the path with the same slashed state as it was 
   // provided by the valler. That is, if it was slashed we should slash it.
   // If it was not, we should remove any trailing slash (except for root).
   if (wasSlahed)
   {
      slash(buff);
   }
   else
   if (isSlashed(buff))
   {
      int bufflen = buff.length();
      if (bufflen <= 1)
         ; // We have a root (without drive), so don't remove the slash.
      else
      if (buff[bufflen-2] == ':')
         ; // We have a root (with drive), so don't remove the slash.
      else
         buff.removeLastChar();
   }

   // ---
   path = keepDrive + buff;
   translateSlashes(path);
   return ok;
}

bool GVfs::isDirectoryEmpty ( const GString& dir, 
                              GVfs::IDE_Ignore ignore,
                              bool* ok ) const
{
   if (ok != null)
      *ok = false; // Until the opposite has been proven.
   bool ret = true;
   // Make sure the directory is slashed, in order to find first contained 
   // item and not the directory it self.
   GString dir_ = dir;
   slash(dir_); 
   GFileItem fitem;
   int hdir = findFirst(fitem, dir_);
   if (hdir == 0)
      return true;
   for (;;)
   {
      const GString& fname = fitem.getName();
      if (!GFile::IsThisDir(fname) && !GFile::IsUpDir(fname))
      {
         switch (ignore)
         {
            case IDE_IgnoreDirs:
               if (!fitem.isDirectory())
                  ret = false;
               break;
            case IDE_IgnoreFiles:
               if (fitem.isDirectory())
                  ret = false;
               break;
            case IDE_IgnoreNone:
            default:
               ret = false;
               break;
         }
         if (ret == false)
            break;
      }
      if (!findNext(hdir, fitem))
         break;
   }
   findClose(hdir);
   return ret;
}

GError GVfs::createDirectory ( const GString& dir )
{
   return GError::NotSupported;
}

bool GVfs::match ( const GString& str_, const GString& filter_, bool caseSen ) const
{
   // Set up the local str and filter of which to use by the algorithm.
   // We must do this since the str_ and filter_ specified by arguments
   // are declared const, and we need to change case with respect to 
   // the caseSen argument specified. This is maybe clumsy, but it works.
   GString strCopy;
   GString filterCopy;
   if (!caseSen)
   {
      strCopy = str_;
      strCopy.toUpperCase();
      filterCopy = filter_;
      filterCopy.toUpperCase();
   }
   const GString& str = (caseSen ? str_ : strCopy);
   const GString& filter = (caseSen ? filter_ : filterCopy);

   // ---
   int strIndex = 0; // Index of character in string
   int strLength = str.length(); // Make this a fast one
   int filterIndex = 0; // Index of character in filter string
   int filterLength = filter.length(); // (F)ilter (L)ength

   // Some special logic is needed in order to return the correct
   // result on situations like where the string is "XXXCompComposed" and
   // the filter is "*Composed".
   int lastStarPosFilt = -1;
   int lastStarPosText = -1;

   while (strIndex < strLength && filterIndex < filterLength)
   {
      char filterChar = filter.charAt(filterIndex);
      if (filterChar == '?')
      {
         filterIndex++;
         strIndex++;
      }

      else
      if (filterChar == '*')
      {
         while (++filterIndex < filterLength)
            if (filter.charAt(filterIndex) != '*') // Skip all '*'
               break;

         if (filterIndex >= filterLength)
            return true;

         lastStarPosFilt = filterIndex - 1;
         lastStarPosText = strIndex;

         filterChar = filter.charAt(filterIndex);
         while (strIndex < strLength)
         {
            char chr = str.charAt(strIndex);
            if (filterChar == chr)
               break;
            strIndex++;
         }
      }

      else
      if (filterChar == str.charAt(strIndex))
      {
         filterIndex++;
         strIndex++;
      }

      else
      {
         if (lastStarPosFilt >= 0)
         {
            filterIndex = lastStarPosFilt;
            strIndex = ++lastStarPosText;
         }
         else
            return false;
      }
   }

   while (filterIndex < filterLength)
   {
      char chr = filter.charAt(filterIndex++);
      if (chr != '*' && chr != '?')
         return false;
   }

   if (strIndex >= strLength)
      return true;
   else
      return false;
}

void GVfs::fillList ( GVfs::List& list,
                      const GString& pattern,
                      bool inclFiles,
                      bool inclDirs,
                      bool inclHidden,
                      bool inclSys ) const
{
   list.clear();

   GFileItem fitem;
   int hdir = findFirst(fitem, pattern);
   if  (hdir == 0)
      return;
   do {
      bool incl = true;
      GString fname = fitem.getFileName();
      if (fitem.isDirectory())
      {
         if (!inclDirs || GFile::IsThisDir(fname) || GFile::IsUpDir(fname))
            incl = false;
      }
      else
      {
         if (!inclFiles)
            incl = false;
      }
      if (incl &&
         (inclHidden || !fitem.isHidden()) &&
         (inclSys || fitem.isSystem()))
      {
         ulonglong sizeIdeal = fitem.fileSize;
         ulonglong sizeAllocated = fitem.fileSizeAlloc;
         GVfs::List::Item* item = new GVfs::List::Item(fname, fitem.attr, sizeIdeal, sizeAllocated);
         list.items.add(item);
      }
   } while (findNext(hdir, fitem));
   findClose(hdir);
}

bool GVfs::exist ( const GString& path )
{
   GFileItem fitem;
   int fh = findFirst(fitem, path);
   if (fh == 0)
      return false;
   findClose(fh);
   return true;
}

bool GVfs::existFile ( const GString& path )
{
   GFileItem fitem;
   int fh = findFirst(fitem, path);
   if (fh == 0)
      return false;
   findClose(fh);
   return !fitem.isDirectory();
}

bool GVfs::existDirectory ( const GString& dir )
{
   if (dynamic_cast<GVfsLocal*>(this) != null)
   {
      // Root directory on the local file system always exists.
      // At least if the drive is valid.
      if (dir.length() == 3 && isalpha(dir[0]) && dir[1] == ':' && isSlash(dir[2]))
         return true; 
   }

   // We shopuld check if the directory it self exists, not if the 
   // directory contains any files. Thus, we should make sure that 
   // the specified directory does not end with a slash, because if
   // it does then the findFirst() method will find the first contained
   // file instead of the directory it self.
   GString dir_ = dir;
   if (isSlashed(dir_))
      dir_.removeLastChar();

   // ---
   GFileItem fitem;
   int fh = findFirst(fitem, dir_);
   if (fh == 0)
      return false;
   findClose(fh);
   return fitem.isDirectory();
}

GError GVfs::getFileInfo ( const GString& path, GFileItem& fitem )
{
   int fh = findFirst(fitem, path);
   if (fh == 0)
      return GError::FileNotFound;
   findClose(fh);
   return GError::Ok;
}

GString GVfs::getShortenedDirOrPath ( const GString& originalPath )
{
   // Probably attempting to move/copy a HPFS filename to a FAT drive.
   // Try to automatically convert the filename, but keep the original
   // long filename as part of the EA's of the file.
   GString drive, dir, fname, fext;
   GFile::SplitPath(originalPath, &drive, &dir, &fname, &fext);
   GString longFName = fname + fext;
   GString shortFName = fname;
   GString shortFExt = fext;

   // Remove all unsupported characters from the file name
   int len = shortFName.length();
   for (int i=len-1; i>=0; i--)
      if (GFile::LegalChars.indexOf(shortFName[i]) < 0)
         shortFName.removeCharAt(i);

   // Remove all unsupported characters from the file extension
   len = shortFExt.length();
   for (int i=len-1; i>=0; i--)
      if (GFile::LegalChars.indexOf(shortFExt[i]) < 0)
         shortFExt.removeCharAt(i);

   if (shortFName.length() > 4)
      shortFName = shortFName.substring(0, 4);

   if (shortFExt.length() > 3) // No more than four letters in extension (not including the dot, which has been cleaned out)
      shortFExt = shortFExt.substring(0, 3);

   if (fext != "" && fext[0] == '.')
      shortFExt.insert('.', 0); // Insert the dot in front of extension

   if (shortFName.equalsIgnoreCase(fname) && shortFExt.equalsIgnoreCase(fext))
      return originalPath;
   else
      return getUniqueFileName(drive + dir, shortFName + "~", shortFExt);
}

GString GVfs::getUniqueFileName ( const GString& dir, const GString& prefix, const GString& extention )
{
   for (;;)
   {
      GString fname = prefix;
      for (int i=prefix.length(); i<8; i++)
         fname += GFile::LegalFNChars[rand() % GFile::LegalFNChars.length()];
      GString slash = isSlashed(dir) ? GString::Empty : GFile::SlashStr;
      GString ret = dir + slash + fname + extention;
      if (!exist(ret))
         return ret;
   }
}

GVfs::File* GVfs::preparePhysicalFile ( const GString& fileName, 
                                        WorkStatus& stat,
                                        const GString& prefix )
{
   // Open the VFS-file of which to read and copy to a temporary file.
   GFileInputStream inStream(*this, fileName, false);

   // Create and open the temporary physical file of where to write.
   GVfsLocal physicalVfs;
   const GString physicalPath = physicalVfs.createTemporaryFile(prefix);
   GFileOutputStream outStream(physicalVfs, physicalPath, true, true, false);

   // Copy the file.
   try {
      const int buffSize = 1024*64;
      GBuffer<byte> buff(buffSize);
      longlong writtenBytesTotally = 0;
      while (!stat.isCancelRequested())
      {
         int read = inStream.read(buff.theBuffer, buffSize);
         if (read <= 0)
            break;
         outStream.write(buff.theBuffer, 1, read);
         writtenBytesTotally += read;
         stat.setFinishedCount(writtenBytesTotally);
      }
   } catch (...) {
      DeletionHandle hdel = 0; // Local VFS supports zero-handle (delete directly).
      physicalVfs.removeFile(hdel, physicalPath, false);
      throw;
   }

   // Return the information for the now prepared physical file.
   if (stat.isCancelRequested())
   {
      DeletionHandle hdel = 0; // Local VFS supports zero-handle (delete directly).
      physicalVfs.removeFile(hdel, physicalPath, false);
      return null;
   }
   else
   {
      // Set timestamp on the output file, to match the zip'ed file.
      // This is not very critical, so just ignore any error codes.
      GFileItem fitem;
      if (getFileInfo(fileName, fitem) == GError::Ok)
      {
         fitem.path = physicalPath;
         physicalVfs.writeAttrAndTimes(null, fitem);
      }
      return new GVfs::File(fileName, physicalPath, false, true);
   }
}

GError GVfs::moveOrRenameFile ( const GString& existingName, 
                                const GString& newName,
                                bool allowCopy )
{
   return GError::NotSupported;
}

GString GVfs::getFullVirtualPathTo ( const GString& path ) const
{
   if (parentVfs == null)
      return path;

   // Get a list of all the nested VFSs in the correct order.
   GArray<GVfs> v;
   for (const GVfs* vfs = parentVfs; vfs != null; vfs = vfs->parentVfs)
      v.add(const_cast<GVfs*>(vfs), false);

   GString buff(128);
   for (int i=0, num=v.getCount(); i<num; i++)
   {
      const GVfs& fs = v.get(i);
      if (i > 0)
      {
         GFile::Slash(buff);
         buff += fs.getLogicalSelfName();
      }
      GString cur = fs.getCurrentDirectory(true);
      if (cur != "")
      {
         GFile::Slash(buff);
         buff += cur;
      }
   }

   slash(buff);
   buff += path;
   return buff;
}
