/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_VIRTUALFILESYSTEM
#define __GLIB_VIRTUALFILESYSTEM

#include "glib/primitives/aptr.h"
#include "glib/primitives/GAtomicCounter.h"
#include "glib/vfs/GFileItem.h"
#include "glib/sys/GError.h"

/**
 *
 * @author  Leif Erik Larsen
 * @since   2002.07.28
 */
class GVfs : public GObject
{
   public:

      /** 
       * File Handle type used by e.g. {@link #openFile} and {@link #closeFile}. 
       * @see GVfs#AutoFileHandleCloser
       */
      typedef int FileHandle;

      /**
       * Used with {@link #removeFile} and {@link #removeDirectory}
       * in order to make it possible for every VFS to implement 
       * speed optimized removal of lists of files and/or directories.
       *
       * @author  Leif Erik Larsen
       * @since   2005.08.31
       */
      typedef int DeletionHandle;

      /**
       * Supported arguments for the "ignore" parameter 
       * of {@link #isDirectoryEmpty}.
       *
       * @author  Leif Erik Larsen
       * @since   2006.02.14
       * @see     #isDirectoryEmpty
       */
      enum IDE_Ignore { IDE_IgnoreNone, IDE_IgnoreDirs, IDE_IgnoreFiles };

      /** Supported mode options, to be used with {@link #openFile}. */
      enum OF_Mode { Mode_ReadOnly, Mode_ReadWrite };

      /** Supported file create options, to be used with {@link #openFile}. */
      enum OF_Create { Create_Never, Create_IfNew, Create_IfExist, Create_Always };

      /** Supported share options, to be used with {@link #openFile}. */
      enum OF_Share { Share_DenyNone, Share_DenyRead, Share_DenyWrite, Share_DenyAll };

      /** Flag-flags, to be used with {@link #openFile}. */
      enum { OF_FLAG_SEQUENTIAL_ACCESS = 0x0000 }; // Default
      enum { OF_FLAG_RANDOM_ACCESS = 0x0001 };
      enum { OF_FLAG_APPEND = 0x0002 };
      enum { OF_FLAG_DONT_OPEN_IF_EXIST = 0x0004 };

      /** Possible action codes, to be returned in the "actionTaken" argument of {@link #openFile}. */
      enum OF_ActionTaken { Action_None, Action_Opened, Action_Created, Action_Truncated };

      /** 
       * To be used with {@link #getFileAttributes} and 
       * {@link #setFileAttributes}.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.11
       */
      enum FileAttributes
      {
         FAttrNormal = FILE_NORMAL,
         FAttrDirectory = FILE_DIRECTORY,
         FAttrArchive = FILE_ARCHIVED,
         FAttrSystem = FILE_SYSTEM,
         FAttrReadOnly = FILE_READONLY,
         FAttrHidden = FILE_HIDDEN,
         FAttrAll = FAttrNormal | FAttrDirectory | FAttrArchive | FAttrSystem | FAttrReadOnly | FAttrHidden,
         FAttrError = -1
      };

      /**
       * This class can be used to contain a {@link GVfs#FDeletionHandle} 
       * that is to be automatically closed upon scope exit.
       *
       * An example of typical usage:
       *
       * <pre>
       *    GVfs& vfs = ...; // A reference to a Virtual File System object.
       *    GVfs::AutoDeletionHandleCloser delCloser(vfs, vfs.openDeletion(), false);
       *    vfs.removeFile(delCloser.hdel, "file1.tmp", false);
       *    vfs.removeFile(delCloser.hdel, "file2.tmp", false);
       *    vfs.removeFile(delCloser.hdel, "file3.tmp", false);
       *    vfs.performDeletion(delCloser.hdel);
       * </pre>
       *
       * @author  Leif Erik Larsen
       * @since   2005.08.31
       */
      class AutoDeletionHandleCloser : public GObject
      {
         public:

            /** The VFS of where the contained deletion handle was opened. */
            GVfs& fs;

            /** The deletion handle it self. */
            const GVfs::DeletionHandle hdel;

            /** True if we shall automatically call {@link GVfs#performDeletion} upon destruction. */
            bool autoPerformDeletion;

            AutoDeletionHandleCloser ( GVfs& fs, GVfs::DeletionHandle hdel, bool autoPerformDeletion );
            virtual ~AutoDeletionHandleCloser ();

         private:

            /** Disable the copy-constructor. */
            AutoDeletionHandleCloser ( const AutoDeletionHandleCloser& src ) : fs(src.fs), hdel(0) {}
      };

      /**
       * This class can be used to contain a {@link GVfs#FileHandle} that 
       * is to be automatically closed upon scope exit.
       *
       * @author  Leif Erik Larsen
       * @since   2005.08.25
       */
      class AutoFileHandleCloser : public GObject
      {
         public:

            /** The VFS of where the contained file handle was opened. */
            GVfs& fs;

            /** The file handle it self. */
            const GVfs::FileHandle fh;

            AutoFileHandleCloser ( GVfs& fs, GVfs::FileHandle fh );
            virtual ~AutoFileHandleCloser ();

         private:

            /** Disable the copy-constructor. */
            AutoFileHandleCloser ( const AutoFileHandleCloser& src ) : fs(src.fs), fh(0) {}
      };

      /**
       * This class is used to represent a single virtual file object
       * that has been prepared by the virtual file system for physical
       * operations so that it can temporarily be seen as a physical file.
       *
       * For instance, the ZIP-archive virtual file system will
       * unzip one contained file item into some temporary physical
       * directory and give the client an instance of this class to let
       * the client temporarily see the virtual file as a physical file
       * on the physical file system.
       *
       * If the file had to be e.g. unpacked or downloaded (depending
       * on the virtual file system) the file object's physical path will
       * automatically be deleted when the file object comes out of scope,
       * but of course only if the object has the ownership of the physical
       * file. The physical file ownership is correctly transferred by the
       * copy constructor and the assignment operator of this class.
       *
       * @author  Leif Erik Larsen
       * @since   2002.08.27
       * @see     GVfs#prepareFile
       */
      class File : public GObject
      {
         private:

            GString virtualPath;
            GString physicalPath;
            bool isdir;
            mutable bool deletePhysicalFileOnDestroy;

         public:

            File ();

            File ( const GString& virtualPath,
                   const GString& physicalPath,
                   bool isdir, bool deletePhysicalFileOnDestroy );

            /**
             * The copy constructor will take over the ownership of the
             * file in case {@link #deletePhysicalFileOnDestroy} is true.
             */
            File ( const File& file );

            /**
             * The destructor will automatically delete the physical
             * file if {@link #deletePhysicalFileOnDestroy} is true.
             */
            virtual ~File ();

         public:

            /**
             * The assigment operator will take over the ownership of the
             * file in case {@link #deletePhysicalFileOnDestroy} is true.
             */
            const File& operator= ( const File& file );

         public:

            const GString& getVirtualPath () const { return virtualPath; }
            const GString& getPhysicalPath () const { return physicalPath; }
            bool isEmpty () const { return virtualPath == GString::Empty; }
      };

      /**
       * Class used to let the GUI-thread access status information of some 
       * VFS operation that was called by a background thread. Typically 
       * for the GUI-thread to be able to show a progress dialog while 
       * waiting for the VFS-operation to finish, or to provide the 
       * user with something like a cancel button.
       * 
       * @author  Leif Erik Larsen
       * @since   2005.01.04
       * @see     GVfs#preparePhysicalFile
       */
      class WorkStatus : public GObject
      {
         private:

            bool volatile cancelled;
            GAtomicCounter<longlong> countFinished;

         public:

            WorkStatus ();
            virtual ~WorkStatus ();

            /**
             * Usually to be called by the GUI-thread to update some 
             * progress dialog while the VFS operation is still working.
             *
             * @author  Leif Erik Larsen
             * @since   2005.01.04
             * @see     #setFinishedCount
             */
            longlong getFinishedCount () const;

            /**
             * To be periodically called by the VFS operation method to 
             * check if the GUI-thread has called {@link #requestCancel}
             * and cancel it self if that is the case.
             *
             * The GUI-thread will also call this method. Typically just 
             * after the VFS operation method has finished, to test if it
             * did actually finished normally or if it finished due to 
             * a cancel-request.
             *
             * @author  Leif Erik Larsen
             * @since   2005.01.04
             * @return  True if the method {@link #requestCancel} has been
             *          called on this object by any thread (usually the 
             *          GUI-thread only will ever call that method).
             * @see     #requestCancel
             */
            bool isCancelRequested () const;

            /**
             * Usually to be called by the GUI-thread in order to request 
             * some VFS operation to cancel it self. Useful only if the VFS 
             * operation method was called by some background thread.
             * 
             * @author  Leif Erik Larsen
             * @since   2005.01.04
             * @see     #isCancelRequested
             */
            void requestCancel ();

            /**
             * Usually to be called by the GUI-thread to update some 
             * progress dialog while the VFS operation is still working.
             *
             * @author  Leif Erik Larsen
             * @since   2005.01.04
             * @see     #getFinishedCount
             */
            void setFinishedCount ( longlong count );
      };

      /**
       * This class is used to represent a set of file items that was
       * found to match the pattern criterias specified to
       * {@link GVfs#fillList}.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.08
       */
      class List : public GObject
      {
         public:

            class Item : public GObject
            {
               friend class GFile;
               friend class List;

               private:

                  /** The filename including extension, but without directory. */
                  GString fname;

                  /** Flags that defines the type and status of the file. */
                  int flags;

                  /** Ideal space actually needed by the file. */
                  ulonglong sizeIdeal;

                  /** Allocated space, including wastage due to clusters, etc. */
                  ulonglong sizeAllocated;

               public:

                  Item ( const char* fname,
                         int flags,
                         ulonglong sizeIdeal,
                         ulonglong sizeAllocated );

                  virtual ~Item ();

               public:

                  const GString& getFName () const;
                  int getFlags () const;
                  ulonglong getSizeIdeal () const;
                  ulonglong getSizeAllocated () const;
                  bool isDirectory () const;
                  bool isReadOnly () const;
                  bool isHidden () const;
                  bool isSystem () const;
            };

         public:

            GArray<GVfs::List::Item> items;

         public:

            List ();
            virtual ~List ();

         public:

            void clear ();
            int size () const;
            const GVfs::List::Item& operator[] ( int index ) const;
      };

   protected:

      /**
       * The parent VFS as of where this VFS is contained.
       * If the parent VFS is null then this is the root VFS,
       * which is usually an instance of {@link GVfsLocal}.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.23
       */
      class GVfs* parentVfs;

   protected:

      explicit GVfs ( class GVfs* parentVfs );

   public:

      virtual ~GVfs ();

   public:

      /**
       * Close the file handle recently returned by {@link #openFile}.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.09
       * @return  The error code or success indicator.
       * @see     #openFile
       */
      virtual GError closeFile ( FileHandle hfile ) = 0;

      /**
       * Close the deletion handle recently returned by {@link #openDeletion}.
       *
       * @author  Leif Erik Larsen
       * @since   2005.08.31
       * @return  The error code or success indicator.
       * @see     #openDeletion
       */
      virtual GError closeDeletion ( DeletionHandle hdel ) = 0;

      /** 
       * Return true if the path contains at least one slash character. 
       *
       * @author  Leif Erik Larsen
       * @since   2005.09.01
       */
      virtual bool containsAnySlash ( const GString& path ) const = 0;

      /**
       * Create the specified directory on the file system.
       *
       * The default implementation does nothing but return 
       * {@link GError::NotSupported}.
       *
       * @author  Leif Erik Larsen
       * @since   2005.09.04
       * @see     #removeDirectory
       */
      virtual GError createDirectory ( const GString& dir );

      /**
       * Return true if and only if the specified file or directory 
       * name do actually exist on this file system. This method depends 
       * on {@link #findFirst} being correctly implemented by the subclass.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.23
       * @return  True if it does exist, or else false.
       * @see     #existDirectory
       * @see     #existFile
       */
      bool exist ( const GString& path );

      /**
       * Test if the specified directory is an existing directory (and not
       * a file) or not. This method depends on {@link #findFirst} being 
       * correctly implemented by the subclass.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.23
       * @return  True if it is an existing directory, or else return false.
       * @see     #existFile
       * @see     #exist
       */
      bool existDirectory ( const GString& dir );

      /**
       * Test if the specified path references an existing file (and not
       * a directory) or not. This method depends on {@link #findFirst} 
       * being correctly implemented by the subclass.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.23
       * @return  True if it is an existing file, or else return false.
       * @see     #existDirectory
       * @see     #exist
       */
      bool existFile ( const GString& path );

      /**
       * Load the filenames that match the specified filename search pattern 
       * and attributes, from the file system.
       *
       * Note that the list will not contain the this (".") and the
       * parent ("..") directory upon return. This means that this method
       * will automatically filter those two items so that the calling
       * client can simply ignore their existense.
       *
       * The default implementation of this method is a generic algorithm
       * that uses {@link #findFirst}, {@link #findNext} and 
       * {@link #findClose}. It can be overridden, e.g. if the subclass 
       * can provide a more effective implementation.
       *
       * @author  Leif Erik Larsen
       * @since   2004.12.23
       * @param   list       The object of where to store the found filenames
       *                     in order to return them to the caller. We will
       *                     always clear the content of the provided list
       *                     before we start loading filenames. In case of 
       *                     any error the provided list will always be 
       *                     empty and cleared upon return.
       * @param   pattern    The filename search pattern with or without
       *                     wildcard letters and with or without directory.
       *                     For most file systems the pattern is a FAT16-
       *                     compatible filename pattern with or without
       *                     wildcard characters such as ? and *.
       * @param   inclFiles  True if we shall include files in the list.
       * @param   inclDirs   True if we shall include directories in the list.
       * @param   inclHidden True if we shall include hidden files/dirs.
       * @param   inclSys    True if we shall include system files/dirs.
       * @see     GVfs#fillList
       */
      virtual void fillList ( GVfs::List& list,
                              const GString& pattern,
                              bool inclFiles,
                              bool inclDirs,
                              bool inclHidden,
                              bool inclSys ) const;

      /**
       * Find the first filename item matching the specified directory,
       * and prepare for consecutive calls to {@link #findNext}
       * and {@link #findClose}.
       *
       * The caller must always make sure to call {@link #findClose}
       * when the filename item loading progress has finished, regardless
       * how and when it is finished.
       *
       * If possible, the ".." directory should always be the item
       * that is returned by this method, if it exists in the
       * (specified-) virtual directory. All virtual file systems should
       * have such an item, except for {@link GVfsLocal} if
       * it is the root directory that is its current directory.
       *
       * @author  Leif Erik Larsen
       * @since   2002.08.22
       * @param   fitem           The file item object of where to put the
       *                          data of the first item.
       * @param   path            The path of which filename items to 
       *                          find. With or without standard FAT wildcard
       *                          characters in the filename part of the path.
       *                          An empty string means "all files in the 
       *                          current directory". If the path specified 
       *                          contains a trailing slash then we will 
       *                          assume the path is a directory and we will
       *                          find all files contained in that directory.
       * @param   cancelSem       If not null, the implementation should
       *                          periodically check the bool that is pointed
       *                          to by this parameter and make sure to
       *                          cancel any time consuming initialization
       *                          (if any) if the content of the byte is
       *                          true. It might be set true by some
       *                          background thread under the control of the
       *                          calling code for instance if the user
       *                          selects something like a cancel-command.
       * @param   preLoadCounter  If not null, the implementation should
       *                          periodically increment the int that is
       *                          pointed to by this parameter so that some
       *                          background thread under the control of the
       *                          calling code can monitor some progress.
       *                          For instance, the ZIP-file virtual file
       *                          system will increment the int once for
       *                          each file items that is preloaded from
       *                          the central directory of the ZIP-file.
       * @return  The hdir of which to be used on consecutive calls to
       *          {@link #findNext} and {@link #findClose}, or a zero
       *          value in case of any error.
       * @see     #findNext
       * @see     #findClose
       * @see     #fillList
       */
      virtual int findFirst ( GFileItem& fitem, 
                              const GString& path = GString::Empty, 
                              bool* cancelSem = null, 
                              int* preLoadCounter = null ) const = 0;

      virtual bool findFirstMightNeedLongTime () const = 0;

      virtual bool findNext ( int hdir, GFileItem& fitem ) const = 0;
      
      virtual void findClose ( int hdir ) const = 0;

      /**
       * Get a reference to the current directory of this file system.
       *
       * @author  Leif Erik Larsen
       * @since   2002.07.30
       * @param   slash   True if the returned directory string should always
       *                  contain a trailing slash, even if it is not the 
       *                  root directory. If it is the root directory that is
       *                  current then this parameter is ignored by most
       *                  implementing subclasses. E.g. for archive files the
       *                  root directory is always returned as an empty string
       *                  (see {@link GVfsArchiveFile#getCurrentDirectory}), 
       *                  and for the standard file system the root directory 
       *                  will always include a trailing slash regardless of
       *                  this parameter (e.g. "C:\").
       */
      virtual GString getCurrentDirectory ( bool slash ) const = 0;

      /**
       * Get the attributes (readonly, hidden, etc.) of the specified file.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.11
       * @param   path      The path of which file to get attributes.
       * @param   errorCode Returns GError::Ok on success, or else the
       *                    system dependent error code. If this argument
       *                    is null then we will not touch it.
       * @return  The attributes of the file, or {@link #FAttrError} on 
       *          any error. Possible flags are those defined by 
       *          {@link #FileAttributes}.
       * @see     #setFileAttributes
       * @see     #loadFileInfo
       */
      virtual int getFileAttributes ( const GString& path, GError* errorCode = null ) = 0;

      /**
       * Get the file information of the specified file.
       *
       * @author  Leif Erik Larsen
       * @since   2005.08.25
       * @param   path      The path of which file to get info about.
       * @param   fitem     On output: Contains the loaded file info.
       * @return  GError::Ok on success, or else the error code.
       * @see     #loadFileInfo
       * @see     #findFirst
       */
      GError getFileInfo ( const GString& path, GFileItem& fitem );

      /**
       * Get the current seek position of the specified file (not pipe).
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.11
       * @param   hfile  Handle of which file to get the seek position.
       * @param   err    Will return the system dependent error code in case
       *                 of any error. This parameter can be null, in which 
       *                 case we will not touch it.
       * @return  The current seek position of the file represented by the 
       *          specified handle, or -1 in case of any error.
       * @see     #setFileSeekPosFromStart
       */
      virtual longlong getFileSeekPos ( GVfs::FileHandle hfile, GError* err ) = 0;

      /**
       * Get the size of the specified open file.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.14
       * @param   hfile  Handle of which file to get the size.
       * @param   err    Will return the system dependent error code in case
       *                 of any error. This parameter can be null, in which 
       *                 case we will not touch it.
       * @return  The total size of the file represented by the specified 
       *          handle, or -1 in case of any error.
       * @see     #setFileSize
       * @see     #loadFileInfo
       */
      virtual longlong getFileSize ( GVfs::FileHandle hfile, GError* err ) = 0;

      /**
       * Get the short name of the implementing file system.
       * Typical return values are e.g.: "FAT", "FAT32", "NTFS", "HPFS", 
       * "ZIP", "FTP", etc.
       *
       * @author  Leif Erik Larsen
       * @since   2004.12.23
       */
      virtual const GString& getFileSystemName () const = 0;

      /**
       * Get the current number of free space (in bytes) of the "drive" of 
       * the Virtual File System. The returned number might or might not 
       * respect the physical free size. Some file systems have no physical 
       * "size" and might return a value less than the size of data that 
       * is possible to put on it. E.g. if the drive is configured to 
       * compress data. Another example is RAM disks, which typically 
       * reports a much larger number of free space than what are 
       * actually available. The subclass might also return zero, but
       * this doesn't neccessarily mean that there is "no free space".
       *
       * @author  Leif Erik Larsen
       * @since   2007.02.05
       */
      virtual longlong getFreeDriveSpace () const = 0;

      /**
       * Assume the specified path is for a file withing the VFS, and 
       * get a fully qualified path from the root VFS and through all 
       * parental VFSs up to and including this one. E.g. if the 
       * specified path is "subdir/myfile.txt" and the VFS is a ZIP-file
       * located at "C:\zips\myzip.zip" then this method will return 
       * "C:\zips\myzip.zip/subdir/myfile.txt".
       *
       * @author  Leif Erik Larsen
       * @since   2007.02.11
       */
      GString getFullVirtualPathTo ( const GString& path ) const;

      /**
       * Get the logical name of the file that defines the content of the
       * virtual file system. This is usually the filename only, without
       * directory.
       *
       * One special exception is for the {@link GVfsLocal},
       * which this method will return nothing but an empty string.
       *
       * @author  Leif Erik Larsen
       * @since   2002.08.20
       */
      virtual GString getLogicalSelfName () const = 0;

      /**
       * Get the fully qualified physical name of the file that defines
       * the content of the virtual file system.
       *
       * E.g. for an archive file system this method should return
       * the path of the archive file.
       *
       * One special exception is for the {@link GVfsLocal},
       * which this method will return nothing but an empty string.
       *
       * @author  Leif Erik Larsen
       * @since   2002.08.20
       */
      virtual GString getPhysicalSelfName () const = 0;

      /**
       * Get the root directory specification string.
       *
       * @author  Leif Erik Larsen
       * @since   2002.07.30
       */
      virtual GString getRootDir () const = 0;

      /**
       * This function is used to get a file or directory name that is
       * guaranteed to be compatible with the FAT file system.
       *
       * It can be used if <i>originalPath</i> is known to be
       * incompatible with the underlying file system. For instance
       * if the filename is to long for the FAT file system.
       *
       * @author  Leif Erik Larsen
       * @since   2005.09.04
       * @see     #getUniqueFileName
       */
      GString getShortenedDirOrPath ( const GString& originalPath );

      /**
       * Get the string (usually a one character string) used to 
       * separate directory elements on this virtual file system.
       * On Windows System FS this is a backslash. On most archive 
       * files, and Linux System FS this is a forward slash.
       *
       * @author  Leif Erik Larsen
       * @since  2006.02.02
       */
      virtual const GString& getSlashStr () const = 0;

      /**
       * Find a filename that is not already in use in the specified
       * directory.
       *
       * The file is not created on the file system, so there is a 
       * theoretical possibility that the filename is no longer unique on 
       * after this method has returned (if the underlying system is 
       * a multitasking system).
       *
       * @author  Leif Erik Larsen
       * @since   2005.09.04
       * @param   dir       The directory of where to find an unique filename.
       * @param   prefix    Prefix of which to use on the name of the 
       *                    temporary file. Usually this should an empty 
       *                    string (e.g. {@link GString#Empty}).
       * @param   extention Extention of which to use on the name of the
       *                    temporary file. Usually this should be ".tmp".
       * @return  The full path of the unique filename.
       * @see     #getShortenedDirOrPath
       */
      GString getUniqueFileName ( const GString& dir, 
                                  const GString& prefix, 
                                  const GString& extention );

      /**
       * Walk the specified directory (which can contain a drive
       * specification as well) so that for instance
       * "C:\OS2\SYSTEM\..\..\OS2" change to "C:\OS2".
       *
       * Another example is that "C:\4OS2\.\4OS2"
       * will change to "C:\4OS2\4OS2".
       *
       * <b>Note</b> that this method will respect the "current directory"
       * of a drive, in case a relative directory is specified in path.
       * The path specified should therefore always be a fully qualified
       * absolute path, with or without a drive specification.
       *
       * @author  Leif Erik Larsen
       * @since   2002.08.04
       * @param   srcDir     The directory to walk.
       * @param   walkedDir  Returns the result of the walk.
       * @see     #TestDirectoryCircularity
       */
      virtual GError getWalkedPath ( const GString& srcDir, GString& walkedDir ) const;

      /**
       * Return true if and only if it is the root directory of the 
       * file system that is the current directory.
       *
       * @author  Leif Erik Larsen
       * @since   2004.05.04
       */
      virtual bool isCurrentDirectoryTheRoot () const = 0;

      /**
       * Check if the specified directory contains any files or 
       * directories and return true if and only if it doesn't.
       *
       * @author  Leif Erik Larsen
       * @since   2004.08.06
       * @param   dir  The directory of which to test, with or without 
       *               a trailing slash.
       * @param   ignore {@link #IDE_IgnoreNone} if we shall return true 
       *               if and only if the specified directory does not
       *               container either any files nor directories.<br>
       *               {@link #IDE_IgnoreDirs} if we shall return true only
       *               if the specified directory contains no files.<br>
       *               {@link #IDE_IgnoreFiles} if we shall return true only
       *               if the specified directory contains no directories.
       * @param   ok   If an error occured (e.g. the specified 
       *               directory does not exist) then we will return 
       *               false in this parameter, or else we will 
       *               return true in this parameter. Can be null,
       *               in which case we will not touch it at all.
       * @return  True if the specified directory does not contain 
       *          any files or directories other than "." and/or "..".
       *          Else we will return false.
       */
      bool isDirectoryEmpty ( const GString& dir, 
                              GVfs::IDE_Ignore ignore = IDE_IgnoreNone,
                              bool* ok = null ) const;

      /**
       * This method is called to check if a call to 
       * {@link #preparePhysicalFile} will possibly take so much
       * time that it is recommended that the caller should call it
       * in a worker background thread.
       *
       * @author  Leif Erik Larsen
       * @since   2002.09.05
       */
      virtual bool isFilePreparationPossiblyLengthy () const = 0;

      /**
       * Test if the specified character should be treated as a 
       * directory separator (slash-) character.
       *
       * @author  Leif Erik Larsen
       * @since   2004.12.23
       * @see     GFile#IsSlash
       * @see     #slash
       */
      virtual bool isSlash ( char chr ) const = 0;

      /**
       * Test if the specified path contains a trailing slash.
       *
       * @author  Leif Erik Larsen
       * @since   2005.09.01
       * @see     #isSlash
       * @see     #slash
       */
      bool isSlashed ( const GString& path ) const;

      /**
       * Load and return the attributes, size, path and time stamp(s) 
       * of the file represented by the specified file handle.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.29
       * @param   hfile  The handle of which file to load information.
       * @param   info   The object of where to return the loaded information.
       * @return  The system dependent error code.
       * @see     #getFileInfo
       * @see     #getFileAttributes
       * @see     #getFileSize
       * @see     #writeAttrAndTimes
       */
      virtual GError loadFileInfo ( GVfs::FileHandle hfile, GFileItem& info ) = 0;

      /**
       * Test if specified string does match the specified filter, which
       * may contain one or more FAT wildcards (that is '*' and '?' as
       * supported by the well known FAT file system).
       *
       * @author Leif Erik Larsen
       * @since  2005.03.08
       */
      virtual bool match ( const GString& str, const GString& filter, bool caseSen ) const;

      /**
       * Move or rename an existing file or directory, but fail if the
       * destination is not on the same VFS is this, or if the device/drive 
       * is not the same as of the source.
       *
       * The default implementation of this method does nothing 
       * but return {@link GError#NotSupported}.
       *
       * @author  Leif Erik Larsen
       * @since   2005.08.26
       * @param   existingName The path of the source file.
       * @param   newName      The path of the destination name.
       * @param   allowCopy    True if a "copy-first-then-remove-source"
       *                       operation is ok. That is if the destination 
       *                       drive device is not the same as the source
       *                       drive device. If false is specified then we 
       *                       will never attempt to move if the move require
       *                       such a "copy first". Mark that such a
       *                       "copy first" operation is not supported on 
       *                       all operationg systems.
       */
      virtual GError moveOrRenameFile ( const GString& existingName, 
                                        const GString& newName,
                                        bool allowCopy );

      /**
       * Open a new deletion handle to be used with 
       * {@link #removeFile} and/or {@link #removeDirectory}.
       *
       * @author  Leif Erik Larsen
       * @since   2005.08.31
       * @return  The handle of the opened deletion handle on success, 
       *          or else null on any error.
       * @see     #closeDeletion
       * @see     #performDeletion
       */
      virtual DeletionHandle openDeletion () = 0;

      /**
       * Open the specified file in the VFS.
       * <p>
       * The returned file handle might be used with functions such
       * as {@link #readFromFile}, {@link #writeToFile} and
       * {@link #setFileSize}.
       * <p>
       * Always remember to pass the returned file handle to
       * {@link #closeFile} when it is no longer needed. Except if
       * the returned file handle is null (du to an error) of
       * course.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.09
       * @param   path        The path of which file to open. The format 
       *                      of the path depends on the VFS.
       * @param   errorCode   Returns the system dependent error code, in
       *                      case of any error (that is; if the returned
       *                      file handle is null). Can be null. If it is
       *                      null we will not return the error code.
       * @param   modeOpt     Open in read and/or write mode.
       * @param   createOpt   Whether or not the file is to be created.
       *                      The file is never created if the mode option
       *                      does not open the file for writing. If the 
       *                      file is opened for read-only access the file 
       *                      will not be "created" anyway.
       * @param   shareOpt    Whether other processes is allowed to access
       *                      the file while being open by this application.
       * @param   flagsOpt    One or more of 
       *                      {@link #OF_FLAG_SEQUENTIAL_ACCESS},
       *                      {@link #OF_FLAG_RANDOM_ACCESS},
       *                      {@link #OF_FLAG_APPEND} (only applicable 
       *                      if file is opened for sequential writing).
       * @param   actionTaken Returns the action taken, if not null.
       * @return  The handle of the opened file on success, or else
       *          null on any error.
       * @see     #closeFile
       */
      virtual FileHandle openFile ( const GString& path, 
                                    GError* errorCode = null,
                                    OF_Mode modeOpt = Mode_ReadOnly,
                                    OF_Create createOpt = Create_Never,
                                    OF_Share shareOpt = Share_DenyWrite,
                                    int flagsOpt = OF_FLAG_SEQUENTIAL_ACCESS,
                                    OF_ActionTaken* actionTaken = null ) = 0;

      /**
       * Perform all file and/or directory deletion operations requested
       * on the specified deletion handle.
       *
       * @author  Leif Erik Larsen
       * @since   2005.08.31
       * @param   hdel The handle of which deletion operations to perform.
       * @return  GError::Ok on success (all deletion operations performed
       *          successfully) or else an error code on any error.
       * @see     #openDeletion
       */
      virtual GError performDeletion ( DeletionHandle hdel ) = 0;

      /**
       * Prepare the specified filename for physical file operations,
       * and return a new file item object representing the prepared 
       * physical file on the physical (local) file system.
       *
       * @author  Leif Erik Larsen
       * @since   2002.08.27
       * @param   fileName  The name of which file to prepare. 
       *                    Must be either the filename only (in which case
       *                    the file must be contained in the current 
       *                    directory of the VFS) or a fully qualified 
       *                    path of the file inside the VFS.
       * @param   stat      Object that can be used by some other thread 
       *                    to periodically get some status information 
       *                    about the progress, until this method finish.
       * @param   prefix    Optional prefix of the destination physical file.
       *                    See {@link GVfsLocal#createTemporaryFile} for 
       *                    more details. You can safely give an empty 
       *                    string here (e.g. {@link GString#Empty}).
       * @return  A new file item object created from the heap, or null
       *          if the user cancelled the operation. The returned object 
       *          is always created from the heap and ownership belongs 
       *          to the caller.
       * @throws  GIllegalStateException if this method is not supported by
       *                    this virtual file system.
       * @throws  GIOException in case of any IO error during the preparation.
       * @see     #isFilePreparationPossiblyLengthy
       */
      virtual File* preparePhysicalFile ( const GString& fileName, 
                                          WorkStatus& stat, 
                                          const GString& prefix );

      /**
       * Reads data from the specified file handle, which must have been 
       * opened for reading by {@link #openFile}, and positioned if needed.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.09
       * @param   hfile  Handle of which file stream to read from.
       * @param   buff   Buffer of where to put the read data.
       * @param   numBytesToRead  Number of bytes to read, maximum.
       * @param   numBytesActuallyRead  Returns the number of bytes actually
       *                          read. If this parameter is null then we
       *                          will not touch it. In case of end-of-file
       *                          we will return a zero value here.
       * @return  The system dependent error code.
       * @see     #writeToFile
       */
      virtual GError readFromFile ( GVfs::FileHandle hfile, 
                                    void* buff, 
                                    int numBytesToRead, 
                                    int* numBytesActuallyRead = null ) = 0;

      /**
       * Delete the directory that is represented by the last
       * filename element of the specified path, from this file system.
       *
       * <b>Note:</b> Depending on the underlying system, this method will
       * possibly not succeed if the directory is not empty and the 
       * <i>trashCan</i> parameter is false. Because we will not do any 
       * recursive operation here. If <i>trashCan</i> is true then the 
       * directory will probably not actually be deleted, but rather moved 
       * to the system dependent trash can. In that case we will therefore 
       * usually succeed even if the directory is not empty. However, if 
       * the trash can mechanism on the system is disabled then we might 
       * not succeed even if the <i>trashCan</i> parameter is true. Note 
       * also that on some systems (at least on Windows) the directory 
       * will not be moved to the trash can if the path specified is not
       * fully qualified, even if the <i>trashCan</i> parameter is true.
       *
       * <b>Note also:</b> That not all file systems has a trash can.
       * The trash can is probably supported only by the local file 
       * system ({@link GVfsLocal}). It is up to the subclassing file 
       * system to choose if - and how - such a trash can is actually 
       * supported.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.22
       * @param   hdel     The deletion handle of where to perform the 
       *                   delete operation. The VFS must support 
       *                   a zero handle here, in order to perform the 
       *                   deletion directly. But for speed optimized 
       *                   deletion of sets of files and/or directories,
       *                   some VFSs requires a non-zero handle here.
       *                   See the overriding implementation for 
       *                   VFS-specific details. 
       *                   E.g. {@link GVfsLocal#removeDirectory}.
       * @param   path     The path of which directory to delete.
       * @param   trashCan True if we shall prefer moving the directory to
       *                   something like a Recycle Bin (on Windows) or 
       *                   Trash Can (on OS/2).
       * @param   ownerWin Handle of which window to use as the owner of 
       *                   the progress dialog, in case the system will 
       *                   display such a progress dialog. Can be null.
       *                   This parameter will be used only if <i>trashCan</i>
       *                   is true. Else it is always ignored.
       * @return  The system dependent error code.
       * @see     #removeFile
       */
      virtual GError removeDirectory ( DeletionHandle hdel, 
                                       const GString& path, 
                                       bool trashCan,
                                       HWND ownerWin = null ) = 0;

      /**
       * Delete the file or directory that is represented by the last
       * filename element of the specified path, from this file system.
       *
       * <b>Note:</b> If <i>trashCan</i> is true then the 
       * file will probably not actually be deleted, but rather moved 
       * to the system dependent trash can. On some systems (at least on
       * Windows) the file will not be moved to the trash can if the path 
       * specified is not fully qualified, even if the <i>trashCan</i> 
       * parameter is true.
       *
       * <b>Note also:</b> That not all file systems has a trash can.
       * The trash can is probably supported only by the local file 
       * system ({@link GVfsLocal}). It is up to the subclassing file 
       * system to choose if - and how - such a trash can is actually 
       * supported.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.22
       * @param   hdel     The deletion handle of where to perform the 
       *                   delete operation. The VFS must support 
       *                   a zero handle here, in order to perform the 
       *                   deletion directly. But for speed optimized 
       *                   deletion of sets of files and/or directories,
       *                   some VFSs requires a non-zero handle here.
       *                   See the overriding implementation for 
       *                   VFS-specific details. 
       *                   E.g. {@link GVfsLocal#removeFile}.
       * @param   path     The path of which file to delete.
       * @param   trashCan True if we shall prefer moving the directory to
       *                   something like a Recycle Bin (on Windows) or 
       *                   Trash Can (on OS/2).
       * @return  The system dependent error code, or 0 on success.
       * @see     #removeDirectory
       */
      virtual GError removeFile ( DeletionHandle hdel,
                                  const GString& path,
                                  bool trashCan ) = 0;

      /**
       * Set the current directory of the file system.
       *
       * @author  Leif Erik Larsen
       * @since   2002.08.04
       */
      virtual GError setCurrentDirectory ( const GString& dir ) = 0;

      /**
       * Set the attributes (readonly, hidden, etc.) of the specified file.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.11
       * @param   path    The path of which file to set attributes.
       * @param   attr    The attribute bits of which to set.
       *                  Possible flags are those defined by 
       *                  {@link #FileAttributes}.
       * @return  The system dependent error code.
       * @see     #getFileAttributes
       */
      virtual GError setFileAttributes ( const GString& path, int attr ) = 0;

      /**
       * Set the seek position of the specified file, relative to the
       * current position of the file.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.11
       * @param   hfile           The handle of which file to seek.
       * @param   distanceToMove  Number of bytes to seek, relative to the
       *                          current position of the file. Use a
       *                          negative value to seek backwards.
       *                          Specifying zero will do nothing but
       *                          return GError::Ok.
       * @return  The system dependent error code.
       * @see     #setFileSeekPosFromStart
       * @see     #setFileSeekPosFromEnd
       * @see     #getFileSeekPos
       */
      virtual GError setFileSeekPosFromCurrent ( GVfs::FileHandle hfile, 
                                                 longlong distanceToMove ) = 0;

      /**
       * Set the seek position of the specified file, relative to the
       * end position of the file.
       *
       * @author  Leif Erik Larsen
       * @since   2003.07.27
       * @param   hfile           The handle of which file to seek.
       * @param   distanceToMove  Number of bytes to seek backward from the
       *                          end of the file. Use a value of 0 to seek
       *                          to the end it self.
       * @return  The system dependent error code.
       * @see     #setFileSeekPosFromStart
       * @see     #setFileSeekPosFromCurrent
       * @see     #getFileSeekPos
       */
      virtual GError setFileSeekPosFromEnd ( GVfs::FileHandle hfile, 
                                             longlong distanceToMove ) = 0;

      /**
       * Set the seek position of the specified file, relative to the
       * start position of the file.
       *
       * @author  Leif Erik Larsen
       * @since   2003.07.27
       * @param   hfile           The handle of which file to seek.
       * @param   distanceToMove  Number of bytes to seek foreward from
       *                          the start of the file.
       * @return  The system dependent error code.
       * @see     #setFileSeekPosFromCurrent
       * @see     #setFileSeekPosFromEnd
       * @see     #getFileSeekPos
       */
      virtual GError setFileSeekPosFromStart ( GVfs::FileHandle hfile, 
                                               longlong distanceToMove ) = 0;

      /**
       * Set the size of the specified open file.
       * The file must have been opened for random write access.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.11
       * @return  The system dependent error code in case of any error.
       * @see     #getFileSize
       */
      virtual GError setFileSize ( GVfs::FileHandle hfile, longlong size ) = 0;

      /**
       * Append a slash character to the specified path, if needed.
       *
       * A slash will not be appended if the path specified is empty,
       * or if the path specified already has a trailing slash.
       *
       * The slash character depends on the implementing file system
       * and operating system. Typically it is a either a backslash 
       * or a normal slash.
       *
       * @author  Leif Erik Larsen
       * @since   2004.12.18
       * @see     GFile#Slash
       * @see     #isSlash
       */
      virtual GString& slash ( GString& dir ) const = 0;

      /**
       *
       * @author  Leif Erik Larsen
       * @since   2002.08.04
       * @return  {@link GError#Ok} if there are no directory circularity
       *          possibilities, or else {@link GError#CircularityRequested}. 
       *          It can also possibly return some other error code if we 
       *          fail to query the current drive and directory or if we 
       *          fail to "walk" the specified directory in memory.
       * @see     #getWalkedPath
       */
      static GError TestDirectoryCircularity ( class GVfs& srcVfs, 
                                               const GString& srcDir, 
                                               class GVfs& dstVfs,
                                               const GString& dstDir );

      /**
       * Make sure every slash character (if any) in the specified path 
       * are compatible with the slashes of this virtual file system.
       *
       * E.g. on the standard file system of OS/2 and Windows this method 
       * will convert all foreward slashes to backslahes, while on the 
       * standard file system on Linux all backslahes will be converted 
       * to foreward slashes. Some archive file VFSs (e.g. ZIP) also uses  
       * foreward slashes for directory separators.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.23
       */
      virtual void translateSlashes ( GString& path ) const = 0;

      /**
       * Walk the specified path in memory, following the specified
       * directory.
       *
       * For instance, if path="C:/my/dir" and walkDir="../test"
       * then we will return "C:/my/test".
       *
       * <b>Note</b> that this method will not respect the "current directory"
       * of a drive, in case a relative directory is specified in path.
       * The path specified should therefore always be a fully qualified
       * absolute path, with or without a drive specification.
       *
       * @author  Leif Erik Larsen
       * @since   2002.08.04
       * @param   path    On input: The starting point of the path of which
       *                  to be walked.
       *                  On output: The result path of the walk operation.
       * @param   walkDir The directory of which to walk.
       * @return  True on success, or else false on any error. The most
       *          likely error is that the walkDir is illegal, for instance
       *          if it contains to many ".."'s so that we must
       *          walk "outside" the path.
       * @see     #getWalkedPath
       */
      virtual bool walkPath ( GString& path, const GString& walkDir ) const;

      /**
       * Write the file attributes and time stamps represented by the
       * specified file item, to the specified file handle or directory. 
       * The file handle, if not null, must be a valid system dependent 
       * handle of a file that is still open, and which has been opened 
       * in write mode.
       *
       * <b>Mark: </b> Win95/98 has no way to change the time stamps on 
       * a directory. Thus, if the specified path is a directory we will 
       * not attempt setting its time stamps. Since this is not an error 
       * (but a limitation of the underlying system) we will not return 
       * an error but instead just silently ignore this and set just 
       * the attributes (not file stamps).
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.29
       * @param   hfile   Handle of the file. Opened in write-mode.
       *                  If this handle is null we will attempt using
       *                  the "path" argument to define which file of 
       *                  which to update. In order to change the 
       *                  attributes and time stamps of a directory, 
       *                  this argument must always be null.
       * @param   info    The file information of which to write.
       * @param   path    The path (usually a directory) of which file
       *                  to update, in case "hfile" is null.
       * @return  The system dependent error code.
       * @see     #loadFileInfo
       */
      virtual GError writeAttrAndTimes ( GVfs::FileHandle hfile, 
                                         const class GFileItem& info,
                                         const GString& path = GString::Empty ) = 0;

      /**
       * Writes data to the specified file handle, which must have been
       * opened for writing by {@link #openFile}, and positioned if needed.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.10
       * @param   hfile  Handle of which file stream to write to.
       * @param   buff   Buffer of which bytes to write.
       * @param   numBytesToWrite Number of bytes to read, maximum.
       * @param   numBytesActuallyWritten  Returns the number of bytes
       *                          actually written. If this parameter is
       *                          null then we will not touch it.
       * @return  GError::Ok on success, or else the system dependent
       *          error code.
       * @see     #readFromFile
       */
      virtual GError writeToFile ( GVfs::FileHandle hfile, 
                                   const void* buff, 
                                   int numBytesToWrite, 
                                   int* numBytesActuallyWritten = null ) = 0;

      virtual bool supportsChangeFileNameCase () const = 0;
      virtual bool isFileNameCasePreserved () const = 0;
      virtual bool isFileNameCaseSensitive () const = 0;
};

#endif
