/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_VFSSTACK
#define __GLIB_VFSSTACK

#include "glib/util/GStack.h"
#include "glib/vfs/GVfs.h"

/**
 * Class used to represent a stack of one or more VFSs involved in
 * a specific directory path. The peek element in the stack is always 
 * the VFS of the last (rightmost) part of the virtual path.
 * The root element is always an instance of {@link GVfsLocal}, 
 * and represents the physical file system of the underlying system.
 *
 * @author  Leif Erik Larsen
 * @since   2005.09.01
 */
class GVfsStack : public GStack<GVfs>
{
   public:

      GVfsStack ();
      virtual ~GVfsStack ();

   public:

      class GVfsLocal& backToRoot ();
      GString getFullVirtualPath ( bool slash ) const;
      bool isRootOnly () const;
      void pop ();
      void push ( GVfs* v, bool autoDelete = true );
      const class GVfsLocal& root () const;
      class GVfsLocal& root ();
      const class GVfs& peek () const;
      class GVfs& peek ();
      GError walkPath ( const GString& pathToWalk, GString* autoSelect = null );

   protected:

      /** 
       * Used by {@link #walkPath} to check if a file is a supported 
       * archive file of some format. By default, this method does nothing 
       * but return false. It is up to the overriding subclass to 
       * implement support for various archive file formats.
       *
       * @author  Leif Erik Larsen
       * @since   2005.09.01
       */
      virtual bool isSupportedZipOrArchiveFile ( const GString& path ) const;

      /** 
       * Used by {@link #walkPath} to walk into an archive file 
       * of some format. By default, this method does nothing but 
       * return {@link GError#NotSupported}. It is up to the 
       * overriding subclass to implement support for various 
       * archive file formats.
       *
       * @author  Leif Erik Larsen
       * @since   2005.09.01
       */
      virtual GError walkIntoArchiveFile ( const GString& fname );
};

#endif
