/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __LCMD_DIRPROGRESSBASE
#define __LCMD_DIRPROGRESSBASE

#include "glib/gui/GWorkerThread.h"

/**
 * An (optionally-) recursive directory file item processor class that makes 
 * it easy to write code that should do some operation on a set of files in 
 * a directory or directory tree. The file operation(s) are done in a 
 * background thread that is automatically managed by this class.
 *
 * Example of commands using this class are: 
 * {@link LCmdMainWindow#cmdChooseFileAttrib},
 * {@link LCmdMainWindow#cmdChooseFileNameCase},
 * {@link LCmdMainWindow#cmdChooseFileTime}.
 *
 * We extend two classes:
 * <b>1)</b>
 * <i>GDialogMessageHandler</i> is extended in order to have the
 * method {@link #handleDialogMessage} called to handle messages for the 
 * dialog that is used to let the user edit the progress properties as must 
 * be configured by the user <i>before</i> the progress background
 * thread is started.
 * <b>2)</b>
 * <i>GWorkerThread</i> is extended in order to have the method 
 * {@link #runTheWorkerThread} called in order to perform the background 
 * operation in a background thread. The default implementation of that 
 * method will call {@link #handleFileItem} once for each matching file 
 * item that the directory processor is configured to handle.
 *
 * @since   2000.07.12
 * @author  Leif Erik Larsen
 */
class LCmdDirProgressBase : public GDialogMessageHandler, 
                            public GWorkerThread
{
   public:

      /** Working file panel. */
      class LCmdFilePanel* panel;

      /** The Virtual File System of where to perform. */
      class GVfs& vfs;

      /** Desc. to show in dialog to edit which attributes to change. */
      GString cDescription;

      /** Text to show in title bar of progress bar window. */
      GString progressTitle;

      /** True if disable the "Include Subdirs" from dialog. */
      bool bDisableInclSubdirs;

      /** True if user request us to recurse through subdirectories also. */
      bool bInclSubdirs;

      /** True if we shall use the file filter options in <i>lcmd->options.fileFilter.*</i>. */
      bool useFilter;

      /** True if user canceled dialog before progress started. */
      bool bCanceled;

      /** Current working directory of progress (secondary thread). */
      GString curDir;

      /** Subclass must set this to true if we shall reread filename items into the panel when progress has finsihed. */
      bool rereadPanelWhenFinish;

      /** Subclass must set this to true if we shall recalc the panel (item widths, column widths, etc.) when progress has finished. */
      bool recalcPanelWhenFinish;

      /** True if we shall auto skip all furter files that we can't process due to some error. */
      bool skipAllByDefault;

      /** True if we shall select item in panel which is about being processed by the background thread. */
      bool selectProcessingItem;

      /** True if we shall operate on marked items only (except for eventually files in sub-directories). */
      bool markedItemsOnly;

      /** Title text to use in the "Now updating" group box of the progress bar window. */
      const char* titleStatus;

      /** Pointer to which file filter to use if useFilter is true. */
      class LCmdDlgFileFilterProperties* pFilter;

      /** Starting directory without pre- nor postfix backslash (that is "" to start from the root directory), or NULL to use the current directory of the specified file panel as the start directory. */
      const char* startDir;

   protected:

      /** True as long as there is no error. */
      volatile bool statusOK;

   private:

      /** Flags used to special what kind of files to look for with DosFindFirst/Next. */
      bool inclFiles;
      bool inclDirs;
      bool inclHidden;
      bool inclSys;

   protected:

      /**
       * @author  Leif Erik Larsen
       * @since   2005.03.08
       * @param   panel   The file panel for which this progress is to 
       *                  perform. Can be null, in which case no file panel
       *                  will be updated when progress has finished. 
       *                  A null argument here is probably used only
       *                  with {@link LCmdDirCache}.
       */
      explicit LCmdDirProgressBase ( class GVfs& vfs,
                                     class LCmdFilePanel* panel,
                                     bool markedOnly = true,
                                     bool selectProc = true,
                                     const char* titleStatus = " Now Updating ",
                                     const char* startDir = null,
                                     bool inclFiles = true,
                                     bool inclDirs = true,
                                     bool inclHidden = true,
                                     bool inclSys = true );

   public:

      virtual ~LCmdDirProgressBase ();

   public:

      bool setup ();

      /**
       * Start and execute the directory processor in a background thread,
       * while the progress dialog is shown to the user.
       *
       * The secondary thread will be executed upon WM_INITDLG of the
       * progress bar window.
       *
       * @param  dlgID   Resource ID of which dialog to execute to let
       *                 user edit progress options. This dialog will be
       *                 skipped if an empty string is specified.
       * @param  title   The title text of the progress window.
       * @param  drive   Drive (1=A, 2=B, 3=C, etc.) of which to process. If
       *                 a value of zero is specified then we will use the
       *                 current drive and directory of the file panel. Else
       *                 we will use the specified drive and the starting
       *                 directory as was given to the constructor of this
       *                 object.
       * @return True on success, or else false on any error or cancel.
       */
      bool startTheDirectoryProcessor ( const GString& dlgID, 
                                        const GString& title, 
                                        int drive = 0 );

      /**
       * Do a call to handleFileItem(), but do the wrapper logic in
       * addition so that the special handleFileItem() can be as simple
       * as possible.
       */
      bool wrapHandleFileItem ( class LCmdFileItem* pFile, 
                                int fileIndex, 
                                bool *bSkipped );

      bool recurseSubDir ();

      bool processFilePanel ();

      bool setPathInfo ( const class LCmdFileItem& fitem, 
                         bool* skipped, 
                         bool* itemUpdated );

   private:

      /**
       * This is the main entry point of the secondary thread.
       * Overrides {@link GWorkerThread#runTheWorkerThread}.
       */
      virtual void runTheWorkerThread ( class GWorkerThread& worker );

      /**
       * Overrides {@link GWorkerThread#onWorkerThreadInitDialog}.
       */
      virtual void onWorkerThreadInitDialog ( class GWorkerThread& worker, 
                                              class GDialogPanel& monitor );

      /**
       * Overrides {@link GWorkerThread#onWorkerThreadCommand}.
       */
      virtual void onWorkerThreadCommand ( class GWorkerThread& worker, 
                                           class GDialogPanel& monitor, 
                                           const GString& compID );

      /**
       * Overrides {@link GWorkerThread#onWorkerThreadUserEvent}.
       */
      virtual void onWorkerThreadUserEvent ( class GWorkerThread& worker, 
                                             class GDialogPanel& monitor, 
                                             const GString& msgID, 
                                             GObject* userParam );

   protected:

      /**
       * This is the method which is to be specially written by the
       * subclassing class.
       *
       * It will be called once for each file in the directory, including
       * subdirectories if bInclSubdirs is set to be true.
       * 'fileIndex' will be -1 for items that doesn't come from the file panel
       * but rather comes from one of the recursive subdirectories.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.12
       * @return  True on success, or else false on cancel or any error.
       */
      virtual bool handleFileItem ( class LCmdFileItem* pFile, 
                                    int fileIndex, 
                                    bool* bSkipped, 
                                    bool* itemUpdated ) = 0;

      /**
       * The message handler of the dialog box of where user can edit the
       * properties of the progress that will be handled by handleFileItem().
       *
       * Subclass must call this method for all messages, even if the
       * message was also handled by the subclass.
       */
      virtual bool handleDialogMessage ( class GDialogMessage& msg );
};

#endif // #ifndef __LCMD_DIRPROGRESSBASE
