/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdDlgCompareDirs.h"
#include "lcmd/LCmdFilePanel.h"
#include "lcmd/LCmdOptions.h"

#include "glib/GProgram.h"
#include "glib/gui/GDialogPanel.h"
#include "glib/gui/event/GDialogMessage.h"
#include "glib/io/GIOException.h"

LCmdDlgCompareDirs::LCmdDlgCompareDirs ( LCmdFilePanel& panel )
                   :LCmdDirProgressBase(panel.vfs.peek(), &panel, false, false, "%TxtCompareDirs_NowComparing"),
                    pOpt(&LCmdOptions::GetOptions().cmpDirs), // Make this a fast one
                    panel1(panel),
                    panel2(panel.getOppositePanel())
{
}

LCmdDlgCompareDirs::~LCmdDlgCompareDirs ()
{
}

bool LCmdDlgCompareDirs::handleFileItem ( LCmdFileItem* pFile,
                                          int fileIndex,
                                          bool* /*bSkipped*/,
                                          bool* itemUpdated )
{
   LCmdFileItem& pFile1 = *pFile;

   *itemUpdated = false;

   // Check if item is contained in both panels.
   const GString& fname = pFile1.getName();
   const GString& fext = pFile1.getExtension();
   int idx = panel2.findItem(fname, fext);
   if (idx < 0)
   {
      if (pOpt->tagUnmatched)
         panel1.toggleTag(fileIndex, false);
      return statusOK;
   }

   LCmdFileItem& pFile2 = panel2.items.get(idx);

   // Check if both items are of the same type (file or directory).
   if (pFile1.isDirectory() != pFile2.isDirectory())
   {
      if (pOpt->tagUnmatched)
         panel1.toggleTag(fileIndex, false);
      return statusOK;
   }

   // Compare the time stamp.
   if (pOpt->timeStampOpt != TAGTMST_IGNORE)
   {
      const GTimeStamp& time1 = pFile1.timeWrite;
      const GTimeStamp& time2 = pFile2.timeWrite;

      switch (pOpt->timeStampOpt)
      {
         case TAGTMST_NEWER:
         {
            if (time1 > time2)
            {
               panel1.toggleTag(fileIndex, false);
               return statusOK;
            }
            break;
         }

         case TAGTMST_OLDER:
         {
            if (time1 < time2)
            {
               panel1.toggleTag(fileIndex, false);
               return statusOK;
            }
            break;
         }

         case TAGTMST_DIFF:
         {
            if (time1 != time2)
            {
               panel1.toggleTag(fileIndex, false);
               return statusOK;
            }
            break;
         }

         case TAGSIZE_IGNORE:
         default:
            break;
      }
   }

   // Compare the size.
   if (pOpt->sizeOpt != TAGSIZE_IGNORE)
   {
      switch (pOpt->sizeOpt)
      {
         case TAGSIZE_LARGER:
         {
            if (pFile1.fileSize > pFile2.fileSize)
            {
               panel1.toggleTag(fileIndex, false);
               return statusOK;
            }
            break;
         }

         case TAGSIZE_SMALLER:
         {
            if (pFile1.fileSize < pFile2.fileSize)
            {
               panel1.toggleTag(fileIndex, false);
               return statusOK;
            }
            break;
         }

         case TAGSIZE_DIFF:
         {
            if (pFile1.fileSize != pFile2.fileSize)
            {
               panel1.toggleTag(fileIndex, false);
               return statusOK;
            }
            break;
         }

         case TAGSIZE_IGNORE:
         default:
            break;
      }
   }

   // Compare contents of the two files.
   if (pOpt->compareContents && !pFile1.isDirectory() && !pFile2.isDirectory())
   {
      GVfs& vfs1 = panel1.vfs.peek();
      GVfs& vfs2 = panel2.vfs.peek();
      GString path1 = pFile1.getFullPath();
      GString path2 = pFile2.getFullPath();

      try {
         if (!GFile::FilesEquals(vfs1, path1, vfs2, path2))
            panel1.toggleTag(fileIndex, false);
      } catch (GIOException& /*e*/) {
         // TODO: Show error message (?)
         // statusOK = false;
         // Tag the file to let the user see that we could not compare
         // the content of the files.
         panel1.toggleTag(fileIndex, false);
      }
   }

   return statusOK;
}

void LCmdDlgCompareDirs::execute ()
{
   // Execute the secondary thread as soon as the progress bar has been created
   if (startTheDirectoryProcessor("DlgCompareDirs", "%Txt_DlgCompareDirs_TitleProgBar"))
   {
      if (panel->markedFilesCount == 0)
      {
         // "The directories appear to be identical."
         GStringl msg("%TxtCompareDirs_DirsAppearIdentical");
         GProgram& prg = GProgram::GetProgram();
         GWindow& mwin = prg.getMainWindow();
         mwin.showMessageBox(msg, GMessageBox::TYPE_INFO);
      }
   }
}

bool LCmdDlgCompareDirs::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG:
      {
         LCmdDirProgressBase::handleDialogMessage(msg);
         dlg.setComponentValue(GInteger::ToString(122 + pOpt->timeStampOpt), true);
         dlg.setComponentValue(GInteger::ToString(105 + pOpt->sizeOpt), true);
         dlg.setComponentValue("111", pOpt->tagUnmatched);
         dlg.setComponentValue("113", pOpt->compareContents);
         dlg.setComponentFocus("DLG_OK");
         return true;
      }

      case GM_COMMAND:
      {
         LCmdDirProgressBase::handleDialogMessage(msg);
         GString cmdID = msg.getParam1String();
         if (cmdID == "DLG_OK")
         {
            // ---
            if (dlg.getComponentBoolValue("122"))
               pOpt->timeStampOpt = TAGTMST_IGNORE;
            else
            if (dlg.getComponentBoolValue("123"))
               pOpt->timeStampOpt = TAGTMST_NEWER;
            else
            if (dlg.getComponentBoolValue("124"))
               pOpt->timeStampOpt = TAGTMST_OLDER;
            else
            if (dlg.getComponentBoolValue("125"))
               pOpt->timeStampOpt = TAGTMST_DIFF;

            // ---
            if (dlg.getComponentBoolValue("105"))
               pOpt->sizeOpt = TAGSIZE_IGNORE;
            else
            if (dlg.getComponentBoolValue("106"))
               pOpt->sizeOpt = TAGSIZE_LARGER;
            else
            if (dlg.getComponentBoolValue("107"))
               pOpt->sizeOpt = TAGSIZE_SMALLER;
            else
            if (dlg.getComponentBoolValue("108"))
               pOpt->sizeOpt = TAGSIZE_DIFF;

            // ---
            pOpt->tagUnmatched = dlg.getComponentBoolValue("111");
            pOpt->compareContents = dlg.getComponentBoolValue("113");

            // Unselect all items (both directories and files) of both
            // panels so that initially no items are selected.
            {
              LCmdFilePanelSelectOptions filt;
              filt.filter = "*";
              filt.inclFiles = true;
              filt.inclDirs = true;
              panel1.unselect(filt);
              panel2.unselect(filt);
            }

            dlg.dismiss(cmdID);
         }
         return true;
      }

      default:
         return LCmdDirProgressBase::handleDialogMessage(msg);
   }
}
