/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdFilePanel.h"
#include "lcmd/LCmdFilePanelFrame.h"
#include "lcmd/LCmdDirCache.h"
#include "lcmd/LCmdCopy.h"
#include "lcmd/LCmdDlgFileDetails.h"
#include "lcmd/LCmdDlgSelectFiles.h"
#include "lcmd/LCmdDynaFileSearch.h"
#include "lcmd/LCmdProcess.h"
#include "lcmd/LCmdSubClient.h"
#include "lcmd/LCmdVfsFilePreparator.h"
#include "lcmd/LCmd.h"

#include "glib/gui/GScrollbar.h"
#include "glib/gui/GUserStringDlg.h"
#include "glib/gui/border/GEmptyBorder.h"
#include "glib/gui/layout/GBorderLayout.h"
#include "glib/gui/event/GUserMessage.h"
#include "glib/gui/dragndrop/GDragInfo.h"
#include "glib/gui/dragndrop/GDragOverAnswer.h"
#include "glib/util/GIsHereTracker.h"
#include "glib/util/GStringUtil.h"
#include "glib/util/GMath.h"
#include "glib/sys/GSystem.h"
#include "glib/vfs/GVfsZip.h"
#include "glib/exceptions/GIllegalStateException.h"

DEFINE_COMMAND_TABLE(LCmdFilePanel);
   ADD_COMMAND("cmdCycleIconTypes",  cmdCycleIconTypes);
   ADD_COMMAND("cmdSortByType",      cmdSortByType);
   ADD_COMMAND("cmdSortByDate",      cmdSortByDate);
   ADD_COMMAND("cmdSortByExtension", cmdSortByExtension);
   ADD_COMMAND("cmdSortByName",      cmdSortByName);
   ADD_COMMAND("cmdSortBySize",      cmdSortBySize);
   ADD_COMMAND("cmdSortByTime",      cmdSortByTime);
END_COMMAND_TABLE;

LCmdFilePanel::LCmdFilePanel ( LCmdFilePanelFrame& frame,
                               const GString& constraints )
              :GWindow(&frame,
                       "FilePanel",
                       constraints,
                       WS2_IGNORE_COLORS_AND_FONT_PROFILE | WS2_OS2Y | WS2_USE_SAME_PROFILE_SECTION_NAME_AS_PARENT),
               recentDragOverItemIndex(-1),
               iconDirUp(null),
               iconDirDir(null),
               iconDirZip(null),
               iconFileExe(null),
               iconFileDoc(null),
               iconFileFile(null),
               isDragOverEmphasised(false),
               frameWin(frame),
               headerWin(frame.headerWin),
               infoBar(frame.infoBar),
               hWndObjRenderer(null),
               items(*this),
               markedFilesCount(0),
               sizeOfMarkedFiles(0),
               makeDir(),
               colors(),
               selectOpt(),
               sortOpt(),
               filter(),
               startup(),
               view(),
               iconSize(0),
               objRender(*this),
               info(*this),
               currentView(null),
               previousViewMode(LCmdFilePanelViewOptions::VIEWMODE_BRIEF),
               tabs("Views", GBorderLayout::CENTER, *this),
               viewBrief(*this, "Brief"),
               viewWide(*this, "Wide"),
               viewFull(*this, "Full"),
#if __ENABLE_TREE_VIEW
               viewTree(*this, "Tree"),
#endif
               viewInfo(*this, "Info")
{
   // Initialize the "fp1" and the "fp2" instance variables of the
   // Larsen Commander object as early as possible. This is probably
   // the earliest point possible of where to initiate them.
   static int CountInstances = 0;
   if (++CountInstances == 1)
   {
      lcmd->fp1 = this;
      lcmd->curPanel = this;
   }
   else
   {
      lcmd->fp2 = this;
   }

   // ---
   setInsets(new GInsets(1, 1, 1, 1), true);
   setBorder(new GEmptyBorder(GColor::DGRAY.getLighter(64)), true);
   setLayoutManager(new GBorderLayout(), true);

   // ---
   iconDirUp = GIcon::GetIcon("IDP_DIRUP");
   iconDirDir = GIcon::GetIcon("IDP_DIRDIR");
   iconDirZip = GIcon::GetIcon("IDP_DIRZIP");
   iconFileExe = GIcon::GetIcon("IDP_FILEEXE");
   iconFileDoc = GIcon::GetIcon("IDP_FILEDOC");
   iconFileFile = GIcon::GetIcon("IDP_FILEFILE");

   // ---
   items.addFileItemContainerListener(this);

   // ---
   tabs.addTab("Brief", "%DlgOptView_MdBrief", GString::Empty, false, "IDP_TBBRIEF", &viewBrief, false);
   tabs.addTab("Full", "%DlgOptView_MdFull", GString::Empty, false, "IDP_TBFULL", &viewFull, false);
   tabs.addTab("Wide", "%DlgOptView_MdWide", GString::Empty, false, "IDP_TBWIDE", &viewWide, false);
#if __ENABLE_TREE_VIEW
   tabs.addTab("Tree", "%DlgOptView_MdTree", GString::Empty, false, "IDP_TBTREE", &viewTree, false);
#endif
   tabs.addTab("Info", "%DlgOptView_MdInfo", GString::Empty, false, "IDP_TBINFO", &viewInfo, false);
   tabs.addValueChangeListener(this);
   tabs.setPreferredSize(new GDimension(400, 250), true);
   tabs.setTabsVisible(false); // Hide the tabs bar and tab page frame controls. */

   // In order to prevent initial NullPointerException, set the default view.
   currentView = &viewBrief;
   previousViewMode = LCmdFilePanelViewOptions::VIEWMODE_BRIEF;

   // Set the initial default font and color options.
   defaultBackgroundColor = colors.itemBck;
   defaultForegroundColor =  colors.itemFileTxt;

   // Activate the popup menu of which to display when user
   // right-click on a file item or on some other space of the
   // file panel.
   setPopupMenu("FileItemContextMenu", lcmd->isUseFancyPopupMenues());

   // The file panel must use the current directory by default.
   queryCurrentDir(frameWin);
}

LCmdFilePanel::~LCmdFilePanel ()
{
   currentView = null;
   tabs.removeValueChangeListener(this);
}

LCmdFilePanel::VfsStack::VfsStack ()
{
}

LCmdFilePanel::VfsStack::~VfsStack ()
{
}

bool LCmdFilePanel::VfsStack::isSupportedZipOrArchiveFile ( const GString& path ) const
{
   return GFileItem::IsZipOrArchiveFile(path);
}

GError LCmdFilePanel::VfsStack::walkIntoArchiveFile ( const GString& fname )
{
   try {
      GVfs& parentVfs = peek();
      GVfsLocal& localVfs = root();
      GVfs* archive = LCmdFilePanel::CreateVfs(parentVfs, fname, localVfs);
      push(archive);
      return GError::Ok;
   } catch (GIOException& e) {
      return e.getSystemErrorCode();
   }
}

GVfs* LCmdFilePanel::CreateVfs ( GVfs& parentVfs, const GString& path, GVfsLocal& localVfs )
{
   // Create a physical temporary working version of the file,
   // for physical action. It is best to do this, so we can be sure 
   // that all kind of file operations are 100% supported during 
   // open and read of the archive file format, since we then can 
   // open and read the physical file instead of the virtual file 
   // directly inside the current VFS.
   GString prefix = "lc-";
   GVfs::WorkStatus stat;
   GVfs::File* vfile = parentVfs.preparePhysicalFile(path, stat, prefix);
   if (vfile == null) // If user did cancel. Should never happen, but in case.
      gthrow_(GIOException("Cancelled by user.", GError::FunctionFailed));

   // TODO: Assume the file is a ZIP-file, since only ZIP-files are supported by now.
   LCmdOptions& opt = lcmd->options;
   return new GVfsZip(localVfs, parentVfs, vfile, opt.vfsZip.toolPath, opt.vfsZip.argsDelete);
}

void LCmdFilePanel::writeProfile ( const GString& sectName, bool force )
{
   GSectionBag& ini = lcmd->getIniProperties();
   LCmdOptions& opt = lcmd->options;

   // ---
   GString sn = sectName;
   ini.putString(sn, "CurDir", getCurrentVfsDirectory(false), force || opt.saveOnExit.currentDirs);
   ini.putString(sn, "PrevDir", previousDir, force || opt.saveOnExit.currentDirs);
   ini.putString(sn, "SelectedItem", getCurItemName(), force || opt.saveOnExit.curSelFiles);

   // ---
   sn = sectName + ".Sorting";
   ini.putInt(sn, "What0", sortOpt.what[0], force || opt.saveOnExit.panelModeSort);
   ini.putInt(sn, "What1", sortOpt.what[1], force || opt.saveOnExit.panelModeSort);
   ini.putInt(sn, "What2", sortOpt.what[2], force || opt.saveOnExit.panelModeSort);
   ini.putInt(sn, "What3", sortOpt.what[3], force || opt.saveOnExit.panelModeSort);
   ini.putInt(sn, "How0", sortOpt.how[0], force || opt.saveOnExit.panelModeSort);
   ini.putInt(sn, "How1", sortOpt.how[1], force || opt.saveOnExit.panelModeSort);
   ini.putInt(sn, "How2", sortOpt.how[2], force || opt.saveOnExit.panelModeSort);
   ini.putInt(sn, "How3", sortOpt.how[3], force || opt.saveOnExit.panelModeSort);

   // ---
   sn = sectName + ".View";
   ini.putBool(sn, "Headerbar", view.showHeaderbar, force || opt.saveOnExit.visibleState);
   ini.putBool(sn, "Infobar", view.showInfobar, force || opt.saveOnExit.visibleState);
   ini.putBool(sn, "Icons", view.showItemIcon, force || opt.saveOnExit.visibleState);
   ini.putBool(sn, "Columnsbar", view.showColumnsBar, force || opt.saveOnExit.visibleState);
   ini.putInt(sn, "IconType", view.showIcon, force || opt.saveOnExit.visibleState);
   ini.putInt(sn, "Mode", view.viewMode, force || opt.saveOnExit.visibleState);
   ini.putBool(sn, "DriveButt", view.showHeaderDrivButt, force || opt.saveOnExit.visibleState);
   ini.putBool(sn, "UpDirButt", view.showHeaderUpDirButt, force || opt.saveOnExit.visibleState);
   ini.putBool(sn, "RootDirButt", view.showHeaderRootButt, force || opt.saveOnExit.visibleState);

   // ---
   sn = sectName + ".Startup";
   ini.putString(sn, "StartupDir", startup.dir, force || opt.saveOnExit.otherOptions);
   ini.putBool(sn, "StartupInCurDir", startup.useCurrentDir, force || opt.saveOnExit.otherOptions);
   ini.putBool(sn, "UseStartupDir", startup.useDir, force || opt.saveOnExit.otherOptions);

   // ---
   sn = sectName + ".Filter";
   ini.putString(sn, "Include", filter.include, force || opt.saveOnExit.otherOptions);
   ini.putString(sn, "Exclude", filter.exclude, force || opt.saveOnExit.otherOptions);
   ini.putBool(sn, "HideArchived", filter.hideArchive, force || opt.saveOnExit.otherOptions);
   ini.putBool(sn, "HideHidden", filter.hideHidden, force || opt.saveOnExit.otherOptions);
   ini.putBool(sn, "HideSystem", filter.hideSystem, force || opt.saveOnExit.otherOptions);
   ini.putBool(sn, "HideReadOnly", filter.hideReadOnly, force || opt.saveOnExit.otherOptions);
   ini.putBool(sn, "ShowDirs", filter.showDirs, force || opt.saveOnExit.otherOptions);
   ini.putBool(sn, "ShowFiles", filter.showFiles, force || opt.saveOnExit.otherOptions);

   // ---
   sn = sectName + ".Tagging";
   ini.putBool(sn, "TagFiles", selectOpt.inclFiles, force || opt.saveOnExit.otherOptions);
   ini.putBool(sn, "TagDirs", selectOpt.inclDirs, force || opt.saveOnExit.otherOptions);

   // ---
   sn = sectName + ".Colors";
   ini.putColor(sn, "ItemBck", colors.itemBck, force || opt.saveOnExit.colors);
   ini.putColor(sn, "ItemBckMarked", colors.itemBckMarked, force || opt.saveOnExit.colors);
   ini.putColor(sn, "ItemDirTxt", colors.itemDirTxt, force || opt.saveOnExit.colors);
   ini.putColor(sn, "ItemFileTxt", colors.itemFileTxt, force || opt.saveOnExit.colors);
   ini.putColor(sn, "ItemReadOnlyTxt", colors.itemReadOnlyTxt, force || opt.saveOnExit.colors);
   ini.putColor(sn, "ItemSysHiddenTxt", colors.itemSysHiddenTxt, force || opt.saveOnExit.colors);
   ini.putColor(sn, "ItemArchiveTxt", colors.itemArchiveTxt, force || opt.saveOnExit.colors);
   ini.putColor(sn, "ItemSelectedBckActive", colors.itemSelectedBckActive, force || opt.saveOnExit.colors);
   ini.putColor(sn, "ItemSelectedBckActiveMarked", colors.itemSelectedBckActiveMarked, force || opt.saveOnExit.colors);
   ini.putColor(sn, "ItemSelectedBckInactive", colors.itemSelectedBckInactive, force || opt.saveOnExit.colors);
   ini.putColor(sn, "ItemSelectedThinInactive", colors.itemSelectedThinInactive, force || opt.saveOnExit.colors);
   ini.putColor(sn, "ItemSelectedThinActive", colors.itemSelectedThinActive, force || opt.saveOnExit.colors);
   ini.putColor(sn, "HeaderActiveBck", colors.headerActiveBck, force || opt.saveOnExit.colors);
   ini.putColor(sn, "HeaderActiveTxt", colors.headerActiveTxt, force || opt.saveOnExit.colors);
   ini.putColor(sn, "HeaderInactiveBck", colors.headerInactiveBck, force || opt.saveOnExit.colors);
   ini.putColor(sn, "HeaderInactiveTxt", colors.headerInactiveTxt, force || opt.saveOnExit.colors);
   ini.putColor(sn, "HeaderActiveBckWhenInsideVfs", colors.headerActiveBckWhenInsideVfs, force || opt.saveOnExit.colors);
   ini.putColor(sn, "HeaderActiveTxtWhenInsideVfs", colors.headerActiveTxtWhenInsideVfs, force || opt.saveOnExit.colors);
   ini.putColor(sn, "HeaderInactiveBckWhenInsideVfs", colors.headerInactiveBckWhenInsideVfs, force || opt.saveOnExit.colors);
   ini.putColor(sn, "HeaderInactiveTxtWhenInsideVfs", colors.headerInactiveTxtWhenInsideVfs, force || opt.saveOnExit.colors);

   // ---
   // The default GWindow-implementation will write the current font-
   // and the current foreground and background color- options.
   GWindow::writeProfile(sectName, force);
}

void LCmdFilePanel::queryProfile ( const GString& sectName )
{
   GSectionBag& ini = lcmd->getIniProperties();

   // ---
   GString sn = sectName + ".Sorting";
   sortOpt.what[0] = (LCmdFilePanelSortOptions::SORT_WHAT) ini.getInt(sn, "What0", sortOpt.what[0]);
   sortOpt.what[1] = (LCmdFilePanelSortOptions::SORT_WHAT) ini.getInt(sn, "What1", sortOpt.what[1]);
   sortOpt.what[2] = (LCmdFilePanelSortOptions::SORT_WHAT) ini.getInt(sn, "What2", sortOpt.what[2]);
   sortOpt.what[3] = (LCmdFilePanelSortOptions::SORT_WHAT) ini.getInt(sn, "What3", sortOpt.what[3]);
   sortOpt.how[0] = (LCmdFilePanelSortOptions::SORT_HOW) ini.getInt(sn, "How0", sortOpt.how[0]);
   sortOpt.how[1] = (LCmdFilePanelSortOptions::SORT_HOW) ini.getInt(sn, "How1", sortOpt.how[1]);
   sortOpt.how[2] = (LCmdFilePanelSortOptions::SORT_HOW) ini.getInt(sn, "How2", sortOpt.how[2]);
   sortOpt.how[3] = (LCmdFilePanelSortOptions::SORT_HOW) ini.getInt(sn, "How3", sortOpt.how[3]);

   // ---
   sn = sectName + ".View";
   view.showHeaderbar = ini.getBool(sn, "Headerbar", view.showHeaderbar);
   view.showInfobar = ini.getBool(sn, "Infobar", view.showInfobar);
   view.showItemIcon = ini.getBool(sn, "Icons", view.showItemIcon);
   view.showColumnsBar = ini.getBool(sn, "Columnsbar", view.showColumnsBar);
   view.showIcon = (LCmdFilePanelViewOptions::SHOWICON) ini.getInt(sn, "IconType", view.showIcon);
   view.showHeaderDrivButt = ini.getBool(sn, "DriveButt", view.showHeaderDrivButt);
   view.showHeaderUpDirButt = ini.getBool(sn, "UpDirButt", view.showHeaderUpDirButt);
   view.showHeaderRootButt = ini.getBool(sn, "RootDirButt", view.showHeaderRootButt);
   try {
      int viewMode = ini.getInt(sn, "Mode", view.viewMode);
      setViewMode(LCmdFilePanelViewOptions::VIEWMODE(viewMode));
   } catch (GIllegalArgumentException& /*e*/) {
      // Nothing critical. User has probably hacked the ini-file.
      // Just continue with the default view mode active.
   }

   // ---
   // Which initial directory to use?
   sn = sectName + ".Startup";
   startup.dir = ini.getString(sn, "StartupDir", startup.dir);
   startup.useCurrentDir = ini.getBool(sn, "StartupInCurDir", startup.useCurrentDir);
   startup.useDir = ini.getBool(sn, "UseStartupDir", startup.useDir);
   if (!startup.useCurrentDir)
   {
      GString dir;
      if (startup.useDir)
         dir = startup.dir;
      else
         dir = ini.getString(sectName, "CurDir", getCurrentVfsDirectory(false));
      walkDir(dir, true);
      previousDir = ini.getString(sectName, "PrevDir", previousDir);
   }

   // ---
   // Which item to make initially selected.
   initiallySelectedItemName = ini.getString(sectName, "SelectedItem");

   // ---
   sn = sectName + ".Filter";
   filter.include = ini.getString(sn, "Include", filter.include);
   filter.exclude = ini.getString(sn, "Exclude", filter.exclude);
   filter.hideArchive = ini.getBool(sn, "HideArchived", filter.hideArchive);
   filter.hideHidden = ini.getBool(sn, "HideHidden", filter.hideHidden);
   filter.hideSystem = ini.getBool(sn, "HideSystem", filter.hideSystem);
   filter.hideReadOnly = ini.getBool(sn, "HideReadOnly", filter.hideReadOnly);
   filter.showDirs = ini.getBool(sn, "ShowDirs", filter.showDirs);
   filter.showFiles = ini.getBool(sn, "ShowFiles", filter.showFiles);

   // ---
   sn = sectName + ".Tagging";
   selectOpt.inclFiles = ini.getBool(sn, "TagFiles", selectOpt.inclFiles);
   selectOpt.inclDirs = ini.getBool(sn, "TagDirs", selectOpt.inclDirs);
   if (!selectOpt.inclFiles && !selectOpt.inclDirs)
      selectOpt.inclFiles = true;

   // ---
   sn = sectName + ".Colors";
   colors.itemBck = ini.getColor(sn, "ItemBck", colors.itemBck);
   colors.itemBckMarked = ini.getColor(sn, "ItemBckMarked", colors.itemBckMarked);
   colors.itemDirTxt = ini.getColor(sn, "ItemDirTxt", colors.itemDirTxt);
   colors.itemFileTxt = ini.getColor(sn, "ItemFileTxt", colors.itemFileTxt);
   colors.itemReadOnlyTxt = ini.getColor(sn, "ItemReadOnlyTxt", colors.itemReadOnlyTxt);
   colors.itemSysHiddenTxt = ini.getColor(sn, "ItemSysHiddenTxt", colors.itemSysHiddenTxt);
   colors.itemArchiveTxt = ini.getColor(sn, "ItemArchiveTxt", colors.itemArchiveTxt);
   colors.itemSelectedBckActive = ini.getColor(sn, "ItemSelectedBckActive", colors.itemSelectedBckActive);
   colors.itemSelectedBckActiveMarked = ini.getColor(sn, "ItemSelectedBckActiveMarked", colors.itemSelectedBckActiveMarked);
   colors.itemSelectedBckInactive = ini.getColor(sn, "ItemSelectedBckInactive", colors.itemSelectedBckInactive);
   colors.itemSelectedThinInactive = ini.getColor(sn, "ItemSelectedThinInactive", colors.itemSelectedThinInactive);
   colors.itemSelectedThinActive = ini.getColor(sn, "ItemSelectedThinActive", colors.itemSelectedThinActive);
   colors.headerActiveBck = ini.getColor(sn, "HeaderActiveBck", colors.headerActiveBck);
   colors.headerActiveTxt = ini.getColor(sn, "HeaderActiveTxt", colors.headerActiveTxt);
   colors.headerInactiveBck = ini.getColor(sn, "HeaderInactiveBck", colors.headerInactiveBck);
   colors.headerInactiveTxt = ini.getColor(sn, "HeaderInactiveTxt", colors.headerInactiveTxt);
   colors.headerActiveBckWhenInsideVfs = ini.getColor(sn, "HeaderActiveBckWhenInsideVfs", colors.headerActiveBckWhenInsideVfs);
   colors.headerActiveTxtWhenInsideVfs = ini.getColor(sn, "HeaderActiveTxtWhenInsideVfs", colors.headerActiveTxtWhenInsideVfs);
   colors.headerInactiveBckWhenInsideVfs = ini.getColor(sn, "HeaderInactiveBckWhenInsideVfs", colors.headerInactiveBckWhenInsideVfs);
   colors.headerInactiveTxtWhenInsideVfs = ini.getColor(sn, "HeaderInactiveTxtWhenInsideVfs", colors.headerInactiveTxtWhenInsideVfs);

   // ---
   // The default GWindow-implementation will load and activate the
   // font- and the foreground and background color- options. It is
   // important that the super implementation is called <i>after</i> we
   // have finisged load the other file panel options so that the super
   // implementation will set the correct initial colors, which we did
   // load by the above code.
   GWindow::queryProfile(sectName);
}

bool LCmdFilePanel::isCurrentPanel () const
{
   return this == lcmd->curPanel;
}

GString LCmdFilePanel::getPanelName ()
{
   if (this == lcmd->fp1)
      return GStringl("%Txt_FP_LeftPanel"); // Left Panel
   else
      return GStringl("%Txt_FP_RightPanel"); // Right Panel
}

LCmdFilePanel& LCmdFilePanel::getOppositePanel () const
{
   if (this == lcmd->fp1)
      return *lcmd->fp2;
   else
      return *lcmd->fp1;
}

bool LCmdFilePanel::isLeftPanel () const
{
   return this == lcmd->fp1;
}

bool LCmdFilePanel::isRightPanel () const
{
   return this == lcmd->fp2;
}

LCmdFilePanel& LCmdFilePanel::GetLeftPanel ()
{
   return *lcmd->fp1;
}

LCmdFilePanel& LCmdFilePanel::GetRightPanel ()
{
   return *lcmd->fp2;
}

LCmdFilePanel& LCmdFilePanel::GetCurrentPanel ()
{
   return *lcmd->curPanel;
}

bool LCmdFilePanel::fillList ( bool autoShowErrMsg, bool synchPaintPanel )
{
   if (items.isExecuting())
   {
      if (GLog::Filter(GLog::TEST))
         GLog::Log(this, "%s: Already running!", GVArgs(__FUNCTION__));
      return false; // Only one filename reader thread can run at once!
   }

   // Set current drive and directory.
   if (!activatePanelDriveAndDir(false, null, autoShowErrMsg))
      return false;

   // ---
   if (GLog::Filter(GLog::DEBUG))
      GLog::Log(this, "%s: Starting filename reader worker thread...", GVArgs(__FUNCTION__));

   // Since the progres dialog will be centered on its owner window
   // we don't want the file panel to be the owner if the file panels
   // are not currently visible. This is in order to prevent the
   // progress dialog to pop up outside the frame window are of our
   // application. Thus, if the file panels are currently hidden,
   // use the frame window as the owner, else use the file panel.
   GWindow* ownerw = this;
   if (!lcmd->options.panelsAreVisible)
      ownerw = &lcmd->cmdCont.conmon;

   // Start the filename reader worker thread.
   lcmd->mainWin.pushStatusbarText("%StrStatusbar_Status_ReadingFilenames");
   items.synchPaintPanel = synchPaintPanel;
   items.workModal(*ownerw);
   lcmd->mainWin.popStatusbarText();

   // ---
   if (GLog::Filter(GLog::DEBUG))
      GLog::Log(this, "%s: Filename reader worker thread has finished.", GVArgs(__FUNCTION__));

   LCmdFilePanel& oppositePanel = getOppositePanel();
   oppositePanel.info.updateUponReread();

   // ---
   LCmdFilePanelModeAbstract& curView = getCurrentView();
   curView.itemsListHasBeenRefreshed();

   // <i>initiallySelectedItemName</i> is non-empty only upon the
   // first time this method is called on each file panel object.
   // That is to initially activate the same item as of which was the
   // selected item upon the last time that the program was exited.
   if (initiallySelectedItemName != GString::Empty &&
      (lcmd->options.saveOnExit.everything || lcmd->options.saveOnExit.curSelFiles))
   {
      selectItem(initiallySelectedItemName);
      initiallySelectedItemName = GString::Empty;
   }

   // Invalidate all parts of the file panel that needs to be repainted.
   frameWin.headerWin.invalidateAll(true);
   frameWin.infoBar.updateAllFileItemInfoCells();

   return true;
}

bool LCmdFilePanel::reRead ( bool synchPaintPanel, int selectIndex )
{
   if (items.isExecuting())
      return false; // Only one filename reader thread can run at once!

   // We must also remember the name and index of the currently selected
   // filename item so that we can re-select it after the panel has been
   // re-read. If this filename item does no longer exist after reread,
   // then we will select the closest item by index instead.
   GString currentItemName = getCurItemName();
   int currentItemIndex = getCurrentSelectedIndex();

   // Read the filenames into the panel.
   if (!fillList(true, synchPaintPanel))
      return false;

   // Update the infobar cell that shows the total number of bytes 
   // in the marked files.
   if (markedFilesCount > 0)
      infoBar.updateAllFileItemInfoCells();

   // Select the item, as requested.
   if (selectIndex == -2)
   {
      ;
   }
   else
   if (selectIndex == -1)
   {
      if (selectItem(currentItemName) == -1)
      {
         if (currentItemIndex >= items.getCount())
            currentItemIndex = items.getCount() - 1;
         if (currentItemIndex >= 0)
            selectItem(currentItemIndex);
      }
   }
   else
   if (selectIndex >= 0)
   {
      selectItem(selectIndex);
   }

   return true;
}

bool LCmdFilePanel::walkDir ( const GString& pathToWalk_ref_, bool temp )
{
   // Use a copy of the referenced path, in case the path references the 
   // "previousDir" instance variable it self.
   const GString pathToWalk_ = pathToWalk_ref_;
   const GString prevDir_ = getCurrentVfsDirectory(false);

   // ---
   GString autoSelect;
   GError err = vfs.walkPath(pathToWalk_, &autoSelect);
   if (err != GError::Ok)
   {
      if (temp)
         return false;
      
      bool showErrMsg = true;
      GVfs& fs = vfs.peek();
      if (!fs.containsAnySlash(pathToWalk_))
      {
         // Open a dialog showing all directories which name matches the
         // specified filename, and let the user select a directory of
         // which to activate.
         GString src = lcmd->dirCache->pickFromList(pathToWalk_);
         if (src != GString::Empty)
            return walkDir(src, false);
         showErrMsg = false;
      }
      
      // Reactivate original directory.
      GString selItem = getCurItemName();
      err = vfs.walkPath(prevDir_);
      fillList();
      if (err == GError::Ok)
      {
         if (selItem != "")
            selectItem(selItem);
      }
      
      // "Error activating directory '%s'.\nMessage from the system: %s"
      if (showErrMsg)
      {
         GStringl errorMsg("%Txt_FP_Error_CantActivateDir", GVArgs(pathToWalk_).add(err.getErrorMessage()));
         showMessageBox(errorMsg, GMessageBox::TYPE_ERROR);
      }
      
      return false;
   }

   if (!temp)
   {
      // Add the current directory to the Directory History and remember 
      // it as the "previous" directory.
      previousDir = prevDir_;
      lcmd->cmdEntry.dirHist.add(prevDir_);

      // Add the newly activated directory to the Dynamic Directory Cache.
      if (lcmd->options.dirCache.addDirsDuringWalk)
      {
         if (!vfs.isRootOnly() && !lcmd->options.dirCache.includeVfsDirs)
         {
            // We are currently is some VFS (like ZIP, FT, etc.) and the 
            // user has configured us to not include such locations in
            // Dynamic Directory Cache. Respect it.
         }
         else
         {
            GString activatedDir = getCurrentVfsDirectory(false);
            lcmd->dirCache->addDirectoryToCache(activatedDir, true);
         }
      }

      // Update the drive button, and other elements of the headerbar window.
      headerWin.layout();

      // Fill in the filenames of the current dir in selected drive.
      if (!fillList())
         return false;

      // Autoselect the parent directory item in case we did go upward.
      if (autoSelect != "")
         selectItem(autoSelect);
   }

   return true;
}

void LCmdFilePanel::doEnter ( int idx, bool forceExecItem )
{
   if (idx <= -1)
      idx = getCurrentSelectedIndex();

   // If the user is currently about doing a dynamic filename
   // text search and has pressed the <enter> key then dismiss the dynamic
   // search text entry box and perform a normal <enter>.
   if (headerWin.isPerformingDynamicSearch())
      headerWin.dismissDynamicSearch();

   // If the Command Line Prompt Window is on and it contains some text then
   // we should perform the current text as either 1) input to a child
   // process that is currently waiting for data to its STDIN, 2) an internal
   // command, or 3) an external command.
   if (!forceExecItem && 
      (lcmd->cmdLine.isVisible() || !lcmd->options.panelsAreVisible))
   {
      GString cmd = lcmd->cmdEntry.getText();
      LCmdProcessLauncher* child = lcmd->cmdEntry.childWaitingForInput;
      if (child != null)
      {
         // Essentially the same code as in LCmdMainWindow::cmdFeedChildProcessStdIn()
         lcmd->cmdLine.setText(GString::Empty);
         int lcount = lcmd->conmon.getLinesCount();
         lcmd->conmon.gotoPos(lcount, 0);
         cmd += GString::EOL; // "\r\n"
         child->addTextToChildStdInQueue(cmd);
         lcmd->cmdEntry.childWaitingForInput = null;
         lcmd->cmdEntry.updateCommandLineColours();
         lcmd->mainWin.setStatusbarText(GString::Empty);
         return;
      }
      else
      if (cmd != GString::Empty)
      {
         // Do not attempt to run command if current drive and/or directory
         // is not activable, except if the command is to change drive only.
         bool isChangeDriveCommand = (cmd.length() == 2 && isalpha(cmd[0]) && cmd[1] == ':');
         if (!isChangeDriveCommand)
         {
            if (!activatePanelDriveAndDir(true))
               return;
         }

         int flags = LCmdCmdLineEntry::ECF_DO_ECHO;
         if (lcmd->options.conVarious.defaultCloseConOnExit)
            flags |= LCmdCmdLineEntry::ECF_CLOSEON_EXIT;

         lcmd->mainWin.setStatusbarText(GString::Empty);
         lcmd->conmon.gotoPos(lcmd->conmon.getLinesCount(), 0);
         lcmd->cmdEntry.executeCommand(cmd, GString::Empty, flags);
         lcmd->cmdLine.setText(GString::Empty);
         return;
      }
   }

   // ---
   lcmd->mainWin.setStatusbarText(GString::Empty);

   // Make sure that the correct directory is current.
   if (!activatePanelDriveAndDir(true))
      return;

   // In any other case we must try to execute the current selected
   // item in the active panel. If there are no current selected item
   // then we can't do this either. In that case we have nothing to do
   // but return in silence.
   if (idx < 0 || !lcmd->options.panelsAreVisible)
      return;

   // ---
   bool sysLaunch = false; // Until the opposite is proven.
   LCmdFileItem& fitem = items.get(idx);
   GString fname = fitem.getFileName();
   if (fitem.isDirectory() || fitem.isZipOrArchiveFile())
   {
      if (fname.endsWith(".zip", true) && lcmd->options.vfsZip.sysLaunchZipFiles)
         sysLaunch = true;
      // TODO: More code and tests will come here when we get richer VFS support!
      if (!sysLaunch)
         walkDownDir(idx);
   }
   else
   {
      sysLaunch = true;
   }

   // Launch as a system shell object if requested.
   if (sysLaunch)
      startSelectedProgram(idx);
}

bool LCmdFilePanel::startSelectedProgram ( int idx )
{
   if (idx <= -1)
      idx = getCurrentSelectedIndex();
   if (idx < 0)
      return false;

   LCmdFileItem& fitem = items.get(idx);
   if (fitem.isDirectory())
   {
      GString dir = getCurrentSysDirectory(false);
      GFile::Slash(dir);
      dir += fitem.getFileName();
      GFile::Slash(dir);
      return GSystem::OpenShellObject(dir);
   }

   // Execute current selected program, or open current selected file.
   LCmdCmdLineEntry& entry = lcmd->cmdLine.entry;
   int flags = LCmdCmdLineEntry::ECF_FORCE_NEW_SESSION |
               LCmdCmdLineEntry::ECF_RUN_AS_SHELL_OBJECT;
   if (lcmd->options.conVarious.defaultCloseConOnExit)
      flags |= LCmdCmdLineEntry::ECF_CLOSEON_EXIT;
   GString path = getCurItemPath(idx);
   if (path.containsWhiteSpace())
      path.insert('"').operator+=('"'); // Must be quoted, due to whitespace.
   return entry.executeCommand(path, GString::Empty, flags);
}

bool LCmdFilePanel::walkDownDir ( int idx )
{
   if (!lcmd->options.panelsAreVisible)
      return false;

   if (idx <= -1)
      idx = getCurrentSelectedIndex();
   if (idx < 0)
      return false;

   LCmdFileItem& fitem = items.get(idx);
   if (!fitem.isDirectory() && !fitem.isZipOrArchiveFile())
      return false;

   GString fname = getCurItemName(idx);
   if (fname == GString::Empty)
      return false;

   return walkDir(fname);
}

bool LCmdFilePanel::walkUpDir ()
{
   GVfs& fs = vfs.peek();
   if (vfs.isRootOnly() && fs.isCurrentDirectoryTheRoot())
      return false; // Root is active. Can't walk up from the root!
   else
      return walkDir("..");
}

void LCmdFilePanel::setShowHiddenFiles ( bool flag )
{
   if (flag == !filter.hideHidden && flag == !filter.hideSystem)
      return;

   filter.hideHidden = !flag;
   filter.hideSystem = !flag;
   reRead();
}

bool LCmdFilePanel::isShowHiddenFiles ()
{
   return !filter.hideHidden && !filter.hideSystem;
}

void LCmdFilePanel::setViewMode ( LCmdFilePanelViewOptions::VIEWMODE mode )
{
   if (mode == view.viewMode)
      return; // Specified view mode is already active.

   // We should automatically select the same filename item in the new 
   // view mode. So get the current selected index before we change the mode.
   int prevSel = getCurrentSelectedIndex();
   previousViewMode = view.viewMode;

   // ---
   GString id;
   switch (mode)
   {
      case LCmdFilePanelViewOptions::VIEWMODE_BRIEF: 
         currentView = &viewBrief;
         id = "Brief"; 
         break;

      case LCmdFilePanelViewOptions::VIEWMODE_WIDE: 
         currentView = &viewWide;
         id = "Wide"; 
         break;

      case LCmdFilePanelViewOptions::VIEWMODE_FULL: 
         currentView = &viewFull;
         id = "Full"; 
         break;

#if __ENABLE_TREE_VIEW
      case LCmdFilePanelViewOptions::VIEWMODE_TREE: 
         currentView = &viewTree;
         id = "Tree"; 
         break;
#endif

      case LCmdFilePanelViewOptions::VIEWMODE_INFO: 
         currentView = &viewInfo;
         id = "Info"; 
         break;

      default: 
         gthrow_(GIllegalArgumentException("Unknown view mode: " + GInteger::ToString(mode)));
   }

   // ---
   if (id != "")
   {
      view.viewMode = mode;
      tabs.activateTab(id);
      LCmdFilePanelModeAbstract& curView = getCurrentView();
      headerWin.invalidateClientAreaForText();
#if __ENABLE_TREE_VIEW
      if (view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_TREE ||
          previousViewMode == LCmdFilePanelViewOptions::VIEWMODE_TREE)
      {
         reRead(false, 0); 
      }
#endif
      curView.onViewHasBeenActivated();
      if (prevSel >= 0)
         selectItem(prevSel);
      else
         onItemSelectionHasChanged();
      if (id == "Info")
      {
         LCmdFilePanel& oppositePanel = getOppositePanel();
         oppositePanel.onItemSelectionHasChanged();
      }
   }
}

int LCmdFilePanel::findNextMatchingItem ( const GString& str, int startIndex )
{
   const int strLength = str.length();
   const int num = items.getCount();
   for (int count=0, i=startIndex+1; count<num; count++, i++)
   {
      if (i < 0)
         i = 0;
      else
      if (i >= num)
         i = 0;

      GString fname = items[i].getFileName();
      if (str.equalsIgnoreCaseNum(fname, strLength))
         return i;
   }

   return -1;
}

int LCmdFilePanel::findPrevMatchingItem ( const GString& str, int startIndex )
{
   const int num = items.getCount();
   for (int count=0, i=startIndex-1; count<num; count++, i--)
   {
      if (i < 0)
         i = num - 1;
      else
      if (i >= num)
         i = num - 1;

      if (str.equalsIgnoreCaseNum(items[i].getFileName(), str.length()))
         return i;
   }

   return -1;
}

bool LCmdFilePanel::findItemByKey ( char chr )
{
   GString str("%c", GVArgs(chr));
   int curSel = getCurrentSelectedIndex();
   int idx = findNextMatchingItem(str, curSel);
   if (idx >= 0)
   {
      selectItem(idx);
      return true;
   }

   return false;
}

bool LCmdFilePanel::startDynamicSearch ( char chr )
{
   if (!lcmd->options.panelsAreVisible)
      return false; // Can not perform dynamic search when panels are off.

   if (view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_INFO)
      return false; // Can not perform dynamic search when info-mode is on.

   // Show/activate the dynamic filename searcher (entry field in the
   // headerbr of each panel.
   if (!headerWin.isPerformingDynamicSearch())
   {
      GString text(chr);

      // Change the font of the entryfield to be the same as
      // is used by the header window.
      headerWin.dynaFileSearch->setFontNameSize(headerWin.getFontNameSize());

      // Change the text in two steps so that the caret gets moved behind the inserted text
      int len = headerWin.dynaFileSearch->getTextLength();
      headerWin.dynaFileSearch->setSelection(0, len);
      headerWin.dynaFileSearch->replaceSelection(text);

      // Finally, show and activate the window of where to edit the search string
      headerWin.dynaFileSearch->setVisible();
      headerWin.dynaFileSearch->grabFocus();

      return true;
   }

   return false;
}

int LCmdFilePanel::calcIconSize ()
{
   if (view.showItemIcon)
   {
      switch (view.showIcon)
      {
         case LCmdFilePanelViewOptions::SHOWICON_SMALL:
            iconSize = GSystem::GetSystemMetrics(GSystem::SMID_CXICONSMALL);
            break;

         case LCmdFilePanelViewOptions::SHOWICON_LARGE:
            iconSize = GSystem::GetSystemMetrics(GSystem::SMID_CXICON);
            break;

         case LCmdFilePanelViewOptions::SHOWICON_INTERNAL:
            // Get the size (width) of the largest (widest) icon.
            // Usually all the "internal" icons are of the same size,
            // but it can happen that the user has defined his/her own
            // icons, in which case we have to perform this code anyway.
            iconSize = 14;
            iconSize = GMath::Max(iconSize, iconDirUp->getWidth());
            iconSize = GMath::Max(iconSize, iconDirDir->getWidth());
            iconSize = GMath::Max(iconSize, iconDirZip->getWidth());
            iconSize = GMath::Max(iconSize, iconFileExe->getWidth());
            iconSize = GMath::Max(iconSize, iconFileDoc->getWidth());
            iconSize = GMath::Max(iconSize, iconFileFile->getWidth());
            break;
      }
   }
   else
   {
      iconSize = 0;
   }

   return iconSize;
}

bool LCmdFilePanel::activatePanel ()
{
   if (this != lcmd->curPanel)
   {
      lcmd->mainWin.cmdActivateOppositePanel();
      return true;
   }
   else
   {
      lcmd->mainWin.ensureFocusOK();
      return false;
   }
}

void LCmdFilePanel::setActive ( bool force )
{
   GWindow::setActive(force);
   activatePanel();
}

bool LCmdFilePanel::copyOrMovePanelFiles ( bool move, bool renameByDefault )
{
   aptr<GArray<LCmdCopyFileItem> > items = makeArrayOfCopyFileItems();
   if (items.get() == null)
      return false;
   return copyOrMovePanelFiles_(move, renameByDefault, *items);
}

bool LCmdFilePanel::copyOrMovePanelFiles_ ( bool move, 
                                            bool renameByDefault, 
                                            GArray<LCmdCopyFileItem>& copyItems )
{
   // Set current disk and directory before copy/move, to make sure that
   // any relative paths will be OK.
   if (!activatePanelDriveAndDir(true))
      return false;

   if (copyItems.getCount() <= 0)
      return false;

   int selectedIndex = -1;
   int curSel = getCurrentSelectedIndex();
   LCmdFilePanel& dstPanel = getOppositePanel();

   GString destPath;
   if (renameByDefault)
   {
      if (curSel >= 0)
         destPath = items[curSel].getFileName();
      else
         destPath = GString::Empty;
   }
   else
   if (markedFilesCount > 0)
   {
      // renameByDefault = false; // Can't rename more than one file at once
      if (dstPanel.view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_INFO)
         destPath = GString::Empty;
      else
         destPath = dstPanel.getCurrentSysDirectory(true);
   }
   else
   if (curSel >= 0)
   {
      selectedIndex = curSel;
      LCmdFileItem& fitem = items[curSel];
      if (fitem.isUpDir())
         return false; // Can't copy the ".." directory!
      if (dstPanel.view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_INFO)
      {
         destPath = fitem.getFileName();
      }
      else
      {
         GString dir = dstPanel.getCurrentSysDirectory(true);
         GString fname = fitem.getFileName();
         GFile f(dir, fname);
         destPath = f.getFullPath();
      }
   }
   else
   {
      return true; // Nothing to copy!
   }

   // ---
   // Ask the user to specify a destination file, directory or path.
   LCmdCopy::SpecifyDestDir dest(move, *this, destPath, copyItems);
   lcmd->dlgCopy.execute(lcmd->mainWin, dest);
   if (dest.iError != LCmdCopy::ERR_NONE)
      return false;

   // ---
   GVfs& vfsSrc = vfs.peek();
   GVfs& vfsDst = dstPanel.vfs.peek();

   // ---
   bool renameOnly = false;
   bool removeEAName = false;
   GString walkedDestPath(256);
   vfsSrc.getWalkedPath(dest.destPath, walkedDestPath);
   const int num = copyItems.getCount();
   if (num > 1 && !vfsDst.existDirectory(walkedDestPath))
   {
      // Destination must be a valid directory: '%s'
      GStringl msg("%Txt_Copy_Error_DestMustBeValid", GVArgs(walkedDestPath));
      lcmd->mainWin.showMessageBox(msg, GMessageBox::TYPE_ERROR);
      return false;
   }

   // Make the destination path's
   GString destFilenameIfSingleFileOnly;
   for (int i=0; i<num; i++)
   {
      GString srcPath = copyItems[i].getSourcePath();
      bool itemHasBeenAddedToList = false;

      if (num == 1)
      {
         // Test if the full path of the source and the full path of the
         // destination is the same. In that case, do some special logic.
         if (walkedDestPath.compare(srcPath, !GFile::IsCaseSensitiveFS()) == 0)
         {
            if (!move)
            {
               // Since filenames in OS/2 are not truly case sensitive there
               // are no chance that this copy attempt will succeed. Thus,
               // we are better stopping the operation with an error message.
               GStringl msg("%Txt_Copy_Error_CantCopyToSelf"); // Cannot copy a file into it self!
               lcmd->mainWin.showMessageBox(msg, GMessageBox::TYPE_ERROR);
               return false;
            }

            // Full source path and full destination path is the same.
            // This probably means that the user want to change the case of
            // one or more characters in the source file or directory name.
            if (walkedDestPath == srcPath)
               return true; // No operation, since case already is the same.

            renameOnly = true;
            copyItems[i].setDestPath(walkedDestPath);
            itemHasBeenAddedToList = true;
         }
      }

      GString dstPath(256);
      if (!itemHasBeenAddedToList)
      {
         if (renameByDefault)
         {
            dstPath = walkedDestPath;
            if (vfsDst.existDirectory(dstPath))
            {
               GFile::Slash(dstPath);
               dstPath += GFile::GetFileName(srcPath);
               copyItems[i].setDestPath(dstPath);
            }
            else
            if (num == 1)
            {
               copyItems[i].setDestPath(dstPath);
               if (move)
                  renameOnly = true;
            }
         }
         else
         if (// Don't assume destination path "as is" if it has a trailing slash.
             !GFile::IsSlashed(dest.destPath) &&
             (
                (// If source is a directory and destination is not existing.
                 vfsSrc.existDirectory(srcPath) &&
                 !vfsDst.exist(dest.destPath))

                ||

                (// If source is a file and destination is not a directory then assume destination "as is".
                vfsSrc.existFile(srcPath) &&
                !vfsDst.existDirectory(dest.destPath))

                ||

                (// If source is a directory with the same name as the destination then assume destination "as is".
                GFile::GetFileName(srcPath).compare(GFile::GetFileName(dest.destPath), !GFile::IsCaseSensitiveFS()) == 0 &&
                vfsSrc.existDirectory(srcPath))
             ))
         {
            dstPath = dest.destPath;
            copyItems[i].setDestPath(dstPath);
         }
         else
         {
            dstPath = dest.destPath;
            vfsDst.slash(dstPath);
            dstPath += GFile::GetFileName(srcPath);
            copyItems[i].setDestPath(dstPath);
         }
      }

      if (num == 1)
      {
         // Always remove the ".LONGNAME" EA from the destination if
         // the target filename is different from the source filename.
         GString fname1 = GFile::GetFileName(srcPath);
         GString fname2 = GFile::GetFileName(dstPath);
         if (!fname1.equalsIgnoreCase(fname2))
            removeEAName = true;
         destFilenameIfSingleFileOnly = fname2;
      }
   }

   // Some suspect code to make the automatic selection of the
   // renamed file as smooth as possible.
   GString autoSelect1, autoSelect2;
   if (renameOnly)
   {
      removeEAName = true;
      if (lcmd->curPanel == lcmd->fp1)
         autoSelect1 = destFilenameIfSingleFileOnly;
      if (lcmd->curPanel == lcmd->fp2)
         autoSelect2 = destFilenameIfSingleFileOnly;
   }

   // Do the real copy/move work...
   int countSkipped = 0;
   bool ok = LCmdCopy::CopyOrMoveFiles(vfsSrc, vfsDst, copyItems, move, *this, true, renameOnly, removeEAName, autoSelect1, autoSelect2, &countSkipped);

   // When you rename a single file item without moving it to a new directory
   // then Larsen Commander should automatically select the destination file
   // item for you when the rename has been successfully done.
   if (ok && renameOnly && destFilenameIfSingleFileOnly != "" && countSkipped == 0)
      selectedIndex = findItem(destFilenameIfSingleFileOnly);
   if (selectedIndex >= 0)
      selectItem(selectedIndex);

   return ok;
}

LCmdFileItem* LCmdFilePanel::getCurItem ( int idx )
{
   if (idx <= -1)
      idx = getCurrentSelectedIndex();
   if (idx < 0 || idx >= items.getCount())
      return null;
   return &items.get(idx);
}

GString LCmdFilePanel::getCurItemName ( int idx )
{
   if (idx <= -1)
      idx = getCurrentSelectedIndex();
   if (idx < 0)
      return GString::Empty;
   LCmdFileItem& fitem = items.get(idx);
   return fitem.getFileName();
}

GString LCmdFilePanel::getCurItemPath ( int idx )
{
   if (idx <= -1)
      idx = getCurrentSelectedIndex();
   if (idx < 0)
      return GString::Empty;
   const LCmdFileItem& fitem = items.get(idx);
   GString dir = getCurrentVfsDirectory(true);
   GString fname = fitem.getFileName();
   return dir + fname;
}

GString LCmdFilePanel::getCurrentSysDirectory ( bool slash ) const
{
   const GVfsLocal& fs = vfs.root();
   return fs.getCurrentDirectory(slash);
}

GString LCmdFilePanel::getCurrentVfsDirectory ( bool slash ) const
{
   return vfs.getFullVirtualPath(slash);
}

int LCmdFilePanel::findItem ( const GString& fname, const GString& fext )
{
   GVfs& fs = vfs.peek();
   bool ignoreCase = !fs.isFileNameCaseSensitive();

   const int num = items.getCount();
   for (int i=0; i<num; i++)
   {
      LCmdFileItem& fitem = items[i];
      const GString& fn = fitem.getName();
      if (!fn.equalsString(fname, ignoreCase))
         continue;
      const GString& fe = fitem.getExtension();
      if (!fe.equalsString(fext, ignoreCase))
         continue;
      return i;
   }

   return -1;
}

int LCmdFilePanel::findItem ( const GString& fname )
{
   GVfs& fs = vfs.peek();
   bool ignoreCase = !fs.isFileNameCaseSensitive();

   int fnameLen = fname.length();
   if (fnameLen == 0)
      return -1;

   // This code is highly optimized, preventing usage of temporary objects.
   const int num = items.getCount();
   for (int i=0; i<num; i++)
   {
      LCmdFileItem& fitem = items[i];
      const GString& fn = fitem.getName();
      const GString& fe = fitem.getExtension();
      if (fnameLen == fn.length() + fe.length())
         if (fname.beginsWith(fn, ignoreCase))
            if (fname.endsWith(fe, ignoreCase))
               return i;
   }

   return -1;
}

bool LCmdFilePanel::queryCurrentDir ( GWindow& errpWin )
{
   GVfsLocal& fs = vfs.root();
   try {
      GString dir = fs.getCurrentDirectory(false);
      GError err = fs.setCurrentDirectory(dir);
      return err == GError::Ok;
   } catch (APIRET& rc) {
      // Can not get the current directory on drive %c:.\nMessage from the system: %s
      GStringl msg("%Txt_FP_Error_CantGetCurDirOnDrive", GVArgs(fs.getCurrentDriveLetter()).add(GSystem::GetApiRetString(rc)));
      errpWin.showMessageBox(msg, GMessageBox::TYPE_ERROR);
      return false;
   }
}

bool LCmdFilePanel::activatePreviousDir ()
{
   GString dir = getCurrentVfsDirectory(false);
   if (previousDir == dir)
      return true;
   return walkDir(previousDir);
}

bool LCmdFilePanel::activatePanelDriveAndDir ( bool rereadIfError,
                                               bool* err,
                                               bool autoShowErrMsg )
{
   if (!vfs.isRootOnly())
   {
      if (err != null)
         *err = false; // Everything performed successfully!
      return true;
   }

   bool ret = true; // Until the opposite has been proven
   if (err != null)
      *err = true; // Until the opposite has been proven

   GVfsLocal& fs = vfs.root();

   GString dir = getCurrentSysDirectory(false);
   GError rc = fs.setCurrentDirectory(dir);
   if (rc != GError::Ok)
   {
      if (autoShowErrMsg && lcmd->finishedStartup)
      {
         // Error activating directory '%s'.\nMessage from the system: %s
         GStringl msg("%Txt_FP_Error_CantActivateDir", GVArgs(dir).add(rc.getErrorMessage()));
         lcmd->mainWin.showMessageBox(msg, GMessageBox::TYPE_ERROR);
      }

      GString root = fs.getRootDir();
      fs.setCurrentDirectory(root);
      headerWin.invalidateAll(true);
      GString curdir = fs.getCurrentDirectory(false);
      GFile::SetCurrentDir(curdir);
      if (rereadIfError)
         ret = fillList(false);
   }
   else
   {
      if (err != null)
         *err = false; // Everything performed successfully!
   }

   // Disable the ".." and "\" button on the Header Bar when the root directory is active.
   bool isRoot = vfs.isRootOnly() && fs.isCurrentDirectoryTheRoot();
   GString cdir = fs.getCurrentDirectory(false);
   headerWin.upDirButt->setEnabled(!isRoot);
   headerWin.rootButt->setEnabled(!isRoot);

   // Request the header bar of the panel to be redrawn if the current
   // directory has changed.
   if (dir.equalsIgnoreCase(cdir))
      headerWin.driveButt->invalidateAll(true);
   else
      headerWin.invalidateAll(true);

   // Also request the Command Line Window to update it self so that it will
   // always show the current directory in the Prompt.
   lcmd->cmdLine.currentPathHasChanged();

   // Update the text on the title bar as well, e.g.: "Larsen Commander - C:\OS2"
   GString curPath = getCurrentSysDirectory(false);
   GString titleText = GString("%s - %s", GVArgs("Larsen Commander").add(curPath));
   lcmd->mainWin.setText(titleText);

   // Update the statusbar cell that shows the free drive space.
   lcmd->mainWin.updateStatusbarDriveFreeCell();

   return ret; // Drive and/or directory may have been changed.
}

void LCmdFilePanel::tagItem ( int index, bool tag, bool repaint )
{
   if (index < 0 || index >= items.getCount())
      return;

   LCmdFileItem& fitem = items.get(index);
   if (fitem.isUpDir()) // Don't mark the "up" dir
      return;

   if (!fitem.isMarked() == !tag)
      return; // Item is already toggled on/off

   fitem.setMarked(tag);

   // Keep trach of the number and total size of all marked items.
   if (fitem.isMarked())
   {
      markedFilesCount += 1;
      sizeOfMarkedFiles += fitem.fileSize;
   }
   else
   {
      markedFilesCount -= 1;
      sizeOfMarkedFiles -= fitem.fileSize;
   }

   if (repaint)
   {
      // Request the file item and the infobar window to be repainted.
      // The infobar must also be updated because it is used to show the
      // current number and total size of all marked items.
      infoBar.updateAllFileItemInfoCells();
      drawItem(index);
   }
}

bool LCmdFilePanel::toggleTag ( int index, bool repaint )
{
   if (index < 0 || index >= items.getCount())
      return false;

   LCmdFileItem& fitem = items.get(index);
   if (fitem.isUpDir()) // Don't mark the "up" dir
      return false;

   bool mark = !fitem.isMarked();
   tagItem(index, mark, repaint);
   return true;
}

void LCmdFilePanel::drawItem ( int itemIndex )
{
   LCmdFilePanelModeAbstract& curView = getCurrentView();
   curView.drawItem(itemIndex);
}

int LCmdFilePanel::select ( LCmdFilePanelSelectOptions& filt )
{
   int count = 0;

   const int nrOfItems = items.getCount();
   for (int i=0; i<nrOfItems; i++)
   {
      LCmdFileItem& fitem = items[i];
      if (fitem.isMarked())
         continue;

      if (fitem.isDirectory())
      {
         if (fitem.isUpDir())
            continue;

         if (!filt.inclDirs)
            continue;
      }
      else
      if (!filt.inclFiles)
         continue;

      GString fileName = fitem.getFileName();
      if (GFile::IsFileNameInFilterList(fileName, filt.filter))
      {
         count++;
         fitem.setMarked(true);
         markedFilesCount += 1;
         sizeOfMarkedFiles += fitem.fileSize;
      }
   }

   if (count > 0)
   {
      invalidateAll(true);
      infoBar.updateAllFileItemInfoCells();
   }

   return count;
}

int LCmdFilePanel::unselect ( LCmdFilePanelSelectOptions& filt )
{
   int count = 0;

   const int nrOfItems = items.getCount();
   for (int i=0; i<nrOfItems; i++)
   {
      LCmdFileItem& fitem = items[i];
      if (!fitem.isMarked())
         continue;

      if (fitem.isDirectory())
      {
         if (fitem.isUpDir())
            continue;

         if (!filt.inclDirs)
            continue;
      }
      else
      if (!filt.inclFiles)
         continue;

      GString fileName = fitem.getFileName();
      if (GFile::IsFileNameInFilterList(fileName, filt.filter))
      {
         count++;
         fitem.setMarked(false);
         markedFilesCount -= 1;
         sizeOfMarkedFiles -= fitem.fileSize;
      }
   }

   if (count > 0)
   {
      invalidateAll(true);
      infoBar.updateAllFileItemInfoCells();
   }

   return count;
}

void LCmdFilePanel::unselectAll ()
{
   bool any = false;
   for (int i=0, count=items.getCount(); i<count; i++)
   {
      LCmdFileItem& fitem = items.get(i);
      if (!fitem.isMarked())
         continue;
      fitem.setMarked(false);
      any = true;
   }

   if (any)
   {
      markedFilesCount = 0;
      LCmdFilePanelModeAbstract& view = getCurrentView();
      view.invalidateAll(true);
      infoBar.updateAllFileItemInfoCells();
   }
}

void LCmdFilePanel::onItemSelectionHasChanged ()
{
   LCmdFilePanel& op = getOppositePanel();
   if (op.view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_INFO)
   {
      int prevWidest = op.info.widthOfWidestItem();
      op.info.updateUponSelection();
      int newWidest = op.info.widthOfWidestItem();
      if (prevWidest != newWidest)
         op.viewInfo.updateScrollBarRange();
   }
   else
   {
      // If current item has changed, then request the text area of the infobar
      // to show the information about newly selected item
      if (markedFilesCount <= 0)
         infoBar.updateAllFileItemInfoCells();
   }
}

int LCmdFilePanel::getCurrentSelectedIndex () const
{
   LCmdFilePanel& self = const_cast<LCmdFilePanel&>(*this); // Throw away const.
   const LCmdFilePanelModeAbstract& curView = self.getCurrentView();
   return curView.getCurrentSelectedIndex();
}

int LCmdFilePanel::selectItem ( int index )
{
   const int num = items.getCount();
   if (num <= 0 || index >= num)
      return -1;

   int prevSelectedItem = getCurrentSelectedIndex();

   LCmdFilePanelModeAbstract& curView = getCurrentView();
   switch (index)
   {
      case NavigateUp: curView.navigateUp(); break;
      case NavigateDown: curView.navigateDown(); break;
      case NavigateLeft: curView.navigateLeft(); break;
      case NavigateRight: curView.navigateRight(); break;
      case NavigatePageUp: curView.navigatePageUp(); break;
      case NavigatePageDown: curView.navigatePageDown(); break;
      case NavigateHome: curView.navigateHome(); break;
      case NavigateEnd: curView.navigateEnd(); break;
      default: curView.navigateRandom(index); break;
   }

   if (getCurrentSelectedIndex() != prevSelectedItem)
   {
      // Let "everyone" know that the selection has changed.
      onItemSelectionHasChanged();
   }

   return getCurrentSelectedIndex();
}


int LCmdFilePanel::selectItem ( const GString& name )
{
   int idx = findItem(name);
   if (idx >= 0)
      selectItem(idx);
   return idx;
}

void LCmdFilePanel::cmdSelectGroup ()
{
   LCmdDlgSelectFiles dlg(*this, selectOpt, "%Txt_DlgSelectFiles_TitleSel");
   if (dlg.execute())
      select(selectOpt);
}

void LCmdFilePanel::cmdUnselectGroup ()
{
   LCmdDlgSelectFiles dlg(*this, selectOpt, "%Txt_DlgSelectFiles_TitleUnsel");
   if (dlg.execute())
      unselect(selectOpt);
}

void LCmdFilePanel::cmdInvertSelection ()
{
   markedFilesCount = 0;
   sizeOfMarkedFiles = 0;

   const int nrOfItems = items.getCount();
   for (int i=0; i<nrOfItems; i++)
   {
      LCmdFileItem& fitem = items[i];

      if (fitem.isDirectory())
      {
         if (fitem.isUpDir())
            continue;

         if (!selectOpt.inclDirs)
         {
            // User has customized us not to toggle the marking of directories.
            // But we must still remember to count those directories eventually
            // already marked.
            if (fitem.isMarked())
            {
               markedFilesCount += 1;
               sizeOfMarkedFiles += fitem.fileSize;
            }
            continue;
         }
      }

      if (fitem.isMarked())
      {
         fitem.setMarked(false);
      }
      else
      {
         fitem.setMarked(true);
         markedFilesCount += 1;
         sizeOfMarkedFiles += fitem.fileSize;
      }
   }

   LCmdFilePanelModeAbstract& view = getCurrentView();
   view.invalidateAll(true);
   infoBar.updateAllFileItemInfoCells();
}

void LCmdFilePanel::cmdSetIconMode ( LCmdFilePanelViewOptions::SHOWICON mode, 
                                     bool visible, 
                                     bool force )
{
   if (!force && view.showIcon == mode && view.showItemIcon == visible)
      return;

   view.showIcon = mode;
   view.showItemIcon = visible;

   items.freeItemIcons();
   items[0].briefWidth = 0; // Force BRIEF mode to recalculate item widths.
   calcIconSize(); // Update the "iconSize" variable.
   LCmdFilePanelModeAbstract& view = getCurrentView();
   view.layout();
   invalidateAll(true);

   int sel = getCurrentSelectedIndex();
   selectItem(0); // Just to force reselection of current item
   selectItem(sel);
}

void LCmdFilePanel::cmdToggleShowIcons ( int toggle )
{
   LCmdFilePanelViewOptions::SHOWICON mode = view.showIcon;
   bool visible = view.showItemIcon;

   if (toggle == -1)
   {
      if (!visible)
      {
         visible = true;
         switch (mode)
         {
            case LCmdFilePanelViewOptions::SHOWICON_INTERNAL:
                 mode = LCmdFilePanelViewOptions::SHOWICON_SMALL;
                 break;

            case LCmdFilePanelViewOptions::SHOWICON_SMALL:
                 mode = LCmdFilePanelViewOptions::SHOWICON_LARGE;
                 break;

            case LCmdFilePanelViewOptions::SHOWICON_LARGE:
                 mode = LCmdFilePanelViewOptions::SHOWICON_INTERNAL;
                 break;
         }
      }
      else
      {
         switch (mode)
         {
            case LCmdFilePanelViewOptions::SHOWICON_INTERNAL:
                 mode = LCmdFilePanelViewOptions::SHOWICON_SMALL;
                 break;

            case LCmdFilePanelViewOptions::SHOWICON_SMALL:
                 mode = LCmdFilePanelViewOptions::SHOWICON_LARGE;
                 break;

            case LCmdFilePanelViewOptions::SHOWICON_LARGE:
                 visible = false;
                 break;
         }
      }
   }
   else
   {
      visible = toggle ? true : false;
   }

   cmdSetIconMode(mode, visible);
}

void LCmdFilePanel::cmdOpenSysFolderCurDir ()
{
   GString dir = getCurrentSysDirectory(false);
   GFile::Slash(dir);
   GSystem::OpenShellObject(dir);
}

void LCmdFilePanel::cmdCreateSysObject ()
{
   if (!vfs.isRootOnly())
   {
      // Command not supported by the current VFS.
      GString msg("%VfsErrCommandNotSupported");
      lcmd->mainWin.showMessageBox(msg, GMessageBox::TYPE_ERROR);
      return;
   }

   GVfs& fs = vfs.peek();
   GString dstPath = getCurItemPath();
   GFileItem fitem(fs, dstPath);
   GString workDir = fitem.getDirectory();
   GStringl leaderStr("%TxtDlgCreateObjectOnDesktop_Leader");
   GStringl titleStr("%TxtDlgCreateObjectOnDesktop_Title");
   GStringl defaultStr("%TxtDlgCreateObjectOnDesktop_DefaultName_OS2", GVArgs(fitem.getFileName()));
   GString name = GUserStringDlg::Get(this, titleStr, leaderStr, defaultStr);
   if (name != "")
      GSystem::CreateSystemShellObject(dstPath, workDir, name);
}

void LCmdFilePanel::cmdChangeDrive ()
{
   GWindow* ownerWin = &frameWin;
   if (!lcmd->options.panelsAreVisible)
      ownerWin = &lcmd->cmdCont.conmon;

   GStringl title("%Txt_DlgChooseDrive_Title", GVArgs(getPanelName()));
   GVfsLocal& fs = vfs.root();
   int prevDrive = fs.getCurrentDriveInfo().drive;
   int drive = lcmd->drives.execute(ownerWin, prevDrive, title);
   if (drive > 0)
      activateDrive(drive);
}

void LCmdFilePanel::cmdShowFileInfo ()
{
   if (view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_INFO)
      return; // Not supported when info-mode is on.

   int curSel = getCurrentSelectedIndex();
   if (curSel >= 0)
   {
      const LCmdFileItem& fitem = items.get(curSel);
      LCmdDlgFileDetails dlg(fitem);
      dlg.execute(&lcmd->mainWin);
   }
}

bool LCmdFilePanel::calcItemRect ( int itemIndex, GRectangle& rect ) const
{
   LCmdFilePanel& self = const_cast<LCmdFilePanel&>(*this); // Throw away const.
   LCmdFilePanelModeAbstract& curView = self.getCurrentView();
   return curView.calcItemRect(itemIndex, rect);
}

void LCmdFilePanel::sortList ( bool keepSelectedItem, bool calcColumns )
{
   GString prevSelectedName;
   if (keepSelectedItem)
      prevSelectedName = getCurItemName();
   items.sort();
   if (calcColumns)
   {
      LCmdFilePanelModeAbstract& curView = getCurrentView();
      curView.layout();
   }
   LCmdFilePanelModeAbstract& view = getCurrentView();
   view.invalidateAll(true);
   if (keepSelectedItem)
      selectItem(prevSelectedName);
}

bool LCmdFilePanel::activateDrive ( char drive, bool temp )
{
   int idrive = int(GCharacter::ToUpperCase(drive) - 'A' + 1);
   return activateDrive(idrive, temp);
}

bool LCmdFilePanel::activateDrive ( int drive, bool temp )
{
   if (!temp)
   {
      // Add the current directory to the Directory History and remember it
      // as the "previous" directory.
      previousDir = getCurrentSysDirectory(false);
      if (lcmd->finishedStartup)
         lcmd->cmdEntry.dirHist.add(previousDir);
   }

   GString dir;
   try {
      dir = GFile::GetCurrentDir(drive);
   } catch (APIRET& rc) {
      GStringl msg("%Txt_FP_Error_CantActivateDrive", GVArgs(drive+'A'-1).add(GSystem::GetApiRetString(rc)));
      lcmd->mainWin.showMessageBox(msg, GMessageBox::TYPE_ERROR);
      return false;
   }

   GVfsLocal& fs = vfs.backToRoot();
   GError rc = fs.setCurrentDirectory(dir);
   if (rc != GError::Ok)
   {
      // Could not activate the "current directory" of that drive.
      // If the "current directory" is not the root then retry by 
      // activating the root of that drive.
      if (dir.length() > 3 && dir[1] == ':' && GFile::IsSlash(dir[2]))
      {
         dir.cutTailFrom(3);
         rc = fs.setCurrentDirectory(dir);
      }
      if (rc != GError::Ok)
      {
         // The root could not be activated either. 
         // This probably means that the drive is invalid.
         GStringl msg("%Txt_FP_Error_CantActivateDrive", GVArgs(drive+'A'-1).add(rc.getErrorMessage()));
         lcmd->mainWin.showMessageBox(msg, GMessageBox::TYPE_ERROR);
         return false;
      }
   }

   // Update visual size of the drives-button, if needed.
   headerWin.layout();

   // Get the current directory of selected drive.
   bool ok = queryCurrentDir(lcmd->mainWin);
   if (ok)
   {
      // If the current view mode is INFO then toggle it now.
      if (view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_INFO)
         setViewMode(LCmdFilePanelViewOptions::VIEWMODE_INFO);

      // Fill in the filenames of the current dir in selected drive,
      // but only if this is not a temporary activation.
      if (!temp)
         ok = fillList();
   }

   return ok;
}

aptr<GArray<LCmdCopyFileItem> > LCmdFilePanel::makeArrayOfCopyFileItems ()
{
   GArray<LCmdCopyFileItem>* ret = null;
   if (markedFilesCount > 0)
   {
      ret = new GArray<LCmdCopyFileItem>(markedFilesCount);
      int num = items.getCount();
      for (int i=0; i<num; i++)
      {
         LCmdFileItem& file = items.get(i); // Make this a fast one
         if (file.isMarked() && !file.isUpDir())
         {
            GString srcPath = file.getFullPath();
            LCmdCopyFileItem* cfi = new LCmdCopyFileItem(srcPath, null);
            ret->add(cfi);
         }
      }
   }
   else
   if (getCurrentSelectedIndex() >= 0)
   {
      ret = new GArray<LCmdCopyFileItem>(1);
      int curSel = getCurrentSelectedIndex();
      LCmdFileItem& file = items.get(curSel);
      if (!file.isUpDir())
      {
         GString srcPath = file.getFullPath();
         LCmdCopyFileItem* cfi = new LCmdCopyFileItem(srcPath, null);
         ret->add(cfi);
      }
   }

   if (ret != null && ret->getCount() <= 0)
   {
      delete ret;
      ret = null;
   }

   return aptr<GArray<LCmdCopyFileItem> >(ret);
}

void LCmdFilePanel::sortBy ( LCmdFilePanelSortOptions::SORT_WHAT what, LCmdFilePanelSortOptions::SORT_HOW how )
{
   int startIdx = 0;
   if (what != LCmdFilePanelSortOptions::PSW_TYPE &&
       sortOpt.what[0] == LCmdFilePanelSortOptions::PSW_TYPE)
   {
      startIdx = 1;
   }

   if (what == sortOpt.what[startIdx])
   {
      if (how == sortOpt.how[startIdx])
         return;
      else
      switch (how)
      {
         case LCmdFilePanelSortOptions::PSH_ASCENDING:
              sortOpt.how[startIdx] = LCmdFilePanelSortOptions::PSH_ASCENDING;
              break;
         case LCmdFilePanelSortOptions::PSH_DESCENDING:
              sortOpt.how[startIdx] = LCmdFilePanelSortOptions::PSH_DESCENDING;
              break;
         case LCmdFilePanelSortOptions::PSH_TOGGLE:
         default:
              sortOpt.how[startIdx] = (sortOpt.how[startIdx] == LCmdFilePanelSortOptions::PSH_ASCENDING ? LCmdFilePanelSortOptions::PSH_DESCENDING : LCmdFilePanelSortOptions::PSH_ASCENDING);
              break;
      }
   }
   else
   {
      if (how == LCmdFilePanelSortOptions::PSH_TOGGLE)
      {
         // Can't toggle when sort-ID is about to change.
         // Therefore, use the default direction.
         how = sortOpt.how[startIdx];
      }

      int endIdx = 3; // Number of sorting items are always four
      for (int i=endIdx; i>startIdx; i--)
      {
         sortOpt.what[i] = sortOpt.what[i-1];
         sortOpt.how[i] = sortOpt.how[i-1];
      }

      sortOpt.what[startIdx] = what;
      sortOpt.how[startIdx] = how;
   }

   sortList(true, true);
}

void LCmdFilePanel::reverseSortOrder ()
{
   if (sortOpt.what[0] != LCmdFilePanelSortOptions::PSW_TYPE)
      sortOpt.how[0] = (sortOpt.how[0] == LCmdFilePanelSortOptions::PSH_ASCENDING ? LCmdFilePanelSortOptions::PSH_DESCENDING : LCmdFilePanelSortOptions::PSH_ASCENDING);

   sortOpt.how[1] = (sortOpt.how[1] == LCmdFilePanelSortOptions::PSH_ASCENDING ? LCmdFilePanelSortOptions::PSH_DESCENDING : LCmdFilePanelSortOptions::PSH_ASCENDING);
   sortOpt.how[2] = (sortOpt.how[2] == LCmdFilePanelSortOptions::PSH_ASCENDING ? LCmdFilePanelSortOptions::PSH_DESCENDING : LCmdFilePanelSortOptions::PSH_ASCENDING);
   sortOpt.how[3] = (sortOpt.how[3] == LCmdFilePanelSortOptions::PSH_ASCENDING ? LCmdFilePanelSortOptions::PSH_DESCENDING : LCmdFilePanelSortOptions::PSH_ASCENDING);

   sortList(true, true);
}

void LCmdFilePanel::allItemsHaveBeenRemoved ()
{
   markedFilesCount = 0;
   sizeOfMarkedFiles = 0;
}

void LCmdFilePanel::valueHasChanged ( GChangeableValueContainer& cvc )
{
   if (&cvc != &tabs)
      return;

   GString tabIdxStr = cvc.queryValue();
   int tabIdx = GInteger::ParseInt(tabIdxStr);
   GWindow& w = tabs.getIndexedTab(tabIdx);
   GString wname = w.getName();

   // Identify the new view mode.
   LCmdFilePanelViewOptions::VIEWMODE newMode;
   if (wname == "Brief")
      newMode = LCmdFilePanelViewOptions::VIEWMODE_BRIEF;
   else
   if (wname == "Wide")
      newMode = LCmdFilePanelViewOptions::VIEWMODE_WIDE;
   else
   if (wname == "Full")
      newMode = LCmdFilePanelViewOptions::VIEWMODE_FULL;
#if __ENABLE_TREE_VIEW
   else
   if (wname == "Tree")
      newMode = LCmdFilePanelViewOptions::VIEWMODE_TREE;
#endif
   else
   if (wname == "Info")
      newMode = LCmdFilePanelViewOptions::VIEWMODE_INFO;
   else
      gthrow_(GIllegalStateException("Unknown tab: " + w.getName()));

   // Activate the new view mode.
   setViewMode(newMode);
}

LCmdFilePanelModeAbstract& LCmdFilePanel::getCurrentView ()
{
   if (currentView == null)
      gthrow_(GIllegalStateException("currentView==null"));
   return *currentView;
}

void LCmdFilePanel::cmdCycleIconTypes ( GAbstractCommand* /*cmd*/ )
{
   cmdToggleShowIcons(-1);
}

void LCmdFilePanel::cmdSortByType ( GAbstractCommand* /*cmd*/ )
{
   sortBy(LCmdFilePanelSortOptions::PSW_TYPE, LCmdFilePanelSortOptions::PSH_TOGGLE);
}

void LCmdFilePanel::cmdSortByDate ( GAbstractCommand* /*cmd*/ )
{
   sortBy(LCmdFilePanelSortOptions::PSW_DATE, LCmdFilePanelSortOptions::PSH_TOGGLE);
}

void LCmdFilePanel::cmdSortByExtension ( GAbstractCommand* /*cmd*/ )
{
   sortBy(LCmdFilePanelSortOptions::PSW_EXTENTION, LCmdFilePanelSortOptions::PSH_TOGGLE);
}

void LCmdFilePanel::cmdSortByName ( GAbstractCommand* /*cmd*/ )
{
   sortBy(LCmdFilePanelSortOptions::PSW_NAME, LCmdFilePanelSortOptions::PSH_TOGGLE);
}

void LCmdFilePanel::cmdSortBySize ( GAbstractCommand* /*cmd*/ )
{
   sortBy(LCmdFilePanelSortOptions::PSW_SIZE, LCmdFilePanelSortOptions::PSH_TOGGLE);
}

void LCmdFilePanel::cmdSortByTime ( GAbstractCommand* /*cmd*/ )
{
   sortBy(LCmdFilePanelSortOptions::PSW_TIME, LCmdFilePanelSortOptions::PSH_TOGGLE);
}

void LCmdFilePanel::cmdSortByUnsorted ( GAbstractCommand* /*cmd*/ )
{
   sortBy(LCmdFilePanelSortOptions::PSW_UNSORTED, LCmdFilePanelSortOptions::PSH_TOGGLE);
}

void LCmdFilePanel::clearEmphasisOfItem ( int itemIndex, GGraphics& g )
{
   if (isDragOverEmphasised)
   {
      GRectangle rect = getWindowRect();
      g.drawEmphasiseBox(rect, GColor::BLACK, 2, true);
      isDragOverEmphasised = false;
   }
   else
   if (itemIndex >= 0)
   {
      GRectangle rect;
      if (calcItemRect(itemIndex, rect))
      {
         LCmdFileItem& pFile = items.get(itemIndex);
         int flags = pFile.getInternalFlags();
         if (flags & LCmdFileItem::FINFO_EMPHASIZED)
         {
            LCmdFilePanelModeAbstract& curView = getCurrentView();
            curView.drawItem(itemIndex, g, rect, true);
            pFile.setInternalFlags(flags & ~LCmdFileItem::FINFO_EMPHASIZED);
         }
      }

      recentDragOverItemIndex = -1;
   }
}

bool LCmdFilePanel::onUserMessage ( GUserMessage& msg )
{
   GString id = msg.getParam1String();
   if (id != "UM_DROPFILES")
      return GWindow::onUserMessage(msg);
   LCmdFilePanelObjRenderer::DropInfo* di = (LCmdFilePanelObjRenderer::DropInfo*) msg.getParam2();
   LCmdFilePanelObjRenderer::DROPOPTYPE opType = di->opType;
   GArray<LCmdCopyFileItem>& items = *di->items;
   /* ---
   for (int i=0, num=items.getCount(); i<num; i++)
   {
      LCmdCopyFileItem* fi = items.get(i);
      GString src = fi->getSourcePath();
      GString dst = fi->getDestPath();
      lcmd->conmon.appendTextFromGuiThread("%s\t%s\n", GVArgs(src).add(dst));
   }
   --- */
   GVfsLocal vfsSrc;
   GVfs& vfsDst = vfs.peek();
   bool move = (opType == LCmdFilePanelObjRenderer::DROPOPTYPE_MOVE);
   LCmdCopy::CopyOrMoveFiles(vfsSrc, vfsDst, items, move, *this, false, false, false);
   return true;
}

class DragFiles : public GObject
{
   private:

      /** Reference to the owner file panel. */
      LCmdFilePanel& panel;

      /** Array of copy of file items of which to drag. */
      GArray<LCmdFileItem> items;

   public:

      explicit DragFiles ( LCmdFilePanel& panel );
      virtual ~DragFiles ();

   public:

      void addItem ( const LCmdFileItem& file );

      bool isEmpty () const;

      /** This is a modal method that will not return until the drag operation is finished. */
      bool startDrag ();
};

DragFiles::DragFiles ( LCmdFilePanel& panel ) 
          :panel(panel) 
{
}

DragFiles::~DragFiles ()
{
}

void DragFiles::addItem ( const LCmdFileItem& file ) 
{ 
   items.add(new LCmdFileItem(file)); 
}

bool DragFiles::isEmpty () const
{
   return items.isEmpty();
}

bool DragFiles::startDrag ()
{
   int numItems = items.getCount();
   DRAGINFO* di = ::DrgAllocDraginfo(numItems);
   if (di == null)
      return false;

   GString cdir = panel.getCurrentSysDirectory(true);

   for (int i=0; i<numItems; i++)
   {
      LCmdFileItem& item = items.get(i); // Make this a fast one.

      DRAGITEM dItem;
      memset(&dItem, 0, sizeof(dItem));
      dItem.hstrRMF = ::DrgAddStrHandle("<DRM_OS2FILE,DRF_UNKNOWN>");
      dItem.fsSupportedOps = DO_COPYABLE | DO_MOVEABLE | DO_LINKABLE;
      dItem.hwndItem = panel.getHWND();
      dItem.hstrSourceName = ::DrgAddStrHandle(item.getFileName());
      dItem.hstrTargetName = dItem.hstrSourceName;
      dItem.hstrContainerName = ::DrgAddStrHandle(cdir);

      // Set the type of the drag item (hstrType). It is often ok to leave
      // this variable as null, but for some target applications it must be 
      // specified. So we should always specify it. E.g. the "DragText"
      // application needs it.
      //
      // For folders, the type should always be "Unknown".
      // For files, it would normally be the file's .TYPE EA - but it doesn't
      // have to be. There are 3 ways we can deal with this:
      //
      // 1) Regardless of file type, always use "Plain Text".
      // 2) For common file types, deduce the type from the extension;
      //    for everything else, use "Plain Text".
      // 3) Read the .TYPE EA & use it;  if there are multiple types,
      //    separate them with commas;  if there's no EA, use "Plain Text".
      //
      // While using the .TYPE EA takes more work than the other methods, 
      // using it ensures that target windows will handle the dropped file 
      // correctly. E.g. dropping a URL object on Mozilla will load the page 
      // rather than displaying the URL.
      //
      // Thanks to Rich Walsh (the author of "DragText" and "Remote WPS")
      // for bringing this information.
      if (item.isDirectory())
      {
         // For folders, the type should always be "Unknown".
         dItem.hstrType = ::DrgAddStrHandle(DRT_UNKNOWN);
      }
      else
      {
         // Keep it simple by now, and use alternative 1) as defined above.
         dItem.hstrType = ::DrgAddStrHandle(DRT_TEXT);
      }

      ::DrgSetDragitem(di, &dItem, sizeof(dItem), i);
   }

   const int MAXIMAGES = 5;
   DRAGIMAGE images[MAXIMAGES];
   memset(&images, 0, sizeof(images));
   int iconOffsetFactor = (panel.view.showIcon == LCmdFilePanelViewOptions::SHOWICON_LARGE ? 10 : 5);
   int numImages = (numItems > MAXIMAGES ? MAXIMAGES : numItems);
   for (int i=0; i<numImages; i++)
   {
      GString fullPath = items.get(i).getFullPath();
      images[i].cb = sizeof(images[i]);
      images[i].hImage = WinLoadFileIcon(fullPath, false);
      images[i].fl = DRG_ICON;
      images[i].cxOffset = USHORT(i*iconOffsetFactor);
      images[i].cyOffset = USHORT(i*iconOffsetFactor);
   }

   HWND hWndTarget = ::DrgDrag(panel.getHWND(), di, images, numImages, VK_ENDDRAG, null);
   if (hWndTarget == null)
   {
      ::DrgDeleteDraginfoStrHandles(di);
      ::DrgFreeDraginfo(di);
   }

   if (panel.view.showIcon == LCmdFilePanelViewOptions::SHOWICON_INTERNAL)
   {
      // Free the system icons that was locally loaded for the purpose of the
      // drag operation only.
      for (int i=0; i<numImages; i++)
         if (images[i].hImage)
            ::WinFreeFileIcon(images[i].hImage);
   }

   return true;
}

void LCmdFilePanel::handleBeginDrag ( int /*xpos*/, int /*ypos*/ )
{
   static bool IsHere = false;
   if (IsHere)
   {
      // Can not begin a drag operation when another drag operation is currently in progress!
      GStringl msg("%Txt_FP_Error_CantDragWhenAnotherDrag");
      lcmd->mainWin.showMessageBox(msg, GMessageBox::TYPE_ERROR);
      return;
   }
   GIsHereTracker isHereTracker(IsHere, true);

   DragFiles drg(*this);

   if (markedFilesCount > 0)
   {
      int num = items.getCount();
      for (int i=0; i<num; i++)
      {
         const LCmdFileItem& file = items.get(i); // Make this a fast one
         if (file.isMarked() && !file.isUpDir())
            drg.addItem(file);
      }
   }
   else
   {
      int curSel = getCurrentSelectedIndex();
      if (curSel >= 0)
      {
         const LCmdFileItem& file = items.get(curSel);
         if (!file.isUpDir())
            drg.addItem(file);
      }
   }

   if (!drg.isEmpty())
      drg.startDrag();
}

void LCmdFilePanel::handleDragLeave ( GDragInfo& /*di*/, GGraphics& g )
{
   clearEmphasisOfItem(recentDragOverItemIndex, g);
}

void LCmdFilePanel::handleDragOver ( GDragInfo& dragInfo, GGraphics& g, int xpos, int ypos, GDragOverAnswer& answ )
{
   LCmdFilePanelModeAbstract& curView = getCurrentView();
   int itemIndex = curView.calcItemIdxFromPos(xpos, ypos);
   LCmdFileItem* item = null;
   if (itemIndex >= 0)
      item = &items.get(itemIndex);

   DRAGINFO& di = dragInfo.di;
   if (di.hwndSource == getHWND())
   {
      if (item != null)
      {
         if (item->isUpDir() || !(item->isDirectory() || item->isProgramFileName()))
         {
            // Can not drop objects to the same panel as is the source panel,
            // except if the objects are drop on a program file or on a
            // sub-directory item.
            if (recentDragOverItemIndex >= 0 && recentDragOverItemIndex != itemIndex)
               clearEmphasisOfItem(recentDragOverItemIndex, g);
            answ.canDrop = GDragOverAnswer::DOANSW_NODROP;
            return;
         }
      }
      else
      {
         // Can not drop objects to the same panel as is the source panel.
         if (recentDragOverItemIndex >= 0 && recentDragOverItemIndex != itemIndex)
            clearEmphasisOfItem(recentDragOverItemIndex, g);
         answ.canDrop = GDragOverAnswer::DOANSW_NODROP;
         return;
      }
   }

   if (objRender.hWndObj == null)
   {
      // The renderer object window has, of some reason, failed to be created.
      // Therefore we can never take any dropping objects anyway.
      if (recentDragOverItemIndex >= 0 && recentDragOverItemIndex != itemIndex)
         clearEmphasisOfItem(recentDragOverItemIndex, g);
      answ.canDrop = GDragOverAnswer::DOANSW_NODROP;
      return;
   }

   answ.canDrop = GDragOverAnswer::DOANSW_DROP;
   switch (di.usOperation)
   {
      case DO_COPY:
           answ.defaultOp = GDragOverAnswer::DOANSW_COPY;
           break;

      case DO_MOVE:
      case DO_DEFAULT:
      case DO_UNKNOWN:
           answ.defaultOp = GDragOverAnswer::DOANSW_MOVE;
           break;

      default:
           answ.canDrop = GDragOverAnswer::DOANSW_NODROPOP;
           return;
   }

   if (item != null && item->isProgramFileName())
   {
      if (answ.defaultOp != GDragOverAnswer::DOANSW_MOVE)
      {
         // Dropping objects on a program file is only alloweed if the
         // current drag operation is "MOVE".
         itemIndex = -1;
      }
   }

   if (recentDragOverItemIndex >= 0 && recentDragOverItemIndex != itemIndex)
      clearEmphasisOfItem(recentDragOverItemIndex, g);

   if (itemIndex < 0)
   {
      if (!isDragOverEmphasised)
      {
         GRectangle r = getWindowRect();
         g.drawEmphasiseBox(r, GColor::BLACK, 2, true);
         isDragOverEmphasised = true;
      }
   }
   else
   {
      GRectangle rect;
      if (calcItemRect(itemIndex, rect))
      {
         if (isDragOverEmphasised)
            clearEmphasisOfItem(-1, g);

         // Paint the empasis of the new dragged over item
         LCmdFileItem& pFile = items.get(itemIndex);
         int flags = pFile.getInternalFlags();
         if (!(flags & LCmdFileItem::FINFO_EMPHASIZED))
         {
            curView.drawItem(itemIndex, g, rect, true);
            pFile.setInternalFlags(flags | LCmdFileItem::FINFO_EMPHASIZED);
         }
      }
   }

   recentDragOverItemIndex = itemIndex;

   // Decide whether or not we are capable
   // of letting the user drop the object(s) at this location.

   for (int c1=0; c1<di.cditem; c1++)
   {
      DRAGITEM* dragItem = DrgQueryDragitemPtr(&di, c1);
      if (dragItem != null)
      {
         if (!DrgVerifyRMF(dragItem, "DRM_OS2FILE", NULL))
         {
            answ.canDrop = GDragOverAnswer::DOANSW_NEVERDROP; // Can not handle this object!
            clearEmphasisOfItem(recentDragOverItemIndex, g);
            break;
         }
         if (answ.defaultOp == GDragOverAnswer::DOANSW_COPY && !(dragItem->fsSupportedOps & DO_COPYABLE) ||
             answ.defaultOp == GDragOverAnswer::DOANSW_MOVE && !(dragItem->fsSupportedOps & DO_MOVEABLE))
         {
            answ.canDrop = GDragOverAnswer::DOANSW_NODROPOP; // Can not handle this object using the current drag operation!
            break;
         }
      }
   }
}

void LCmdFilePanel::handleDrop ( GDragInfo& dragInfo, GGraphics& g, int xpos, int ypos )
{
   clearEmphasisOfItem(recentDragOverItemIndex, g);

   // Post the drop operation to the object window to perform the
   // rendering in a separate thread.
   dragInfo.di.xDrop = SHORT(xpos);
   dragInfo.di.yDrop = SHORT(ypos);
   objRender.postDropOperation(dragInfo);
}
