/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __LCMD_FILEPANEL
#define __LCMD_FILEPANEL

#include "glib/gui/GDecoratedWindow.h"
#include "glib/gui/GTabbedPanel.h"
#include "glib/gui/dragndrop/GDragDropHandler.h"
#include "glib/util/GValueChangeListener.h"
#include "glib/vfs/GVfsStack.h"
#include "lcmd/LCmdInfoPanel.h"
#include "lcmd/LCmdFileItemContainer.h"
#include "lcmd/LCmdFilePanelViewOptions.h"
#include "lcmd/LCmdFilePanelSortOptions.h"
#include "lcmd/LCmdFilePanelFilterOptions.h"
#include "lcmd/LCmdFilePanelStartupOptions.h"
#include "lcmd/LCmdFilePanelColorOptions.h"
#include "lcmd/LCmdFilePanelSelectOptions.h"
#include "lcmd/LCmdFilePanelObjRenderer.h"
#include "lcmd/LCmdFilePanelModeBrief.h"
#include "lcmd/LCmdFilePanelModeWide.h"
#include "lcmd/LCmdFilePanelModeFull.h"
#include "lcmd/LCmdFilePanelModeTree.h"
#include "lcmd/LCmdFilePanelModeInfo.h"
#include "lcmd/LCmdMakeDir.h"

/**
 * This window class is used to represent a file panel.
 *
 * There are always two and only two instances of this class in
 * Larsen Commander. One for the left- and one for the right file panel.
 *
 * @author  Leif Erik Larsen
 * @since   1998.10.09
 */
class LCmdFilePanel : public GWindow, 
                      public LCmdFileItemContainer::Listener,
                      public GValueChangeListener,
                      public GDragDropHandler
{
   friend class LCmdFileItem;
   friend class LCmdFileItemContainer;

   private:

      /** Index of recent item underneath the DM_DRAGOVER message. */
      int recentDragOverItemIndex;

      /** The "internal" icon used to represent ".." directories. */
      const GIcon* iconDirUp;

      /** The "internal" icon used to represent directories in general. */
      const GIcon* iconDirDir;

      /** The "internal" icon used to represent archive files (virtual directories). */
      const GIcon* iconDirZip;

      /** The "internal" icon used to represent executable files. */
      const GIcon* iconFileExe;

      /** The "internal" icon used to represent files that are associated to any application, program or script. */
      const GIcon* iconFileDoc;

      /** The "internal" icon used to represent files of unknown type. */
      const GIcon* iconFileFile;

      /**
       * The name of the item that should be initially selected after
       * the next time that the filenames list has been filled.
       * If this is an empty string then we will selected the very
       * first item. This is done by {@link #fillList}.
       */
      GString initiallySelectedItemName;

      /** True if the DM_DRAGOVER border is currently painted. */
      bool isDragOverEmphasised;

   public:

      /** A reference to our frame window, which is our parent window. */
      class LCmdFilePanelFrame& frameWin;

      /** The Header Bar of the Panel. */
      class LCmdFilePanelHeader& headerWin;

      /** The info/status bar window of this panel. */
      class LCmdFilePanelInfoBar& infoBar;

      /** Handle of the object window to render file objects in a background thread. */
      HWND hWndObjRenderer;

      /** The dynamic array of file items/objects. */
      LCmdFileItemContainer items;

      /** Number of marked files in this panel. */
      int markedFilesCount;

      /** Total size of all marked files. */
      longlong sizeOfMarkedFiles;

      /** CB for the "Make Directory" command. */
      LCmdMakeDir makeDir;

      /** Colors to be used by this panel. */
      LCmdFilePanelColorOptions colors;

      /** CB for the "Select" command. */
      LCmdFilePanelSelectOptions selectOpt;

      /** Sorting options of this panel. */
      LCmdFilePanelSortOptions sortOpt;

      /** Which files to show/hide. */
      LCmdFilePanelFilterOptions filter;

      /** Startup options for the panel. */
      LCmdFilePanelStartupOptions startup;

      /** Options that has to do with view/display. */
      LCmdFilePanelViewOptions view;

      /** Pixels height and width of a file icon (depends on View.ShowIcon). */
      int iconSize;

      /** The object window to render dragged files in a background thread. */
      LCmdFilePanelObjRenderer objRender;

      /** The Object used to show info in the INFO mode. */
      LCmdInfoPanel info;

      /** Previous directory, including drive. */
      GString previousDir;

      /**
       * @author  Leif Erik Larsen
       * @since   2005.10.22
       */
      class VfsStack : public GVfsStack
      {
         public:
            VfsStack ();
            virtual ~VfsStack ();
            virtual bool isSupportedZipOrArchiveFile ( const GString& path ) const;
            virtual GError walkIntoArchiveFile ( const GString& fname );
      };

      /** The stack of VFSs involved in the current directory of the file panel. */
      VfsStack vfs;

      /** The current active view tab. */
      class LCmdFilePanelModeAbstract* currentView;

      /** The previously active view tab. Can be null. */
      LCmdFilePanelViewOptions::VIEWMODE previousViewMode;

      /** The tabs with each available file items view mode (brief, wide, full, etc.). */
      GTabbedPanel tabs;

      /** The BRIEF view. */
      LCmdFilePanelModeBrief viewBrief;

      /** The WIDE view. */
      LCmdFilePanelModeWide viewWide;

      /** The FULL view. */
      LCmdFilePanelModeFull viewFull;

      /** The TREE view. */
#if __ENABLE_TREE_VIEW
      LCmdFilePanelModeTree viewTree;
#endif

      /** The INFO view. */
      LCmdFilePanelModeInfo viewInfo;

      /**
       * These are the ID's used to select items via the keyboard arrow keys.
       *
       * @author  Leif Erik Larsen
       * @since   2006.01.26
       */
      enum Navigate
      {
         NavigateUp = -10000,
         NavigateDown,
         NavigateLeft,
         NavigateRight,
         NavigatePageUp,
         NavigatePageDown,
         NavigateHome,
         NavigateEnd
      };

   public:

      LCmdFilePanel ( class LCmdFilePanelFrame& frame, 
                      const GString& constraints );

      virtual ~LCmdFilePanel ();

   public:

      /**
       * Used with drag-n-drop, to clear the temporary selection of a file 
       * item which is underneath the mouse cursor during a drag operation.
       *
       * @author  Leif Erik Larsen
       * @since   2007.01.07
       */
      void clearEmphasisOfItem ( int itemIndex, class GGraphics& g );

      /**
       * Handle the menu command IDM_LEFTDRIVE or IDM_RIGHTDRIVE.
       */
      void cmdChangeDrive ();

      /**
       * Create WPS Object on desktop for the current selected file item.
       */
      void cmdCreateSysObject ();

      /**
       * Make every marked item (in specified panel) unmarked,
       * and vice verca.
       */
      void cmdInvertSelection ();

      /**
       * Open the current directory of specified file panel as a
       * Work Place Shell object window.
       */
      void cmdOpenSysFolderCurDir ();

      /**
       * Pop up a dialog of where user can specify which files to
       * mark using wildcards.
       */
      void cmdSelectGroup ();

      /**
       * Set the current icon view mode.
       *
       * @param  mode    The new icon mode of which to activate.
       * @param  visible True to set icons visible, or else false to make
       *                 the icons hidden.
       * @param  force   True if we shall force a recalculation even if the
       *                 specified icon mode and visibility is the same as
       *                 is already active.
       * @see    #cmdToggleShowIcons
       */
      void cmdSetIconMode ( LCmdFilePanelViewOptions::SHOWICON mode, 
                            bool visible = true, 
                            bool force = false );

      /**
       * Show some detailed information about the current selected file item.
       */
      void cmdShowFileInfo ();

      /**
       * Toggle or cycle the current icon mode.
       *
       * @param  toggle  -1 to cycle/toggle current state, false to
       *                 toggle off or true to toggle on.
       * @see    #cmdSetIconMode
       */
      void cmdToggleShowIcons ( int toggle );

      /**
       * Pop up a dialog of where user can specify which files to
       * unmark using wildcards.
       */
      void cmdUnselectGroup ();

      // ---
      void cmdCycleIconTypes ( class GAbstractCommand* cmd = null );
      void cmdSortByType ( class GAbstractCommand* cmd = null );
      void cmdSortByDate ( class GAbstractCommand* cmd = null );
      void cmdSortByExtension ( class GAbstractCommand* cmd = null );
      void cmdSortByName ( class GAbstractCommand* cmd = null );
      void cmdSortBySize ( class GAbstractCommand* cmd = null );
      void cmdSortByTime ( class GAbstractCommand* cmd = null );
      void cmdSortByUnsorted ( class GAbstractCommand* cmd = null );
      DECLARE_COMMAND_TABLE(LCmdFilePanel);

   private:

      /**
       * Part of {LCmdFileItemContainer#Listener}.
       * Called by {@link LCmdFileItemContainer#freeItems} only.
       *
       * @author  Leif Erik Larsen
       * @since   2006.01.24
       */
      virtual void allItemsHaveBeenRemoved ();

      /**
       * This is the "kernel" method of <i>copyOrMovePanelFiles()</i>.
       * It is to be used by that method only.
       * @see #copyOrMovePanelFiles
       */
      bool copyOrMovePanelFiles_ ( bool move, 
                                   bool renameByDefault, 
                                   GArray<LCmdCopyFileItem>& copyItems );

      /**
       * @author  Leif Erik Larsen
       * @since   2007.01.07
       */
      virtual bool onUserMessage ( class GUserMessage& msg );

      /** 
       * Part of {@link GValueChangeListener}. 
       * Listens for value-vhange events from {@link #tabs}.
       * 
       * @author  Leif Erik Larsen
       * @since   2006.01.26
       */
      virtual void valueHasChanged ( class GChangeableValueContainer& cvc );

   public:

      /**
       * Activate the specified drive on the specified file panel.
       *
       * @param  drive  1=A, 2=B, 3=C, etc.
       * @param  temp   True if the drive change is a temporary part of
       *                a more complicated directory activation algoritm.
       *                Will be true when called by {@link #walkDir}.
       *                If this parameter is true then we will not update
       *                the directory history, and we will not update the
       *                content of the file panel.
       * @return True on success, or else false on any error.
       * @see    #activateDrive(char, bool)
       */
      bool activateDrive ( int drive, bool temp = false );

      /**
       * Same as {@link #activateDrive(int, bool)}, except this overloaded
       * version of the method takes a character that represents the
       * drive of which to activate ('A', 'B', 'C', etc.).
       */
      bool activateDrive ( char drive, bool temp = false );

      /**
       * Activate the specified file panel.
       *
       * @return True if current panel has changed, or else
       *         false if specified panel was the active panel already.
       * @see    #setActive
       */
      bool activatePanel ();

      /**
       * Make the current drive and directory of specified file panel
       * active. If, fore some reason, the directory can't be activated
       * (for example if it doesn't exist), then activate the root
       * directory instead. In that case, also set 'cCurDir' so
       * that it always contains which directory has actually been activated
       * upon return. Therefore, this function will always activate some
       * directory.
       *
       * @param  rereadIfError  True if we shall automatically call
       *                        {@link #fillList} in case we have to activate
       *                        another directory due to some error when trying
       *                        to activate the "current" directory. Or else
       *                        false. Mark that we will not call
       *                        {@link #fillList} if we fail to activate the
       *                        "current" drive, regardless of this parameter.
       *                        This parameter is almost always to be true.
       * @param  pErr           Will return false if the originally current
       *                        drive or directory was activated successfully
       *                        without any error of any kind. If we had to
       *                        adjust the current drive then this bool will
       *                        be returned as true. If a NULL pointer is
       *                        specified we will not touch it at all.
       * @param  autoShowErrMsg True if we shall automatically show an error
       *                        message in case of any error, or else false.
       *                        Typically, this parameter should be true.
       * @return True on success (panel directory may have been error
       *         corrected), or else false on any error (panel drive or
       *         directory may be undefined). We will automatically show
       *         the correct error message.
       */
      bool activatePanelDriveAndDir ( bool rereadIfError, 
                                      bool* pErr = null, 
                                      bool autoShowErrMsg = true );

      /**
       * Activate the directory that was active before the current
       * directory was activate.
       */
      bool activatePreviousDir ();

      /**
       * Calculate the value to be stored in the variable {@link #iconSize}.
       */
      int calcIconSize ();

      /**
       * Calculate the rectangle of where the indexed file item is
       * contained within the file panel.
       *
       * @return   True if the indexed item is within visible area,
       *           or else false.
       */
      bool calcItemRect ( int itemIndex, class GRectangle& rect ) const;

      /**
       * @param  move            True if we shall move, or else false to copy.
       * @param  renameByDefault True if we shall move/copy to a new filename
       *                         in the same directory as of the current file
       *                         panel, by default.
       * @return True on success, or else false on any error.
       */
      bool copyOrMovePanelFiles ( bool move, bool renameByDefault = false );

      /**
       * Common code to create a Virtual File System object based on the 
       * specified path to an archive file. The created VFS is allocated
       * on the heap, and the caller receives the ownership of it.
       *
       * @author  Leif Erik Larsen
       * @since   2006.02.15
       * @param   parentVfs The VFS of where the archive file is contained.
       * @param   path      Path of the archive file to load. The path is 
       *                    relative to the specified parent VFS only. Can 
       *                    also be a relative path from the current 
       *                    directory of the parent VFS.
       * @param   localVfs  The Local File System is needed in order for some
       *                    internal usage by this static method.
       * @throws  GIOException in case of any error loading the archive file.
       */
      static GVfs* CreateVfs ( GVfs& parentVfs, const GString& path, GVfsLocal& localVfs );
                        

      /**
       * Perform an exit-command, executing the command line text (if any)
       * or the current selected filename item.
       *
       * @author  Leif Erik Larsen
       * @since   2006.01.16
       * @param   idx   Index of which filename item to start, or 
       *                -1 in order to start the current selected item.
       * @param   forceExecItem If this parameter is false then we will 
       *                execute the command line text, if any, or else the
       *                selected file item. If it is true then we will 
       *                execute the current selected item even if there
       *                are some text in the command line. This
       *                parameter is true typically  only if we are
       *                called due to a mouse button double click.
       */
      void doEnter ( int idx = -1, bool forceExecItem = false );

      /**
       * Draw the indexed filename item.
       *
       * @author  Leif Erik Larsen
       * @since   2006.01.24
       */
      void drawItem ( int itemIndex );

      /**
       * Reread filenames from current directory into table of filenames in
       * specified file panel.
       *
       * This is the inner level of the method used to reread filenames.
       * It is usually to be called by {@link #reRead}.
       *
       * @param  autoShowErrMsg  True if we shall automatically show an error
       *                         message in case of any error, or else false.
       *                         Typically, this parameter should be true.
       * @param  synchPaintPanel True if we shall synchronously paint the 
       *                         "Reading filenames..." message on the file 
       *                         panel face while initiating the filename 
       *                         reader. This should probably be true only
       *                         when called due to an explicit user request,
       *                         in order to make sure that the user will
       *                         visually "see" that somethings actually 
       *                         happen. Otherwise this parameter is 
       *                         probably to be false (default) in order 
       *                         to prevent unwanted "flashing".
       * @return True on success, or else false on any error.
       * @see    #reRead
       */
      bool fillList ( bool autoShowErrMsg = true, bool synchPaintPanel = false );

      /**
       * Find the index of the named file item which name equals the
       * specified filename and extension, ignoring character cases.
       *
       * @param  fname    The full filename of which to find.
       * @return Index of the found filename item, or -1 if it
       *         doesn't exist.
       */
      int findItem ( const GString& fname );

      /**
       * Find the index of the named file item which name equals the
       * specified filename and extension, ignoring character cases.
       *
       * @return Index of the found filename item, or -1 if it
       *         doesn't exist.
       */
      int findItem ( const GString& fname, const GString& fext );

      /**
       * Find the next item that starts with the specified character key.
       *
       * @param  cChar  Actual character input.
       * @return True if we found another item that starts with the
       *         specified character key or else false.
       */
      bool findItemByKey ( char cChar );

      /**
       * Find the index of the next item which filename matches the
       * characters given in "str" (case insensitive), starting from
       * the given index plus one.
       *
       * @return The index of the found item, or else -1 if no items
       *         matches the specified string.
       */
      int findNextMatchingItem ( const GString& str, int startIndex );

      /**
       * Find the index of the previous item which filename matches the
       * characters given in pcStr (case insensitive), starting from the
       * given index minus one.
       *
       * @return The index of the found item, or else -1 if no items
       *         matches the specified string.
       */
      int findPrevMatchingItem ( const GString& str, int startIndex );

      /**
       * Get a reference to the current active file panel.
       *
       * @author  Leif Erik Larsen
       * @since   2004.11.15
       * @see     #GetLeftPanel
       * @see     #GetRightPanel
       * @see     #getOppositePanel
       */
      static LCmdFilePanel& GetCurrentPanel ();

      /**
       * Get the index of the current selected filename item in the current
       * view mode, or -1 if there is currently no such selected item.
       *
       * @author  Leif Erik Larsen
       * @since   2006.01.24
       */
      int getCurrentSelectedIndex () const;

      /**
       * Get the current directory, including drive, of the file panel.
       *
       * Note that this will always return the physical working directory,
       * not the virtual. This means that it is the working directory as
       * it is to be seen by the underlying system that will be returned.
       * If the user has navigated into some virtual file system in the
       * file panel we will not respect that virtual path in the returned
       * string.
       *
       * @author  Leif Erik Larsen
       * @param   slash  True if we shall make sure that the returned
       *                 string is terminated with a system dependent
       *                 slash character.
       * @see     getCurrentVfsDirectory
       */
      GString getCurrentSysDirectory ( bool slash ) const;

      /**
       * Get the current directory of the file panel, including the 
       * name of any active VFS.
       *
       * @author  Leif Erik Larsen
       * @param   slash  True if we shall make sure that the returned
       *                 string is terminated with a system dependent
       *                 slash character.
       * @see     getCurrentSysDirectory
       */
      GString getCurrentVfsDirectory ( bool slash ) const;

      /**
       * @author  Leif Erik Larsen
       * @since   2006.01.24
       */
      class LCmdFilePanelModeAbstract& getCurrentView ();

      /**
       * Get a pointer to the specified file item, or null 
       * if the specified index is invalid.
       *
       * @author  Leif Erik Larsen
       * @since   2005.01.05
       * @see     #getCurItemName
       * @see     #getCurItemPath
       */
      class LCmdFileItem* getCurItem ( int idx = -1 );

      /**
       * Get the full name of the specified filename item.
       *
       * @author  Leif Erik Larsen
       * @since   2005.01.05
       * @param   idx  Index of which item name to get, or -1 to get the 
       *               name of the current selected item.
       * @return  The full filename (not including path) of the specified
       *          item. If the specified index is illegal then return 
       *          an empty string.
       * @see     #getCurItem
       * @see     #getCurItemPath
       */
      GString getCurItemName ( int idx = -1 );

      /**
       * Get the fully qualified VFS-path of the indexed or current selected 
       * filename item.
       *
       * If the specified index is illegal then return an empty string.
       *
       * @author  Leif Erik Larsen
       * @since   2006.01.16
       * @param   idx  Index of which item path to get, or -1 to get the 
       *               path of the current selected item.
       * @see     #getCurItem
       * @see     #getCurItemName
       */
      GString getCurItemPath ( int idx = -1 );

      /**
       * Get a reference to the left file panel of Larsen Commander.
       *
       * @author  Leif Erik Larsen
       * @since   2004.11.15
       * @see     #getRightPanel
       * @see     LCmd#fp2
       * @see     #GetCurrentPanel
       */
      static LCmdFilePanel& GetLeftPanel ();

      /**
       * Get a refernce to the opposite of "this" file panel.
       *
       * @see   #GetLeftPanel
       * @see   #GetRightPanel
       */
      LCmdFilePanel& getOppositePanel () const;

      /**
       * Get a reference to the right file panel of Larsen Commander.
       *
       * @author  Leif Erik Larsen
       * @since   2004.11.15
       * @see     #GetLeftPanel
       * @see     LCmd#fp1
       * @see     #GetCurrentPanel
       */
      static LCmdFilePanel& GetRightPanel ();

      /**
       * Get "Left Panel" or "Right Panel" with respect to the
       * current "Swap Panel" state.
       */
      GString getPanelName ();

      // These are common drag-n-drop code used by all the file panel mode classes.
      // These methods implements the interface {@link GDragDropHandler}.
      virtual void handleBeginDrag ( int xpos, int ypos );
      virtual void handleDragLeave ( class GDragInfo& di, class GGraphics& g );
      virtual void handleDragOver ( class GDragInfo& di, class GGraphics& g, int xpos, int ypos, class GDragOverAnswer& answ );
      virtual void handleDrop ( class GDragInfo& di, class GGraphics& g, int xpos, int ypos );

      /**
       * Check if this file panel is the currently active file panel 
       * of Larsen Commander.
       *
       * @author  Leif Erik Larsen
       * @since   2004.11.16
       */
      bool isCurrentPanel () const;

      /**
       * Return true if and only if this file panel is currently acting 
       * as the left file panel of Larsen Commander.
       * 
       * @author  Leif Erik Larsen
       * @since   2004.11.16
       * @see     #isRightPanel
       */
      bool isLeftPanel () const;

      /**
       * Return true if and only if this file panel is currently acting 
       * as the right file panel of Larsen Commander.
       * 
       * @author  Leif Erik Larsen
       * @since   2004.11.16
       * @see     #isLeftPanel
       */
      bool isRightPanel () const;

      /**
       * Return True if hidden and system files are currently to be visible.
       */
      bool isShowHiddenFiles ();

      /**
       * Make a new instance of HArray<LCmdCopyFileItem*> and return a
       * reference to it.
       *
       * @return The returned object will contain the path of all the
       *         currently marked filename items. If there are no currently
       *         marked filenames then it will contain the path of the
       *         current selected file item only. If no file item is
       *         currently selected either then the returned pointer object
       *         will contain a null pointer. Anyway, only the source path
       *         variable of the <i>LCmdCopyFileItem</i> objects in the
       *         returned object will contain a value. The destination path
       *         variables will all be null.
       */
      aptr<GArray<LCmdCopyFileItem> > makeArrayOfCopyFileItems ();

      /**
       * This method is automatically called whenever the current
       * filename item selection has been changed.
       *
       * @author  Leif Erik Larsen
       * @since   2001.03.24
       */
      void onItemSelectionHasChanged ();

      /**
       * Read and store the name of current directory of the drive
       * which is active in specified file panel.
       *
       * @param  errpWin  Which window to use as parent of error message
       *                  box (in case of any error).
       * @return True on success, or else false on any error (we will
       *         show error message).
       */
      bool queryCurrentDir ( GWindow& errpWin );

      /**
       * Load and activate the profile variables for the file panel
       * window, from the specified section name.
       * @see #writeProfile
       */
      void queryProfile ( const GString& sectName );

      /**
       * Reread the current directory of the file panel.
       *
       * @param  synchPaintPanel See {@link #fillList}.
       * @param  selectIndex -2 if we shall not select any item at all when
       *                     reread has finished. This will make the very
       *                     first item the "current selected item", but we
       *                     will not "repaint it". -1 if we shall try to
       *                     keep the same item (by name) as is currently
       *                     selected. Any value >= 0 will make us select
       *                     that indexed item.
       * @return True on success, or else false on any error.
       * @see    #fillList
       */
      bool reRead ( bool synchPaintPanel = false, int selectIndex = -1 );

      void reverseSortOrder ();

      /**
       * Activate (select) the indexed filename item.
       *
       * We will automatically perform all required scrolling logic to
       * always keep the selected item within visible area.
       *
       * @param  index  Index of which filename item to activate, or one
       *                of the NAVIGATE_* macros to handle navigation
       *                from keyboard.
       * @return Index of selected filename item, or -1 if the requested
       *         item does not exist in the list.
       */
      int selectItem ( int index );

      /**
       * Activate (select) the named filename item.
       *
       * We will automatically perform all required scrolling logic to
       * always keep the selected item within visible area.
       *
       * @param  name   The full name of which filename item to activate.
       * @return Index of selected filename item, or -1 if the requested
       *         item does not exist in the list.
       */
      int selectItem ( const GString& name );

      /**
       * Tag all items that match the specified selection options.
       *
       * If any items are tagged already then they will not be
       * affected by this function.
       *
       * @return Number of items actually selected.
       * @see    #unselect
       */
      int select ( class LCmdFilePanelSelectOptions& filt );

      /**
       * Will call the default super implementation and then also call 
       * {@link #activatePanel} afterwards.
       *
       * @author  Leif Erik Larsen
       * @since   2005.12.11
       * @see     #activatePanel
       */
      virtual void setActive ( bool force = false );

      /**
       * Toggle visibility of hidden files.
       */
      void setShowHiddenFiles ( bool flag );

      /**
       * Change to specified view-mode of specified file panel.
       *
       * @param   mode   The view mode of which to activate.
       */
      void setViewMode ( LCmdFilePanelViewOptions::VIEWMODE mode );

      /**
       * Sort the file items in the list by the specified column data.
       *
       * @param  what  ID of which column to sort by.
       * @param  how   How to sort.
       */
      void sortBy ( LCmdFilePanelSortOptions::SORT_WHAT what, 
                    LCmdFilePanelSortOptions::SORT_HOW how );

      /**
       * Sort the filenames in specified panel, using the sorting options of
       * specified panel.
       *
       * We will not reread the filename list, or perform any other I/O
       * operation, so this function will always succeed.
       *
       * This function uses the quick-sorting algorithm.
       *
       * @author  Leif Erik Larsen
       * @since   2006.01.24
       * @param   keepSelectedItem True if the currently selected filename 
       *                  item should remain selected after the sort 
       *                  operation.
       * @param   calcColumns True if called by the GUI thread, in which 
       *                  case we can safely perform GUI operations such
       *                  as updating the columns. False if called by the
       *                  filename reader background thread.
       */
      void sortList ( bool keepSelectedItem, bool calcColumns );

      /**
       * Open the edit field of where user can search for a named file.
       *
       * @param  cChar  Actual character input.
       * @return True if OK or else false if the Dynamic Search is
       *         already in progress or if any other error.
       */
      bool startDynamicSearch ( char cChar );

      /**
       * Execute the indexed filename item, as if user has pressed
       * the [Enter]-key on keyboard when the indexed item is active.
       *
       * @param  Leif Erik Larsen
       * @since  2006.01.16
       * @param  idx   Index of which filename item to start, or 
       *               -1 in order to start the current selected item.
       * @return True on success or else false on any error.
       */
      bool startSelectedProgram ( int idx = -1 );

      void tagItem ( int index, bool tag = true, bool repaint = true );

      /**
       * Toggle the selection mark of the indexed item. All required
       * repainting and tracking is handled correctly and automatically
       * by this function.
       *
       * @return True if the indexed item was actually toggled (no error)
       *         or else false on any error (e.g. array index out of bounds).
       */
      bool toggleTag ( int index, bool repaint );

      /**
       * Untag all items that match the specified selection options.
       *
       * If any items are untagged already then they will not be
       * affected by this function.
       *
       * @return Number of items actually unselected.
       * @see    #select
       */
      int unselect ( class LCmdFilePanelSelectOptions& filt );

      /**
       * @author  Leif Erik Larsen
       * @since   2006.01.16
       */
      void unselectAll ();

      /**
       * Walk the specified virtual file system path.
       *
       * Typically called when user press the enter-key on a
       * sub-directory within the file panel, or when user has 
       * typed a CD-command with either a simple or a complex path.
       * Also called when user requests to activate one of the
       * directories from the list of stored directories.
       *
       * This method will handle all directories correctly, including
       * relative directories (such as e.g.: "..\..\temp") and directories
       * with drive specification (such as e.g.: "C:\temp" or "C:.\temp").
       * Even if the specified directory defines a drive only (such
       * as e.g.: "D:") the directory is activate correctly. In the last
       * case we will activate the "current directory" of the specified
       * drive.
       *
       * <b>Mark</b> that the specified path can also contain VFS
       * elements such as the name of archive files, in which case we 
       * will open the archive and walk into it just as if it was 
       * a directory.
       *
       * @param   pathToWalk  Path (absolute or relative) of which to walk 
       *                      into. Can even contain a drive specification.
       * @param   temp        True if this is a temporary operation that 
       *                      is part of the file panel initialization 
       *                      called from {@link #queryProfile}. Normally,
       *                      and in all other cases, this parameter
       *                      should be false. If this parameter is true
       *                      then we will not show any error message to 
       *                      the user in case this method returns false.
       * @param   err         If this parameter != null then it will contain 
       *                      the error code of why this method did fail, in 
       *                      case we return false. If we return true then 
       *                      this parameter will always return 
       *                      {@link GError#Ok}. If this parameter is null
       *                      then we will not touch it by any means.
       * @return  True on success, or else false on any error.
       *          In case of any error the caller can assume that we have 
       *          already shown the corresponding error message to the 
       *          user (except if the <i>temp</i> parameter is true).
       */
      bool walkDir ( const GString& pathToWalk, bool temp = false );

      /**
       * Move into the indexed directory.
       *
       * @author  Leif Erik Larsen
       * @since   2006.01.16
       * @param   idx  Index of which file item representing the directory 
       *               of which to walk down into, or -1 in order to 
       *               use the index of the current selected file item.
       * @return  True on success, or else false on any error. Possible
       *          errors are, for instance, that the current selected item is
       *          not a directory, or that the directory activation went 
       *          wrong of some reason.
       */
      bool walkDownDir ( int idx = -1 );

      /**
       * Activate the parent directory.
       *
       * @return True on success, or else false on any error.
       */
      bool walkUpDir ();

      /**
       * Write the profile variables for the file panel window, under the
       * specified section name.
       * @see #queryProfile
       */
      void writeProfile ( const GString& sectName, bool force );
};

#endif // #ifndef __LCMD_FILEPANEL
