/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdFilePanelColumnsBar.h"
#include "lcmd/LCmdFilePanel.h"
#include "lcmd/LCmdOptions.h"

#include "glib/GProgram.h"
#include "glib/sys/GSystem.h"
#include "glib/util/GStringUtil.h"
#include "glib/util/GMath.h"

DEFINE_COMMAND_TABLE(LCmdFilePanelColumnsBar);
   ADD_COMMAND("cmdColumnsbarButtonShowAll", cmdColumnsbarButtonShowAll);
   ADD_COMMAND("cmdColumnsbarButtonDefWidthAll", cmdColumnsbarButtonDefWidthAll);
   ADD_COMMAND("cmdColumnsbarButtonMinWidthAll", cmdColumnsbarButtonMinWidthAll);
END_COMMAND_TABLE;

DEFINE_COMMAND_TABLE(LCmdFilePanelColumnsBar::Button);
   ADD_COMMAND("cmdColumnsbarButtonHide", cmdColumnsbarButtonHide);
   ADD_COMMAND("cmdColumnsbarButtonDefWidth", cmdColumnsbarButtonDefWidth);
   ADD_COMMAND("cmdColumnsbarButtonMinWidth", cmdColumnsbarButtonMinWidth);
END_COMMAND_TABLE;

// --------------------------------------------------------------------------
LCmdFilePanelColumnsBar::Button::Button ( const GString& id,
                                          const GString& shortcutKeyCmdID,
                                          LCmdFilePanelColumnsBar& clmnBar,
                                          LCmdFilePanelSortOptions::SORT_WHAT sortID )
                                :GToolbarButton(id,
                                                GString::Empty,
                                                clmnBar,
                                                id,
                                                GProgram::LoadText("%" + id, GResourceTable::LT_PREFER_TEXT),
                                                GString::Empty,
                                                GString::Empty,
                                                GToolbarButton::IP_RIGHT),
                                 clmnBar(clmnBar),
                                 customizedWidth(0),
                                 shortcutKeyCmdID(shortcutKeyCmdID),
                                 sortID(sortID)
{
}

void LCmdFilePanelColumnsBar::Button::queryProfile ( const GString& sectName )
{
   if (!clmnBar.ignoreIniProfile)
   {
      GProgram& prg = GProgram::GetProgram();
      GSectionBag& ini = prg.getIniProperties();
      if (this == &clmnBar.clmnIcon)
         customizedWidth = 0; // Always use default width on the icon column.
      else
         customizedWidth = ini.getInt(sectName, "Width", customizedWidth);
      setVisible(ini.getBool(sectName, "Visible", true));
   }

   GToolbarButton::queryProfile(sectName);
}

void LCmdFilePanelColumnsBar::Button::writeProfile ( const GString& sectName, bool force )
{
   if (!clmnBar.ignoreIniProfile)
   {
      GProgram& prg = GProgram::GetProgram();
      GSectionBag& ini = prg.getIniProperties();
      LCmdOptions& opt = LCmdOptions::GetOptions();
      ini.putInt(sectName, "Width", customizedWidth, force || opt.saveOnExit.visibleState);
      ini.putBool(sectName, "Visible", isVisible(), force || opt.saveOnExit.visibleState);
   }

   GToolbarButton::writeProfile(sectName, force);
}

int LCmdFilePanelColumnsBar::Button::getMinimumWidth () const
{
   GString str = getText();
   int strw = clmnBar.listWin.getWidthOfString(str);
   return strw + getIconWidth() + 12;
}

int LCmdFilePanelColumnsBar::Button::getPreferredWidth () const
{
   if (!isVisible())
      return 0;
   else
   if (customizedWidth > 0)
      return customizedWidth;
   else
      return calcDefaultWidth();
}

void LCmdFilePanelColumnsBar::Button::setVisible ( bool flag )
{
   if (flag == isVisible())
      return;

   GToolbarButton::setVisible(flag);
   clmnBar.updateTheColumnsbar();
}

bool LCmdFilePanelColumnsBar::Button::isAtLeft ( int xpos, int ypos ) const
{
   if (isFirstVisibleToolbarElement())
      return false;
   GDimension dim = getWindowSize();
   return xpos >= 0 && xpos <= 4 && ypos >= 0 && ypos < dim.height;
}

bool LCmdFilePanelColumnsBar::Button::isAtRight ( int xpos, int ypos ) const
{
   GDimension dim = getWindowSize();
   return xpos < dim.width && xpos >= dim.width - 4 && ypos >= 0 && ypos < dim.height;
}

bool LCmdFilePanelColumnsBar::Button::isMousePosOverridden ( int xpos, int ypos ) const
{
   return isAtLeft(xpos, ypos) || isAtRight(xpos, ypos);
}

bool LCmdFilePanelColumnsBar::Button::onMouseMove ( int xpos, int ypos, const GWindowMessage::InputFlags& flags )
{
   if (!isMouseDown())
   {
      if (isAtRight(xpos, ypos) ||
          isAtLeft(xpos, ypos))
      {
         GSystem::SetMouseCursorShape(GSystem::MCSID_SIZEWE);
         dismissMouseCapture();
         return true; // Yes, we have set the mouse cursor, so don't return false!
      }
   }

   return GToolbarButton::onMouseMove(xpos, ypos, flags);
}

bool LCmdFilePanelColumnsBar::Button::onButton1Down ( int xpos, int ypos, const GWindowMessage::InputFlags& flags )
{
   if (isAtLeft(xpos, ypos))
   {
      // Transfer the event to the "right edge" of our closest left
      // neighbour button that is currently visible.
      int num = clmnBar.getElementCount();
      for (int i=1; i<num; i++)
      {
         if (this != &clmnBar.getElement(i))
            continue;
         if (!dismissMouseCapture())
            return true;
         for (int leftIndex=i-1; leftIndex>=0; leftIndex--)
         {
            Button& butt = dynamic_cast<Button&>(clmnBar.getElement(leftIndex));
            if (!butt.isVisible())
               continue;
            int x = butt.getWindowSize().width - 2;
            return butt.onButton1Down(x, ypos, flags);
         }
         return true;
      }
      return true;
   }

   if (!isAtRight(xpos, ypos))
      return GToolbarButton::onButton1Down(xpos, ypos, flags);

   // Activate our frame window (in case LCmd isn't already active)
   GProgram& prg = GProgram::GetProgram();
   GWindow& mwin = prg.getMainWindow();
   mwin.setActive();

   GWindow* boundWin = clmnBar.getParentWindow();
   if (boundWin == null)
      boundWin = &clmnBar;

   int minX = getWindowPos().x + getMinimumWidth() - 1;
   int maxX = boundWin->getWindowSize().width - 1;
   int startX = getWindowPos().x + getWindowSize().width;
   int pos = boundWin->trackVSplit(minX, maxX, startX, -1);
   if (pos != -1)
   {
      int width = pos - getWindowPos().x;
      setCustomizedWidth(width);
   }

   return true;
}

bool LCmdFilePanelColumnsBar::Button::onButton2Click ( int xpos, int ypos, const GWindowMessage::InputFlags& /*flags*/ )
{
   if (getPopupMenu() == null)
      setPopupMenu("ColumnsbarButtonContextMenu", false);

   // Activate our frame window (in case LCmd isn't already active)
   GProgram& prg = GProgram::GetProgram();
   GWindow& mwin = prg.getMainWindow();
   mwin.setActive();

   if (!dismissMouseCapture())
      return true;

   setVisiblePopupMenu(true, xpos, ypos);
   return true;
}

void LCmdFilePanelColumnsBar::Button::setCustomizedWidth ( int width, bool updateClmnsbar )
{
   if (width == 0)
   {
      customizedWidth = 0;
   }
   else
   {
      int minWidth = getMinimumWidth();
      customizedWidth = GMath::Max(width, minWidth);
   }
   if (updateClmnsbar)
      clmnBar.updateTheColumnsbar();
}

const GIcon* LCmdFilePanelColumnsBar::Button::getIcon () const
{
   LCmdFilePanel& fp = clmnBar.listWin.fpanel;
   int sortOptIdx = 0;
   if (fp.sortOpt.what[0] == LCmdFilePanelSortOptions::PSW_TYPE)
      sortOptIdx = 1; // Files are sorted by type (file/dir), but don't use the ascending/descending icon to show this. Use the icon on the next sort order type instead.
   if (fp.sortOpt.what[sortOptIdx] != sortID)
      return clmnBar.iconDummy; // Not currently sorted by this column, so return the "dummy" icon.
   if (fp.sortOpt.how[sortOptIdx] == LCmdFilePanelSortOptions::PSH_ASCENDING)
      return clmnBar.iconAscending;
   else
      return clmnBar.iconDescending;
}

int LCmdFilePanelColumnsBar::Button::getIconWidth () const
{
   const GIcon* icon = getIcon();
   if (icon == null)
      return 0;
   return icon->getWidth();
}

GString LCmdFilePanelColumnsBar::Button::getTooltipText ()
{
   GString txt(getKeyboardShortcutKeyText());
   GString txtID = "%" + getIDString();
   txt += GProgram::LoadText(txtID, GResourceTable::LT_PREFER_ALTERNATIVE, 0);
   return txt;
}

GString LCmdFilePanelColumnsBar::Button::getKeyboardShortcutKeyText () const
{
   LCmdFilePanel& fp = clmnBar.listWin.fpanel;
   if (!fp.isCurrentPanel())
      return "";
   return getKeyboardShortcutKeyTextForID(shortcutKeyCmdID);
}

void LCmdFilePanelColumnsBar::Button::cmdColumnsbarButtonHide ( GAbstractCommand* /*cmd*/ )
{
   if (getIDString() == "cmdSortByType") // If we are the "Icon" column
      clmnBar.listWin.fpanel.cmdToggleShowIcons(false); // Hide the icon column
   else
      setVisible(false);
}

void LCmdFilePanelColumnsBar::Button::cmdColumnsbarButtonDefWidth ( GAbstractCommand* /*cmd*/ )
{
   customizedWidth = 0;
   clmnBar.updateTheColumnsbar();
}

void LCmdFilePanelColumnsBar::Button::cmdColumnsbarButtonMinWidth ( GAbstractCommand* /*cmd*/ )
{
   if (this == &clmnBar.clmnIcon)
      cmdColumnsbarButtonDefWidth(); // Always use default width on the icon column.
   else
      setCustomizedWidth(1);
}

// --------------------------------------------------------------------------
LCmdFilePanelColumnsBar::ButtonIcon::ButtonIcon ( LCmdFilePanelColumnsBar& clmnBar )
                                    :Button("cmdSortByType", GString::Empty, clmnBar, LCmdFilePanelSortOptions::PSW_UNSORTED)
{
}

const GIcon* LCmdFilePanelColumnsBar::ButtonIcon::getIcon () const
{
   return null;
}

int LCmdFilePanelColumnsBar::ButtonIcon::getPreferredWidth () const
{
   LCmdFilePanel& fp = clmnBar.listWin.fpanel;
   if (!fp.view.showItemIcon)
      return 0;
   return Button::getPreferredWidth();
}

int LCmdFilePanelColumnsBar::ButtonIcon::calcDefaultWidth () const
{
   LCmdFilePanel& fp = clmnBar.listWin.fpanel;
   if (!fp.view.showItemIcon)
      return 0;
   int strw = getMinimumWidth();
   int iconw = fp.iconSize + 8;
   if (fp.view.showIcon == LCmdFilePanelViewOptions::SHOWICON_LARGE)
      iconw += 6;
   return GMath::Max(strw, iconw);
}

void LCmdFilePanelColumnsBar::ButtonIcon::drawCell ( GGraphics& g, LCmdFilePanel& fp, LCmdFileItem& fitem, const GRectangle& r, const GColor& /*bc*/, const GColor& /*fc*/, bool isCurSelected )
{
   if (fp.view.showItemIcon)
   {
      int xpos = r.x + 4;
      int ypos = r.y + ((clmnBar.listWin.list.itemHeight - fp.iconSize) / 2);
      if (isCurSelected)
         ypos -= 1;
      fitem.drawTheIcon(g, xpos, ypos, fp);
   }
}

// --------------------------------------------------------------------------
LCmdFilePanelColumnsBar::ButtonName::ButtonName ( LCmdFilePanelColumnsBar& clmnBar )
                                    :Button("cmdSortByName", "cmdCurPanelSortByName", clmnBar, LCmdFilePanelSortOptions::PSW_NAME)
{
}

int LCmdFilePanelColumnsBar::ButtonName::calcDefaultWidth () const
{
   int w1 = getMinimumWidth();
   int w2 = clmnBar.listWin.list.widthOfWidestItemFName + 15;
   
   // If the extension-column is hidden we will psint the extension
   // as part of this column. So add space for the extension too.
   if (!clmnBar.clmnExt.isVisible())
      w2 += clmnBar.listWin.list.widthOfWidestItemFExt;
   
   // ---
   return GMath::Max(w1, w2);
}

void LCmdFilePanelColumnsBar::ButtonName::drawCell ( GGraphics& g,
                                                     LCmdFilePanel& fp,
                                                     LCmdFileItem& fitem,
                                                     const GRectangle& r,
                                                     const GColor& /*bc*/,
                                                     const GColor& fc,
                                                     bool /*isCurSelected*/ )
{
   GString fname = fitem.getName();
   if (fitem.isDirectory() || !clmnBar.clmnExt.isVisible())
      fname += fitem.getExtension();
   g.drawText(fname, r, fc);
   if (fitem.isHidden() || fitem.isSystem())
   {
      int textw = g.getWidthOfString(fname);
      int ypos = r.y + (r.height/2) - 1;
      int endx = r.x + textw;
      g.setColor(fc);
      g.setLineType(GGraphics::LTSolid);
      g.drawLine(r.x, ypos, endx, ypos);
   }
}

// --------------------------------------------------------------------------
LCmdFilePanelColumnsBar::ButtonExt::ButtonExt ( LCmdFilePanelColumnsBar& clmnBar )
                                   :Button("cmdSortByExtension", "cmdCurPanelSortByExtension", clmnBar, LCmdFilePanelSortOptions::PSW_EXTENTION)
{
}

int LCmdFilePanelColumnsBar::ButtonExt::calcDefaultWidth () const
{
   int w1 = getMinimumWidth();
   int w2 = clmnBar.listWin.list.widthOfWidestItemFExt + 15;
   return GMath::Max(w1, w2);
}

void LCmdFilePanelColumnsBar::ButtonExt::drawCell ( GGraphics& g, LCmdFilePanel& /*fp*/, LCmdFileItem& fitem, const GRectangle& r, const GColor& /*bc*/, const GColor& fc, bool /*isCurSelected*/ )
{
   GString ext;
   if (!fitem.isDirectory())
      ext = fitem.getExtension();
   g.drawText(ext, r, fc);
   if (fitem.isHidden() || fitem.isSystem())
   {
      int textw = g.getWidthOfString(ext);
      int ypos = r.y + (r.height/2) - 1;
      int endx = r.x + textw;
      g.setColor(fc);
      g.setLineType(GGraphics::LTSolid);
      g.drawLine(r.x, ypos, endx, ypos);
   }
}

// --------------------------------------------------------------------------
LCmdFilePanelColumnsBar::ButtonSize::ButtonSize ( LCmdFilePanelColumnsBar& clmnBar )
                                    :Button("cmdSortBySize", "cmdCurPanelSortBySize", clmnBar, LCmdFilePanelSortOptions::PSW_SIZE),
                                     alwaysLeftAlign(false)
{
}

int LCmdFilePanelColumnsBar::ButtonSize::calcDefaultWidth () const
{
   int strw = getMinimumWidth();
   int valuew = clmnBar.listWin.getWidthOfString("9.999.999XX");
   return GMath::Max(valuew, strw);
}

void LCmdFilePanelColumnsBar::ButtonSize::drawCell ( GGraphics& g,
                                                     LCmdFilePanel& /*fp*/,
                                                     LCmdFileItem& fitem,
                                                     const GRectangle& r,
                                                     const GColor& /*bc*/,
                                                     const GColor& fc,
                                                     bool /*isCurSelected*/ )
{
   fitem.drawTheFileSizeCell(g, r, fc, alwaysLeftAlign);
}

// --------------------------------------------------------------------------
LCmdFilePanelColumnsBar::ButtonDate::ButtonDate ( LCmdFilePanelColumnsBar& clmnBar )
                                    :Button("cmdSortByDate", "cmdCurPanelSortByDate", clmnBar, LCmdFilePanelSortOptions::PSW_DATE)
{
}

int LCmdFilePanelColumnsBar::ButtonDate::calcDefaultWidth () const
{
   int strw = getMinimumWidth();
   int valuew = clmnBar.listWin.getWidthOfString("9999.99.99XX");
   return GMath::Max(valuew, strw);
}

void LCmdFilePanelColumnsBar::ButtonDate::drawCell ( GGraphics& g, LCmdFilePanel& /*fp*/, LCmdFileItem& fitem, const GRectangle& r, const GColor& /*bc*/, const GColor& fc, bool /*isCurSelected*/ )
{
   // Format the date with the user customized date format settings of the system.
   GString str = fitem.timeWrite.getDateString();
   g.drawText(str, r, fc, GGraphics::DUMMY_COLOR, GGraphics::RIGHT);
}

// --------------------------------------------------------------------------
LCmdFilePanelColumnsBar::ButtonTime::ButtonTime ( LCmdFilePanelColumnsBar& clmnBar )
                                    :Button("cmdSortByTime", "cmdCurPanelSortByTime", clmnBar, LCmdFilePanelSortOptions::PSW_TIME)
{
}

int LCmdFilePanelColumnsBar::ButtonTime::calcDefaultWidth () const
{
   int strw = getMinimumWidth();
   int valuew = clmnBar.listWin.getWidthOfString("99:99:99 AM");
   return GMath::Max(valuew, strw);
}

void LCmdFilePanelColumnsBar::ButtonTime::drawCell ( GGraphics& g, LCmdFilePanel& /*fp*/, LCmdFileItem& fitem, const GRectangle& r, const GColor& /*bc*/, const GColor& fc, bool /*isCurSelected*/ )
{
   // Format the time with the user customized time format settings of the system.
   GString str = fitem.timeWrite.getTimeString();
   g.drawText(str, r, fc, GGraphics::DUMMY_COLOR, GGraphics::RIGHT);
}

// --------------------------------------------------------------------------
LCmdFilePanelColumnsBar::LCmdFilePanelColumnsBar ( LCmdFilePanelModeFull& listWin, 
                                                   const GString& constraints )
                        :GToolbar("Columns", 
                                  constraints, 
                                  listWin, 
                                  GString::Empty),
                         listWin(listWin),
                         ignoreIniProfile(false),
                         clmnIcon(*this),
                         clmnName(*this),
                         clmnExt(*this),
                         clmnSize(*this),
                         clmnDate(*this),
                         clmnTime(*this),
                         customColumnSuccession(10),
                         iconAscending(GIcon::GetIcon("IDP_SORTASCENDING")),
                         iconDescending(GIcon::GetIcon("IDP_SORTDESCENDING")),
                         iconDummy(GIcon::GetIcon("IDP_SORT_DUMMY"))
{
   // Add all the supported column buttons. They will always be there (in
   // memory), but the user might choose some of them to be hidden.
   addButton(clmnIcon);
   addButton(clmnName);
   addButton(clmnExt);
   addButton(clmnSize);
   addButton(clmnDate);
   addButton(clmnTime);

   setFlatButtonBorders(false);
   setInsets(null, false);
}

LCmdFilePanelColumnsBar::~LCmdFilePanelColumnsBar ()
{
}

void LCmdFilePanelColumnsBar::writeProfile ( const GString& sectName, bool force )
{
   if (ignoreIniProfile)
      return;

   GToolbar::writeProfile(sectName, force);
}

void LCmdFilePanelColumnsBar::queryProfile ( const GString& sectName )
{
   if (ignoreIniProfile)
      return;

   GToolbar::queryProfile(sectName);

   // Make sure that not all columns are hidden. If they are then
   // assume something is wrong and make everyone visible.
   // This code was added at April 6, 2001 in order to fix a possible
   // bug somewhere (not located) that can possibly sometimes cause
   // every column to be hidden. This could probably only happen
   // if an old LCMD.INI file is used, with some incompatibilities
   // with the latest version. Thus, this code is just an attempt to
   // make life easier for those newly upgraded users, if any.
   // A few people reported this bug with Version 1.5 beta #1 and #2.
   bool any = false;
   const int num = getElementCount();
   for (int i=0; i<num; i++)
   {
      GWindow& butt = getElement(i);
      if (dynamic_cast<LCmdFilePanelColumnsBar::ButtonIcon*>(&butt) != null)
         continue; // Ignore the Icon-column.
      any = butt.isVisible();
      if (any)
         break;
   }
   if (!any)
   {
      cmdColumnsbarButtonShowAll();
      cmdColumnsbarButtonDefWidthAll();
   }
}

int LCmdFilePanelColumnsBar::getPreferredHeight () const
{
   if (!isVisible())
      return 0;

   LCmdFilePanelViewOptions& viewOpt = listWin.fpanel.view;
   if (viewOpt.showColumnsBar)
      return 5 + getHeightOfString("X");
   else
      return 0;
}

void LCmdFilePanelColumnsBar::moveButtonsHor ( int movex )
{
   if (movex == 0)
      return;

   const int num = getElementCount();
   if (movex > 0)
   {
      for (int i=num-1; i>=0; i--)
      {
         GToolbarElement& elm = getElement(i);
         GPoint pt = elm.getWindowPos();
         elm.setWindowPos(pt.x + movex, pt.y);
         elm.updateWindow();
      }
   }
   else
   {
      for (int i=0; i<num; i++)
      {
         GToolbarElement& elm = getElement(i);
         GPoint pt = elm.getWindowPos();
         elm.setWindowPos(pt.x + movex, pt.y);
         elm.updateWindow();
      }
   }

   updateWindow();
}

bool LCmdFilePanelColumnsBar::onButton2Click ( int xpos, int ypos, const GWindowMessage::InputFlags& /*flags*/ )
{
   if (getPopupMenu() == null)
      setPopupMenu("ColumnsbarContextMenu", false);

   // Activate our frame window (in case LCmd isn't already active)
   GProgram& prg = GProgram::GetProgram();
   GWindow& mwin = prg.getMainWindow();
   mwin.setActive();

   setVisiblePopupMenu(true, xpos, ypos);
   return true;
}

void LCmdFilePanelColumnsBar::updateTheColumnsbar ()
{
   // Update the items list content and scrollbars.
   layout();
   listWin.invalidateAll(true);
   int prevLColXPos = listWin.list.leftMostColumnXPos;
   listWin.list.calcAllColumns();
   listWin.list.leftMostColumnXPos = prevLColXPos;
   listWin.list.updateScrollBarPos();
}

int LCmdFilePanelColumnsBar::getColumnsCount ()
{
   return getElementCount();
}

LCmdFilePanelColumnsBar::Button& LCmdFilePanelColumnsBar::getColumnButton ( int index )
{
   GWindow& child = getElement(index);
   return dynamic_cast<Button&>(child);
}

void LCmdFilePanelColumnsBar::cmdColumnsbarButtonShowAll ( GAbstractCommand* /*cmd*/ )
{
   const int num = getColumnsCount();
   for (int i=0; i<num; i++)
   {
      Button& b = getColumnButton(i);
      b.GToolbarButton::setVisible(true);
   }
   listWin.fpanel.view.showItemIcon = true; // Restore the icon-column as well.
   listWin.fpanel.calcIconSize();
   updateTheColumnsbar();
}

void LCmdFilePanelColumnsBar::cmdColumnsbarButtonDefWidthAll ( GAbstractCommand* /*cmd*/ )
{
   const int num = getColumnsCount();
   for (int i=0; i<num; i++)
   {
      Button& b = getColumnButton(i);
      b.setCustomizedWidth(0, false);
   }
   updateTheColumnsbar();
}

void LCmdFilePanelColumnsBar::cmdColumnsbarButtonMinWidthAll ( GAbstractCommand* /*cmd*/ )
{
   const int num = getColumnsCount();
   for (int i=0; i<num; i++)
   {
      Button& b = getColumnButton(i);
      if (&b == &clmnIcon)
         b.setCustomizedWidth(0, false); // Always use default width on the icon column.
      else
         b.setCustomizedWidth(1, false);
   }
   updateTheColumnsbar();
}

GToolbarElement* LCmdFilePanelColumnsBar::addElementImpl ( GToolbarElement* elm, bool autoDelete )
{
   GToolbarElement* ret = GToolbar::addElementImpl(elm, autoDelete);
   int posOnScreenIdx = getElementCount() - 1;
   customColumnSuccession.add(posOnScreenIdx);
   return ret;
}

void LCmdFilePanelColumnsBar::setColumnPositionOnScreen ( int currentIdx, int newIdx )
{
   int count = getElementCount();
   if (currentIdx < 0 || currentIdx >= count)
      gthrow_(GIllegalArgumentException("currentIdx=" + GInteger::ToString(currentIdx)));
   if (newIdx < 0 || newIdx >= count)
      gthrow_(GIllegalArgumentException("newIdx=" + GInteger::ToString(newIdx)));
   if (newIdx == currentIdx)
      return;
   customColumnSuccession.swap(currentIdx, newIdx);
   layout();
}

void LCmdFilePanelColumnsBar::layout ()
{
   if (getHWND() == null)
      return;

   GRectangle r = getWindowRect();
   const GInsets& ins = getInsets();
   int xpos = ins.left;
   int ypos = ins.bottom;
   int height = r.height - ins.bottom - ins.top;
   int num = getElementCount();
   for (int i=0; i<num; i++)
   {
      int nextIdx = customColumnSuccession[i];
      GToolbarElement& butt = getElement(nextIdx);
      int width = butt.getPreferredWidth();
      butt.setWindowBounds(xpos, ypos, width, height);
      xpos += width;
   }

   int movex = listWin.list.leftMostColumnXPos;
   moveButtonsHor(movex);
}
