/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __LCMD_FILEPANELHEADER
#define __LCMD_FILEPANELHEADER

#include "glib/gui/GAbstractToolbarWindow.h"
#include "glib/gui/GToolbarButton.h"

/**
 * This class defines the header bar window of a file panel.
 *
 * The header bar always contains three buttons that are all visible by
 * default; the Drive-, Up-Dir-, and the Root-Dir- buttons. In addition,
 * it contains a text entry window that is visible only when the user
 * is about to type in some dynamic filename search text.
 *
 * @author  Leif Erik Larsen
 * @since   1999.04.07
 */
class LCmdFilePanelHeader : public GAbstractToolbarWindow
{
   public:

      /**
       * The base class of all the buttons in the header bar.
       *
       * @author  Leif Erik Larsen
       * @since   2004.09.18
       */
      class Button : public GToolbarButton
      {
         protected:

            LCmdFilePanelHeader& headerBar;

         protected:

            Button ( LCmdFilePanelHeader& ownerTB, 
                     const GString& text, 
                     const GString& iconID = GString::Empty );

            virtual ~Button ();

         public:

            virtual void performAction ();
            virtual void updateIDString () = 0;
            virtual int getPreferredWidth () const;
            virtual bool onButton1Down ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );
      };

      /**
       * The Drive Button.
       *
       * @author  Leif Erik Larsen
       * @since   2004.09.18
       */
      class DriveButton : public Button
      {
         public:

            DriveButton ( LCmdFilePanelHeader& ownerTB );
            virtual ~DriveButton ();

         public:

            void updateIDString ();
            GString getTooltipText () const;
            GString getText () const;
      };

      /**
       * The Up-Directory Button.
       *
       * @author  Leif Erik Larsen
       * @since   2004.09.18
       */
      class UpDirButton : public Button
      {
         public:

            UpDirButton ( LCmdFilePanelHeader& ownerTB );
            virtual ~UpDirButton ();

         public:

            void updateIDString ();
            GString getKeyboardShortcutKeyText () const;
            GString getTooltipText () const;
      };

      /**
       * The Root-Directory Button.
       *
       * @author  Leif Erik Larsen
       * @since   2004.09.18
       */
      class RootButton : public Button
      {
         public:

            RootButton ( LCmdFilePanelHeader& ownerTB );
            virtual ~RootButton ();

         public:

            void updateIDString ();
            GString getKeyboardShortcutKeyText () const;
            GString getTooltipText () const;
      };

      /** A reference to the parent window. */
      class LCmdFilePanelFrame& panelFrame;

      /** Click to choose which drive to activate. */
      DriveButton* driveButt;

      /** Click to walk one directory level up. */
      UpDirButton* upDirButt;

      /** Click to activate the roor directory. */
      RootButton* rootButt;

      /** Overriding {@link GWindow#tooltipPos} if != {@link GTooltip#PosInheritFromParent}. */
      mutable GTooltip::Position overriddenTooltipPos;

      /**
       * Entryfield window of where the user can edit a filename
       * for dynamic filename search in panel.
       *
       * This entryfield is initially hidden, but will be visible when user
       * press ALT-X to start a dynamic search. The size of the entryfield
       * window will be adjusted upon <i>layout()</i>.
       */
      class LCmdDynaFileSearch* dynaFileSearch;

      /** The rectangle of where to paint the title text. */
      GRectangle titleRect;

   public:

      LCmdFilePanelHeader ( class LCmdFilePanelFrame& parentWin, 
                            const GString& constraints );

      virtual ~LCmdFilePanelHeader ();

   protected:

      virtual bool onBackgroundColorChanged ( const GColor& color );
      virtual bool onButton1Down ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );
      virtual bool onForegroundColorChanged ( const GColor& color );
      virtual bool onPaint ( GGraphics& g, const GRectangle& rect );
      virtual bool onMouseMove ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );

   public:

      void layout ();

      const GColor& getPreferredBckColor () const;
      const GColor& getPreferredTxtColor () const;

      /**
       * @author  Leif Erik Larsen
       * @since   2005.12.21
       */
      virtual GTooltip::Position getTooltipPosition () const;

      /**
       * @author  Leif Erik Larsen
       * @since   2005.12.13
       */
      virtual GString getTooltipText () const;

      /**
       * This method is called when the containing file panel is activated
       * or deactivated, to repaint the header with the preferred colors
       * after the activation/deactivation.
       *
       * @author  Leif Erik Larsen
       * @since   2002.08.20
       */
      void invalidateClientAreaForText ();

      /**
       * Get the preferred height of the window, in pixels.
       */
      int getPreferredHeight () const;

      /**
       * Terminate the ongoing dynamic file name search, if any.
       */
      void dismissDynamicSearch ();

      /**
       * Test if the header of the specified file panel is about
       * performing a dynamic file name search.
       *
       * @return True if the dynamic search is currently ongoing,
       *         or else false.
       */
      bool isPerformingDynamicSearch ();
};

#endif // #ifndef __LCMD_FILEPANELHEADER
