/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __LCMD_OPTIONS
#define __LCMD_OPTIONS

#include "glib/util/GArray.h"
#include "glib/primitives/GBoolean.h"

#include "lcmd/LCmdFilePanelStartupOptions.h"
#include "lcmd/LCmdFilePanelFilterOptions.h"
#include "lcmd/LCmdFilePanelSortOptions.h"
#include "lcmd/LCmdFilePanelViewOptions.h"
#include "lcmd/LCmdDlgCompareDirs.h"
#include "lcmd/LCmdDlgFileFilterProperties.h"
#include "lcmd/LCmdDlgQuickDirList.h"
#include "lcmd/LCmdSepSessionApps.h"

/**
 * This class contains most of the user customizable options of the program.
 *
 * @author  Leif Erik Larsen
 * @since   2001.02.20
 */
class LCmdOptions : public GObject
{
   public:

      /**
       * This class is used to store the configuration of the
       * directory cache options.
       *
       * @author  Leif Erik Larsen
       * @since   2007.02.06
       */
      class DirCacheOpts
      {
         public:

            /** True if we shall add visited directories to the cache during CD & Walk operations. */
            bool addDirsDuringWalk;

            /** True if we shall remove from the Directory Cache when directories are deleted or moved by larsen Commander. */
            bool autoStripDirsAfterDirDeletions;

            /** True if we shall track VFS directories as well. */
            bool includeVfsDirs;

         public:

            DirCacheOpts ();

         public:

            void restoreToDefault ();
      };

   public:

      /**
       * This class is used to store the configuration of the
       * confirm options.
       *
       * @author  Leif Erik Larsen
       * @since   2001.03.13
       */
      class ConfirmOpts
      {
         public:

            /** True if ask user before exit. */
            bool prgExit;

            /** True if ask user before delete each top-level file. */
            bool delFile;

            /** True if ask user before delete each individual file in sub-directories. */
            bool delFileInSubDirs;

            /** True if ask user before delete an empty directory. */
            bool delEmptyDir;

            /** True if ask user before delete dir w/files. */
            bool delDirContainingFiles;

            /** True if ask user before delete hid/sys/read file or directory. */
            bool delHidSysReadFile;

            /** True if ask user before overwrite existing file or directory. */
            bool overwriteFileOrDir;

         public:

            ConfirmOpts ();

         public:

            void restoreToDefault ();
      };

   public:

      /**
       * This class is used to store the configuration for which options to
       * automatically save into INI-file upon program exit.
       *
       * @author  Leif Erik Larsen
       * @since   2001.03.13
       */
      class SaveOptOnExit
      {
         public:

            /** True if autosave all options upon exit. */
            bool everything;

            /** True if autosave current directory in left and right panel. */
            bool currentDirs;

            /** True if autosave the table of stored directories (the fast dirlist). */
            bool storedDirs;

            /** True if autosave the directory history. */
            bool dirHist;

            /** True if autosave the command history. */
            bool cmdHist;

            /** True if autosave the directory cache. */
            bool dirCache;

            /** True if autosave which file is currently selected in left and right panel. */
            bool curSelFiles;

            /** True if autosave visibility state of window componets. */
            bool visibleState;

            /** True if autosave panel modes and sort options. */
            bool panelModeSort;

            /** True if autosave frame window position. */
            bool frameWinPos;

            /** True if autosave color options. */
            bool colors;

            /** True if autosave color options. */
            bool fonts;

            /** True if autosave all other options. */
            bool otherOptions;

         public:

            SaveOptOnExit ();

         public:

            void restoreToDefault ();
      };

   public:

      /**
       * This class is used to store the configuration of the
       * file copy/move progress options
       *
       * @author  Leif Erik Larsen
       * @since   2001.03.13
       */
      class FileCopyOpts
      {
         public:

            /** The types of "Preallocate File Space" options that can be selected by user. */
            enum PreallocID { Never = 0, Always, SpecifiedDrives };

            /** True if we shall show statistics in file copy/move progress-bar. */
            bool showStatistics;

            /** True if we shall wait for the user to click "OK" when file copy/move has finished. */
            bool waitOnFinish;

            /** True if we should show a warning message box if target doesn't seem to have enough space. */
            bool warnIfNotEnoughTargetSpace;

            /** True if we shall preallocate destination filesize before start copying file content. */
            PreallocID preallocSpace;

            /** Commaseparated list of which drives to preallocate space on. E.g.: "C:, E:, F:". */
            GString preallocSpaceOnDrives;

            /** Preallocate File Space only if filesize is not larger than this number of bytes. */
            longlong maxPreallocateSize;

            /** Maximum size of the allocated copy-buffer, in kilobytes. */
            int maxBuffSize;

         public:

            FileCopyOpts ();

         public:

            void restoreToDefault ();
            void setPreallocSpaceFromInt ( int value );

            /**
             * Check if file space is to be preallocated for the specified 
             * drive, where driveStr is e.g. "C:" or "D:".
             *
             * @author  Leif Erik Larsen
             * @since   2006.11.06
             */
            bool isPreallocSpaceForDrive ( const GString& driveStr );
      };

   public:

      /**
       * This class is used to store the configuration of how files and
       * directories are to be deleted by default.
       *
       * @author  Leif Erik Larsen
       * @since   2001.04.17
       */
      class FileDelOpts
      {
         public:

            /** Default state of the "Delete to Recycle Bin/Trash Can" flag. */
            bool defaultDeleteToTrashCan;

            /** True if the Delete-key should be used to delete files when command line is empty or hidden. */
            bool useDelKey;

            /** True if the progress dialog should be updated on directories only. */
            bool showDirsProgressOnly;

         public:

            FileDelOpts ();

         public:

            void restoreToDefault ();
      };

   public:

      /**
       * This class is used to store the configuration of how to
       * run Console and Dos programs by default.
       *
       * @author  Leif Erik Larsen
       * @since   2001.04.17
       */
      class ConVariousOpts
      {
         public:

            /** Default state of the "Close console windows on exit". */
            bool defaultCloseConOnExit;

            /** True if start Console programs in fullscreen mode. */
            bool startConFullScreen;

            /** True if start Dos programs in fullscreen mode. */
            bool startDosFullScreen;

         public:

            ConVariousOpts ();

         public:

            void restoreToDefault ();
      };

   public:

      /**
       * This class is used to store various options for VFS in general.
       *
       * @author  Leif Erik Larsen
       * @since   2004.10.29
       */
      class VfsVariousOpts
      {
         public:

            VfsVariousOpts ();

         public:

            void restoreToDefault ();
      };

   public:

      /**
       * This class is used to store options for the ZIP-file VFS.
       *
       * @author  Leif Erik Larsen
       * @since   2004.10.29
       */
      class VfsZipOpts
      {
         public:

            /** True if we shall let the system launch ZIP-file objects by default, when user press enter on a ZIP-file. */
            bool sysLaunchZipFiles;

            /** The path to the ZIP.EXE or corresponding command line tool. */
            GString toolPath;

            /** The arguments to use for performing delete operations via the command line tool. */
            GString argsDelete;

         public:

            VfsZipOpts ();

         public:

            void restoreToDefault ();
      };

   public:

      /**
       * This class is used to store some various options.
       *
       * @author  Leif Erik Larsen
       * @since   2001.04.17
       */
      class VariousOpts
      {
         public:

            /** Number of pixels to add (or subtract, if negative) to the default file item heights. */
            int addToItemHeight;

            /** Number of pixels to add (or subtract, if negative) to the default file item width (used for brief mode only). */
            int addToItemWidth;

            /** Number of milliseconds until auto hiding tooltip box, or zero for "never". */
            int autoHideTooltipMillis;

            /** Number of milliseconds to wait until actually displaying the tooltip box. */
            int delayShowTooltipMillis;

         public:

            VariousOpts ();

         public:

            void restoreToDefault ();
      };

   public:

      /**
       * This class is used to store the configuration of a single
       * extarnal utility program.
       *
       * @author  Leif Erik Larsen
       * @since   2001.03.13
       */
      class ProgOption
      {
         public:

            /** True if the program has an internal alternative. */
            bool internalAvail;

            /** True if we shall use the external program. */
            bool useExternal;

            /** The program module name. */
            GString progName;

            /** The optional program startup parameters. */
            GString params;

            /** The working directory of the program. */
            GString workDir;

         private:

            /** Backup copy of {@link #internalAvail} to use as "default" value in option dialog. */
            bool def_internalAvail;

            /** Backup copy of {@link #useExternal} to use as "default" value in option dialog. */
            bool def_useExternal;

            /** Backup copy of {@link #progName} to use as "default" value in option dialog. */
            GString def_progName;

            /** Backup copy of {@link #params} to use as "default" value in option dialog. */
            GString def_params;

            /** Backup copy of {@link #workDir} to use as "default" value in option dialog. */
            GString def_workDir;

         public:

            ProgOption ();

            ProgOption ( const ProgOption& prog );

            ProgOption ( const char* progName, 
                         const char* params, 
                         const char* workDir, 
                         bool internalAvail, 
                         bool useExternal );

         public:

            ProgOption& operator= ( const ProgOption& prog );
            bool operator== ( const ProgOption& prog );

         private:

            /**
             * Common code to be used by the copy constructor and by the
             * assign operator.
             */
            ProgOption& copyFrom ( const ProgOption& prog );

            /**
             * Initialize all the <i>def_*</i> variables so that they all contains
             * a mirror of their respective variable that they are to "backup".
             */
            void init_def ();

         public:

            /**
             * Restore all the program option variables from their respective
             * <i>def_</i> variable so that the program options are set
             * back to default.
             */
            void restoreToDefault ();

            /**
             * Launch the program giving the specified parameters. In case of any
             * error we will show the corresponding error message to the user
             * automatically.
             *
             * @param  forceNewSession True to force the program to run in a new
             *                         session.
             * @param  noEcho          True to prevent the command from being
             *                         echoed on console monitor.
             * @return True if the program was launched successfully,
             *         or else false on any error.
             */
            bool launchIt ( bool forceNewSession = false, bool noEcho = false ) const;
      };

   public:

      const GString DefaultFontFilePanelHeader;
      const GString DefaultFontFrameTitlebar;
      const GString DefaultFontFrameMenubar;
      const GString DefaultFontConMon;
      const GString DefaultFontPrompt;
      const GString DefaultFontCmdLineEntry;

   public:

      /**
       * Relative size of the first file panel, in 1/1000.
       *
       * The "first" file panel means the panel that is to the "left" by
       * default. It can, however, currently be to the "right", depending
       * on the most recent "Swap Panels" (Ctrl+U) command.
       */
      int panel1RelWidth;

      /** Relative height of the console, in 1/1000. */
      int consoleRelHeight;

      /** True if startup with panel sizes at 50/50. */
      bool panelSizesStart5050;

      /** True if panels should be shown in "Full Screen" mode. */
      bool panelsFullScreen;

      /** True if panels are currently visible. */
      bool panelsAreVisible;

      /** True if we shall use fancy menues (with icons, etc.). */
      bool fancyMenuBars;

      /** True if we shall use fancy menues (with icons, etc.). */
      bool fancyPopupMenues;

      /** Filter used by various recursive file operations, such as "File Attributes", "Time Stamp", "Filename Case", "User Process Files", etc. */
      LCmdDlgFileFilterProperties fileFilter;

      /** Confirmation options. */
      ConfirmOpts confirm;

      /** File copy/move progress options. */
      FileCopyOpts fileCopy;

      /** Options for file and directory deletion. */
      FileDelOpts fileDel;

      /** Options for the dynamic Directory Cache. */
      DirCacheOpts dirCache;

      /** Options that are used to specify how to run Console and Dos programs by default. */
      ConVariousOpts conVarious;

      /** Various options for VFS in general. */
      VfsVariousOpts vfsVarious;

      /** Options for the ZIP-file VFS. */
      VfsZipOpts vfsZip;

      /** Some various options. */
      VariousOpts various;

      /** Options of which to save upon exit. */
      SaveOptOnExit saveOnExit;

      /** Ten stored directories for quick and easy access. */
      LCmdDlgQuickDirList quickDirList;

      /** Options used to adjust the way two directories are compared. */
      LCmdDlgCompareDirs::COMPAREDIRSOPT cmpDirs;

      /** Editor. */
      ProgOption editor;

      /** Viewer. */
      ProgOption viewer;

      /** File Find. */
      ProgOption ffind;
      
      /** File Diff. */
      ProgOption fdiff;
      
      /** Clone Clean. */
      ProgOption cclean;

      /** Undelete. */
      ProgOption undel;

      /** Extended Attributes Viewer/Editor. */
      ProgOption eas;

   public:

      LCmdOptions ( class LCmd& lcmd );
      virtual ~LCmdOptions ();

   public:

      /**
       * Get a reference to the current options of Larsen Commander.
       *
       * @author  Leif Erik Larsen
       * @since   2004.11.15
       * @see     LCmd#options
       */
      static LCmdOptions& GetOptions ();
};

/**
 * This class implements the dialog box that can be used by the user to
 * edit most of the user customizable options of the program.
 *
 * @author  Leif Erik Larsen
 * @since   2001.02.20
 */
class LCmdDlgOptions : public GDialogMessageHandler
{
   public:

      /** LEFT=Left file panel, RIGHT=Right file panel. */
      enum FilePanelID
      {
         LEFT = 0,
         RIGHT
      };

      // ---
      class Startup : public GDialogMessageHandler
      {
         private:

            FilePanelID fpid;
            LCmdFilePanelStartupOptions* current;
            LCmdFilePanelStartupOptions backup;

         public:

            Startup ( FilePanelID fpid );
            virtual ~Startup () {}

         private:

            void setControlValues ( class GDialogPanel& dlg );
            void getControlValues ( class GDialogPanel& dlg );
            virtual bool handleDialogMessage ( class GDialogMessage& msg );
      };

      // ---
      class Filter : public GDialogMessageHandler
      {
         private:

            FilePanelID fpid;
            class LCmdFilePanel* fpanel;
            LCmdFilePanelFilterOptions* current;
            LCmdFilePanelFilterOptions backup;

         public:

            Filter ( FilePanelID fpid );
            virtual ~Filter () {}

         private:

            void setControlValues ( class GDialogPanel& dlg );
            void getControlValues ( class GDialogPanel& dlg );
            virtual bool handleDialogMessage ( class GDialogMessage& msg );
      };

      // ---
      class Sort : public GDialogMessageHandler
      {
         private:

            FilePanelID fpid;
            class LCmdFilePanel* fpanel;
            LCmdFilePanelSortOptions* current;
            LCmdFilePanelSortOptions backup;

            int whatID[4];
            int howID[4];

         public:

            Sort ( FilePanelID fpid );
            virtual ~Sort () {}

         private:

            void setControlValues ( class GDialogPanel& dlg );
            void getControlValues ( class GDialogPanel& dlg );
            virtual bool handleDialogMessage ( class GDialogMessage& msg );
      };

      // ---
      class ViewCmn : public GDialogMessageHandler
      {
         private:

            struct Options
            {
               bool showToolbar;
               bool showKeybar;
               bool showStatbar;
               bool showCmdLine;
               bool panelsFullScreen;
               int  panel1RelWidth;
               int  consoleRelHeight;
               bool panelSizesStart5050;
            };

         private:

            Options current;
            Options backup;

         public:

            ViewCmn () {}
            virtual ~ViewCmn () {}

         private:

            void setControlValues ( class GDialogPanel& dlg );
            void getControlValues ( class GDialogPanel& dlg );
            void activateCurrentSettings ();
            virtual bool handleDialogMessage ( class GDialogMessage& msg );
      };

      // ---
      class View : public GDialogMessageHandler
      {
         private:

            FilePanelID fpid;
            class LCmdFilePanel* fpanel;
            LCmdFilePanelViewOptions* current;
            LCmdFilePanelViewOptions backup;

         public:

            View ( FilePanelID fpid );
            virtual ~View () {}

         private:

            void setControlValues ( class GDialogPanel& dlg );
            LCmdFilePanelViewOptions::VIEWMODE getControlValues ( class GDialogPanel& dlg );
            virtual bool handleDialogMessage ( class GDialogMessage& msg );
      };

      // ---
      class ViewStyle : public GDialogMessageHandler
      {
         private:

            struct Options
            {
               bool toolbarFlatButtons;
               bool toolbarTooltip;
               bool cmdlineFlatButtons;
               bool cmdlineTooltip;
               bool keybarFlatButtons;
               bool keybarTooltip;
               bool headbarsTooltip;
               bool columnbarsTooltip;
               bool infobarsTooltip;
               bool statusbarTooltip;
               bool fancyMenuBars;
               bool fancyPopupMenues;
            };

         private:

            Options current;
            Options backup;

         public:

            ViewStyle () {}
            virtual ~ViewStyle () {}

         private:

            void setControlValues ( class GDialogPanel& dlg );
            void getControlValues ( class GDialogPanel& dlg );
            void activateCurrentSettings ();
            virtual bool handleDialogMessage ( class GDialogMessage& msg );
      };

      // ---
      class Confirm : public GDialogMessageHandler
      {
         private:

            LCmdOptions::ConfirmOpts& current;
            LCmdOptions::ConfirmOpts backup;

         public:

            Confirm ( LCmdOptions::ConfirmOpts& current );
            virtual ~Confirm () {}

         private:

            void setControlValues ( class GDialogPanel& dlg );
            void getControlValues ( class GDialogPanel& dlg );
            virtual bool handleDialogMessage ( class GDialogMessage& msg );
      };

      // ---
      class SaveOpt : public GDialogMessageHandler
      {
         private:

            LCmdOptions::SaveOptOnExit& current;
            LCmdOptions::SaveOptOnExit backup;

         public:

            SaveOpt ( LCmdOptions::SaveOptOnExit& current );
            virtual ~SaveOpt () {}

         private:

            void setControlValues ( class GDialogPanel& dlg );
            void getControlValues ( class GDialogPanel& dlg );
            virtual bool handleDialogMessage ( class GDialogMessage& msg );
      };

      // ---
      class FileCopy : public GDialogMessageHandler
      {
         private:

            LCmdOptions::FileCopyOpts& current;
            LCmdOptions::FileCopyOpts backup;

         public:

            FileCopy ();
            virtual ~FileCopy () {}

         private:

            void setControlValues ( class GDialogPanel& dlg );
            void getControlValues ( class GDialogPanel& dlg );
            void updateEnabledStateOfDrivesEntry ( GDialogPanel& dlg );
            virtual bool handleDialogMessage ( class GDialogMessage& msg );
      };

      // ---
      class ConVarious : public GDialogMessageHandler
      {
         private:

            LCmdOptions::ConVariousOpts& current;
            LCmdOptions::ConVariousOpts backup;

         public:

            ConVarious ( LCmdOptions::ConVariousOpts& current );
            virtual ~ConVarious () {}

         private:

            void setControlValues ( class GDialogPanel& dlg );
            void getControlValues ( class GDialogPanel& dlg );
            virtual bool handleDialogMessage ( class GDialogMessage& msg );
      };

      // ---
      class VfsVarious : public GDialogMessageHandler
      {
         private:

            LCmdOptions::VfsVariousOpts& current;
            LCmdOptions::VfsVariousOpts backup;

         public:

            VfsVarious ( LCmdOptions::VfsVariousOpts& current );
            virtual ~VfsVarious () {}

         private:

            void setControlValues ( class GDialogPanel& dlg );
            void getControlValues ( class GDialogPanel& dlg );
            virtual bool handleDialogMessage ( class GDialogMessage& msg );
      };

      // ---
      class VfsZip : public GDialogMessageHandler
      {
         private:

            LCmdOptions::VfsZipOpts& current;
            LCmdOptions::VfsZipOpts backup;

         public:

            VfsZip ( LCmdOptions::VfsZipOpts& current );
            virtual ~VfsZip () {}

         private:

            void setControlValues ( class GDialogPanel& dlg );
            void getControlValues ( class GDialogPanel& dlg );
            virtual bool handleDialogMessage ( class GDialogMessage& msg );
      };

      // ---
      class Various : public GDialogMessageHandler
      {
         private:

            LCmdOptions::VariousOpts& current;
            LCmdOptions::VariousOpts backup;

         public:

            Various ( LCmdOptions::VariousOpts& current );
            virtual ~Various () {}

         private:

            void setControlValues ( class GDialogPanel& dlg );
            void getControlValues ( class GDialogPanel& dlg );
            virtual bool handleDialogMessage ( class GDialogMessage& msg );
      };

      // ---
      class FileDel : public GDialogMessageHandler
      {
         private:

            LCmdOptions::FileDelOpts& current;
            LCmdOptions::FileDelOpts backup;

         public:

            FileDel ( LCmdOptions::FileDelOpts& current );
            virtual ~FileDel () {}

         private:

            void setControlValues ( class GDialogPanel& dlg );
            void getControlValues ( class GDialogPanel& dlg );
            virtual bool handleDialogMessage ( class GDialogMessage& msg );
      };

      // ---
      class DirCache : public GDialogMessageHandler
      {
         private:

            LCmdOptions::DirCacheOpts& current;
            LCmdOptions::DirCacheOpts backup;

         public:

            DirCache ( LCmdOptions::DirCacheOpts& current );
            virtual ~DirCache () {}

         private:

            void setControlValues ( class GDialogPanel& dlg );
            void getControlValues ( class GDialogPanel& dlg );
            virtual bool handleDialogMessage ( class GDialogMessage& msg );
      };

      // ---
      class FontCmn : public GDialogMessageHandler
      {
         private:

            class Options
            {
               public:

                  GString fontToolBar;
                  GString fontConMon;
                  GString fontPrompt;
                  GString fontCmdLine;
                  GString fontStatusBar;
                  GArray<GString> fontKeyBarButtons;
            };

         private:

            Options backup;

         public:

            FontCmn () {}
            virtual ~FontCmn () {}

         private:

            void setControlValues ( class GDialogPanel& dlg );
            virtual bool handleDialogMessage ( class GDialogMessage& msg );
      };

      // ---
      class Font : public GDialogMessageHandler
      {
         private:

            class Options
            {
               public:

                  GString fontHeaderBar;
                  GString fontBrief;
                  GString fontWide;
                  GString fontFull;
                  GString fontTree;
                  GString fontInfo;
                  GString fontInfoBar;
            };

         private:

            FilePanelID fpid;
            class LCmdFilePanel* fpanel;
            Options backup;

         public:

            Font ( FilePanelID fpid );
            virtual ~Font () {}

         private:

            void setControlValues ( class GDialogPanel& dlg );
            virtual bool handleDialogMessage ( class GDialogMessage& msg );
      };

      // ---
      class Programs : public GDialogMessageHandler
      {
         private:

            LCmdOptions::ProgOption& current;
            LCmdOptions::ProgOption backup;

         public:

            Programs ( LCmdOptions::ProgOption& current );
            virtual ~Programs () {}

         private:

            void setControlValues ( class GDialogPanel& dlg );
            virtual bool handleDialogMessage ( class GDialogMessage& msg );
      };

      // ---
      class Commands : public GDialogMessageHandler
      {
         private:

            GArray<GBoolean> bckEnabledInternal;
            GArray<GBoolean> bckEnabledCmdExe;

         public:

            Commands ();
            virtual ~Commands () {}

         private:

            virtual bool handleDialogMessage ( class GDialogMessage& msg );
      };

      // ---
      class StrBagEd : public GDialogMessageHandler
      {
         private:

            GKeyBag<GString>& current;
            GKeyBag<GString> backup;

         public:

            StrBagEd ( GKeyBag<GString>& current );
            virtual ~StrBagEd () {}

         private:

            void setControlValues ( class GDialogPanel& dlg );
            virtual bool handleDialogMessage ( class GDialogMessage& msg );
      };

      // ---
      class SSA : public GDialogMessageHandler
      {
         private:

            GKeyBag<LCmdSepSessionApps>& current;
            GKeyBag<LCmdSepSessionApps> backup;

         public:

            SSA ( GKeyBag<LCmdSepSessionApps>& current );
            virtual ~SSA () {}

         private:

            virtual bool handleDialogMessage ( class GDialogMessage& msg );
      };

   private:

      /** Will be false until the dialog has been executed for the first time. */
      bool hasBeenExecuted;

      /** The id of which page to initially activate upon dialog execution. */
      GString initialPageID;

      Startup startup1, startup2;
      Filter filter1, filter2;
      Sort sort1, sort2;
      ViewCmn viewCmn;
      View view1, view2;
      ViewStyle viewStyle;
      Confirm confirm;
      SaveOpt saveOpt;
      FileCopy fileCopy;
      FileDel fileDel;
      DirCache dirCache;
      ConVarious conVarious;
      VfsVarious vfsVarious;
      VfsZip vfsZip;
      Various various;
      FontCmn fontCmn;
      Font font1, font2;
      Programs progEdit, progView, progFFind, progFDiff, progCClean;
      Programs progUndel, progEAEdit;
      Commands commands;
      StrBagEd env, aliases;
      SSA ssa;

      aptr<GDialogFrame> dlgFrame;

      class GTabbedPanel& tabs;
      class GTabbedPanel& tabsStartup;
      class GTabbedPanel& tabsFilter;
      class GTabbedPanel& tabsSort;
      class GTabbedPanel& tabsView;
      class GTabbedPanel& tabsVarious;
      class GTabbedPanel& tabsFonts;
      class GTabbedPanel& tabsPrograms;
      class GTabbedPanel& tabsConsole;
      class GTabbedPanel& tabsVfs;

   public:

      LCmdDlgOptions ();
      virtual ~LCmdDlgOptions () {}

   public:

      void executeModal ( const GString& id );
      virtual bool handleDialogMessage ( class GDialogMessage& msg );
      static void CmdJumpToOptionsPage ( const GString& id );
};

#endif // #ifndef __LCMD_OPTIONS
