/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __LCMD_VFSFILEPREPARATOR
#define __LCMD_VFSFILEPREPARATOR

#include "glib/gui/GWorkerThread.h"
#include "glib/vfs/GVfs.h"

/**
 * This class is used to simplify the task of displaying a progress
 * bar for preparing one or more files in a virtual file system for
 * physical action.
 *
 * The progress bar dialog is displayed only if the virtual file system
 * says that the file preparation operation will possibly take some
 * time. This is true e.g. for archive file systems, and false for
 * the standard physical file system.
 *
 * For a sample showing exactly how to use this class, please take
 * a look at the implementation of {@link LCmdMainWindow#cmdViewFile}.
 *
 * @author  Leif Erik Larsen
 * @since   2005.01.05
 */
class LCmdVfsFilePreparator : public GWorkerThread
{
   private:

      /** The original name of the file within VFS. */
      GString originalFileName;

      /** The resulting prepared physical file, only temporarily used. */
      GVfs::File* physicalFileRep;

      /** The size of the physical file, after preparation. */
      longlong fileSize;

      class LCmdFilePanel& fp;
      GVfs& vfs;
      GVfs::WorkStatus workStatus;

   public:

      LCmdVfsFilePreparator ( class LCmdFilePanel& fp );
      virtual ~LCmdVfsFilePreparator ();

      /**
       * Prepare the currently selected file of the specified file panel.
       * The preparation is done in a background thread if necessary.
       *
       * If the current file system of the file panel is the standard
       * physical file system then this will virtually do nothing but
       * return a reference directly to the file on the physical
       * file system.
       *
       * If the file had to be e.g. unpacked or downloaded (depending
       * on the virtual file system) the returned file object's
       * physical path will automatically be deleted when the returned
       * file object comes out of scope.
       *
       * @author  Leif Erik Larsen
       * @since   2005.01.04
       * @return  A file item representing the prepared file on the physical
       *          file system, or null if the user canceled the operation.
       *          The ownership of the returned object belongs to the caller.
       */
      GVfs::File* prepareSelectedFile ( class GWindow& parentWin );

      /**
       * @author  Leif Erik Larsen
       * @since   2005.08.24
       * @throws  GIOException in case of any error preparing the file.
       */
      GVfs::File* prepareSpecifiedFile ( class GWindow& parentWin, const GString& path );

   private:

      GVfs::File* doIt ();
      GVfs::File* prepareIt ( class GWindow& parentWin );
      virtual void runTheWorkerThread ( class GWorkerThread& worker );
      virtual void onWorkerThreadInitDialog ( class GWorkerThread& worker, class GDialogPanel& monitor );
      virtual void onWorkerThreadCommand ( class GWorkerThread& worker, class GDialogPanel& monitor, const GString& cmd );
      virtual void onWorkerThreadUpdateMonitor ( class GWorkerThread& worker, class GDialogPanel& monitor );
};

#endif
